/*******************************************************************
	lex.c : Lexical Analizer

			Coded by Shigeru Hitomi  Mar, 8, 1992
*******************************************************************/

/*******************************************************************
	INCLUDE FILES
*******************************************************************/
#include "stream.h"		/* include stdio.h */
#include <ctype.h>
#include <string.h>
#include <math.h>

#include "defs.h"
#include "prototype.h"
#include "operator.h"
#include "y.tab.h"
#include "module.h"

/*******************************************************************
	DIFINITIONS
*******************************************************************/
#define	NaN	(-1)		/* Not a Number */
#define Return(type)	{top = FALSE; return (type);}

#ifndef HAVE_STRCHR
#define	strchr(s,c)	index(s,c)
extern char    *index();	/* 'strchr()' in some systems */
#endif


/*******************************************************************
	PRIVATE PROTOTYPES
*******************************************************************/
static int   number        _ANSI_ARGS_((void));
static int   token         _ANSI_ARGS_((void));
static int   isunix        _ANSI_ARGS_((char *sbuf));
static int   quoted_string _ANSI_ARGS_((int  quote));
static char *getline       _ANSI_ARGS_((char *str,  int sepa));
static int   follow        _ANSI_ARGS_((int expect, int ifyes, int ifno));
static int   backslash     _ANSI_ARGS_((int c));

/*******************************************************************
	GLOBAL VARIABLES
*******************************************************************/
extern int      lineno;		        /* current line No.      */

int             c;	        	/* global for use by warning() */
BOOLEAN         top = TRUE;	        /* line top              */

char            stream[MAXBUF];    	/* stream buffer         */
char           *streamptr = stream;	/* stream buffer pointer */

/*******************************************************************
	Lexical Analizer
*******************************************************************/

static int
number()
{
  extern Object   Scalar;
  double          d;
  char            sbuf[100], *getnum(), *next;
  int             len;

  UNGETC();		/* '.' or digit return to stream */

  next = getnum(streamptr);	/* (*next) is not a number */
  len = next - streamptr;	/* length of number's string */
  if (len == 0)
    return NaN;	/* Not a Number */

  sbuf[len] = '\0';
  strncpy(sbuf, streamptr, len);	/* number's strings into sbuf */

  sscanf(sbuf, "%lf", &d);/* string to number */
  streamptr = next;	/* set next pointer */

  yylval.sym = install(" ", NUMBER, &d, &Scalar);
  return NUMBER;
}


static int
isunix(sbuf)
     char  *sbuf;
{
  SPACE_SKIP();
  UNGETC();

  switch (c) {
  case '=' or '[' or ':':
    return FALSE;
  case '<':
    return (LOOKUP() == '<') ? FALSE : TRUE;
  default:
    return executable(sbuf);
  }
}

/* token */
static int
token()
{
  char           *prog_name, *execname();
  char            sbuf[ONELINE];
  Symbol         *s, *install();

  UNGETC();
  streamptr = GetToken(streamptr, sbuf, sizeof(sbuf));

  prog_name = execname(sbuf);

  if (prog_name != NULL) {
    char           *ln, *keep = streamptr;
    while (GETC() != '(')
      /* nothing */ ;
    ln = getline(prog_name, ',');
    /* elimination of  ')' */
    ln[strlen(ln) - 1] = '\0';
    yylval.sym = install(" ", STRING, &ln, &String);
    efree(ln);
    efree(prog_name);
    streamptr = keep;
    return SAT_COM;
  }
  /* look up symbol table */
  s = lookup(sbuf);

  if (top) {
    char            sepa = LOOKUP();
    if (s == 0 && isunix(sbuf)) {	/* UNIX command ? */
      char           *ln;
      sepa = (sepa != c) ? sepa : '\0';
      ln = getline(sbuf, sepa);
      yylval.sym = install(" ", STRING, &ln, &String);
      efree(ln);
      return COMMAND;
    }
  }

  /* VAR */
  if (s == 0) {		/* undefined variable */
    s = install(sbuf, UNDEF, NULL, NULL);
  }
  yylval.sym = s;		/* for parser */

  return s->type == UNDEF ? VAR : s->type;
}


/* quoted string */
static int
quoted_string(quote)
     int  quote;
{
  extern Object   String;
  char            sbuf[ONELINE], *p, *argv[1];
  for (p = sbuf; (c = GETC()) != quote; p++) {
    if (c == '\n' || c == EOF)
      execerror("missing quote", " ");
    if (p >= sbuf + sizeof(sbuf) - 1) {
      *p = '\0';
      execerror("string too long", sbuf);
    }
    *p = backslash(c);
  }
  *p = '\0';
  argv[0] = sbuf;
  yylval.sym = install(" ", STRING, argv, &String);
  return STRING;
}


static char   *
getline(str, sepa)
     char   *str;
     int     sepa;
{
  /* command line */
  char            sbuf[ONELINE], *p;
  char           *sym;

  for (p = sbuf;
       (c = GETC()) != '\n' && c != '\0' 
	 /* && c != '}' && c != ';' && c != '&'*/;
       p++) {
    if (p >= sbuf + sizeof(sbuf) - 1) {
      *p = '\0';
      execerror("string too long", sbuf);
    }
    *p = backslash(c);
  }
  UNGETC();		/* for newline */
  *p = '\0';

  /* size = str + sepa + sbuf + '\0' */
  sym = emalloc(strlen(str) + strlen(sbuf) + 2);
  if (str == NULL)
    str = "";
  if (sepa != '\0')
    sprintf(sym, "%s%c%s", str, sepa, sbuf);
  else
    sprintf(sym, "%s%s", str, sbuf);

  return sym;
}

static int
follow(expect, ifyes, ifno)
{	/* look ahead for >=, etc. */
  int             c = GETC();

  if (c == expect) {
    return ifyes;
  }
  UNGETC();
  return ifno;
}



static int
backslash(c)			/* get next char with \'s interpreted */
     int  c;
{
  static char     transtab[] = "b\bf\fn\nr\rt\t";

  if (c != '\\')
    return c;

  c = GETC();
  if (islower(c) && strchr(transtab, c))
    return strchr(transtab, c)[1];
  return c;
}


int
yylex()
{				/* lexical analizer */
  char           *ln;

  SPACE_SKIP();

  /* number */
  if (c == '.' || isdigit(c)) {
    if (number() != NaN)
      Return(NUMBER);
  }
  /* token */
  if (isalpha(c)) {
    int type = token();
    Return(type);	/* Don't write "Return(token());" */
  }
  if (top && c == '/') {	/* cf. /usr/local/bin/tcsh */
    ln = getline("", c);
    yylval.sym = install(" ", STRING, &ln, &String);
    efree(ln);
    Return(COMMAND);
  }

  top = FALSE;
  /* symbols */
  switch (c) {
  case EOF:
    return 0;	/* end of file  */
  case '\\':
    ln = getline("", '\0');
    yylval.sym = install(" ", STRING, &ln, &String);
    efree(ln);
    return COMMAND;
  case '"':
    return quoted_string('"');
  case '`':
    quoted_string('`');
    return UNIX_COM;
    /*
      case '[':
      return INDEX;
      case ':':
      return follow('[', SP, ':');
      */

    /* relative operator */
  case '>':
    return follow('>', 1, 0) ? RPIPE : follow('=', GE, GT);
  case '<':
    return follow('<', 1, 0) ? LPIPE : follow('=', LE, LT);
  case '=':
    return follow('=', EQ, '=');
  case '!':
    return follow('=', NE, NOT);
  case '|':
    return follow('|', OR, '|');
  case '&':
    return follow('&', AND, '&');

    /* assginment operator and '++' , '--' */
  case '+':
    yylval.inst = (Inst *) add;
    return follow('=', 1, 0) ? ASGN_OP : follow('+', INCDEC, '+');
  case '-':
    yylval.inst = (Inst *) sub;
    return follow('=', 1, 0) ? ASGN_OP : follow('-', INCDEC, '-');
  case '*':
    yylval.inst = (Inst *) mul;
    return follow('=', ASGN_OP, '*');
  case '/':
    yylval.inst = (Inst *) Div;
    return follow('=', ASGN_OP, '/');
  case '%':
    yylval.inst = (Inst *) mod;
    return follow('=', ASGN_OP, '%');
  case '^':
    yylval.inst = (Inst *) tpow;
    return follow('=', ASGN_OP, '^');


  case '\n':
/*    lineno++; */
    return '\n';
  case '\0':
    return '\n';
  case ';':
    return ';';
  default:
    return c;
  }
}

/******************************************************************
	End of Lexical Analizer
*******************************************************************/
