#include <stdio.h>
#include <string.h>
#include <signal.h>
#include "defs.h"

#include "prototype.h"
#include "y.tab.h"

static Symbol *symlist = 0;	/* symbol table: linked list */
static Acc        *acc = 0;
int           freeCell = 0;


extern BOOLEAN  indef;
/*--------------------------------------------------------------------*/


Symbol         *
GetSymbolTable()
{
  return symlist;
}

void
PutSymbolTable(tableEntry)
     Symbol         *tableEntry;
{
  symlist = tableEntry;
}


Symbol *
lookup(s)		/* find s in symbol table */
     char           *s;
{
  register Symbol *sp;
  for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
    if ( indef && sp->type == VAR)
      continue;
    if (equal(sp->name, s))
      return sp;
  }
  return 0;			/* 0 ==> not found */
}

Symbol  *
install(s, t, val, super_class)	/* install s in symbol table */
     char           *s;	/* key-name in symbol table  */
     int             t;	/* union members type        */
     void           *val;	/* value                     */
     Object         *super_class;	/* super class object */
{
  register Symbol *sp;

  sp = (Symbol *) emalloc(sizeof(Symbol));
  sp->name = emalloc(strlen(s) + 1); /* +1 for '\0' */
  strcpy(sp->name, s);
  sp->type = t;
  sp->onstack = -1;		/* not on stack */
  sp->obj = NULL;
  if (t == UNDEF || equal(s, " ")) {
    freeCell++;
  }
  if (super_class != NULL) {
    sp->obj = (*super_class->method->new) ();
    (*super_class->method->setvalue) (sp->obj, val, 0, NULL);
  }
  sp->next = symlist;		/* put at front of list */
  symlist = sp;

#ifdef DEBUG
  fprintf(stderr, "install(%s, %lx)\n", s, (unsigned long)sp);
#endif

  return sp;
}


/*--------------------------------------------------------------------*/

static Acc  *
regist_acc(super_class)
     Object         *super_class;
{
  Acc   *ap;
  int    type = TypeofOBJ(super_class);
  int    type2;

  /* Search for target object type */
  for (ap = acc; ap != 0; ap = ap->next) {
    type2 = TypeofOBJ(ap->obj);
    if (ap->obj->link == 0 && type2 == type)
      break;
  }
  if (ap == 0) {		/* no entry in acc list */
    ap = (Acc *) emalloc(sizeof(Acc));
    ap->obj = (Object *) emalloc(sizeof(Object));
    ap->obj = (*super_class->method->new) ();
    ap->next = acc;		/* put at front of list */
    acc = ap;
  }

  return ap;
}

Object  *
newacc(val, dim, index, super_class)	/* install s in symbol table */
     void           *val;	        /* value */
     int             dim, *index;	/* size of val */
     Object         *super_class;	/* super_class */
{
  Acc  *ap;

  ap = regist_acc(super_class);
  if (val != 0)
    /*    (*super_class->method->setvalue)(ap->obj, val, dim, index);*/
    (super_class->method->setvalue)(ap->obj, val, dim, index);

  return ap->obj;
}


Object  *
cpyacc(super_class)		/* install s in symbol table */
     Object         *super_class;	/* super_class */
{
  Acc  *ap = regist_acc(super_class);
  (*super_class->method->copy) (ap->obj, super_class);
  return ap->obj;
}


static void
freeAcc()
{
  register Acc   *ap = acc, *this;
  acc = 0;			/* end of acc-list */
#ifdef DEBUG
  fprintf(stderr,"freeAcc("), fflush(stderr);
#endif
  while (ap != 0) {
    this = ap;			/* free target */
    ap = ap->next;		/* next pointer */
#ifdef DEBUG
    fprintf(stderr,"[%lx]", (unsigned long)this), fflush(stderr);
#endif
    (*this->obj->method->destroy) (this->obj);
    efree((char*)this);
  }
#ifdef DEBUG
  fprintf(stderr,")"), fflush(stderr);
#endif
}


Object  *
BufftoObj(id, super_class)
     int             id;	/* Buffer ID */
     Object         *super_class;	/* Snapshot or Series */
{
  Object *obj;
  int             dim, index[MAX_INDEX];
  Buffer         *buf = ReadBuffer(id, &dim, index);
  if (buf == NULL)
    execerror("illegal return value", "");
  obj = newacc(buf, dim, index, super_class);
  FreeBuffer(buf);
  DestroyBuffer(id);
  return obj;
}

/*--------------------------------------------------------------------*/

#if 0
void
listup()
{				/* list up symbol table */
  register Symbol *sp;

  hrule("-");
  for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
#ifdef LISTUP_ALLOBJ
    xprintf("[%s]", sp->name);
#else
    if (!equal(sp->name, " "))
      xprintf("[%s]", sp->name );
#endif
  }
  xprintf("\n");
  hrule("-");
}
#else
void
listup()
{				/* list up symbol table */
  register Symbol *sp;
  int i,j, flag;
  static char *excepts[] = {"LittleEndian", "BigEndian", "+1", NULL};
  static struct { int no; char *name; } types[] = {
    {SCALAR_T,   "scalar"},   {SERIES_T, "series"},
    {SNAPSHOT_T, "snapshot"}, {STRING_T, "string"},
    {0, NULL}
  };
  static struct { int no; char *name; } classes[] = {
    {FUNCTION, "functions"}, {PROCEDURE, "procedures"}, {0, NULL}
  };

  flag = 0;
  for (i = 0; types[i].name != NULL; i++) {
    for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
      if (!equal(sp->name, " ") && sp->type == VAR ) {
	if (sp->obj != NULL && TypeofOBJ(sp->obj) == types[i].no &&
	    !equal(sp->name, types[i].name)) {
	  if (flag == 0) {
	    xprintf("Defined %s objects:\n", types[i].name); 
	    hrule("-");
	    flag = 1;
	  }
	  xprintf("%s", sp->name );
	  if ( sp->obj->dim > 0 ) {
	    if (!(sp->obj->dim == 1 && sp->obj->index[0] == 1 ))
	      for (j = 0; j < sp->obj->dim && j < MAX_INDEX; j++)
		xprintf("[%d]", sp->obj->index[j] ); 
	  }
	  xprintf("  "); 
	}
      }
    }
    if (flag == 1) {
      flag = 0;
      xprintf("\n\n"); 
    }
  }

  /* Functions / Procedures */
  for (i = 0; classes[i].name != NULL; i++) {
    for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
      if (!equal(sp->name, " ") && sp->type == classes[i].no ) {
	if (flag == 0) {
	  xprintf("Defined %s:\n", classes[i].name);
	  hrule("-");
	  flag = 1;
	}
	xprintf("%s  ", sp->name );
      }
    }
    if (flag == 1) {
      flag = 0;
      xprintf("\n\n"); 
    }
  }

  /* Constants */
  for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
    if (!equal(sp->name, " ") && sp->type == CONSTANT) {
      for (j = 0; excepts[j] != NULL; j++ ) {
	if (equal(sp->name, excepts[j]))
	  break;
      }
      if (excepts[j] == NULL) {
	if (flag==0) {
	  xprintf("Defined constants:\n"); 
	  hrule("-");
	  flag = 1;
	}
	xprintf("%s  ", sp->name );
      }
    }
  }
  if (flag == 1) {
    flag = 0;
    xprintf("\n\n"); 
  }

#ifdef DEBUG
  /* Undefined Objects */
  hrule("-");
  for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
    if (!equal(sp->name, " ") && sp->type == UNDEF ) {
      if (flag==0) {
	xprintf("Undefined symbols:\n"); 
	hrule("-");
	flag = 1;
      }
      xprintf("%s  ", sp->name );
    }
  }
  if (flag == 1) {
    flag = 0;
    xprintf("\n\n"); 
  }
#endif

  /* Modules */
  for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
    if (!equal(sp->name, " ") && sp->type == MODULE_NAME ) {
      if (flag==0) {
	xprintf("Defined command modules:\n"); 
	hrule("-");
	flag = 1;
      }
      xprintf("%s  ", sp->name );
    }
  }
  if (flag == 1) {
    flag = 0;
    xprintf("\n\n"); 
  }
}
#endif

static BOOLEAN
free_ok(this, name)
     register Symbol *this;
     char           *name;
{
  Object         *obj;

#if 0  
  if (incode2(this))	/* is this used other place except in undef_var? */
    return FALSE;
#endif

  if (this->type == UNDEF && !incode(this))	/* undefined  symbol */
    return TRUE;

  if ( name == NULL )		/* fail safe by take */
    return FALSE;

  if (!equal(this->name, name))	/* not target symbol */
    return FALSE;

  obj = getobj(this);
  if (obj != NULL && obj->link != 0) /* linking! (on the stack) */
    return FALSE;

  if (equal(name, " ") && incode(this))	/* linking! (in the code) */
    return FALSE;

  return TRUE;			/* free symbol */
}



int
undef(name)
     char           *name;
{
  register int    n = 0;
  register Symbol *sp = symlist, *pre = NULL, *this;

#ifdef DEBUG
  fprintf(stderr, "undef("), fflush(stderr);
#endif

  while (sp != NULL) {
    this = sp;			/* this is current symbol */
    sp = sp->next;		/* point to next symbol   */

    if (free_ok(this, name)) {
      Object         *obj = getobj(this);

#ifdef DEBUG
      fprintf(stderr, "(");
      if ( this->name != NULL && !equal(this->name, " "))
	fprintf(stderr, "%s,", this->name);
      fprintf(stderr, "%lx)",(unsigned long)this);
      fflush(stderr);
#endif

      if (obj != NULL)
	(*obj->method->destroy) (obj);
      if (incode2(this)) {
#ifdef DEBUG
	fprintf(stderr, "->UNDEF");
	fflush(stderr);
#endif
	this->type = UNDEF;
	this->obj  = NULL;
	pre = this;
	continue; /* not free */
      } else {
	efree(this->name);
	efree((char*)this);
	if (pre == NULL)
	  symlist = sp;
	else
	  pre->next = sp;
	n++;
      }
    } else {
      pre = this;
#ifdef DEBUG
      fprintf(stderr, "<");
      if ( this->name != NULL && !equal(this->name, " "))
	fprintf(stderr, "%s,", this->name);
      fprintf(stderr, "%lx>",(unsigned long)this);
      fflush(stderr);
#endif
    }
  }
#ifdef DEBUG
  fprintf(stderr, ")\n"), fflush(stderr);
#endif
  return n;
}


void
FreeSymbols()
{
  RETSIGTYPE (*istat)() = (RETSIGTYPE (*)()) signal(SIGINT, SIG_IGN);

#ifdef DEBUG
  int n;

  n = undef(" ");
  freeCell -= n;
  fprintf(stderr, "\tundef-symbols-(%d), rest-cells-(%d)\n", n, freeCell);
#else
  freeCell -= undef(" ");
#endif

  freeCell = 0;
  freeAcc();			/* free accumulators */
#ifdef DEBUG

  fprintf(stderr, "\n");
#endif

  signal(SIGINT, istat);
}


char    *
symbol_name(p)
     Inst            p;
{
  Symbol         *sp, *sym = (Symbol *) p;
  static char     tmp[256], *question = "?";

  for (sp = symlist; sp != (Symbol *) 0; sp = sp->next) {
    if (sp == sym) {
      if (!equal(sp->name, " "))
	return sp->name;

      if (sp->obj == 0)
	return 0;

      switch (TypeofOBJ(sp->obj)) {
      case SERIES_T or SNAPSHOT_T:
	sprintf(tmp, "BUF%d", *(int *) sp->obj->val);
	return tmp;
      case SCALAR_T:
	sprintf(tmp, "%g", *(double *) sp->obj->val);
	return tmp;
      case STRING_T:
	sprintf(tmp, "\"%s\"", *(char **) sp->obj->val);
	return tmp;
      default:
	return 0;
      }
    }
  }
#ifdef __alpha
  if (-100 < (long int) p && (long int) p < 100) {
    sprintf(tmp, "(%ld)", (long int)p);
    return tmp;
  }
#else
  if (-100 < (int) p && (int) p < 100) {
    sprintf(tmp, "(%d)", (int)p);
    return tmp;
  }
#endif
  return question;
}


char    *
keyword_search(s, first)	/* find s in symbol table */
     char           *s;
     BOOLEAN         first;
{
  static Symbol  *follow;
  register Symbol *sp, *startpoint;

  if (first)
    startpoint = symlist;
  else
    startpoint = (follow == (Symbol *) 0) ? symlist : follow;

  for (sp = startpoint; sp != (Symbol *) 0; sp = sp->next) {
    if (strncmp(sp->name, s, strlen(s)) == 0) {
      follow = sp->next;
      return (char *) sp->name;
    }
  }
  follow = (Symbol *) 0;
  return 0;			/* 0 ==> not found */
}
