/*******************************************************************
  INCLUDE FILES
  *******************************************************************/
#include <ctype.h>
#include <signal.h>

#include "defs.h"
#include "prototype.h"
#include "stream.h"
#include "y.tab.h"
#include "code.h"

/*******************************************************************
  GLOBAL VARIABLES
  *******************************************************************/
extern char    *progname;	/* for error messages */

extern char    *infile;		/* input file name    */

extern BOOLEAN  indef;
extern int      c;		/* global for use by flush_stream()  */

extern char	errlog[];

extern int PID;                 /* take for intcatch */


/*******************************************************************
  PRIVATE PROTOTYPES
  *******************************************************************/
static void   assign_err   _ANSI_ARGS_((void));
static void   flush_stream _ANSI_ARGS_((void));
static void   ErrMessage   _ANSI_ARGS_((int  ch));

/*******************************************************************
  FUNCTIONS
  *******************************************************************/


/*
 * recovery() in main.c
 */

void
execerror(s, t)			/* recover from run-time error */
     char           *s, *t;
{
  warning(s, t);
  recovery();

  /* This statement not reached */
  fatal("longjmp returned");
}


void
defnonly(s)			/* warn if illegal definition */
     char           *s;
{
  if (!indef)
    execerror(s, "used outside definition");
}

void
yyerror(s)			/* called for yacc syntax error */
     char           *s;
{
  warning(s, (char *) 0);
  flush_stream();
  /* reset in-definition flag */
  if(indef) 
    endSub();

  *mainprogbase = STOP;		/* safety lock */
  pc = mainprogbase;

  if(fin != (FILE *)0)
    fseek(fin, 0L, 2);		/* flush rest of file */
}

void
fatal(msg)
     char           *msg;
{
  fprintf(stderr, "Fatal Error: %s\n", msg);
  die(1);
}


void
intcatch()
{				/* chatch interrupt signal */
#ifdef DEBUG
  fprintf(stderr, "intcatch()\n");
#endif

  signal(SIGINT, SIG_IGN);

  break_comment(FALSE);
  
  ClearDisplayBuffer();
  
  if(indef) 
    endSub();
  
  /* for MODULE/mdl.gets.c */
  if ( PID == 0 ) {
#ifdef CORE
    puts("I am Child in intcatch.");
#endif
    _exit(0);
  }
  
  if (push_console(TELL)) {
    popConsoleIO();
    move_to_eos();
  }
  fprintf(stderr,"\n");		/* newline */

  *mainprogbase = STOP;		/* safty lock */
  *mainprog = STOP;		/* safty lock */
  pc = mainprogbase;

  recovery();

  fatal("longjmp returned");
  return;
}


void
fpecatch()
{				/* chatch floating point exceptions */
  execerror("floating point exception", (char *) 0);
}

static void
assign_err()
{
  int             type = yylval.sym->type;
  xprintf("illegal assignment :");
  switch (type) {
  case NUMBER or STRING or CONSTANT:
    xprintf("constant value");
    break;
  case BLTIN or FUNCTION or PROCEDURE:
    xprintf("function/procedure");
    break;
  default:
    /* Keywors between WHILE and PRINT */
    if (WHILE <= type && type <= PRINT)
      xprintf("keyword");
    else
      xprintf("not valiable");
    break;
  }
}


static void
ErrMessage(ch)
     int   ch;
{
  xprintf("\n\t * * * ");	/* Begin of Error Message */
  switch (ch) {
  case '=':
    assign_err();
    break;
  default:
    /* already read buffer */
    insertc(streamptr, -1, '\0');
    xprintf("[%s", stream);

    /* the buffer still remaining to be read */
    rmnl(streamptr);		/* remove newline */
    xprintf("%c[7m%s%c[0m]", ESC, streamptr, ESC);
    break;
  }
  xprintf(" * * *\n");		/* End of Error Message */
}


static void
flush_stream()
{
  while (c != ';' && c != '\n' && c != EOF)
    c = GETC();			/* flush rest of input line */
  if (c == '\n')
    lineno++;
}


void
warning(s, t)			/* print warning message */
     char           *s, *t;
{
  static FILE    *errfp = 0;
  if (errfp == 0) {
    if((errfp = fopen(errlog, "w")) == 0)
      errfp = (FILE *) (-1);
  }
  xprintf("%s: %s", progname, s);
  if (t)
    xprintf(" %s", t);

  if (infile && strlen(infile) > 0)
    xprintf(" in %s", infile);

  xprintf(" near line %d", lineno);

  if (equal(s, "syntax error"))
    ErrMessage(c);
  else
    xprintf("\n");
  errflush(errfp);		/* flush error stream */
}

/******************************************************************
	End of FILE
*******************************************************************/
