/**
 * $Id: sc_hash.c,v 1.3 2004/06/21 16:44:17 jklein Exp $
 * $Author: jklein $
 * $Log: sc_hash.c,v $
 * Revision 1.3  2004/06/21 16:44:17  jklein
 * Added iterator.
 *
 */


#include "scale/hash.h"
#include "scale/alloc.h"


typedef struct _sc_hash_elt shash_elt_t;

struct _sc_hash_elt {
	spointer key;
	spointer obj;

	shash_elt_t *next;	/* this is for hash_elt which has same hash-key */
};

struct _sc_hash_table {
	shash_elt_t **table;
	size_t count;
	size_t hashsize;
	hashfunc_t hash;
	sc_compf_t cmp;
};


shash_t *
s_hash_alloc(size_t hsize, hashfunc_t hfunc, sc_compf_t cmpf)
{
	shash_t *htable;

	htable = s_malloc(sizeof(shash_t));
	htable->hashsize = (hsize != 0)?hsize:DEFAULT_HASH_SIZE;
	htable->table = s_calloc(htable->hashsize, sizeof(shash_elt_t *));
	htable->hash = hfunc;
	htable->cmp = cmpf;
	htable->count = 0;

	return htable;
}


spointer
s_hash_get(shash_t *htable, const spointer key)
{
	unsigned int hashkey;
	shash_elt_t *elt;

	hashkey = htable->hash(key, htable->hashsize);

	for(elt = htable->table[hashkey]; elt; elt=elt->next) {
		if(htable->cmp(elt->key, key) == 0)
			return elt->obj;
	}
	
	return NULL;
}


void
s_hash_put(shash_t *htable, spointer key, spointer obj)
{
	shash_elt_t *next, *elt;
	unsigned int hashkey;

	elt = s_malloc(sizeof(shash_elt_t));
	elt->key = key;
	elt->obj = obj;

	/* remove same key before adding new one. */
	s_hash_remove(htable, key);

	hashkey = htable->hash(key, htable->hashsize);

	next = htable->table[hashkey];
	htable->table[hashkey] = elt;
	htable->table[hashkey]->next = next;
	htable->count++;
}


/* This function doesn't remove object.
 * It only takes out from a hash-table.
 * So, you have to free it in you own reponsibility.
 */
void
s_hash_remove(shash_t *htable, const spointer key)
{
	shash_elt_t *elt, *prev;
	unsigned int hashkey;

	elt = prev = NULL;

	hashkey = htable->hash(key, htable->hashsize);

	for(elt = htable->table[hashkey]; elt; elt=elt->next) {
		if(htable->cmp(elt->key, key) == 0) {
			if(prev)
				prev->next = elt->next;
			else
				htable->table[hashkey] = elt->next;

			s_freen(elt, sizeof(shash_elt_t));

			htable->count--;
			break;
		}
		prev = elt;
	}
}


size_t
s_hash_size(shash_t *htable)
{
	return htable->count;
}


void 
s_hash_free(shash_t *htable)
{
	s_hash_clear(htable);
	s_freen(htable->table, sizeof(shash_elt_t *) * htable->hashsize);
	s_freen(htable, sizeof(shash_t));
}


void
s_hash_clear(shash_t *htable)
{
	shash_elt_t *elt;
	int i;

	for (i=0; i<htable->hashsize; i++) {
		
		if ((elt = htable->table[i]) != NULL) {
			s_freen(elt, sizeof(shash_elt_t));
			htable->table[i] = NULL;
		}
	}

	htable->count = 0;
}


/* 
 * Alfred V. Aho, Ravi Sethi, Jeffrey D. Ullman 
 * ĸإѥ飲 ˡġ
 * Ҽڡ
 * ºݤ˼¸ơХ󥹤ɤϥåͤ뤳Ȥǧ
 * 
 * char * to hashkey.
 */
unsigned int
s_string_hashfunc(const spointer key, size_t hashsize)
{
	unsigned int h, g;
	const char *p;

	h = g = 0;
	for (p=key; *p; p++) {
		h = (h << 4) + (*p);
		if ((g = h & 0xf0000000)) {
			h = h ^ (g >> 24);
			h = h ^ g;
		}
	}
	return h % hashsize;
}


/* for iterator */

typedef struct {
	shash_elt_t *next;
	size_t cindex;
	size_t done;
} hash_itr_manager_t;


static spointer
_s_hash_iterator_next(siterator_t *itr)
{
	hash_itr_manager_t *mng = (hash_itr_manager_t *)itr->data;
	shash_t *htable = (shash_t *)itr->listdata;
	spointer key;

	if (s_hash_size(htable) == mng->done)
		return NULL;

	key = mng->next->key;
	mng->next = mng->next->next;

	/* find next element */
	while (mng->next == NULL && mng->cindex < htable->hashsize) {
		mng->next = htable->table[++mng->cindex];
	}

	mng->done++;

	return key;
}


static void
_s_hash_iterator_clear(siterator_t *itr)
{
	hash_itr_manager_t *mng = (hash_itr_manager_t *)itr->data;
	shash_t *htable = (shash_t *)itr->listdata;

	mng->done = 0;
	mng->cindex = 0;

	mng->next = htable->table[mng->cindex];

	/* find first element */
	while (mng->next == NULL && mng->cindex < htable->hashsize) {
		mng->next = htable->table[++mng->cindex];
	}
}


static void
_s_hash_iterator_free(siterator_t *itr)
{
	s_freen(itr->data, sizeof(hash_itr_manager_t));
	s_freen(itr, sizeof(siterator_t));
}



siterator_t *
s_hash_keys(shash_t *htable)
{
	siterator_t *itr;

	itr = (siterator_t *)s_malloc(sizeof(siterator_t));
	itr->listdata = htable;
	itr->next  = _s_hash_iterator_next;
	itr->clear = _s_hash_iterator_clear;
	itr->free  = _s_hash_iterator_free;

	itr->data = (hash_itr_manager_t *)s_malloc(sizeof(hash_itr_manager_t));

	itr->clear(itr);

	return itr;
}
