/*
 * $Id: sc_vector.c,v 1.6 2004/06/26 06:29:20 jklein Exp $
 * $Author: jklein $
 * $Log: sc_vector.c,v $
 * Revision 1.6  2004/06/26 06:29:20  jklein
 * Added sort function and IO utilities.
 *
 * Revision 1.5  2004/06/24 15:16:50  jklein
 * sxarray_t was added for abstruction of collection structure.
 *
 * Revision 1.4  2004/06/23 12:59:01  jklein
 * Added scale/btree.h, sc_btree.c.
 *
 * Revision 1.3  2004/06/21 16:44:17  jklein
 * Added iterator.
 *
 */


#include "scale/alloc.h"
#include "scale/vector.h"


#define DEF_VECTOR_SIZE 64
#define DEF_GROUTH_RATIO 32


void
_s_vector_grouth(svector_t *vect, size_t tsize)
{
	if (vect->memsize >= tsize)
		return; /* nothiong to do. */

	vect->array = (spointer *)s_reallocn(vect->array,
										 vect->memsize,
										 sizeof(spointer)*tsize);
	vect->memsize = tsize;
}



svector_t *
s_vector_alloc(void)
{
	return s_vector_allocn(DEF_VECTOR_SIZE);
}


svector_t *		
s_vector_allocn(size_t n) 
{
	svector_t *vect;

	vect = typed_malloc(svector_t);
	vect->count = 0;
	vect->memsize = 0;
	vect->nextidx = 0;
	vect->array = NULL;

	_s_vector_grouth(vect, n);

	return vect;
}


void			
s_vector_free(svector_t *vect) 
{
	s_freen(vect->array, sizeof(spointer) * vect->memsize);
	s_freen(vect, sizeof(svector_t));
}


size_t
s_vector_size(svector_t *vect)
{
	return vect->count;
}


void
s_vector_insert_at(svector_t *vect, uint idx, const spointer contents)
{
	int i;

	if (vect->memsize == vect->nextidx)
		_s_vector_grouth(vect, vect->memsize + DEF_GROUTH_RATIO);

	for (i=vect->count; i>idx; i--) {
		vect->array[i] = vect->array[i-1];
	}

	vect->array[idx] = contents;

	++vect->count;
	++vect->nextidx;
}


void
s_vector_add(svector_t *vect, const spointer contents) 
{
	s_vector_insert_at(vect, vect->count, contents);
}


spointer		
s_vector_get(svector_t *vect, uint idx) 
{
	if (vect->count <= idx)
		return NULL;

	return vect->array[idx];
}


spointer		
s_vector_remove(svector_t *vect, uint idx) 
{
	spointer contents;
	sint i;

	if (vect->count <= idx)
		return NULL;

	contents = vect->array[idx];
	vect->array[idx] = NULL;

	/* need to slide all contents after 'idx'. */
	for (i=idx; i<vect->count-1; i++) {
		vect->array[i] = vect->array[i+1];
	}
	vect->array[vect->count-1] = NULL;
	--vect->count;
	--vect->nextidx;
	
	return contents;
}


sint32			
s_vector_find(svector_t *vect, sc_compf_t compare, spointer contents) 
{
	sint i;

	for (i=0; i<vect->count; i++) {
		if (compare(vect->array[i], contents) == 0)
			return i;
	}
	return -1;
}


sboolean		
s_vector_join(svector_t *dst, svector_t *src) 
{
	size_t newsize;
	sint i;

	newsize = dst->count + src->count;

	if (dst->memsize < newsize)
		_s_vector_grouth(dst, newsize);

	for (i=0; i<src->count; i++) {
		dst->array[dst->nextidx++] = src->array[i];
	}

	dst->count += src->count;

	return true;
}


static int
_s_vector_qsort_pivot(spointer list[], int low, int high, sc_compf_t cmpf)
{
	int k = low + 1;

	while (k <= high && cmpf(list[low], list[k]) == 0)
		k++;

	if (k > high)
		return -1;

	return (cmpf(list[low], list[k]) >= 0)?low:k;
}


static void
_s_vector_qsort(spointer list[], int low, int high, sc_compf_t cmpf)
{
	int p;

	/* done */
	if (low == high) return;

	if ((p = _s_vector_qsort_pivot(list, low, high, cmpf)) != -1) {
		spointer x = list[p];
		int i = low;
		int j = high;

		while (i <= j) {
			while (cmpf(list[i], x) < 0) ++i;
			while (cmpf(list[j], x) > 0) --j;

			/* swap */
			if (i <= j) {
				spointer tmp = list[i];
				list[i] = list[j];
				list[j] = tmp;
				i++, j--;
			}
		}
		_s_vector_qsort(list, low, i-1, cmpf);
		_s_vector_qsort(list, i, high, cmpf);
	}
}


void
s_vector_sort(svector_t *vect, sc_compf_t cmpf)
{
	_s_vector_qsort(vect->array, 0, s_vector_size(vect) - 1, cmpf);
}



/***************************************************************************
 * These functions are for siterator_t interface.
 ***************************************************************************/

typedef struct {
	size_t next_index;
} vector_itr_manager_t;


static spointer
_s_vector_iterator_next(siterator_t *itr)
{
	vector_itr_manager_t *mng = (vector_itr_manager_t *)itr->data;
	svector_t *vect = (svector_t *)itr->listdata;

	if (s_vector_size(vect) == mng->next_index)
		return NULL;

	return vect->array[mng->next_index++];
}


static void
_s_vector_iterator_clear(siterator_t *itr)
{
	vector_itr_manager_t *mng = (vector_itr_manager_t *)itr->data;
	mng->next_index = 0;
}


static void
_s_vector_iterator_free(siterator_t *itr)
{
	s_freen(itr->data, sizeof(vector_itr_manager_t));
	s_freen(itr, sizeof(siterator_t));
}


siterator_t *
s_vector_iterator(svector_t *vect)
{
	siterator_t *itr;

	itr = typed_malloc(siterator_t);
	itr->listdata = vect;
	itr->next = _s_vector_iterator_next;
	itr->clear = _s_vector_iterator_clear;
	itr->free = _s_vector_iterator_free;

	itr->data = typed_malloc(vector_itr_manager_t);

	itr->clear(itr);

	return itr;
}


/***************************************************************************
 * These functions are for sxarray_t interface.
 ***************************************************************************/

#define __XARR_GET_VECT(__xarr) ((svector_t *)__xarr->listdata)



static void
_s_vect_xarr_free(sxarray_t *xarr)
{
	svector_t *vect = __XARR_GET_VECT(xarr);

	if (xarr->is_origin)
		s_vector_free(vect);
	s_freen(xarr, sizeof(sxarray_t));
}


static size_t
_s_vect_xarr_size(sxarray_t *xarr)
{
	return s_vector_size(__XARR_GET_VECT(xarr));
}


static void
_s_vect_xarr_insert_at(sxarray_t *xarr, uint idx, spointer contents)
{
	s_vector_insert_at(__XARR_GET_VECT(xarr), idx, contents);
}


static void
_s_vect_xarr_add(sxarray_t *xarr, spointer contents)
{
	s_vector_add(__XARR_GET_VECT(xarr), contents);
}


static spointer
_s_vect_xarr_get(sxarray_t *xarr, size_t ind)
{
	return s_vector_get(__XARR_GET_VECT(xarr), ind);
}


static spointer
_s_vect_xarr_remove(sxarray_t *xarr, size_t ind)
{
	return s_vector_remove(__XARR_GET_VECT(xarr), ind);
}


static sint32
_s_vect_xarr_find(sxarray_t *xarr, sc_compf_t cmpf, spointer contents)
{
	return s_vector_find(__XARR_GET_VECT(xarr), cmpf, contents);
}


static siterator_t *
_s_vect_xarr_iterator(sxarray_t *xarr)
{
	return s_vector_iterator(__XARR_GET_VECT(xarr));
}


static void
_s_vect_xarr_sort(sxarray_t *xarr, sc_compf_t cmpf)
{
	s_vector_sort(__XARR_GET_VECT(xarr), cmpf);
}


static void
_s_vect_xarr_join(sxarray_t *dest, sxarray_t *src)
{
	svector_t *dvect = __XARR_GET_VECT(dest);
	size_t newsize, srcsize;
	uint i;

	srcsize = s_xarray_size(src);
	if ((newsize = dvect->count + srcsize) > dvect->memsize) {
		_s_vector_grouth(dvect, newsize);
	}

	for (i=0; i<srcsize; i++) {
		dvect->array[dvect->nextidx++] = s_xarray_get(src, i);
	}

	dvect->count += srcsize;
}


sxarray_t *
s_vector_xarray(void)
{
	svector_t *vect = s_vector_alloc();
	sxarray_t *xarr = s_vector_attach_xarray(vect);
	xarr->is_origin = true;
	return xarr;
}


sxarray_t *
s_vector_attach_xarray(svector_t *vect)
{
	sxarray_t *xarr;

	xarr = typed_malloc(sxarray_t);
	xarr->listdata = vect;
	xarr->free     = _s_vect_xarr_free;
	xarr->size     = _s_vect_xarr_size;
	xarr->add      = _s_vect_xarr_add;
	xarr->get      = _s_vect_xarr_get;
	xarr->remove   = _s_vect_xarr_remove;
	xarr->find	   = _s_vect_xarr_find;
	xarr->iterator = _s_vect_xarr_iterator;
	xarr->sort     = _s_vect_xarr_sort;
	xarr->join	   = _s_vect_xarr_join;
	xarr->insert_at = _s_vect_xarr_insert_at;
	
	xarr->is_origin = false;

	return xarr;
}

