
#include "scale/btree.h"
#include "scale/alloc.h"


static sbtree_node_t *
_s_node_alloc(spointer key, spointer contents)
{
	sbtree_node_t *node;

	node = typed_malloc(sbtree_node_t);
	node->key = key;
	node->contents = contents;
	node->lnode = node->rnode = NULL;

	return node;
}


sbtree_t *
s_btree_alloc(sc_compf_t compare)
{
	sbtree_t *btree;

	btree = typed_malloc(sbtree_t);
	btree->count = 0;
	btree->compare = compare;
	btree->head = _s_node_alloc(NULL, NULL);
	btree->end = _s_node_alloc(NULL, NULL);

	btree->head->lnode = btree->head->rnode = btree->end;
	btree->end->lnode = btree->end->rnode = btree->end;

	return btree;
}


static void
_s_btree_node_free(sbtree_node_t *node, sbtree_node_t *end)
{
	if (node == end)
		return;
		
	_s_btree_node_free(node->lnode, end);
	_s_btree_node_free(node->rnode, end);
	s_freen(node, sizeof(sbtree_node_t));
}


void	
s_btree_free(sbtree_t *btree)
{
	_s_btree_node_free(btree->head, btree->end);
	s_freen(btree->end, sizeof(sbtree_node_t));
	s_freen(btree, sizeof(sbtree_t));
}


size_t	
s_btree_size(const sbtree_t *btree)
{
	return btree->count;
}


void	
s_btree_add(sbtree_t *btree, spointer key, spointer conts)
{
	sbtree_node_t *cnode, *pnode; /* child and parent node */
	sboolean save_right = true;
	
	pnode = btree->head;
	cnode = btree->head->rnode;
	
	while (cnode != btree->end) {
		pnode = cnode;
		save_right = (btree->compare(key, pnode->key) > 0);
		cnode = (save_right)?pnode->rnode:pnode->lnode;
	}

	cnode = _s_node_alloc(key, conts);
	cnode->lnode = cnode->rnode = btree->end;

	if (save_right)
		pnode->rnode = cnode;
	else
		pnode->lnode = cnode;
	++btree->count;
}


spointer	
s_btree_get(sbtree_t *btree, spointer key)
{
	sint cmp;
	sbtree_node_t *node;

	btree->end->key = key;
	node = btree->head->rnode;
	
	while ((cmp = btree->compare(key, node->key)) != 0) {
		node = (cmp > 0)?node->rnode:node->lnode;
	}

	return node->contents;
}


spointer	
s_btree_remove(sbtree_t *btree, const spointer key)
{
	sint cmp;
	sbtree_node_t *pnode, *cnode, *tnode;
	spointer contents;
	sboolean save_right = true;

	btree->end->key = key;
	pnode = btree->head;
	cnode = btree->head->rnode;

	while ((cmp = btree->compare(key, cnode->key)) != 0) {
		pnode = cnode;
		cnode = (cmp > 0)?pnode->rnode:pnode->lnode;
		save_right = (cmp > 0);
	}

	contents = cnode->contents;
	tnode = cnode;

	if (cnode->rnode == btree->end) {
		cnode = cnode->lnode;
	}
	else if (cnode->rnode->lnode == btree->end) {
		cnode = cnode->rnode;
		cnode->lnode = tnode->lnode;
	}
	else {
		sbtree_node_t *nextnode; /* smallest node in 'cnode'. */
		nextnode = cnode->lnode;
		while (nextnode->lnode->lnode != btree->end)
			nextnode = nextnode->lnode;
		cnode = nextnode->lnode;
		nextnode->lnode = cnode->rnode;
		cnode->lnode = tnode->lnode;
		cnode->rnode = tnode->rnode;
	}

	s_freen(tnode, sizeof(sbtree_node_t));

	if (save_right)
		pnode->rnode = cnode;
	else
		pnode->lnode = cnode;

	--btree->count;
	return contents;
}


#include "scale/vector.h"
#include "scale/iterator.h"

/**
 * Left-First Search(lfs)$B$K$h$k%$%F%l!<%?9=B$BN!#(B
 * $BFsJ,LZ$G$N(Blfs$B$O$=$N9=B$>e!">:=g$KJB$V!#(B
 */
typedef struct {
	svector_t *nodes;
	size_t next_index;
} btree_lfs_itr_manager;



static spointer
_lfs_itr_next(siterator_t *itr)
{
	btree_lfs_itr_manager *mng;
	sbtree_node_t *node;
	
	mng = (btree_lfs_itr_manager *)itr->data;
	
	if (s_vector_size(mng->nodes) <= mng->next_index)
		return NULL;

	node = (sbtree_node_t *)s_vector_get(mng->nodes, mng->next_index++);

	return node->key;
}


static void
_lfs_itr_clear(siterator_t *itr)
{
	btree_lfs_itr_manager *mng;

	mng = (btree_lfs_itr_manager *)itr->data;

	mng->next_index = 0;
}

static void
_lfs_itr_free(siterator_t *itr)
{
	btree_lfs_itr_manager *mng;

	mng = (btree_lfs_itr_manager *)itr->data;

	s_vector_free(mng->nodes);
	s_freen(mng, sizeof(btree_lfs_itr_manager));
	s_freen(itr, sizeof(siterator_t));
}


static void
_lfs_add_node(svector_t *vect, sbtree_node_t *node, sbtree_node_t *end)
{
	if (node == end)
		return;

	_lfs_add_node(vect, node->lnode, end);
	s_vector_add(vect, node);
	_lfs_add_node(vect, node->rnode, end);
}

static svector_t *
_lfs_vector_alloc(sbtree_t *btree)
{
	svector_t *nodes;

	nodes = s_vector_allocn(s_btree_size(btree));
	_lfs_add_node(nodes, btree->head->rnode, btree->end);
	return nodes;
}


siterator_t *
s_btree_lfs_iterator(sbtree_t *btree)
{
	btree_lfs_itr_manager *mng;
	siterator_t *itr;

	mng = typed_malloc(btree_lfs_itr_manager);
	mng->nodes = _lfs_vector_alloc(btree);

	itr = typed_malloc(siterator_t);
	itr->data = mng;
	itr->next = _lfs_itr_next;
	itr->clear = _lfs_itr_clear;
	itr->free =  _lfs_itr_free;

	itr->clear(itr);

	return itr;
}
