/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *  Copyright (C) 2004-2005 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Based on scim-hangul.
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 */

#define Uses_SCIM_CONFIG_BASE
#define Uses_SCIM_EVENT

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <scim.h>
#include <gtk/scimkeyselection.h>
#include "scim_anthy_prefs.h"
#include "intl.h"

using namespace scim;

#define scim_module_init anthy_imengine_setup_LTX_scim_module_init
#define scim_module_exit anthy_imengine_setup_LTX_scim_module_exit

#define scim_setup_module_create_ui       anthy_imengine_setup_LTX_scim_setup_module_create_ui
#define scim_setup_module_get_category    anthy_imengine_setup_LTX_scim_setup_module_get_category
#define scim_setup_module_get_name        anthy_imengine_setup_LTX_scim_setup_module_get_name
#define scim_setup_module_get_description anthy_imengine_setup_LTX_scim_setup_module_get_description
#define scim_setup_module_load_config     anthy_imengine_setup_LTX_scim_setup_module_load_config
#define scim_setup_module_save_config     anthy_imengine_setup_LTX_scim_setup_module_save_config
#define scim_setup_module_query_changed   anthy_imengine_setup_LTX_scim_setup_module_query_changed

#define DATA_POINTER_KEY "scim-anthy::ConfigPointer"

static GtkWidget * create_setup_window ();
static void        load_config (const ConfigPointer &config);
static void        save_config (const ConfigPointer &config);
static bool        query_changed ();

// Module Interface.
extern "C" {
    void scim_module_init (void)
    {
        bindtextdomain (GETTEXT_PACKAGE, SCIM_ANTHY_LOCALEDIR);
        bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    }

    void scim_module_exit (void)
    {
    }

    GtkWidget * scim_setup_module_create_ui (void)
    {
        return create_setup_window ();
    }

    String scim_setup_module_get_category (void)
    {
        return String ("IMEngine");
    }

    String scim_setup_module_get_name (void)
    {
        return String (_("Anthy"));
    }

    String scim_setup_module_get_description (void)
    {
        return String (_("An Anthy IMEngine Module."));
    }

    void scim_setup_module_load_config (const ConfigPointer &config)
    {
        load_config (config);
    }

    void scim_setup_module_save_config (const ConfigPointer &config)
    {
        save_config (config);
    }

    bool scim_setup_module_query_changed ()
    {
        return query_changed ();
    }
} // extern "C"


// Internal data structure
struct BoolConfigData
{
    const char *key;
    bool        value;
    const char *label;
    const char *title;
    const char *tooltip;
    GtkWidget  *widget;
    bool        changed;
};

struct StringConfigData
{
    const char *key;
    String      value;
    const char *label;
    const char *title;
    const char *tooltip;
    GtkWidget  *widget;
    bool        changed;
};

struct KeyboardConfigPage
{
    const char       *label;
    StringConfigData *data;
};

struct ComboConfigCandidate
{
    const char *label;
    const char *data;
};

enum {
    COLUMN_LABEL = 0,
    COLUMN_VALUE = 1,
    COLUMN_DESC  = 2,
    COLUMN_DATA  = 3,
    N_COLUMNS    = 4,
};

// Internal data declaration.
static bool __have_changed = true;
static GtkWidget   * __widget_key_categories_menu = NULL;
static GtkWidget   * __widget_key_filter          = NULL;
static GtkWidget   * __widget_key_filter_button   = NULL;
static GtkWidget   * __widget_key_list_view       = NULL;
static GtkTooltips * __widget_tooltips            = NULL;

static BoolConfigData __config_bool_common [] =
{
    {
        SCIM_ANTHY_CONFIG_AUTO_CONVERT_ON_PERIOD,
        SCIM_ANTHY_CONFIG_AUTO_CONVERT_ON_PERIOD_DEFAULT,
        N_("Start _conversion on inputting a comma or a period."),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CLOSE_CAND_WIN_ON_SELECT,
        SCIM_ANTHY_CONFIG_CLOSE_CAND_WIN_ON_SELECT_DEFAULT,
        N_("Close candidate window when select a candidate _directly."),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_LEARN_ON_MANUAL_COMMIT,
        SCIM_ANTHY_CONFIG_LEARN_ON_MANUAL_COMMIT_DEFAULT,
        N_("Learn on _manual committing."),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_LEARN_ON_AUTO_COMMIT,
        SCIM_ANTHY_CONFIG_LEARN_ON_AUTO_COMMIT_DEFAULT,
        N_("Learn on a_uto committing."),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_ROMAJI_HALF_SYMBOL,
        SCIM_ANTHY_CONFIG_ROMAJI_HALF_SYMBOL_DEFAULT,
        N_("Use half-width characters for _symbols"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_ROMAJI_HALF_NUMBER,
        SCIM_ANTHY_CONFIG_ROMAJI_HALF_NUMBER_DEFAULT,
        N_("Use half-width characters for _numbers"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHOW_INPUT_MODE_LABEL,
        SCIM_ANTHY_CONFIG_SHOW_INPUT_MODE_LABEL_DEFAULT,
        N_("Show _input mode label"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHOW_TYPING_METHOD_LABEL,
        SCIM_ANTHY_CONFIG_SHOW_TYPING_METHOD_LABEL_DEFAULT,
        N_("Show _typing method label"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHOW_PERIOD_STYLE_LABEL,
        SCIM_ANTHY_CONFIG_SHOW_PERIOD_STYLE_LABEL_DEFAULT,
        N_("Show _period style label"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHOW_DICT_LABEL,
        SCIM_ANTHY_CONFIG_SHOW_DICT_LABEL_DEFAULT,
        N_("Show _dictionary menu label"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHOW_DICT_ADMIN_LABEL,
        SCIM_ANTHY_CONFIG_SHOW_DICT_ADMIN_LABEL_DEFAULT,
        N_("Show _edit dictionary label"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHOW_ADD_WORD_LABEL,
        SCIM_ANTHY_CONFIG_SHOW_ADD_WORD_LABEL_DEFAULT,
        N_("Show _add word label"),
        NULL,
        NULL,
        NULL,
        false,
    },
};
static unsigned int __config_bool_common_num = sizeof (__config_bool_common) / sizeof (BoolConfigData);

static StringConfigData __config_string_common [] =
{
    {
        SCIM_ANTHY_CONFIG_TYPING_METHOD,
        SCIM_ANTHY_CONFIG_TYPING_METHOD_DEFAULT,
        N_("Typing _method: "),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_PERIOD_STYLE,
        SCIM_ANTHY_CONFIG_PERIOD_STYLE_DEFAULT,
        N_("St_yle of comma and period: "),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SPACE_TYPE,
        SCIM_ANTHY_CONFIG_SPACE_TYPE_DEFAULT,
        N_("_Space type: "),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_TEN_KEY_TYPE,
        SCIM_ANTHY_CONFIG_TEN_KEY_TYPE_DEFAULT,
        N_("Input from _ten key: "),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_DICT_ADMIN_COMMAND,
        SCIM_ANTHY_CONFIG_DICT_ADMIN_COMMAND_DEFAULT,
        N_("_Edit dictionary command:"),
        NULL,
        NULL,
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_ADD_WORD_COMMAND,
        SCIM_ANTHY_CONFIG_ADD_WORD_COMMAND_DEFAULT,
        N_("_Add word command:"),
        NULL,
        NULL,
        NULL,
        false,
    },
};
static unsigned int __config_string_common_num = sizeof (__config_string_common) / sizeof (StringConfigData);

static StringConfigData __config_keyboards_edit [] =
{
    {
        SCIM_ANTHY_CONFIG_INSERT_SPACE_KEY,
        SCIM_ANTHY_CONFIG_INSERT_SPACE_KEY_DEFAULT,
        N_("Insert space"),
        N_("Select inserting space keys"),
        N_("The key events to insert a space. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_INSERT_ALT_SPACE_KEY,
        SCIM_ANTHY_CONFIG_INSERT_ALT_SPACE_KEY_DEFAULT,
        N_("Insert alternative space"),
        N_("Select inserting alternative space keys"),
        N_("The key events to insert a alternative space. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_INSERT_HALF_SPACE_KEY,
        SCIM_ANTHY_CONFIG_INSERT_HALF_SPACE_KEY_DEFAULT,
        N_("Insert half space"),
        N_("Select inserting half width space keys"),
        N_("The key events to insert a half width space. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_INSERT_WIDE_SPACE_KEY,
        SCIM_ANTHY_CONFIG_INSERT_WIDE_SPACE_KEY_DEFAULT,
        N_("Insert wide space"),
        N_("Select inserting wide space keys"),
        N_("The key events to insert a wide space. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_COMMIT_KEY,
        SCIM_ANTHY_CONFIG_COMMIT_KEY_DEFAULT,
        N_("Commit"),
        N_("Select commit keys"),
        N_("The key events to commit the preedit string. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CONVERT_KEY,
        SCIM_ANTHY_CONFIG_CONVERT_KEY_DEFAULT,
        N_("Convert"),
        N_("Select convert keys"),
        N_("The key events to convert the preedit string to kanji. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CANCEL_KEY,
        SCIM_ANTHY_CONFIG_CANCEL_KEY_DEFAULT,
        N_("Cancel"),
        N_("Select cancel keys"),
        N_("The key events to cancel preediting or converting. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_BACKSPACE_KEY,
        SCIM_ANTHY_CONFIG_BACKSPACE_KEY_DEFAULT,
        N_("Backspace"),
        N_("Select backspace keys"),
        N_("The key events to delete a character before caret. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_DELETE_KEY,
        SCIM_ANTHY_CONFIG_DELETE_KEY_DEFAULT,
        N_("Delete"),
        N_("Select delete keys"),
        N_("The key events to delete a character after caret. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_caret [] =
{
    {
        SCIM_ANTHY_CONFIG_MOVE_CARET_FIRST_KEY,
        SCIM_ANTHY_CONFIG_MOVE_CARET_FIRST_KEY_DEFAULT,
        N_("Move to first"),
        N_("Select move caret to first keys"),
        N_("The key events to move the caret to the first of preedit string. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_MOVE_CARET_LAST_KEY,
        SCIM_ANTHY_CONFIG_MOVE_CARET_LAST_KEY_DEFAULT,
        N_("Move to last"),
        N_("Select move caret to last keys"),
        N_("The key events to move the caret to the last of the preedit string. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_MOVE_CARET_FORWARD_KEY,
        SCIM_ANTHY_CONFIG_MOVE_CARET_FORWARD_KEY_DEFAULT,
        N_("Move to forward"),
        N_("Select move caret to forward keys"),
        N_("The key events to move the caret to forward. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_MOVE_CARET_BACKWARD_KEY,
        SCIM_ANTHY_CONFIG_MOVE_CARET_BACKWARD_KEY_DEFAULT,
        N_("Move to backward"),
        N_("Select move caret to backward keys"),
        N_("The key events to move the caret to backward. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_segments [] =
{
    {
        SCIM_ANTHY_CONFIG_SELECT_FIRST_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_SELECT_FIRST_SEGMENT_KEY_DEFAULT,
        N_("Select the first segment"),
        N_("Select keys to select the first segment"),
        N_("The key events to select the first segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_LAST_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_SELECT_LAST_SEGMENT_KEY_DEFAULT,
        N_("Select the last segment"),
        N_("Select keys to select the last segment"),
        N_("The key events to select the the last segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_NEXT_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_SELECT_NEXT_SEGMENT_KEY_DEFAULT,
        N_("Select the next segment"),
        N_("Select keys to select the next segment"),
        N_("The key events to select the next segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_PREV_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_SELECT_PREV_SEGMENT_KEY_DEFAULT,
        N_("Select the previous segment"),
        N_("Select keys to select the previous segment"),
        N_("The key events to select the previous segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SHRINK_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_SHRINK_SEGMENT_KEY_DEFAULT,
        N_("Shrink the segment"),
        N_("Select keys to shrink the segment"),
        N_("The key events to shrink the selected segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_EXPAND_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_EXPAND_SEGMENT_KEY_DEFAULT,
        N_("Expand the segment"),
        N_("Select keys to expand the segment"),
        N_("The key events to expand the selected segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_COMMIT_FIRST_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_COMMIT_FIRST_SEGMENT_KEY_DEFAULT,
        N_("Commit the first segment"),
        N_("Select keys to commit the first segment"),
        N_("The key events to commit the first segment. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_COMMIT_SELECTED_SEGMENT_KEY,
        SCIM_ANTHY_CONFIG_COMMIT_SELECTED_SEGMENT_KEY_DEFAULT,
        N_("Commit the selected segment"),
        N_("Select keys to commit the selected segment"),
        N_("The key events to commit the selected segment. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_candidates [] =
{
    {
        SCIM_ANTHY_CONFIG_SELECT_FIRST_CANDIDATE_KEY,
        SCIM_ANTHY_CONFIG_SELECT_FIRST_CANDIDATE_KEY_DEFAULT,
        N_("First candidate"),
        N_("Select the first candidate keys"),
        N_("The key events to select the first candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_LAST_CANDIDATE_KEY,
        SCIM_ANTHY_CONFIG_SELECT_LAST_CANDIDATE_KEY_DEFAULT,
        N_("Last candidate"),
        N_("Select the last candidate keys"),
        N_("The key events to the select last candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_NEXT_CANDIDATE_KEY,
        SCIM_ANTHY_CONFIG_SELECT_NEXT_CANDIDATE_KEY_DEFAULT,
        N_("Next candidate"),
        N_("Select the next candidate keys"),
        N_("The key events to select the next candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_PREV_CANDIDATE_KEY,
        SCIM_ANTHY_CONFIG_SELECT_PREV_CANDIDATE_KEY_DEFAULT,
        N_("Previous candidate"),
        N_("Select the previous candidate keys"),
        N_("The key events to select the previous candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CANDIDATES_PAGE_UP_KEY,
        SCIM_ANTHY_CONFIG_CANDIDATES_PAGE_UP_KEY_DEFAULT,
        N_("Page up"),
        N_("Select page up candidates keys"),
        N_("The key events to switch candidates page up. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CANDIDATES_PAGE_DOWN_KEY,
        SCIM_ANTHY_CONFIG_CANDIDATES_PAGE_DOWN_KEY_DEFAULT,
        N_("Page down"),
        N_("Select page down candidates keys"),
        N_("The key events to switch candidates page down. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_direct_select [] =
{
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_1_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_1_KEY_DEFAULT,
        N_("1st candidate"),
        N_("Select keys to select 1st candidate"),
        N_("The key events to select the 1st candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_2_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_2_KEY_DEFAULT,
        N_("2nd candidate"),
        N_("Select keys to select 2nd candidate"),
        N_("The key events to select the 2nd candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_3_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_3_KEY_DEFAULT,
        N_("3rd candidate"),
        N_("Select keys to select 3rd candidate"),
        N_("The key events to select the 3rd candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_4_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_4_KEY_DEFAULT,
        N_("4th candidate"),
        N_("Select keys to select 4th candidate"),
        N_("The key events to select the 4th candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_5_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_5_KEY_DEFAULT,
        N_("5th candidate"),
        N_("Select keys to select 5th candidate"),
        N_("The key events to select the 5th candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_6_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_6_KEY_DEFAULT,
        N_("6th candidate"),
        N_("Select keys to select 6th candidate"),
        N_("The key events to select the 6th candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_7_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_7_KEY_DEFAULT,
        N_("7th candidate"),
        N_("Select keys to select 7th candidate"),
        N_("The key events to select the 7th candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_8_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_8_KEY_DEFAULT,
        N_("8th candidate"),
        N_("Select keys to select 8th candidate"),
        N_("The key events to select the 8th candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_9_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_9_KEY_DEFAULT,
        N_("9th candidate"),
        N_("Select keys to select 9th candidate"),
        N_("The key events to select the 9th candidate. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_10_KEY,
        SCIM_ANTHY_CONFIG_SELECT_CANDIDATE_10_KEY_DEFAULT,
        N_("10th candidate"),
        N_("Select keys to select 10th candidate"),
        N_("The key events to select the 10th candidate. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_converting [] =
{
    {
        SCIM_ANTHY_CONFIG_CONV_TO_HIRAGANA_KEY,
        SCIM_ANTHY_CONFIG_CONV_TO_HIRAGANA_KEY_DEFAULT,
        N_("Convert to hiragana"),
        N_("Select keys to convert to hiragana"),
        N_("The key events to convert the preedit string to hiragana. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CONV_TO_KATAKANA_KEY,
        SCIM_ANTHY_CONFIG_CONV_TO_KATAKANA_KEY_DEFAULT,
        N_("Convert to katakana"),
        N_("Select keys to convert to katakana"),
        N_("The key events to convert the preedit string to katakana. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CONV_TO_HALF_KATAKANA_KEY,
        SCIM_ANTHY_CONFIG_CONV_TO_HALF_KATAKANA_KEY_DEFAULT,
        N_("Convert to half width"),
        N_("Select keys to convert to half width katakana"),
        N_("The key events to convert the preedit string to half width katakana. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CONV_TO_LATIN_KEY,
        SCIM_ANTHY_CONFIG_CONV_TO_LATIN_KEY_DEFAULT,
        N_("Convert to latin"),
        N_("Select keys to convert to latin"),
        N_("The key events to convert the preedit string to latin. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CONV_TO_WIDE_LATIN_KEY,
        SCIM_ANTHY_CONFIG_CONV_TO_WIDE_LATIN_KEY_DEFAULT,
        N_("Convert to wide latin"),
        N_("Select keys to convert to wide latin"),
        N_("The key events to convert the preedit string to wide latin. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_mode [] =
{
    {
        SCIM_ANTHY_CONFIG_CIRCLE_KANA_MODE_KEY,
        SCIM_ANTHY_CONFIG_CIRCLE_KANA_MODE_KEY_DEFAULT,
        N_("Circle kana mode"),
        N_("Select circle kana mode keys"),
        N_("The key events to circle kana mode. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_LATIN_MODE_KEY,
        SCIM_ANTHY_CONFIG_LATIN_MODE_KEY_DEFAULT,
        N_("Latin mode"),
        N_("Select latin mode keys"),
        N_("The key events to toggle latin mode. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_WIDE_LATIN_MODE_KEY,
        SCIM_ANTHY_CONFIG_WIDE_LATIN_MODE_KEY_DEFAULT,
        N_("Wide latin mode"),
        N_("Select wide latin mode keys"),
        N_("The key events to toggle wide latin mode. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_HIRAGANA_MODE_KEY,
        SCIM_ANTHY_CONFIG_HIRAGANA_MODE_KEY_DEFAULT,
        N_("Hiragana mode"),
        N_("Select hiragana mode keys"),
        N_("The key events to switch to hiragana mode. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_KATAKANA_MODE_KEY,
        SCIM_ANTHY_CONFIG_KATAKANA_MODE_KEY_DEFAULT,
        N_("Katakana mode"),
        N_("Select katakana mode keys"),
        N_("The key events to switch to katakana mode. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_CIRCLE_TYPING_METHOD_KEY,
        SCIM_ANTHY_CONFIG_CIRCLE_TYPING_METHOD_KEY_DEFAULT,
        N_("Circle typing method"),
        N_("Select circle typing method keys"),
        N_("The key events to circle typing method. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_dict [] =
{
    {
        SCIM_ANTHY_CONFIG_DICT_ADMIN_KEY,
        SCIM_ANTHY_CONFIG_DICT_ADMIN_KEY_DEFAULT,
        N_("Edit dictionary"),
        N_("Select edit dictionary keys"),
        N_("The key events to launch dictionary administration tool. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_ADD_WORD_KEY,
        SCIM_ANTHY_CONFIG_ADD_WORD_KEY_DEFAULT,
        N_("Add a word"),
        N_("Select add a word keys"),
        N_("The key events to launch the tool to add a word. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static StringConfigData __config_keyboards_reverse_learning [] =
{
    {
        SCIM_ANTHY_CONFIG_COMMIT_REVERSE_LEARN_KEY,
        SCIM_ANTHY_CONFIG_COMMIT_REVERSE_LEARN_KEY_DEFAULT,
        N_("_Commit:"),
        N_("Select commit keys"),
        N_("The key events to commit the preedit string "
           "with reversing the preference of learning. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_COMMIT_FIRST_SEGMENT_REVERSE_LEARN_KEY,
        SCIM_ANTHY_CONFIG_COMMIT_FIRST_SEGMENT_REVERSE_LEARN_KEY_DEFAULT,
        N_("Commit the _first segment:"),
        N_("Select keys to commit the first segment"),
        N_("The key events to commit the first segment "
           "with reversing the preference of learning. "),
        NULL,
        false,
    },
    {
        SCIM_ANTHY_CONFIG_COMMIT_SELECTED_SEGMENT_REVERSE_LEARN_KEY,
        SCIM_ANTHY_CONFIG_COMMIT_SELECTED_SEGMENT_REVERSE_LEARN_KEY_DEFAULT,
        N_("Commit the _selected segment:"),
        N_("Select keys to commit the selected segment"),
        N_("The key events to commit the selected segment "
           "with reversing the preference of learning. "),
        NULL,
        false,
    },
    {
        NULL,
        "",
        NULL,
        NULL,
        NULL,
        NULL,
        false,
    },
};

static struct KeyboardConfigPage __key_conf_pages[] =
{
    {N_("Mode keys"),          __config_keyboards_mode},
    {N_("Edit keys"),          __config_keyboards_edit},
    {N_("Caret keys"),         __config_keyboards_caret},
    {N_("Segments keys"),      __config_keyboards_segments},
    {N_("Candidates keys"),    __config_keyboards_candidates},
    {N_("Direct select keys"), __config_keyboards_direct_select},
    {N_("Convert keys"),       __config_keyboards_converting},
    {N_("Dictionary keys"),    __config_keyboards_dict},
};
static unsigned int __key_conf_pages_num = sizeof (__key_conf_pages) / sizeof (KeyboardConfigPage);

const int INDEX_SEARCH_BY_KEY = __key_conf_pages_num;
const int INDEX_ALL           = __key_conf_pages_num + 1;

static ComboConfigCandidate typing_methods[] =
{
    {N_("Romaji typing method"), "Roma"},
    {N_("Kana typing method"),   "Kana"},
    {NULL, NULL},
};

static ComboConfigCandidate period_styles[] =
{
    {"\xE3\x80\x81\xE3\x80\x82", "Japanese"},
    {"\xEF\xBC\x8C\xE3\x80\x82", "WideLatin_Japanese"},
    {"\xEF\xBC\x8C\xEF\xBC\x8E", "WideLatin"},
    {",.",                       "Latin"},
    {NULL, NULL},
};

static ComboConfigCandidate space_types[] =
{
    {N_("Wide"),              "Wide"},
    {N_("Half"),              "Half"},
    {N_("Follow input mode"), "FollowMode"},
    {NULL, NULL},
};

static ComboConfigCandidate ten_key_types[] =
{
    {N_("Wide"),              "Wide"},
    {N_("Half"),              "Half"},
    {N_("Follow input mode"), "FollowMode"},
    {NULL, NULL},
};


static void     on_default_editable_changed       (GtkEditable     *editable,
                                                   gpointer         user_data);
static void     on_default_toggle_button_toggled  (GtkToggleButton *togglebutton,
                                                   gpointer         user_data);
static void     on_default_key_selection_clicked  (GtkButton       *button,
                                                   gpointer         user_data);
static void     on_default_combo_changed          (GtkEditable     *editable,
                                                   gpointer         user_data);
static void     on_key_filter_selection_clicked   (GtkButton       *button,
                                                   gpointer         user_data);
static void     on_dict_menu_label_toggled        (GtkToggleButton *togglebutton,
                                                   gpointer         user_data);
static void     on_key_category_menu_changed      (GtkOptionMenu   *omenu,
                                                   gpointer         user_data);
static gboolean on_key_list_view_key_press        (GtkWidget       *widget,
                                                   GdkEventKey     *event,
                                                   gpointer         user_data);
static gboolean on_key_list_view_button_press     (GtkWidget       *widget,
                                                   GdkEventButton  *event,
                                                   gpointer         user_data);
static void     setup_widget_value ();


static BoolConfigData *
find_bool_config_entry (const char *config_key)
{
    if (!config_key)
        return NULL;

    for (unsigned int i = 0; i < __config_bool_common_num; i++) {
        BoolConfigData *entry = &__config_bool_common[i];
        if (entry->key && !strcmp (entry->key, config_key))
            return entry;
    }

    return NULL;
}

static StringConfigData *
find_string_config_entry (const char *config_key)
{
    if (!config_key)
        return NULL;

    for (unsigned int i = 0; i < __config_string_common_num; i++) {
        StringConfigData *entry = &__config_string_common[i];
        if (entry->key && !strcmp (entry->key, config_key))
            return entry;
    }

    return NULL;
}

#if 0
static StringConfigData *
find_key_config_entry (const char *config_key)
{
    for (unsigned int j = 0; j < __key_conf_pages_num; ++j) {
        for (unsigned int i = 0; __key_conf_pages[j].data[i].key; ++ i) {
            StringConfigData *entry = &__key_conf_pages[j].data[i];
            if (entry->key && !strcmp (entry->key, config_key))
                return entry;
        }
    }

    return NULL;
}
#endif

static bool
match_key_event (const KeyEventList &list, const KeyEvent &key)
{
    KeyEventList::const_iterator kit;

    for (kit = list.begin (); kit != list.end (); ++kit) {
        if (key.code == kit->code && key.mask == kit->mask)
             return true;
    }
    return false;
}

static GtkWidget *
create_check_button (const char *config_key)
{
    BoolConfigData *entry = find_bool_config_entry (config_key);
    if (!entry)
        return NULL;

    entry->widget = gtk_check_button_new_with_mnemonic (_(entry->label));
    gtk_container_set_border_width (GTK_CONTAINER (entry->widget), 4);
    g_signal_connect (G_OBJECT (entry->widget), "toggled",
                      G_CALLBACK (on_default_toggle_button_toggled),
                      entry);
    gtk_widget_show (entry->widget);

    if (!__widget_tooltips)
        __widget_tooltips = gtk_tooltips_new();
    if (entry->tooltip)
        gtk_tooltips_set_tip (__widget_tooltips, entry->widget,
                              _(entry->tooltip), NULL);

    return entry->widget;
}

#define APPEND_ENTRY(data, i)                                                  \
{                                                                              \
    label = gtk_label_new (NULL);                                              \
    gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _((data)->label));    \
    gtk_widget_show (label);                                                   \
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);                       \
    gtk_misc_set_padding (GTK_MISC (label), 4, 0);                             \
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, i, i+1,                  \
                      (GtkAttachOptions) (GTK_FILL),                           \
                      (GtkAttachOptions) (GTK_FILL), 4, 4);                    \
    (data)->widget = gtk_entry_new ();                                         \
    gtk_label_set_mnemonic_widget (GTK_LABEL (label), (data)->widget);         \
    g_signal_connect ((gpointer) (data)->widget, "changed",                    \
                      G_CALLBACK (on_default_editable_changed),                \
                      (data));                                                 \
    gtk_widget_show ((data)->widget);                                          \
    gtk_table_attach (GTK_TABLE (table), (data)->widget, 1, 2, i, i+1,         \
                      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),                \
                      (GtkAttachOptions) (GTK_FILL), 4, 4);                    \
                                                                               \
    if (!__widget_tooltips)                                                    \
        __widget_tooltips = gtk_tooltips_new();                                \
    if ((data)->tooltip)                                                       \
        gtk_tooltips_set_tip (__widget_tooltips, (data)->widget,               \
                              _((data)->tooltip), NULL);                       \
}

static GtkWidget *
create_combo (const char *config_key, gpointer candidates_p,
              GtkWidget *table, gint idx)
{
    StringConfigData *entry = find_string_config_entry (config_key);
    if (!entry)
        return NULL;

    GtkWidget *label;

    label = gtk_label_new_with_mnemonic (_(entry->label));
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
    gtk_misc_set_padding (GTK_MISC (label), 4, 0);
    gtk_table_attach (GTK_TABLE (table), label, 0, 1, idx, idx + 1,
                      (GtkAttachOptions) (GTK_FILL),
                      (GtkAttachOptions) (GTK_FILL), 4, 4);
    gtk_widget_show (label);

    entry->widget = gtk_combo_new ();
    gtk_label_set_mnemonic_widget (GTK_LABEL (label),
                                   GTK_COMBO (entry->widget)->entry);
    gtk_combo_set_value_in_list (GTK_COMBO (entry->widget), TRUE, FALSE);
    gtk_combo_set_case_sensitive (GTK_COMBO (entry->widget), TRUE);
    gtk_entry_set_editable (GTK_ENTRY (GTK_COMBO (entry->widget)->entry), FALSE);
    gtk_widget_show (entry->widget);
    gtk_table_attach (GTK_TABLE (table), entry->widget, 1, 2, idx, idx + 1,
                      (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
                      (GtkAttachOptions) (GTK_FILL), 4, 4);
    g_object_set_data (G_OBJECT (GTK_COMBO (entry->widget)->entry), DATA_POINTER_KEY,
                       (gpointer) candidates_p);

    g_signal_connect ((gpointer) GTK_COMBO (entry->widget)->entry, "changed",
                      G_CALLBACK (on_default_combo_changed),
                      entry);

    if (!__widget_tooltips)
        __widget_tooltips = gtk_tooltips_new();
    if (entry->tooltip)
        gtk_tooltips_set_tip (__widget_tooltips, entry->widget,
                              _(entry->tooltip), NULL);

    return entry->widget;
}

static void
append_key_bindings (GtkTreeView *treeview, gint idx, const gchar *filter)
{
    GtkListStore *store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));
    KeyEventList keys1, keys2;
    
    if (filter && *filter)
        scim_string_to_key_list (keys1, filter);

    if (idx < 0 || idx >= (gint) __key_conf_pages_num)
        return;

    for (unsigned int i = 0; __key_conf_pages[idx].data[i].key; i++) {
        if (filter && *filter) {
            scim_string_to_key_list (keys2, __key_conf_pages[idx].data[i].value.c_str());
            KeyEventList::const_iterator kit;
            bool found = true;
            for (kit = keys1.begin (); kit != keys1.end (); ++kit) {
                if (!match_key_event (keys2, *kit)) {
                    found = false;
                    break;
                }
            }
            if (!found)
                continue;
        }

        GtkTreeIter iter;
        gtk_list_store_append (store, &iter);
        gtk_list_store_set (store, &iter,
                            COLUMN_LABEL, _(__key_conf_pages[idx].data[i].label),
                            COLUMN_VALUE, __key_conf_pages[idx].data[i].value.c_str (),
                            COLUMN_DESC,  _(__key_conf_pages[idx].data[i].tooltip),
                            COLUMN_DATA, &__key_conf_pages[idx].data[i],
                            -1);
    }
}

static void
key_list_view_popup_key_selection (GtkTreeView *treeview)
{
    GtkTreeModel *model = gtk_tree_view_get_model (treeview);
    GtkTreePath *treepath = NULL;
    GtkTreeIter iter;

    gtk_tree_view_get_cursor (treeview, &treepath, NULL);
    if (!treepath) return;
    gtk_tree_model_get_iter (model, &iter, treepath);
    gtk_tree_path_free (treepath);

    StringConfigData *data;
    gtk_tree_model_get (model, &iter,
                        COLUMN_DATA, &data,
                        -1);
    if (data) {
        GtkWidget *dialog = scim_key_selection_dialog_new (_(data->title));
        gint result;

        scim_key_selection_dialog_set_keys
            (SCIM_KEY_SELECTION_DIALOG (dialog),
             data->value.c_str());

        result = gtk_dialog_run (GTK_DIALOG (dialog));

        if (result == GTK_RESPONSE_OK) {
            const gchar *keys = scim_key_selection_dialog_get_keys
                (SCIM_KEY_SELECTION_DIALOG (dialog));

            if (!keys) keys = "";

            if (strcmp (keys, data->value.c_str())) {
                data->value = keys;
                gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                                    COLUMN_VALUE, data->value.c_str(),
                                    -1);
                data->changed = true;
                __have_changed = true;
            }
        }

        gtk_widget_destroy (dialog);
    }
}

static GtkWidget *
create_common_page ()
{
    GtkWidget *vbox, *table, *widget;

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    table = gtk_table_new (4, 2, FALSE);
    gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
    gtk_widget_show (table);

    /* typing method */
    widget = create_combo (SCIM_ANTHY_CONFIG_TYPING_METHOD,
                           (gpointer) &typing_methods,
                           table, 0);

    /* period style */
    widget = create_combo (SCIM_ANTHY_CONFIG_PERIOD_STYLE,
                           (gpointer) &period_styles,
                           table, 1);

    /* space_style */
    widget = create_combo (SCIM_ANTHY_CONFIG_SPACE_TYPE,
                           (gpointer) &space_types,
                           table, 2);

    /* ten key_style */
    widget = create_combo (SCIM_ANTHY_CONFIG_TEN_KEY_TYPE,
                           (gpointer) &ten_key_types,
                           table, 3);

    /* auto convert */
    widget = create_check_button (SCIM_ANTHY_CONFIG_AUTO_CONVERT_ON_PERIOD);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    /* close candidate window on select */
    widget = create_check_button (SCIM_ANTHY_CONFIG_CLOSE_CAND_WIN_ON_SELECT);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    return vbox;
}

static GtkWidget *
create_romaji_page ()
{
    GtkWidget *vbox, *widget;

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    /* symbol */
    widget = create_check_button (SCIM_ANTHY_CONFIG_ROMAJI_HALF_SYMBOL);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    /* number */
    widget = create_check_button (SCIM_ANTHY_CONFIG_ROMAJI_HALF_NUMBER);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    return vbox;
}

static GtkWidget *
create_learning_page ()
{
    GtkWidget *vbox, *vbox2, *hbox, *alignment, *table;
    GtkWidget *widget, *label, *button;

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("<b>Enable/Disable learning</b>"));
    gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 24, 0);
    gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    vbox2 = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (alignment), vbox2);
    gtk_widget_show (vbox2);

    /* maual commit */
    widget = create_check_button (SCIM_ANTHY_CONFIG_LEARN_ON_MANUAL_COMMIT);
    gtk_box_pack_start (GTK_BOX (vbox2), widget, FALSE, FALSE, 4);

    /* auto commit */
    widget = create_check_button (SCIM_ANTHY_CONFIG_LEARN_ON_AUTO_COMMIT);
    gtk_box_pack_start (GTK_BOX (vbox2), widget, FALSE, FALSE, 4);

    /* key preference */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("<b>Key preferences to commit "
                             "with reversing learning preference</b>"));
    gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 24, 0);
    gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    table = gtk_table_new (3, 3, FALSE);
    gtk_container_add (GTK_CONTAINER (alignment), table);
    gtk_widget_show (table);

    for (unsigned int i = 0; __config_keyboards_reverse_learning[i].key; i++) {
        StringConfigData *entry = &__config_keyboards_reverse_learning[i];
        APPEND_ENTRY (entry, i);
        gtk_entry_set_editable (GTK_ENTRY (entry->widget), FALSE);
        button = gtk_button_new_with_label ("...");
        gtk_widget_show (button);
        gtk_table_attach (GTK_TABLE (table), button, 2, 3, i, i + 1,
                          GTK_FILL, GTK_FILL, 4, 4);
        gtk_label_set_mnemonic_widget (GTK_LABEL (label), button);
        g_signal_connect ((gpointer) button, "clicked",
                          G_CALLBACK (on_default_key_selection_clicked),
                          entry);
    }

    return vbox;
}

static GtkWidget *
create_toolbar_page ()
{
    GtkWidget *vbox, *hbox, *label, *widget;

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    /* show/hide toolbar label */
    widget = create_check_button (SCIM_ANTHY_CONFIG_SHOW_INPUT_MODE_LABEL);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    widget = create_check_button (SCIM_ANTHY_CONFIG_SHOW_TYPING_METHOD_LABEL);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    widget = create_check_button (SCIM_ANTHY_CONFIG_SHOW_PERIOD_STYLE_LABEL);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    /* dictionary menu */
    widget = create_check_button (SCIM_ANTHY_CONFIG_SHOW_DICT_LABEL);
    g_signal_connect ((gpointer) widget, "toggled",
                      G_CALLBACK (on_dict_menu_label_toggled),
                      NULL);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);
    gtk_widget_show (hbox);
    label = gtk_label_new ("    ");
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);
    widget = create_check_button (SCIM_ANTHY_CONFIG_SHOW_DICT_ADMIN_LABEL);
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 4);
    gtk_widget_show (hbox);
    label = gtk_label_new ("    ");
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);
    widget = create_check_button (SCIM_ANTHY_CONFIG_SHOW_ADD_WORD_LABEL);
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 0);

    // set initial state
    on_dict_menu_label_toggled (GTK_TOGGLE_BUTTON (widget), NULL);

    return vbox;
}

static GtkWidget *
create_dict_page (void)
{
    GtkWidget *table;
    GtkWidget *label;
    StringConfigData *entry;

    table = gtk_table_new (2, 2, FALSE);
    gtk_widget_show (table);

    // dict admin command
    entry = find_string_config_entry (SCIM_ANTHY_CONFIG_DICT_ADMIN_COMMAND);
    APPEND_ENTRY(entry, 0);

    // add word command
    entry = find_string_config_entry (SCIM_ANTHY_CONFIG_ADD_WORD_COMMAND);
    APPEND_ENTRY(entry, 1);

    return table;
}

static GtkWidget *
create_keyboard_page (void)
{
    GtkWidget *vbox, *hbox;

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 4);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    gtk_widget_show(hbox);

    // category menu
    GtkWidget *label = gtk_label_new_with_mnemonic (_("_Group:"));
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);
    gtk_widget_show (label);

    GtkWidget *omenu = gtk_option_menu_new ();
    __widget_key_categories_menu = omenu;
    gtk_box_pack_start (GTK_BOX (hbox), omenu, FALSE, FALSE, 2);
    gtk_widget_show (omenu);

    gtk_label_set_mnemonic_widget (GTK_LABEL (label), omenu);

    GtkWidget *menu = gtk_menu_new ();

    GtkWidget *menuitem;

    for (unsigned int i = 0; i < __key_conf_pages_num; i++) {
        menuitem = gtk_menu_item_new_with_label (_(__key_conf_pages[i].label));
        gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
        gtk_widget_show (menuitem);
    }

    menuitem = gtk_menu_item_new_with_label (_("Search by key"));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show (menuitem);

    menuitem = gtk_menu_item_new_with_label (_("all"));
    gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
    gtk_widget_show (menuitem);

    gtk_option_menu_set_menu (GTK_OPTION_MENU (omenu), menu);
    gtk_widget_show (menu);

    GtkWidget *entry = gtk_entry_new ();
    __widget_key_filter = entry;
    gtk_entry_set_editable (GTK_ENTRY (entry), FALSE);
    gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 2);
    gtk_widget_show(entry);

    GtkWidget *button = gtk_button_new_with_label ("...");
    __widget_key_filter_button = button;
    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (on_key_filter_selection_clicked), entry);
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 2);
    gtk_widget_show (button);

    // key bindings view
    GtkWidget *scrwin = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrwin),
                                         GTK_SHADOW_IN);
    gtk_container_set_border_width (GTK_CONTAINER (scrwin), 4);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scrwin),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start (GTK_BOX (vbox), scrwin, TRUE, TRUE, 2);
    gtk_widget_show (scrwin);

    GtkListStore *store = gtk_list_store_new (N_COLUMNS,
                                              G_TYPE_STRING,
                                              G_TYPE_STRING,
                                              G_TYPE_STRING,
                                              G_TYPE_POINTER);
    GtkWidget *treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
    __widget_key_list_view = treeview;
    gtk_container_add (GTK_CONTAINER (scrwin), treeview);
    gtk_widget_show (treeview);

    GtkCellRenderer *cell;
    GtkTreeViewColumn *column;
    cell = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Feature"), cell,
                                                       "text", COLUMN_LABEL,
                                                       NULL);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (column, 120);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

    cell = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Key bindings"), cell,
                                                       "text", COLUMN_VALUE,
                                                       NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

    cell = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (_("Description"), cell,
                                                       "text", COLUMN_DESC,
                                                       NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

    // connect signals
    g_signal_connect (G_OBJECT (omenu), "changed",
                      G_CALLBACK (on_key_category_menu_changed), treeview);
    g_signal_connect (G_OBJECT (treeview), "key-press-event",
                      G_CALLBACK (on_key_list_view_key_press), NULL);
    g_signal_connect (G_OBJECT (treeview), "button-press-event",
                      G_CALLBACK (on_key_list_view_button_press), NULL);

    return vbox;
}

static GtkWidget *
create_setup_window ()
{
    static GtkWidget *window = NULL;

    if (!window) {
        GtkWidget *notebook = gtk_notebook_new();
        gtk_notebook_popup_enable(GTK_NOTEBOOK(notebook));
        gtk_widget_show (notebook);
        window = notebook;
        gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);

        // Create the common page.
        GtkWidget *page = create_common_page ();
        GtkWidget *label = gtk_label_new (_("Common"));
        gtk_widget_show (label);
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

        // Create the key bind pages.
        page = create_keyboard_page ();
        label = gtk_label_new (_("Key bindings"));
        gtk_widget_show (label);
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

        // Create the romaji page.
        page = create_romaji_page ();
        label = gtk_label_new (_("Romaji Typing"));
        gtk_widget_show (label);
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

        // Create the learning page.
        page = create_learning_page ();
        label = gtk_label_new (_("Learning"));
        gtk_widget_show (label);
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

        // Create the toolbar page.
        page = create_toolbar_page ();
        label = gtk_label_new (_("Toolbar"));
        gtk_widget_show (label);
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

        // Create the dictionary page.
        page = create_dict_page ();
        label = gtk_label_new (_("Dictionary"));
        gtk_widget_show (label);
        gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

        // for preventing enabling left arrow.
        gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook), 1);
        gtk_notebook_set_current_page (GTK_NOTEBOOK (notebook), 0);

        setup_widget_value ();
    }

    return window;
}

static void
setup_combo_value (GtkCombo *combo, const String & str)
{
    GList *list = NULL;
    const char *defval = NULL;

    ComboConfigCandidate *data
        = static_cast<ComboConfigCandidate*>
        (g_object_get_data (G_OBJECT (GTK_COMBO(combo)->entry),
                            DATA_POINTER_KEY));

    for (unsigned int i = 0; data[i].label; i++) {
        list = g_list_append (list, (gpointer) _(data[i].label));
        if (!strcmp (data[i].data, str.c_str ()))
            defval = _(data[i].label);
    }

    gtk_combo_set_popdown_strings (combo, list);
    g_list_free (list);

    if (defval)
        gtk_entry_set_text (GTK_ENTRY (combo->entry), defval);
}

#include "scim_anthy_style_file.h"
static void
setup_widget_value ()
{
#if 0 // test for style file
    AnthyStyleFile file;
    file.load (SCIM_ANTHY_STYLEDIR "/msime.sty");
#endif

    for (unsigned int i = 0; i < __config_bool_common_num; i++) {
        BoolConfigData &entry = __config_bool_common[i];
        if (entry.widget)
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (entry.widget),
                                          entry.value);
    }

    for (unsigned int i = 0; i < __config_string_common_num; i++) {
        StringConfigData &entry = __config_string_common[i];
        if (entry.widget && GTK_IS_COMBO (entry.widget))
            setup_combo_value (GTK_COMBO (entry.widget), entry.value);
        else if (entry.widget && GTK_IS_ENTRY (entry.widget))
            gtk_entry_set_text (GTK_ENTRY (entry.widget),
                                entry.value.c_str ());
    }

    for (unsigned int j = 0; j < __key_conf_pages_num; ++j) {
        for (unsigned int i = 0; __key_conf_pages[j].data[i].key; ++ i) {
            if (__key_conf_pages[j].data[i].widget) {
                gtk_entry_set_text (
                    GTK_ENTRY (__key_conf_pages[j].data[i].widget),
                    __key_conf_pages[j].data[i].value.c_str ());
            }
        }
    }

    for (unsigned int i = 0; __config_keyboards_reverse_learning[i].key; i++) {
        if (__config_keyboards_reverse_learning[i].widget) {
            gtk_entry_set_text (
                GTK_ENTRY (__config_keyboards_reverse_learning[i].widget),
                __config_keyboards_reverse_learning[i].value.c_str ());
        }
    }

#if 0 // test for style file
    // clear all key bindings
    for (unsigned int j = 0; j < __key_conf_pages_num; ++j) {
        for (unsigned int i = 0; __key_conf_pages[j].data[i].key; ++ i)
            __key_conf_pages[j].data[i].value = "";
    }

    // reset key bindings
    AnthyStyleLines lines;
    AnthyStyleLines::iterator it;
    file.get_entry_list (lines, "KeyBindings");
    for (it = lines.begin (); it != lines.end (); it++) {
        if (it->get_type () != ANTHY_STYLE_LINE_KEY)
            continue;
        String key, fullkey;
        it->get_key (key);
        fullkey = String ("/IMEngine/Anthy/") + key;
        StringConfigData *entry = find_key_config_entry (fullkey.c_str ());
        if (entry)
            it->get_value (entry->value);
        else
            std::cerr << "No entry for : " << key << std::endl;
    }
#endif

    gtk_option_menu_set_history (GTK_OPTION_MENU (__widget_key_categories_menu), 0);
    gtk_widget_set_sensitive (__widget_key_filter, FALSE);
    gtk_widget_set_sensitive (__widget_key_filter_button, FALSE);
    GtkTreeModel *model;
    model = gtk_tree_view_get_model (GTK_TREE_VIEW (__widget_key_list_view));
    gtk_list_store_clear (GTK_LIST_STORE (model));
    append_key_bindings (GTK_TREE_VIEW (__widget_key_list_view), 0, NULL);
}

static void
load_config (const ConfigPointer &config)
{
    if (config.null ())
        return;

    for (unsigned int i = 0; i < __config_bool_common_num; i++) {
        BoolConfigData &entry = __config_bool_common[i];
        entry.value = config->read (String (entry.key), entry.value);
    }

    for (unsigned int i = 0; i < __config_string_common_num; i++) {
        StringConfigData &entry = __config_string_common[i];
        entry.value = config->read (String (entry.key), entry.value);
    }

    for (unsigned int j = 0; j < __key_conf_pages_num; ++ j) {
        for (unsigned int i = 0; __key_conf_pages[j].data[i].key; ++ i) {
            __key_conf_pages[j].data[i].value =
                config->read (String (__key_conf_pages[j].data[i].key),
                              __key_conf_pages[j].data[i].value);
        }
    }

    for (unsigned int i = 0; __config_keyboards_reverse_learning[i].key; i++) {
        StringConfigData &entry = __config_keyboards_reverse_learning[i];
        entry.value = config->read (String (entry.key), entry.value);
    }

    setup_widget_value ();

    for (unsigned int i = 0; i < __config_bool_common_num; i++)
        __config_bool_common[i].changed = false;

    for (unsigned int i = 0; i < __config_string_common_num; i++)
        __config_string_common[i].changed = false;

    for (unsigned int j = 0; j < __key_conf_pages_num; j++) {
        for (unsigned int i = 0; __key_conf_pages[j].data[i].key; ++ i)
            __key_conf_pages[j].data[i].changed = false;
    }

    for (unsigned int i = 0; __config_keyboards_reverse_learning[i].key; i++)
        __config_keyboards_reverse_learning[i].changed = false;

    __have_changed = false;
}

static void
save_config (const ConfigPointer &config)
{
    if (config.null ())
        return;

    for (unsigned int i = 0; i < __config_bool_common_num; i++) {
        BoolConfigData &entry = __config_bool_common[i];
        if (entry.changed)
            entry.value = config->write (String (entry.key), entry.value);
        entry.changed = false;
    }

    for (unsigned int i = 0; i < __config_string_common_num; i++) {
        StringConfigData &entry = __config_string_common[i];
        if (entry.changed)
            entry.value = config->write (String (entry.key), entry.value);
        entry.changed = false;
    }

    for (unsigned int j = 0; j < __key_conf_pages_num; j++) {
        for (unsigned int i = 0; __key_conf_pages[j].data[i].key; ++ i) {
            if (__key_conf_pages[j].data[i].changed)
                config->write (String (__key_conf_pages[j].data[i].key),
                               __key_conf_pages[j].data[i].value);
            __key_conf_pages[j].data[i].changed = false;
        }
    }

    for (unsigned int i = 0; __config_keyboards_reverse_learning[i].key; i++) {
        StringConfigData &entry = __config_keyboards_reverse_learning[i];
        if (entry.changed)
            entry.value = config->write (String (entry.key), entry.value);
        entry.changed = false;
    }

    __have_changed = false;
}

static bool
query_changed ()
{
    return __have_changed;
}


static void
on_default_toggle_button_toggled (GtkToggleButton *togglebutton,
                                  gpointer         user_data)
{
    BoolConfigData *entry = static_cast<BoolConfigData*> (user_data);

    if (entry) {
        entry->value = gtk_toggle_button_get_active (togglebutton);
        entry->changed = true;
        __have_changed = true;
    }
}

static void
on_default_editable_changed (GtkEditable *editable,
                             gpointer     user_data)
{
    StringConfigData *entry = static_cast <StringConfigData*> (user_data);

    if (entry) {
        entry->value = String (gtk_entry_get_text (GTK_ENTRY (editable)));
        entry->changed = true;
        __have_changed = true;
    }
}

static void
on_default_key_selection_clicked (GtkButton *button,
                                  gpointer   user_data)
{
    StringConfigData *data = static_cast <StringConfigData*> (user_data);

    if (data) {
        GtkWidget *dialog = scim_key_selection_dialog_new (_(data->title));
        gint result;

        scim_key_selection_dialog_set_keys (
            SCIM_KEY_SELECTION_DIALOG (dialog),
            gtk_entry_get_text (GTK_ENTRY (data->widget)));

        result = gtk_dialog_run (GTK_DIALOG (dialog));

        if (result == GTK_RESPONSE_OK) {
            const gchar *keys = scim_key_selection_dialog_get_keys (
                            SCIM_KEY_SELECTION_DIALOG (dialog));

            if (!keys) keys = "";

            if (strcmp (keys, gtk_entry_get_text (GTK_ENTRY (data->widget))))
                gtk_entry_set_text (GTK_ENTRY (data->widget), keys);
        }

        gtk_widget_destroy (dialog);
    }
}

static void
on_default_combo_changed (GtkEditable *editable,
                          gpointer user_data)
{
    StringConfigData *entry = static_cast<StringConfigData*> (user_data);
    ComboConfigCandidate *data = static_cast<ComboConfigCandidate*>
        (g_object_get_data (G_OBJECT (editable),
                            DATA_POINTER_KEY));

    if (!entry) return;
    if (!data) return;

    const char *label =  gtk_entry_get_text (GTK_ENTRY (editable));

    for (unsigned int i = 0; data[i].label; i++) {
        if (label && !strcmp (_(data[i].label), label)) {
            entry->value = data[i].data;
            entry->changed = true;
            __have_changed = true;
            break;
        }
    }
}

static void
on_dict_menu_label_toggled (GtkToggleButton *togglebutton,
                            gpointer         user_data)
{
    bool active = gtk_toggle_button_get_active (togglebutton);

    BoolConfigData *entry;
    entry = find_bool_config_entry (SCIM_ANTHY_CONFIG_SHOW_DICT_ADMIN_LABEL);
    if (entry->widget)
        gtk_widget_set_sensitive (entry->widget, active);
    entry = find_bool_config_entry (SCIM_ANTHY_CONFIG_SHOW_ADD_WORD_LABEL);
    if (entry->widget)
        gtk_widget_set_sensitive (entry->widget, active);
}

static void
on_key_category_menu_changed (GtkOptionMenu *omenu, gpointer user_data)
{
    GtkTreeView *treeview = GTK_TREE_VIEW (user_data);
    GtkListStore *store = GTK_LIST_STORE (gtk_tree_view_get_model (treeview));

    gtk_list_store_clear (store);

    gint idx = gtk_option_menu_get_history (omenu);

    bool use_filter = false;

    if (idx >= 0 && idx < (gint) __key_conf_pages_num) {
        append_key_bindings (treeview, idx, NULL);

    } else if (idx == INDEX_SEARCH_BY_KEY) {
        // search by key
        use_filter = true;
        const char *str = gtk_entry_get_text (GTK_ENTRY (__widget_key_filter));
        for (unsigned int i = 0; i < __key_conf_pages_num; i++)
            append_key_bindings (treeview, i, str);

    } else if (idx == INDEX_ALL) {
        // all
        for (unsigned int i = 0; i < __key_conf_pages_num; i++)
            append_key_bindings (treeview, i, NULL);
    }

    gtk_widget_set_sensitive (__widget_key_filter,        use_filter);
    gtk_widget_set_sensitive (__widget_key_filter_button, use_filter);
}

static void
on_key_filter_selection_clicked (GtkButton *button,
                                 gpointer   user_data)
{
    GtkEntry *entry = static_cast <GtkEntry*> (user_data);

    if (entry) {
        GtkWidget *dialog = scim_key_selection_dialog_new (_("Set key filter"));
        gint result;

        scim_key_selection_dialog_set_keys (
            SCIM_KEY_SELECTION_DIALOG (dialog),
            gtk_entry_get_text (entry));

        result = gtk_dialog_run (GTK_DIALOG (dialog));

        if (result == GTK_RESPONSE_OK) {
            const gchar *keys = scim_key_selection_dialog_get_keys (
                            SCIM_KEY_SELECTION_DIALOG (dialog));

            if (!keys) keys = "";

            if (strcmp (keys, gtk_entry_get_text (entry)))
                gtk_entry_set_text (entry, keys);

            GtkTreeModel *model;
            model = gtk_tree_view_get_model (GTK_TREE_VIEW (__widget_key_list_view));
            gtk_list_store_clear (GTK_LIST_STORE (model));
            for (unsigned int i = 0; i < __key_conf_pages_num; i++)
                append_key_bindings (GTK_TREE_VIEW (__widget_key_list_view),
                                     i, keys);
        }

        gtk_widget_destroy (dialog);
    }
}

static gboolean
on_key_list_view_key_press (GtkWidget *widget, GdkEventKey *event,
                            gpointer user_data)
{
    GtkTreeView *treeview = GTK_TREE_VIEW (widget);

    switch (event->keyval) {
    case GDK_Return:
    case GDK_KP_Enter:
        key_list_view_popup_key_selection (treeview);
        break;
    }

    return FALSE;
}

static gboolean
on_key_list_view_button_press (GtkWidget *widget, GdkEventButton *event,
                               gpointer user_data)
{
    GtkTreeView *treeview = GTK_TREE_VIEW (widget);

    if (event->type == GDK_2BUTTON_PRESS) {
        key_list_view_popup_key_selection (treeview);
        return TRUE;
    }

    return FALSE;
}
/*
vi:ts=4:nowrap:ai:expandtab
*/
