/*
 *  Copyright (C) 2006 Takashi Nakamoto <bluedwarf@bpost.plala.or.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define Uses_SCIM_UTILITY

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <scim.h>
#include "scim_ruby_util.h"
#include "intl.h"

int atoi(const String &str, int base)
{
    int ret = 0;
    int i = 0;
    int len = str.length();

    if( base > 36 )
	return 0;

    if( base <= 10 )
    {
	for(i; i<len; i++)
	{
	    if( str[i] < '0' || str[i] > '9' )
		break;
	    ret = ret * base + (str[i] - '0');
	}
    }
    else
    {
	for(i; i<len; i++)
	{
	    if( str[i] >= '0' && str[i] <= '9' )
		ret = ret * base + (str[i] - '0');
	    else if( str[i] >= 'a' && str[i] <= ('a' + base - 11) )
		ret = ret * base + (str[i] - 'a' + 10);
	    else if( str[i] >= 'A' && str[i] <= ('A' + base - 11) )
		ret = ret * base + (str[i] - 'A' + 10);
	    else
		break;
	}
    }

    return ret;
}

void replace_all(String &str, const String &src, const String &des)
{
    int pos = 0;
    int src_len = src.length();
    int des_len = des.length();

    while( ( pos = str.find(src, pos) ) != String::npos)
    {
	str.replace( pos, src_len, des );
	pos += des_len;
    }
}

void unquote(String &str)
{
    if( str[ str.length()-1 ] != '"' ||
	str[0] != '"' )
    {
	throw Exception("Failed to unquote; Not quoted string was specified.");
    }

    // strip double quotation mark
    str.erase(str.length()-1,1);
    str.erase(0,1);

    // read from the first character
    int pos = 0;
    char tmp[1];
    int tmp_i;
    while( ( pos = str.find("\\", pos) ) != String::npos)
    {
	switch( str[pos+1] )
	{
	case '\\': // back slash (0x5c)
	    str.replace( pos, 2, "\\" );
	    pos++;
	    break;
	case '"': // double quote (0x22)
	    str.replace( pos, 2, "\"" );
	    pos++;
	    break;
	case 't': // horizontal tab (0x09)
	    str.replace( pos, 2, "\t" );
	    pos++;
	    break;
	case 'n': // line feed (0x0a)
	case 'r': // carriage return (0x0d)
	    str.replace( pos, 2, "\r\n" );
	    pos+=2;
	    break;
	case 's': // space (0x20)
	    str.replace( pos, 2, " " );
	    pos++;
	    break;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7': // octadecimal code
	    tmp_i = atoi( str.substr( pos+1, 3 ), 8);
	    if( tmp_i > 127 )
		tmp[0] = tmp_i - 256;
	    else
		tmp[0] = tmp_i;
	    str.replace( pos, 4, tmp );
	    pos+=1;
	    break;
	case 'x': // hexadecimal code ("\xnn": n=0-9a-fA-F)
	    tmp_i = atoi( str.substr( pos+1, 3 ), 8);
	    if( tmp_i > 127 )
		tmp[0] = tmp_i - 256;
	    else
		tmp[0] = tmp_i;
	    str.replace( pos, 4, tmp );
	    pos+=1;
	    break;
	case 'b': // back space (0x08)
	case 'v': // vertical tab (0x0b)
	case 'f': // form feed (0x0c)
	case 'a': // bell (0x07)
	case 'e': // escape (0x1b)
	    str.replace( pos, 2, "" ); // ignore
	    break;
	default: // other characters
	    tmp[0] = str[pos+1];
	    str.replace( pos, 2, tmp );
	    pos++;
	}
    }
}
