/*
 *  Copyright (C) 2006 Takashi Nakamoto <bluedwarf@bpost.plala.or.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define Uses_SCIM_UTILITY

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <termios.h>

#include <scim.h>
#include "scim_ruby_interpreter.h"
#include "scim_ruby_util.h"
#include "intl.h"
#include "errno.h"


#ifndef SCIM_RUBY_SCIMRUBY_LIB
    #define SCIM_RUBY_SCIMRUBY_LIB           (SCIM_RUBYDIR"/scimruby.rb")
#endif

#ifndef SCIM_RUBY_COMPLETION_LIB
    #define SCIM_RUBY_COMPLETION_LIB           (SCIM_RUBYDIR"/completion.rb")
#endif

RubyInterpreter::RubyInterpreter(const String &irb_command,
				 const String &required_libs,
				 const String &pre_include_modules)
    :m_fp_input(NULL),
     m_fp_output(NULL),
     m_irb_command(irb_command),
     m_history_size(10),
     m_can_predict(false)
{
    open_irb();

    if(load_library(SCIM_RUBY_COMPLETION_LIB))
	m_can_predict = true; // for completion
    if(!load_library(SCIM_RUBY_SCIMRUBY_LIB))
	throw Exception("Fatal error; failed to find scimruby.rb");
    load_required_libraries(required_libs);
    include_modules(pre_include_modules);
}

RubyInterpreter::~RubyInterpreter()
{
    exit_irb();
}

void RubyInterpreter::open_irb()
{
#define R (0)
#define W (1)
#define PARENT_READ (read_pipe[R])
#define CHILD_WRITE (read_pipe[W])
#define CHILD_READ (write_pipe[R])
#define PARENT_WRITE (write_pipe[W])

    int read_pipe[2]; // read pipe for parent
    int write_pipe[2]; // write pipe for parent
    int pid;

    if( pipe(read_pipe) < 0 )
    {
	throw Exception("Fatal error; failed to open reading pipe for the parent process.");
	return;
    }
    if( pipe(write_pipe) < 0 )
    {
	throw Exception("Fatal error; failed to open writing pipe for the parent process.");
	close(read_pipe[R]);
	close(read_pipe[W]);
	return;
    }

    child_write_fd = CHILD_READ;

    pid = fork();
    if( pid < 0 )
    {
	switch(errno)
	{
	case EAGAIN:
	    throw Exception("Fatal error [fork()]; some resources are temporarily not available (EAGAIN).");
	    break;
	case ENOMEM:
	    throw Exception("Fatal error [fork()]; shortage of memory area (ENOMEM).");
	default:
	    throw Exception("Fatal error [fork()]; failed to make a cihld process without known cause.");
	}
	close(PARENT_READ);
	close(CHILD_WRITE);
	close(CHILD_READ);
	close(PARENT_WRITE);
	return;
    }

    if( pid == 0 )
    {
	// child; launch irb
	close(PARENT_WRITE);
	close(PARENT_READ);
	dup2(CHILD_READ, 0);
	dup2(CHILD_WRITE, 1);
	close(CHILD_READ);
	close(CHILD_WRITE);

	// ToDo: this chunk of code to prevent echo back does not work!
	/*
	// echo off
	struct termios stored_settings;
	struct termios new_settings;
	tcgetattr(0,&stored_settings);
	new_settings = stored_settings;
	new_settings.c_lflag &= (~ECHO);
	tcsetattr(0,TCSANOW,&new_settings);
	*/

	if(execlp(m_irb_command.c_str(), "--noprompt", "-f", NULL) < 0)
	    // "-f" means that scim-ruby doesn't load ~/.irbrc
	{
	    switch(errno)
		throw Exception( strerror(errno) );

	    // irb command cannot be found
	    close(CHILD_READ);
	    close(CHILD_WRITE);
	    _exit(1);
	}
	_exit(0);
    }
    else
    {
	// parent
	close(CHILD_READ);
	close(CHILD_WRITE);

	m_fp_input = fdopen(PARENT_WRITE, "w");
	m_fp_output = fdopen(PARENT_READ, "r");
    }
}

void
RubyInterpreter::load_required_libraries(const String &required_libs)
{
    // required_libs is a string which has library names separated by comma

    String str = required_libs;
    int index = 0;

    while((index = str.find(',', 0)) != String::npos)
    {
	if(index)
	    load_library( str.substr(0,index) );
	str.erase(0, index+1);
    }
    if(str.length() != 0)
    {
	load_library(str);
    }
}

bool
RubyInterpreter::load_library(const String &library)
{
    if( library.find("'") != String::npos )
    {
	// library name must not have single quoatation mark
	return false;
    }

    // set code to send
    String src = String("require '") + library + String("'\n\n");

    // send code to irb
    String res = send_code_to_irb(src);

    // delete first line, echo back
    int first_lf = res.find('\n');
    res.erase(0, first_lf+1);

    if( res.find("true") != String::npos )
	return true;
    else
	return false;
}

void
RubyInterpreter::include_modules(const String &modules)
{
    // modules is a string which has module names separated by comma

    String str = modules;
    int index = 0;

    while((index = str.find(',', 0)) != String::npos)
    {
	if(index)
	    include_module( str.substr(0,index) );
	str.erase(0, index+1);
    }
    if(str.length() != 0)
    {
	include_module(str);
    }
}

void
RubyInterpreter::include_module(const String &module)
{
    // ToDo: check module name is valid

    // set code to send
    String src = String("include ") + module + String("\n\n");

    // send code to irb
    send_code_to_irb(src);
}

String
RubyInterpreter::send_code_to_irb(const String &code)
{
    char buf[256];

    if(	fputs(code.c_str(), m_fp_input) == EOF )
    {
	throw Exception("IO Error [fputs()]; failed to send Ruby code to irb process.");
    }

    if( fflush( m_fp_input ) != 0 )
	throw Exception( strerror(errno) );

    // get output string
    String ret;
    while( fgets(buf, sizeof(buf), m_fp_output) )
    {
	if(strcmp(buf, "\n") == 0)
	    return ret;

	ret += buf;
    }

    throw Exception("IO Error [fgets()]; unkown error occured while reading the result of sent Ruby code.");
}

void RubyInterpreter::exit_irb()
{
    char buf[16];

    fputs("exit\n\n", m_fp_input);
    fclose(m_fp_input);

    // get rest of buffer
    while( fgets(buf, sizeof(buf), m_fp_output) )
    {};
    fclose(m_fp_output);

    // ToDo: wait for the child process exit correctly
    wait(0);
}

void
RubyInterpreter::get_methods(const WideString &wstr_object, list<WideString> &methods)
{
    methods.clear();
    String object = utf8_wcstombs(wstr_object);

    // set code to send
    String code = object + String(".methods.join(',')\n\n");

    // ToDo: optimize reading process or sorting process

    String methods_str = send_code_to_irb(code);

    // delete first line, echo back
    int first_lf = methods_str.find('\n');
    methods_str.erase(0, first_lf+1);

    // delete the tail character, line feed code
    methods_str.erase(methods_str.length()-1, 1);

    // unquote
    unquote( methods_str );

    // get method list splitted by comma(,)
    int comma_pos, start_pos = 0;

    while( (comma_pos = methods_str.find(",", start_pos)) != String::npos)
    {
	methods.push_back( utf8_mbstowcs(methods_str.substr(start_pos, comma_pos-start_pos)) );
	start_pos = comma_pos + 1;
    }

    // ToDo: sort "methods" more quickly
    methods.sort();
}

void
RubyInterpreter::get_candidates(const WideString &wstr_code, list<WideString> &candidates)
{
    candidates.clear();

    if( !m_can_predict )
	return; // nothing to do

    // escape single quotation mark (replace ' by \')
    String code = utf8_wcstombs( wstr_code );
    replace_all( code, String("'"), String("\\'") );

    // prevent the problem that the appended single quotation mark will be 
    // escaped by "\" on the tail of wstr_code
    if( code[code.length()-1] == '\\' )
	code.replace( code.length() - 1, 1, "\\\\");

    // Set code to send.
    // See data/completion.rb for the detail of this code.
    code = String("ScimRuby::complete('")
	+ code
	+ String("').join(\"\\x1f\")\n\n"); // the delimiter is \x1f (Unit Separator)

    String candidates_str = send_code_to_irb(code);

    // delete first line, echo back
    int first_lf = candidates_str.find('\n');
    candidates_str.erase(0, first_lf+1);

    // delete the tail character, line feed code
    candidates_str.erase(candidates_str.length()-1, 1);

    // unquote
    unquote( candidates_str );

    // get method list splitted by comma(,)
    int delim_pos, start_pos = 0;

    // split by \x1f (Unit Separator)
    while( (delim_pos = candidates_str.find((char)(0x1f), start_pos)) != String::npos)
    {
	candidates.push_back( utf8_mbstowcs(candidates_str.substr(start_pos, delim_pos-start_pos)) );
	start_pos = delim_pos + 1;
    }

    WideString last = utf8_mbstowcs(candidates_str.substr(start_pos));
    if(last.length())
	candidates.push_back( last );

// unnecessary; the candidates list is sorted by data/completion.rb
//    candidates.sort();
}

WideString
RubyInterpreter::execute_ruby_code(const WideString &wsrc,
				   const bool save_as_history)
{
    // ToDo: confirm that the wsrc is valid Ruby code in one line

    // set code to send
    String escaped = escape_single_quote(utf8_wcstombs(wsrc));
    String src = String("(begin; eval('") + escaped + String("'); rescue Exception; ScimRuby::exception_handler($!); end).to_scimruby\n\n");

    String res = send_code_to_irb(src);

    // delete first line, echo back
    int first_lf = res.find('\n');
    if(first_lf == WideString::npos)
	return WideString();
    res.erase(0, first_lf+1);

    // delete the tail character, line feed code
    res.erase(res.length()-1, 1);

    // unquote
    unquote( res );

    // save executed code as history
    if( save_as_history )
    {
	m_history_list.push_front(wsrc);
	while( m_history_list.size() > m_history_size )
	    m_history_list.pop_back();
    }

    return utf8_mbstowcs(res);
}

bool
RubyInterpreter::valid_syntax(const WideString &wsrc)
{
    String command;
    String src = utf8_wcstombs(wsrc);
    // escape '=>\' and \=>\\
    command = String("ScimRuby.valid_syntax?('") + src + String("')");
    String res = send_code_to_irb(command);

    return res == String("true");
}

void
RubyInterpreter::setHistorySize(int size)
{
    if( size < m_history_size )
	m_history_list.resize( size );
    m_history_size = size;
}

list<WideString>
&RubyInterpreter::get_history()
{
    return m_history_list;
}
