/*
 *  Copyright (C) 2006 Takashi Nakamoto <bluedwarf@bpost.plala.or.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Based on scim-imengine-skeleton.
 * Copyright (C) Hiroyuki Ikezoe <poincare@ikezoe.net>
 * Copyright (C) 2004 - 2005 Takuro Ashie <ashie@homa.ne.jp>
 */

/*
 * The original code is scim_uim_imengine.cpp in scim-uim-0.1.3. 
 * Copyright (C) 2004 James Su <suzhe@tsinghua.org.cn>
 */

#define Uses_SCIM_UTILITY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_LOOKUP_TABLE
#define Uses_SCIM_CONFIG_BASE

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <scim.h>
#include "scim_ruby_imengine_factory.h"
#include "scim_ruby_imengine.h"
#include "scim_ruby_prefs.h"
#include "scim_ruby_keybind.h"
#include "intl.h"

#define PROP_PREFIX "/IMEngine/Ruby"
#define PROP_INPUT_MODE PROP_PREFIX"/InputMode"
#define PROP_INPUT_MODE_RUBY PROP_INPUT_MODE"/Ruby"
#define PROP_INPUT_MODE_DIRECT PROP_INPUT_MODE"/Direct"

RubyInstance::RubyInstance (RubyFactory *factory,
                                    const String    &encoding,
                                    int              id)
    : IMEngineInstanceBase (factory, encoding, id),
      m_factory (factory),
      m_prev_key (0,0),
      m_input_mode(PROP_INPUT_MODE_RUBY),
      m_caret_pos(0),
      m_candidate_type(HIDDEN)
{
    SCIM_DEBUG_IMENGINE(1) << "Create RUBY Instance : ";

    install_properties();
}

RubyInstance::~RubyInstance ()
{
}

bool
RubyInstance::process_key_event (const KeyEvent& key)
{
    SCIM_DEBUG_IMENGINE(2) << "process_key_event.\n";
    KeyEvent newkey;

    // ignore key release.
    if (key.is_key_release ()) {
        return true;
    }

    // ignore modifier keys
    if (key.code == SCIM_KEY_Shift_L || key.code == SCIM_KEY_Shift_R ||
        key.code == SCIM_KEY_Control_L || key.code == SCIM_KEY_Control_R ||
        key.code == SCIM_KEY_Alt_L || key.code == SCIM_KEY_Alt_R)
        return false;

    // direct input mode
    if(m_input_mode == PROP_INPUT_MODE_DIRECT)
    {
        // changing mode
        if(m_factory->getKeyBind()->match_keys(key, CHANGE_MODE))
        {
	    switch_input_mode();
            return true;
        }

        return false;
    }

    // ruby interpretation mode
    if(process_key_event_lookup_keybind(key))
        return true;

    if(m_preedit_string.length())
        return process_key_event_with_preedit(key);
    else
        return process_key_event_without_preedit(key);
}

bool
RubyInstance::process_key_event_lookup_keybind (const KeyEvent &key)
{
    if(m_factory->getKeyBind()->match_keys(key, KAKUTEI))
    {
	if(m_lookup_table.number_of_candidates())
	{
	    // get selected candidate
	    int pos = m_lookup_table.get_cursor_pos();
	    WideString cand = m_lookup_table.get_candidate(pos);

	    if( m_candidate_type == COMPLETION )
	    {
		// replace the string piece by the complemented name
		WideString piece = m_lookup_table.get_candidate(0);
		int piece_pos = m_preedit_string.rfind(piece, m_caret_pos);
		if(piece_pos == WideString::npos)
		    return false; // ToDo: it is an unexpected error, isn't it?
		m_preedit_string.replace(piece_pos,
					 m_caret_pos-piece_pos,
					 cand);

		update_preedit_string(m_preedit_string);
		move_preedit_caret( piece_pos + cand.length() );
	    
		hide_candidate();

		return true;
	    }
	    else if( m_candidate_type == HISTORY )
	    {
		// just add selected candidate to preedit string
		m_preedit_string.insert( m_caret_pos, cand );
		update_preedit_string( m_preedit_string );

		m_caret_pos += cand.length();
		move_preedit_caret( m_caret_pos );

		show_preedit_string();
		hide_candidate();

		return true;
	    }
	}
	else
	{
	    // execute Ruby code in preedit window
	    if(!m_preedit_string.length())
		return false;
	    try
	    {
		WideString res =
		    m_factory->getInterpreter()->execute_ruby_code(m_preedit_string);
		commit_string(res);

		reset();
		hide_candidate();

		return true;
	    }
	    catch( RubyExecutionException &e )
	    {
		reset();
		hide_candidate();

		String error_message = e.message;

		// only one line string are permitted for aux string
		int first_lf = error_message.find('\n');
		if(first_lf != String::npos)
		    error_message = error_message.substr(0, first_lf-1);

		// show error message on aux area
		m_aux_string_base = utf8_mbstowcs(error_message);
		update_aux_string(m_aux_string_base);
		show_aux_string();

		return true;
	    }
	}
    }
    else if(m_factory->getKeyBind()->match_keys(key, SHOW_HISTORY))
    {
	if( m_candidate_type != HISTORY )
	{
	    // show history candidate
	    update_history_candidate();
	}
	else
	{
	    // hide history candidate if it is already shown
	    hide_candidate();
	}

	return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, PASTE_CLIPBOARD))
    {
	// retrive a utf-8 string from an external command
	String ret = String();
	char buf[256];
	FILE *cmd = popen( "xclip -o -selection clipboard -quiet", "r" );

	if( !cmd )
	{
	    // ToDo: error handling
	    return false;
	}

	while( fgets(buf, sizeof(buf), cmd) )
	    ret += buf;
	pclose( cmd );

	// add the retrieved string to the preedit sring
	if( !ret.length() )
	    return false;

	hide_candidate();

	WideString str = utf8_mbstowcs( ret );
        m_preedit_string.insert( m_caret_pos, str );
	update_preedit_string( m_preedit_string );
	move_preedit_caret( m_caret_pos + str.length() );

	return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, BACKSPACE))
    {
        if(!m_preedit_string.length())
            return false;

        // The caret position is the beginning of preedit string.
        // Nothing to do.
        if(m_caret_pos == 0)
            return true;

        // backspace
        m_preedit_string.erase(m_caret_pos-1, 1);
        update_preedit_string(m_preedit_string);
        move_preedit_caret( m_caret_pos - 1 );

	// show candidate window if suitable candidates exist
	update_completion_candidate();

        // The preedit string is emtpy.
        // Hide preedit window.
        if(!m_preedit_string.length())
            reset();

        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, DELETE))
    {
        if(!m_preedit_string.length())
            return false;

        // The caret position is the end of preedit string.
        // Nothing to do.
        if(m_caret_pos == m_preedit_string.length())
            return true;

        // delete
        m_preedit_string.erase(m_caret_pos, 1);
        update_preedit_string(m_preedit_string);
        // there is no need to move the caret

        // The preedit string is emtpy.
        // Hide preedit window.
        if(!m_preedit_string.length())
            reset();

        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, GO_FORWARD))
    {
        if(!m_preedit_string.length())
            return false;

        // The caret position is the end of preedit string.
        // Nothing to do.
        if(m_caret_pos == m_preedit_string.length())
            return true;

	// hide candidate window
	hide_candidate();

        // move caret forward in preedit window
        move_preedit_caret( m_caret_pos + 1 );
        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, GO_BACK))
    {
        if(!m_preedit_string.length())
            return false;

        // The caret position is the beginning of preedit string.
        // Nothing to do.
        if(m_caret_pos == 0)
            return true;

	// hide candidate window
	hide_candidate();

        // move caret backward in preedit window
        move_preedit_caret( m_caret_pos - 1 );
        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, GOTO_HEAD))
    {
        if(!m_preedit_string.length())
            return false;

	// hide candidate window
	hide_candidate();

        // move caret the head of line
        move_preedit_caret( 0 );
        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, GOTO_END))
    {
        if(!m_preedit_string.length())
            return false;

	// hide candidate window
	hide_candidate();

        // move caret the head of line
        move_preedit_caret( m_preedit_string.length() );
        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, SELECT_UP))
    {
	if(!m_lookup_table.number_of_candidates())	
	    return false;

	m_lookup_table.cursor_up();
	update_lookup_table(m_lookup_table);

	return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, SELECT_DOWN))
    {
	if(!m_lookup_table.number_of_candidates())	
	    return false;

	m_lookup_table.cursor_down();
	update_lookup_table(m_lookup_table);

	return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, NEXT_PAGE))
    {
	if(!m_lookup_table.number_of_candidates())	
	    return false;

	m_lookup_table.page_down();
	update_lookup_table(m_lookup_table);

	return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, PREVIOUS_PAGE))
    {
	if(!m_lookup_table.number_of_candidates())	
	    return false;

	m_lookup_table.page_up();
	update_lookup_table(m_lookup_table);

	return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, CLEAR))
    {
        // clear
        reset();

        if(!m_preedit_string.length())
            return false;

        return true;
    }
    else if(m_factory->getKeyBind()->match_keys(key, CHANGE_MODE))
    {
        if(m_input_mode == PROP_INPUT_MODE_RUBY)
        {
            if(m_preedit_string.length() != 0)
            {
                // just commit preedit string beforehand
                hide_preedit_string();
                commit_string(m_preedit_string);
                reset();
            }

	    switch_input_mode();
        }
        else
	    switch_input_mode();

        return true;
    }

    return false;
}

bool
RubyInstance::process_key_event_without_preedit (const KeyEvent &key)
{
    ucs4_t ucs4_key = key.get_unicode_code();

    if(ucs4_key &&
       (key.mask == SCIM_KEY_NullMask || key.mask == SCIM_KEY_ShiftMask) &&
       key.code != SCIM_KEY_Return &&
       key.code != SCIM_KEY_Tab)
    {
        // show preedit window with the specified key
        m_preedit_string += ucs4_key;
        update_preedit_string(m_preedit_string);
        show_preedit_string();
        move_preedit_caret( m_caret_pos + 1 );

	// show candidate window if suitable candidates exist
	update_completion_candidate();

        return true;
    }
    else
    {
	reset();
	return false;
    }
}

bool
RubyInstance::process_key_event_with_preedit (const KeyEvent &key)
{
    ucs4_t ucs4_key = key.get_unicode_code();

    if(!ucs4_key)
        return false;

    if(key.mask == SCIM_KEY_NullMask || key.mask == SCIM_KEY_ShiftMask)
    {
        // insert the specified character to preedit string
        m_preedit_string.insert(m_caret_pos, 1, ucs4_key);
        update_preedit_string(m_preedit_string);
        move_preedit_caret( m_caret_pos + 1 );

	// show candidate window if suitable candidates exist
	update_completion_candidate();

        return true;
    }
    else
        return false;
}

void
RubyInstance::move_preedit_caret (unsigned int pos)
{
    m_caret_pos = pos;
    update_preedit_caret(pos);
}

void
RubyInstance::reset ()
{
    SCIM_DEBUG_IMENGINE(2) << "reset.\n";

    m_lookup_table.clear ();
    move_preedit_caret( 0 );
    update_preedit_string(m_preedit_string = WideString());
    hide_candidate ();
    hide_preedit_string ();
}

void
RubyInstance::focus_in ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_in.\n";

    hide_aux_string ();
    install_properties();
}

void
RubyInstance::focus_out ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_out.\n";

    // reset
    m_lookup_table.clear();
    move_preedit_caret( 0 );
    update_preedit_string(m_preedit_string = WideString());
    hide_candidate();
    hide_preedit_string();
}

void
RubyInstance::trigger_property (const String &property)
{
    String prop_type = property.substr( 0, property.find_last_of('/') );
    
    if( prop_type == PROP_INPUT_MODE && property != m_input_mode )
    {
	switch_input_mode();
    }

    SCIM_DEBUG_IMENGINE(2) << "trigger_property : " << property << "\n";
}

void
RubyInstance::update_completion_candidate()
{
    m_lookup_table.clear();

    if( !m_factory->useCompletion() )
    {
	hide_candidate();
	return;
    }

    m_aux_string_base =
	m_factory->getReference()->getCandidates(m_lookup_table,
						 m_preedit_string,
						 m_caret_pos);
    m_candidate_type = COMPLETION;

    update_candidate();
}

void
RubyInstance::update_history_candidate()
{
    m_lookup_table.clear();

    // get history list from an interpreter
    list<WideString> history_list = m_factory->getInterpreter()->get_history();
    list<WideString>::iterator p = history_list.begin();
    while( p != history_list.end() )
    {
	m_lookup_table.append_candidate(*p);
	p++;
    }

    m_aux_string_base = utf8_mbstowcs(_("History"));
    m_candidate_type = HISTORY;

    update_candidate();
}

void
RubyInstance::update_candidate()
{
    if(m_lookup_table.number_of_candidates())
    {
	m_lookup_table.show_cursor(true);
	show_lookup_table();
	update_lookup_table(m_lookup_table);

	// aux string shown below the preedit string
	update_aux_string(m_aux_string_base);
	show_aux_string();
    }
    else
    {
	hide_candidate();
    }
}

void
RubyInstance::hide_candidate()
{
    m_lookup_table.clear();
    m_lookup_table.show_cursor(false);
    hide_lookup_table();
    m_aux_string_base = WideString();
    hide_aux_string();
    m_candidate_type = HIDDEN;
}

void
RubyInstance::install_properties()
{
    if( m_properties.size() <= 0 )
    {
	Property prop;

	prop = Property( PROP_INPUT_MODE, "R",
			 String(""), _("Input mode") );
	m_properties.push_back( prop );

	prop = Property( PROP_INPUT_MODE_RUBY, _("Ruby interpretation mode"),
			 String(""), _("Ruby interpretation mode") );
	m_properties.push_back( prop );

	prop = Property( PROP_INPUT_MODE_DIRECT, _("Direct input mode"),
			 String(""), _("Direct input mode") );
	m_properties.push_back( prop );
    }

    register_properties( m_properties );
}

void
RubyInstance::switch_input_mode()
{
    PropertyList::iterator p = std::find( m_properties.begin(),
					  m_properties.end(),
					  PROP_INPUT_MODE );

    if( m_input_mode == PROP_INPUT_MODE_RUBY )
    {
	m_input_mode = PROP_INPUT_MODE_DIRECT;
	p->set_label( "D" );
    }
    else
    {
	m_input_mode = PROP_INPUT_MODE_RUBY;
	p->set_label( "R" );
    }

    update_property( *p );
}
