/*
 *  Copyright (C) 2006 Takashi Nakamoto <bluedwarf@bpost.plala.or.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Based on scim-imengine-skeleton.
 * Copyright (C) Hiroyuki Ikezoe <poincare@ikezoe.net>
 * Copyright (C) 2004 - 2005 Takuro Ashie <ashie@homa.ne.jp>
 */

/*
 * The original code is scim_uim_imengine.cpp in scim-uim-0.1.3. 
 * Copyright (C) 2004 James Su <suzhe@tsinghua.org.cn>
 */

#define Uses_SCIM_UTILITY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_CONFIG_BASE

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <scim.h>
#include "scim_ruby_imengine_factory.h"
#include "scim_ruby_imengine.h"
#include "scim_ruby_prefs.h"
#include "scim_ruby_keybind.h"
#include "scim_ruby_interpreter.h"
#include "scim_ruby_reference.h"
#include "scim_ruby_util.h"
#include "intl.h"

#define scim_module_init ruby_LTX_scim_module_init
#define scim_module_exit ruby_LTX_scim_module_exit
#define scim_imengine_module_init ruby_LTX_scim_imengine_module_init
#define scim_imengine_module_create_factory ruby_LTX_scim_imengine_module_create_factory

#define SCIM_CONFIG_IMENGINE_RUBY_UUID     "/IMEngine/Ruby/UUID-"

#ifndef SCIM_RUBY_ICON_FILE
    #define SCIM_RUBY_ICON_FILE           (SCIM_ICONDIR"/scim-ruby.png")
#endif

static ConfigPointer _scim_config (0);

extern "C" {
    void scim_module_init (void)
    {
        bindtextdomain (GETTEXT_PACKAGE, SCIM_RUBY_LOCALEDIR);
        bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    }

    void scim_module_exit (void)
    {
        _scim_config.reset ();
    }

    uint32 scim_imengine_module_init (const ConfigPointer &config)
    {
        SCIM_DEBUG_IMENGINE(1) << "Initialize Ruby Engine.\n";

        _scim_config = config;

        return 1;
    }

    IMEngineFactoryPointer scim_imengine_module_create_factory (uint32 engine)
    {
        RubyFactory *factory = 0;

        try {
            factory = new RubyFactory (String ("other,ja_JP"),
                                       String ("379cdb00-43ac-44d1-8ec5-98b2b0463cdb"),
                                       _scim_config);
        } catch (...) {
            delete factory;
            factory = 0;
        }

        return factory;
    }
}


RubyFactory::RubyFactory (const String &lang,
                                const String &uuid,
                                const ConfigPointer &config)
    : m_uuid (uuid),
      m_config (config),
      m_use_completion(false)
{
    SCIM_DEBUG_IMENGINE(1) << "Create Ruby Factory :\n";
    SCIM_DEBUG_IMENGINE(1) << "  Lang : " << lang << "\n";
    SCIM_DEBUG_IMENGINE(1) << "  UUID : " << uuid << "\n";

    if (lang.length() >= 2)
        set_languages (lang);

    // intialization

    // unreloadable configuration
    String irb_command = config->read(String(SCIM_RUBY_CONFIG_IRB_COMMAND),
                                      String(SCIM_RUBY_CONFIG_IRB_COMMAND_DEFAULT));
    String ri_command = config->read(String(SCIM_RUBY_CONFIG_RI_COMMAND),
                                     String(SCIM_RUBY_CONFIG_RI_COMMAND_DEFAULT));
    String required_libs = config->read(String(SCIM_RUBY_CONFIG_REQUIRED_LIBRARIES),
                                        String(SCIM_RUBY_CONFIG_REQUIRED_LIBRARIES_DEFAULT));
    String pre_include_modules = config->read(String(SCIM_RUBY_CONFIG_PRE_INCLUDE_MODULES),
                                        String(SCIM_RUBY_CONFIG_PRE_INCLUDE_MODULES_DEFAULT));

    m_key_bind = new KeyBind();

    m_interpreter = new RubyInterpreter(irb_command, required_libs, pre_include_modules);
    m_reference = new RubyReference(m_interpreter, ri_command);

    // reloadable configuration
    reload_config (m_config);
    m_reload_signal_connection
        = m_config->signal_connect_reload (slot (this, &RubyFactory::reload_config));
}

RubyFactory::~RubyFactory ()
{
    delete m_key_bind;
    delete m_interpreter;
    delete m_reference;
}

WideString
RubyFactory::get_name () const
{
    return utf8_mbstowcs (String ("Ruby"));
}

WideString
RubyFactory::get_authors () const
{
    String tmp = String(_("Authors of scim-ruby"));
    tmp += ":\n";
    tmp += "  ";
    tmp += _("(C) 2006 Takashi Nakamoto <bluedwarf@bpost.plala.or.jp>");
    tmp += "\n";
    tmp += "\n";

    return utf8_mbstowcs( tmp );
}

WideString
RubyFactory::get_credits () const
{
    String tmp = String(_("Contributors of scim-ruby"));
    tmp += ":\n";
    tmp += "Nobody? Please help me!";
    tmp += "\n";
    tmp += "\n";

    return utf8_mbstowcs( tmp );
}

WideString
RubyFactory::get_help () const
{
    // ToDo: write help
    return WideString ();
}

String
RubyFactory::get_uuid () const
{
    return m_uuid;
}

String
RubyFactory::get_icon_file () const
{
    return String (SCIM_RUBY_ICON_FILE);
}

String
RubyFactory::get_language () const
{
    return scim_validate_language ("other");
}

IMEngineInstancePointer
RubyFactory::create_instance (const String &encoding, int id)
{
    return new RubyInstance (this, encoding, id);
}

KeyBind *
RubyFactory::getKeyBind()
{
    return m_key_bind;
}

RubyInterpreter *
RubyFactory::getInterpreter()
{
    return m_interpreter;
}

RubyReference *
RubyFactory::getReference()
{
    return m_reference;
}

void
RubyFactory::reload_config (const ConfigPointer &config)
{
    if (!config) return;
    
    String str;

    // common keys
    str = config->read(String(SCIM_RUBY_CONFIG_KAKUTEI_KEY),
                       String(SCIM_RUBY_CONFIG_KAKUTEI_KEY_DEFAULT));
    m_key_bind->set_keys(str, KAKUTEI);
    str = config->read(String(SCIM_RUBY_CONFIG_SHOW_HISTORY_KEY),
                       String(SCIM_RUBY_CONFIG_SHOW_HISTORY_KEY_DEFAULT));
    m_key_bind->set_keys(str, SHOW_HISTORY);
    str = config->read(String(SCIM_RUBY_CONFIG_PASTE_CLIPBOARD_KEY),
                       String(SCIM_RUBY_CONFIG_PASTE_CLIPBOARD_KEY_DEFAULT));
    m_key_bind->set_keys(str, PASTE_CLIPBOARD);

    // mode keys
    str = config->read(String(SCIM_RUBY_CONFIG_CHANGE_MODE_KEY),
                       String(SCIM_RUBY_CONFIG_CHANGE_MODE_KEY_DEFAULT));
    m_key_bind->set_keys(str, CHANGE_MODE);

    // keys related to the caret
    str = config->read(String(SCIM_RUBY_CONFIG_BACKSPACE_KEY),
                       String(SCIM_RUBY_CONFIG_BACKSPACE_KEY_DEFAULT));
    m_key_bind->set_keys(str, BACKSPACE);
    str = config->read(String(SCIM_RUBY_CONFIG_DELETE_KEY),
                       String(SCIM_RUBY_CONFIG_DELETE_KEY_DEFAULT));
    m_key_bind->set_keys(str, DELETE);
    str = config->read(String(SCIM_RUBY_CONFIG_GO_FORWARD_KEY),
                       String(SCIM_RUBY_CONFIG_GO_FORWARD_KEY_DEFAULT));
    m_key_bind->set_keys(str, GO_FORWARD);
    str = config->read(String(SCIM_RUBY_CONFIG_GO_BACK_KEY),
                       String(SCIM_RUBY_CONFIG_GO_BACK_KEY_DEFAULT));
    m_key_bind->set_keys(str, GO_BACK);
    str = config->read(String(SCIM_RUBY_CONFIG_GOTO_HEAD_KEY),
                       String(SCIM_RUBY_CONFIG_GOTO_HEAD_KEY_DEFAULT));
    m_key_bind->set_keys(str, GOTO_HEAD);
    str = config->read(String(SCIM_RUBY_CONFIG_GOTO_END_KEY),
                       String(SCIM_RUBY_CONFIG_GOTO_END_KEY_DEFAULT));
    m_key_bind->set_keys(str, GOTO_END);
    str = config->read(String(SCIM_RUBY_CONFIG_SELECT_UP_KEY),
                       String(SCIM_RUBY_CONFIG_SELECT_UP_KEY_DEFAULT));
    m_key_bind->set_keys(str, SELECT_UP);
    str = config->read(String(SCIM_RUBY_CONFIG_SELECT_DOWN_KEY),
                       String(SCIM_RUBY_CONFIG_SELECT_DOWN_KEY_DEFAULT));
    m_key_bind->set_keys(str, SELECT_DOWN);
    str = config->read(String(SCIM_RUBY_CONFIG_NEXT_PAGE_KEY),
                       String(SCIM_RUBY_CONFIG_NEXT_PAGE_KEY_DEFAULT));
    m_key_bind->set_keys(str, NEXT_PAGE);
    str = config->read(String(SCIM_RUBY_CONFIG_PREVIOUS_PAGE_KEY),
                       String(SCIM_RUBY_CONFIG_PREVIOUS_PAGE_KEY_DEFAULT));
    m_key_bind->set_keys(str, PREVIOUS_PAGE);
    str = config->read(String(SCIM_RUBY_CONFIG_CLEAR_KEY),
                       String(SCIM_RUBY_CONFIG_CLEAR_KEY_DEFAULT));
    m_key_bind->set_keys(str, CLEAR);

    String use_completion =  config->read(String(SCIM_RUBY_CONFIG_USE_COMPLETION),
                                        String(SCIM_RUBY_CONFIG_USE_COMPLETION_DEFAULT));
    if( use_completion == "1")
	m_use_completion = true;
    else
	m_use_completion = false;

    String history_size_str =  config->read(String(SCIM_RUBY_CONFIG_HISTORY_SIZE),
					    String(SCIM_RUBY_CONFIG_HISTORY_SIZE_DEFAULT));
    int history_size = atoi(history_size_str);
    m_interpreter->setHistorySize(history_size);
}
