<?php
/**
 * Abstract Dao Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

require_once \sprintf("phar://%s/core/db/DbFactory.php", namespace\PHAR);

abstract class AbstractDao
{
	const SEL_ALL_COLS = 1; // select all fields

	protected $db;
	protected $vars;
	protected $fields;
	protected $aliases;
	protected $funcs;
	protected $wheres;
	protected $groups;
	protected $orders;
	protected $limit;
	protected $offset;
	protected $affected;
	// テーブル名・カラム名を保持
	protected $table;
	protected $columns;

	/**
	 * constructor
	 * @access	protected
	 * @param	\Adelie\Dsn	$dsn
	 * @param	array	$opts
	 */
	protected function __construct (namespace\Dsn $dsn, Array $opts=array())
	{
		$this->db = namespace\DbFactory::create($dsn, $opts);
		$this->init();
		$this->table = "";
		$this->columns = array();
	}

	/**
	 * initialize
	 * @access	public
	 * @return	void
	 */
	public function init ()
	{
		$this->vars = array();
		$this->fields = array();
		$this->aliases = array();
		$this->funcs = array();
		$this->wheres = array();
		$this->groups = array();
		$this->orders = array();
		$this->limit = -1;
		$this->offset = -1;
		$this->affected = 0;
		$this->db->free();
	}

	/**
	 * disconnect
	 * @access	public
	 * @return	void
	 */
	public function close ()
	{
		$this->db->close();
	}

	/**
	 * AUTO_INCREMENTまたはシーケンス値を取得
	 * @access	public
	 * @return	integer
	 * @todo	実装
	 */
	public function getKey ()
	{
		return $this->db->getKey();
	}

	/**
	 * select
	 * @access	public
	 * @return	array
	 */
	public function select ()
	{
		try {
			$wheres = array();
			foreach ($this->wheres as $key=>$val) {
				$operator = $val["ope"];
				$wheres[] = "{$key} {$operator} :{$key}";
			}
			$orderby = array();
			foreach ($this->orders as $key=>$val) {
				$orderby[] = "{$key} {$val}";
			}
			$sql = "SELECT * FROM {$this->table} ";
			if (\count($wheres)) {
				$sql .= "WHERE ".\implode(" AND ", $wheres)." ";
			}
			if (\count($this->groups)) {
				$sql .= "GROUP BY ".\implode(", ", $this->groups);
			}
			if (\count($orderby)) {
				$sql .= "ORDER BY ".\implode(", ", $orderby)." ";
			}
			if ($this->limit>=0) {
				$sql .= "LIMIT {$this->limit} ";
			}
			if ($this->offset>=0) {
				$sql .= "OFFSET {$this->offset} ";
			}
			$this->db->prepare($sql);

			foreach ($this->wheres as $key=>$val) {
				$this->db->bind($key, $val["val"]);
			}
			return $this->db->exec();
		} catch (\Exception $e) {
			throw new namespace\DbException("", 0, $e);
		}
	}

	/**
	 * insert
	 * @access	public
	 * @return	void
	 */
	public function insert ()
	{
		$cols = \array_keys($this->vars);
		$sql  = "INSERT INTO {$this->table} (";
		$sql .= \implode(", ", $cols);
		$sql .= ") VALUES ( ";
		$sql .= ":" . \implode(", :", $cols)." ";
		$sql .= ") ";
		$this->db->prepare($sql);
		foreach ($this->vars as $key=>$val) {
			$this->db->bind($key, $val["var"], $val["type"]);
		}
		$this->db->exec();
	}

	/**
	 * update
	 * @access	public
	 * @return	void
	 */
	public function update ()
	{
		$binds = array();
		// 更新データ
		$fields = array();
		foreach ($this->vars as $key=>$val) {
			$fields[] = "{$key} = :{$key}";
			$binds[$key] = $val;
		}
		// WHERE句
		$wheres = array();
		foreach ($this->wheres as $key=>$val) {
			$wheres[] = "{$key} {$val["ope"]} :w_{$key}";
			$binds["w_{$key}"] = $val["val"];
		}
		// SQL
		$sql  = "UPDATE {$this->table} SET ";
		$sql .= \implode(", ", $fields) . " ";
		if (\count($wheres)) {
			$sql .= "WHERE " . \implode(" AND ", $wheres) . " ";
		}
		$this->db->prepare($sql);
		foreach ($binds as $key=>$val) {
			$this->db->bind($key, $val);
		}
		$this->db->exec();
	}

	/**
	 * delete
	 * @access	public
	 * @return	void
	 */
	public function delete ()
	{
		$binds = array();
		// WHERE句
		$wheres = array();
		foreach ($this->wheres as $key=>$val) {
			$wheres[] = "{$key} {$val["ope"]} :w_{$key}";
			$binds["w_{$key}"] = $val["val"];
		}
		// SQL
		$sql = "DELETE FROM {$this->table} ";
		if ($wheres) {
			$sql .= "WHERE " . \implode(" AND ", $wheres) . " ";
		}

		// 実行
		$this->db->prepare($sql);
		foreach ($binds as $key=>$val) {
			$this->db->bind($key, $val);
		}
		$this->db->exec();
		$this->affected = $this->db->getRowCount();
	}

	/**
	 * initiates a transaction
	 * @access	public
	 * @return	void
	 */
	public function begin ()
	{
		$this->db->prepare("BEGIN");
		$this->db->exec();
		$this->db->free();
	}

	/**
	 * commits a transaction
	 * @access	public
	 * @return	void
	 */
	public function commit ()
	{
		$this->db->prepare("COMMIT");
		$this->db->exec();
	}

	/**
	 * rolls back a transaction
	 * @access	public
	 * @param	string	$savepoint
	 * @return	void
	 */
	public function rollback ($savepoint="")
	{
		$sql = "ROLLBACK " . (\strlen($savepoint) ? "TO SAVEPOINT {$savepoint} " : "");
		$this->db->prepare($sql);
		$this->db->exec();
	}

	/**
	 * savepoint
	 * @access	public
	 * @param	string	$savepoint
	 * @return	void
	 */
	public function savepoint ($savepoint)
	{
		$this->db->prepare("SAVEPOINT {$savepoint} ");
		$this->db->exec();
	}

	/**
	 * savepoint
	 * @access	public
	 * @param	string	$savepoint
	 * @return	void
	 */
	public function release ($savepoint)
	{
		$this->db->prepare("RELEASE SAVEPOINT {$savepoint} ");
		$this->db->exec();
	}

	/**
	 * Truncate
	 * @access	public
	 * @return	void
	 */
	public function truncate ()
	{
		// Polymorphism isn't used because of the simple method.
		$sql = ($this->db->getDriver()=="sqlite") ? "DELETE FROM {$this->table}" : "TRUNCATE TABLE {$this->table}";
		$this->db->prepare($sql);
		$this->db->exec();
	}

	/**
	 * 予め記述しているSQLを使ってクエリーを実行
	 * @access	public
	 * @param	string	$sql_id
	 * @param	mixed	$argv...
	 * @return	DbResult
	 */
	public function sql ($sql_id)
	{
		$args = \func_get_args();
		$sql_id = \array_shift($args);
		$method = \sprintf("do%sSql", namespace\String::toPascal($sql_id));
		if (!\is_callable(array($this, $method))) {
			throw new namespace\DbException("Invalid SQL");
		}

		$sql = \call_user_func_array(array($this, $method), (array)$args);
		$sql = \trim($sql);
		$this->db->prepare($sql);
		return $this->db->exec();
	}
}
