/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altUtil.cpp 1424 2008-11-17 14:21:35Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altUtil.cpp
 * \brief   Utility Function
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altUtil.h"
#include "altStr.h"

#ifdef ALT_WIN
  #include <direct.h>
  #include <share.h>  // for _SH_DENYNO
#endif

#ifdef ALT_LINUX
  #include <time.h>
  #include <sys/stat.h>   // for mkdir
  #include <sys/types.h>  // for mkdir
  #include <fcntl.h>      // for O_RDWR
#endif

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
/**
 * \brief compare string
 *
 * \param szStr1  [! ] string 1
 * \param szStr2  [I ] string 2
 *
 * \return  1   szStr1 > szStr2
 * \return  0   szStr1 = szStr2
 * \return  -1  szStr1 < szStr2
 */
LIBALT_API altInt altStrcmp (
  altChar *       szStr1,
  const altChar * szStr2
)
{
  for (altUInt i = 0; ; i++) {
    if (szStr1[i] > szStr2[i]) {
      return (1);
    }
    else if (szStr1[i] < szStr2[i]) {
      return (-1);
    }
    else if (szStr1[i] == 0x00 && szStr2[i] == 0x00) {
      break;
    }
  }
  return (0);
}

/**
 * \brief copy string
 *
 * \param pDst  [ O] output
 * \param pSrc  [I ] source string
 * \param nSize [I ] string size
 */
LIBALT_API void altStrncpy (
  altChar *       pDst,
  const altChar * pSrc,
  const altUInt   nSize
)
{
#ifdef ALT_WIN
  strncpy_s (pDst, nSize, pSrc, nSize);
#else
  strncpy (pDst, pSrc, nSize);
#endif
}


/**
 * \brief add string
 *
 * \param pDst      [ O] output
 * \param nDstSize  [I ] output buffer size
 * \param pSrc      [I ] source string
 * \param nSize     [I ] string size
 */
LIBALT_API void altStrncat (
  altChar *       pDst,
  const altUInt   nDstSize,
  const altChar * pSrc,
  const altUInt   nSize
)
{
#ifdef ALT_WIN
  strncat_s (pDst, nDstSize, pSrc, nSize);
#else
  strncat (pDst, pSrc, nSize);
#endif
}

/**
 * \brief format string
 *
 * \param pBuf      [IO] buffer
 * \param nBufSize  [I ] buffer size
 * \param szFormat  [I ] format
 * \param oArgs     [I ] arguments
 *
 * \return  string length
 */
LIBALT_API const altInt altVsnprintf (
  altChar *       pBuf,
  const altUInt   nBufSize,
  const altChar * szFormat,
  va_list &       oArgs
)
{
  altInt nRet;

#ifdef ALT_WIN
  nRet = vsnprintf_s (pBuf, nBufSize, _TRUNCATE, szFormat, oArgs);
#else
  nRet = vsnprintf (pBuf, nBufSize, szFormat, oArgs);
#endif

  return nRet;
}

/**
 * \brief format string
 *
 * \param pBuf      [IO] buffer
 * \param nBufSize  [I ] buffer size
 * \param szFormat  [I ] format
 * \param oArgs     [I ] arguments
 *
 * \return  string length
 */
LIBALT_API const altInt altVswnprintf (
  altWChar *        pBuf,
  const altUInt     nBufSize,
  const wchar_t *   szFormat,
  va_list &         oArgs
)
{
  altInt nRet;

#ifdef ALT_WIN
  nRet = vswprintf_s ((wchar_t *)pBuf, nBufSize, szFormat, oArgs);
#else
  nRet = vswprintf ((wchar_t *)pBuf, nBufSize, szFormat, oArgs);
#endif

  return nRet;
}

/**
 * \brief sleep
 *
 * \param msec  [I ] sleep milliseconds
 */
LIBALT_API void altSleep (
  const DWORD dwMilliseconds
)
{
#ifdef ALT_WIN
  Sleep (dwMilliseconds);
#endif

#ifdef ALT_LINUX
  struct timespec oReq;
  struct timespec oRem;

  oReq.tv_sec = dwMilliseconds / 1000;
  oReq.tv_nsec = (dwMilliseconds % 1000) * 1000 * 1000;

  altInt nRet = nanosleep (& oReq, & oRem);
  while (nRet != 0) {
    nRet = nanosleep (& oRem, & oRem);
  }
#endif
}

/**
 * \brief create directory
 *
 * \param szDirName  [I ] directory name
 *
 * \return true   success
 * \return false  error
 */
LIBALT_API const altBool altMkDir (
  const altChar * szDirName
)
{
  altInt nRet;

#ifdef ALT_WIN
   nRet = _mkdir (szDirName);
#endif

#ifdef ALT_LINUX
  nRet = mkdir (szDirName, 0777);
#endif

  return (nRet == 0);
}

/**
 * \brief create directory recursive
 *
 * \param szDirName  [I ] directory name
 *
 * \return 0  success
 * \return -1 error
 */
LIBALT_API const altBool altMkDirR (
  const altChar * szDirName
)
{
  altStr  sDir;
  altStr  sDirName = szDirName;
  altInt  nIndex = sDirName.FindChar ("\\/");

  while (nIndex > 0) {
    // create directory
    sDir = sDirName.SubStr (0, nIndex - 1);
    altMkDir (sDir.GetCStr());
    nIndex = sDirName.FindChar (nIndex + 1, "\\/");
  }

  // create directory
  if (! altMkDir (szDirName)) {
    return (false);
  }
 
  return (true);
}

///
/// \brief	open file
///
/// \param	pFile		    [ O] pointer of file stream
/// \param	szFileName	[I ] file name
/// \param	szMode		  [I ] open mode
///
void altFOpen (
  FILE * &        pFile,
  const altChar * szFileName,
  const altChar * szMode
)
{
#ifdef ALT_WIN
	pFile = _fsopen (szFileName, szMode, _SH_DENYNO);
#endif

#ifdef ALT_LINUX
	pFile = fopen (szFileName, szMode);
#endif
}

///
/// \brief	open file
///
/// \param	pFile		    [ O] pointer of file stream
/// \param	szFileName	[I ] file name
/// \param	szMode		  [I ] open mode
///
void altFOpen (
  FILE * &          pFile,
  const altWChar *  szFileName,
  const altWChar *  szMode
)
{
#ifdef ALT_WIN
	pFile = _wfsopen ((const wchar_t *)szFileName, (const wchar_t *)szMode, _SH_DENYNO);
#endif

#ifdef ALT_LINUX
	pFile = fopen ((const char *)szFileName, (const char *)szMode);
#endif
}

///
/// \brief  Get wide string length
///
/// \param  szWStr  [I ] Wide string
///
/// \return Wide string length
///
altUInt altWStrLen (
  const altWChar *  szWStr
)
{
  if (szWStr == NULL) {
    return (0);
  }

  altUInt   nLen = 0;
  altWChar  wChar;

  for (altUInt i = 0; ; i++) {
    memcpy (& wChar, & szWStr[i], sizeof (wChar));
    if (wChar == 0x0000) {
      break;
    }
    nLen++;
  }
  return (nLen);
}


/**
 * \brief Copy wide string
 *
 * \param pDst  [ O] output
 * \param pSrc  [I ] source string
 * \param nSize [I ] string size
 */
LIBALT_API void altWStrncpy (
  altWChar *        pDst,
  const altWChar *  pSrc,
  const altUInt     nSize
)
{
  if (pDst == NULL) {
    return;
  }
  if (pSrc == NULL) {
    return;
  }
  if (nSize == 0) {
    return;
  }
  for (altUInt i = 0; i < nSize; i++) {
    if (pSrc[i] == 0x0000) {
      pDst[i] = 0x0000;
      break;
    }
    pDst[i] = pSrc[i];
  }
  pDst[nSize - 1] = 0x0000;
}
