/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altWStr.cpp 1426 2009-09-09 01:53:25Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altWStr.cpp
 * \brief   Wide string
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altWStr.h"
#include "altBaseMem.h"
#include "altUtil.h"
#include "altLog.h"

#ifdef ALT_DOTNET
#using <system.dll>

using namespace System;
using namespace System::Text::RegularExpressions;
#endif

#include <string>

#ifdef ALT_LINUX
#include <iconv.h>
#endif
#ifdef ALT_WIN
#include "altBase/iconv/iconv.h"
#endif


/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altWStr::altWStr() :
m_szStr (NULL),
m_nLen (0)
{
  ALT_BASE_WSTRALLOC (m_szStr, "");
}

///
/// \brief  Constructor
///
/// \param  c [I ] Character
///
LIBALT_API altWStr::altWStr(const altWChar c) :
m_szStr (NULL),
m_nLen (1)
{
  altWChar  szBuf[2];
  szBuf[0] = c;
  szBuf[1] = 0x0000;
  ALT_BASE_WSTRALLOC (m_szStr, szBuf);
}

///
/// \brief  Constructor
///
/// \param  szStr [I ] string
///
LIBALT_API altWStr::altWStr(const altWChar * szStr) :
m_szStr (NULL),
m_nLen (0)
{
  m_nLen = altWStrLen (szStr);
  ALT_BASE_WSTRALLOC (m_szStr, szStr);
}

///
/// \brief  Constructor
///
/// \param  szStr [I ] string
/// \param  nLen  [I ] length
///
LIBALT_API altWStr::altWStr(const altWChar * szStr, const altUInt nLen) :
m_szStr (NULL),
m_nLen (nLen)
{
  ALT_BASE_WSTRNALLOC (m_szStr, szStr, nLen);
}

///
/// \brief  Constructor
///
/// \param  szStr [I ] string
///
LIBALT_API altWStr::altWStr(const wchar_t * szStr) :
m_szStr (NULL),
m_nLen (0)
{
  m_nLen = altWStrLen ((altWChar *)szStr);
  ALT_BASE_WSTRALLOC (m_szStr, szStr);
}


///
/// \brief  Copy Constructor
///
/// \param  oStr [I ] string
///
LIBALT_API altWStr::altWStr(const altWStr & oStr) :
m_szStr (NULL),
m_nLen (0)
{
  m_nLen = oStr.m_nLen;
  ALT_BASE_WSTRALLOC (m_szStr, oStr.m_szStr);
}

///
/// \brief  Copy Constructor
///
/// \param  oStr [I ] string
///
LIBALT_API altWStr::altWStr(const altStr & oStr) :
m_szStr (NULL),
m_nLen (0)
{
  (* this) = oStr;
}

///
/// \brief  Destructor
///
LIBALT_API altWStr::~altWStr()
{
  ALT_BASE_FREE (m_szStr);
  m_nLen = 0;
}
  
///
/// \brief  = operator
///
/// \param  oStr  [I ] string object
///
/// \return this object
///
LIBALT_API altWStr &  altWStr::operator =(const altWStr & oStr)
{
  if (m_szStr == oStr.m_szStr) {
    return (* this);
  }
  ALT_BASE_FREE (m_szStr);
  m_nLen = oStr.m_nLen;
  ALT_BASE_WSTRALLOC (m_szStr, oStr.m_szStr);
  return (* this);
}

///
/// \brief  = operator
///
/// \param  oStr  [I ] string object
///
/// \return this object
///
LIBALT_API altWStr &  altWStr::operator =(const altStr & oStr)
{
  if (m_szStr == (altWChar *)oStr.GetCStr()) {
    return (* this);
  }

  ALT_BASE_FREE (m_szStr);
  m_nLen = oStr.GetLen();

#ifdef ALT_WIN
  ALT_BASE_MALLOC (m_szStr, (m_nLen + 1) * sizeof (altWChar));
  size_t  nSize = (m_nLen + 1) * sizeof (altWChar);
  mbstowcs_s (& nSize, (wchar_t *)m_szStr, m_nLen + 1, oStr.GetCStr(), oStr.GetLen());
#endif
#ifdef ALT_LINUX
  mbstowcs ((wchar_t *)m_szStr, oStr.GetCStr(), oStr.GetLen());
#endif

  m_nLen = altWStrLen (m_szStr);

  return (* this);
}

///
/// \brief  + operator
///
/// \param  oStr  [I ] string object
///
/// \return this object
///
LIBALT_API const altWStr altWStr::operator +(const altWStr & oStr)
{
  altWStr sTmp;
  sTmp += (* this);
  sTmp += oStr;
  return (sTmp);
}

///
/// \brief  += operator
///
/// \param  oStr  [I ] string object
///
/// \return this object
///
LIBALT_API const altWStr & altWStr::operator +=(const altWStr & oStr)
{
  ALT_BASE_EXALLOC (m_szStr, (m_nLen + 1) * sizeof (altWChar), (m_nLen + oStr.m_nLen + 1) * sizeof (altWChar));
  memcpy (m_szStr + m_nLen, oStr.m_szStr, oStr.m_nLen * sizeof (altWChar));
  m_nLen += oStr.m_nLen;
  return (* this);
}

///
/// \brief  == operator
///
/// \param  oStr  [I ] string object
///
/// \return true  same string
/// \return false other string
///
LIBALT_API const altBool altWStr::operator ==(const altWStr & oStr) const
{
  if (m_szStr == NULL) {
    return (false);
  }
  if (m_szStr == oStr.m_szStr) {
    return (true);
  }
  if (m_nLen != oStr.m_nLen) {
    return (false);
  }
  if (memcmp (m_szStr, oStr.m_szStr, m_nLen) == 0) {
    return (true);
  }
  return (false);
}

///
/// \brief  != operator
///
/// \param  oStr  [I ] string object
///
/// \return true  other string
/// \return false same string
///
LIBALT_API const altBool altWStr::operator !=(const altWStr & oStr) const
{
  altBool b = ((* this) == oStr);
  return (! b);
}

///
/// \brief  < operator
///
/// \param  oStr  [I ] string object
///
/// \return true  
/// \return false 
///
LIBALT_API const altBool altWStr::operator <(const altWStr & oStr) const
{
  if (m_szStr == oStr.m_szStr) {
    return (false);
  }

  for (altUInt i = 0; i < m_nLen && i < oStr.m_nLen; i++) {
    if (m_szStr[i] > oStr.m_szStr[i]) {
      return (false);
    }
    else if (m_szStr[i] < oStr.m_szStr[i]) {
      return (true);
    }
  }

  if (m_nLen == oStr.m_nLen) {
    return (false);
  }
  else if (m_nLen < oStr.m_nLen) {
    return (true);
  }

  return (false);
}

///
/// \brief  <= operator
///
/// \param  oStr  [I ] string object
///
/// \return true  
/// \return false 
///
LIBALT_API const altBool altWStr::operator <=(const altWStr & oStr) const
{
  if (m_szStr == oStr.m_szStr) {
    return (true);
  }

  for (altUInt i = 0; i < m_nLen && i < oStr.m_nLen; i++) {
    if (m_szStr[i] > oStr.m_szStr[i]) {
      return (false);
    }
    else if (m_szStr[i] < oStr.m_szStr[i]) {
      return (true);
    }
  }

  if (m_nLen == oStr.m_nLen) {
    return (true);
  }
  else if (m_nLen < oStr.m_nLen) {
    return (true);
  }

  return (false);
}

///
/// \brief  > operator
///
/// \param  oStr  [I ] string object
///
/// \return true  
/// \return false 
///
LIBALT_API const altBool altWStr::operator >(const altWStr & oStr) const
{
  if (m_szStr == oStr.m_szStr) {
    return (false);
  }

  for (altUInt i = 0; i < m_nLen && i < oStr.m_nLen; i++) {
    if (m_szStr[i] > oStr.m_szStr[i]) {
      return (true);
    }
    else if (m_szStr[i] < oStr.m_szStr[i]) {
      return (false);
    }
  }

  if (m_nLen == oStr.m_nLen) {
    return (false);
  }
  else if (m_nLen < oStr.m_nLen) {
    return (false);
  }

  return (true);
}

///
/// \brief  >= operator
///
/// \param  oStr  [I ] string object
///
/// \return true  
/// \return false 
///
LIBALT_API const altBool altWStr::operator >=(const altWStr & oStr) const
{
  if (m_szStr == oStr.m_szStr) {
    return (true);
  }

  for (altUInt i = 0; i < m_nLen && i < oStr.m_nLen; i++) {
    if (m_szStr[i] > oStr.m_szStr[i]) {
      return (true);
    }
    else if (m_szStr[i] < oStr.m_szStr[i]) {
      return (false);
    }
  }

  if (m_nLen == oStr.m_nLen) {
    return (true);
  }
  else if (m_nLen < oStr.m_nLen) {
    return (false);
  }

  return (true);
}

///
/// \brief  [] operator
///
/// \param  nIndex [I ] index
///
/// \return character of index position
///
LIBALT_API altWChar & altWStr::operator [](const altUInt nIndex) const
{
#if 0
  if (nIndex >= m_nLen) {
    return (0x0000);
  }
#endif
  return (m_szStr[nIndex]);
}

///
/// \brief  Convert Encoding
///
/// \param  szFrom [I ] From
/// \param  szTo   [I ] To
///
/// \return true success  
/// \return false error
///
LIBALT_API altBool altWStr::Convert(const altChar * szFrom, const altChar * szTo)
{
  iconv_t nfd = iconv_open (szFrom, szTo);

  altUInt     nSize = m_nLen * sizeof (altWChar);
  altWChar *  pBuf = NULL;
  size_t      nBufSize = nSize * 4;

  ALT_BASE_MALLOC (pBuf, nBufSize);

  altWChar *  pIn = m_szStr;
  altWChar *  pOut = pBuf;

  altStr  sBuf;
  ALT_LOG_P (ALT_D_DEBUG, sBuf.Format ("%s -> %s", szFrom, szTo));

  ALT_LOG_P (ALT_D_DEBUG, sBuf.Format ("%d", nBufSize));
#ifdef ALT_LINUX
  size_t s = iconv (nfd, (char **)& pIn, & nSize, (char **)& pOut, & nBufSize);
#endif
#ifdef ALT_WIN
  size_t s = iconv (nfd, (const char **)& pIn, & nSize, (char **)& pOut, & nBufSize);
#endif
  if (s < 0) {
    ALT_LOG_P (ALT_E_ERROR, sBuf.Format ("%d:%s", errno, strerror (errno)));
    return (false);
  }
  ALT_LOG_P (ALT_D_DEBUG, sBuf.Format ("%d", nBufSize));

  (* pOut) = 0x0000;

  ALT_BASE_FREE (m_szStr);
  ALT_BASE_WSTRALLOC (m_szStr, pBuf);
  m_nLen = altWStrLen (m_szStr);
  ALT_BASE_FREE (pBuf);

  iconv_close (nfd);

  return (true);
}

///
/// \brief  Get C styte string
///
/// \return pointer of string
///
LIBALT_API const altWChar * altWStr::GetCStr() const
{
  return (m_szStr);
}

///
/// \brief  Get string length
///
/// \return string length
///
LIBALT_API const altUInt altWStr::GetLen() const
{
  return (m_nLen);
}

///
/// \brief  Get buffer size
///
/// \return buffer size
///
LIBALT_API const altUInt altWStr::GetSize() const
{
  return (m_nLen * sizeof (altWChar));
}

///
/// \brief  Search string
///
/// \param  sStr  [I ] search string
///
/// \return found index
/// \return -1  not found
///
LIBALT_API altInt altWStr::Find(const altWStr & sStr) const
{
#if 0
  /// Using Boyer-Moore Algorithm
  altUInt aSkip[256 * 256];
  altUInt nStrLen = sStr.GetLen();
  if (nStrLen == 0) {
    return (-1);
  }

  // create skip table
  for (altUInt i = 0; i < ALT_ARRAY_SIZE (aSkip); i++) {
    aSkip[i] = nStrLen;
  }
  for (altUInt i = 0; i < nStrLen - 1; i++) {
    aSkip[static_cast<altUInt>(sStr[i])] = nStrLen - (i + 1);
  }

  altUInt i = nStrLen - 1;
  altUInt nLen = GetLen();

  while (i < nLen) {
    altUInt j = nStrLen - 1;

    while (m_szStr[i] == sStr[j]) {

      // string found
      if (j == 0) {
        return i;
      }

      i--;
      j--;
    }

    if (aSkip[static_cast<altUInt>(m_szStr[i])] > nStrLen - j) {
      i += aSkip[static_cast<altUInt>(m_szStr[i])];
    }
    else {
      i += (nStrLen - j);
    }
  }
#endif

  for (altInt i = 0; i < (altInt)(m_nLen); i++) {
    altInt nMatch = 0;
    for (altInt j = 0; j < (altInt)(sStr.m_nLen); j++) {
      if (m_szStr[i] == sStr[j] && m_szStr[i] != 0x0000) {
        nMatch++;
      }
    }
    if (nMatch == (altInt)sStr.m_nLen) {
      return (i);
    }
  }
  
  // string not found
  return (-1);
}

///
/// \brief  Search String from last
///
/// \param  sStr  [I ] search string
///
/// \return found index
/// \return -1    not found
///
LIBALT_API altInt altWStr::FindLastOf(const altWStr & sStr) const
{
  /// Using Boyer-Moore Algorithm
  altUInt aSkip[256 * 256];
  altUInt nStrLen = sStr.GetLen();

  // create skip table
  for (altUInt i = 0; i < ALT_ARRAY_SIZE (aSkip); i++) {
    aSkip[i] = nStrLen;
  }
  for (altUInt i = 0; i < nStrLen - 1; i++) {
    aSkip[static_cast<altUInt>(sStr[i])] = nStrLen - (i + 1);
  }

  altUInt nLen = GetLen();
  altUInt i = nLen - 1;

  while (i >= (nStrLen - 1)) {
    altUInt j = nStrLen - 1;

    while (m_szStr[i] == sStr[j]) {

      // string found
      if (j == 0) {
        return i;
      }

      i--;
      j--;
    }

    if (aSkip[static_cast<altUInt>(m_szStr[i])] > nStrLen - j) {
      i -= aSkip[static_cast<altUInt>(m_szStr[i])];
    }
    else {
      i -= (nStrLen - j);
    }
  }

  // string not found
  return (-1);
}

///
/// \brief  Search character
///
/// \param  sCharList  [I ] character list
///
/// \return found index
/// \return -1  not found
///
LIBALT_API altInt altWStr::FindChar(const altWStr & sCharList) const
{
  altUInt nLen = GetLen();
  altUInt nListLen = sCharList.GetLen();

  for (altUInt i = 0; i < nLen; i++) {
    for (altUInt j = 0; j < nListLen; j++) {
      if (m_szStr[i] == sCharList[j]) {
        return (altInt)(i);
      }
    }
  }
  return (-1);
}

///
/// \brief  Search character
///
/// \param  nBegin     [I ] begin index
/// \param  sCharList  [I ] character list
///
/// \return found index
/// \return -1  not found
///
LIBALT_API altInt altWStr::FindChar(const altUInt nBegin, const altWStr & sCharList) const
{
  altUInt nLen = GetLen();
  altUInt nListLen = sCharList.GetLen();

  for (altUInt i = nBegin; i < nLen; i++) {
    for (altUInt j = 0; j < nListLen; j++) {
      if (m_szStr[i] == sCharList[j]) {
        return (altInt)(i);
      }
    }
  }
  return (-1);
}

///
/// \brief  Search character from last
///
/// \param  sCharList  [I ] character list
///
/// \return found index
/// \return -1  not found
///
LIBALT_API altInt altWStr::FindLastOfChar(const altWStr & sCharList) const
{
  altUInt nLen = GetLen();
  altUInt nListLen = sCharList.GetLen();

  for (altUInt i = nLen - 1; i >= 0; i--) {
    for (altUInt j = 0; j < nListLen; j++) {
      if (m_szStr[i] == sCharList[j]) {
        return (i);
      }
    }
  }

  return (-1);
}

///
/// \brief  sub string
///
/// \param  nIndex  [I ] begin index
///
/// \return string
///
LIBALT_API altWStr altWStr::SubStr(const altUInt nIndex) const
{
  altWStr sSubStr;

  if ((* this)[nIndex] == 0x00) {
    return sSubStr;
  }

  sSubStr = & m_szStr[nIndex];

  return (sSubStr);
}

///
/// \brief  sub string
///
/// \param  nBegin  [I ] begin index
/// \param  nEnd    [I ] end index
///
/// \return string
///
LIBALT_API altWStr altWStr::SubStr(const altUInt nBegin, const altUInt nEnd) const
{
  if (nBegin > nEnd) {
    altWStr sRet;
    return (sRet);
  }
  altWChar * pSubStr = NULL;
  ALT_BASE_DUPALLOC (pSubStr, m_szStr, m_nLen * sizeof (altWChar));

  // no memory
  if (pSubStr == NULL) {
    altWStr sRet;
    return (sRet);
  }

  // Revision end index
  altUInt nLen = m_nLen;

  altUInt nRealEnd = nEnd;
  if (nRealEnd >= nLen) {
    nRealEnd = nLen - 1;
  }

  altUInt nSubStrLen = nRealEnd - nBegin + 1;

  // create sub string
  if (nBegin > 0) {
    memmove (pSubStr, & pSubStr[nBegin], nBegin * sizeof (* pSubStr));
  }
  else {
    memcpy (pSubStr, & pSubStr[nBegin], nSubStrLen * sizeof (* pSubStr));
  }
  pSubStr[nSubStrLen - 1] = 0x0000;

  altWStr sSubStr = pSubStr;
  ALT_BASE_FREE (pSubStr);

  return sSubStr;
}

///
/// \brief  replace
///
/// \param  sTarget   [I ] target string
/// \param  sReplaced [I ] replace string
///
/// \return 0 target not found
/// \return 1 target replaced
///
LIBALT_API altUInt altWStr::Replace(const altWStr & sTarget, const altWStr & sReplaced)
{
  altInt  nIndex = this->Find (sTarget);
  if (nIndex < 0) {
    return (0);
  }

  if (sTarget.GetLen() != sReplaced.GetLen()) {
    altUInt     nNowSize = (m_nLen + 1) * sizeof (altWChar);
    altUInt     nNewSize = nNowSize + ((sReplaced.GetLen() - sTarget.GetLen()) * sizeof (altWChar));
    altWChar *  pPtr = NULL;
    
    altUInt nRet = ALT_BASE_MALLOC (pPtr, nNewSize);
    if (nRet != ALT_SUCCESS) {
      return (0);
    }
    if (nIndex > 0) {
      memcpy (pPtr, m_szStr, nIndex);
    }
    memcpy (& pPtr[nIndex], sReplaced.GetCStr(), sReplaced.GetLen() * sizeof (altWChar));
    memcpy (& pPtr[nIndex + sReplaced.GetLen()], & m_szStr[nIndex + sTarget.GetLen()], nNowSize - ((nIndex + sTarget.GetLen()) * sizeof (altWChar)));

    ALT_BASE_FREE (m_szStr);
    m_szStr = pPtr;
  }
  else {
    memcpy (& m_szStr[nIndex], sReplaced.GetCStr(), sReplaced.GetLen() * sizeof (altWChar));
  }

  return (1);
}

///
/// \brief  replace all
///
/// \param  sTarget   [I ] target string
/// \param  sReplaced [I ] replace string
///
/// \return replaced count
///
LIBALT_API altUInt altWStr::ReplaceAll(const altWStr & sTarget, const altWStr & sReplaced)
{
  altUInt nRet;
  altUInt nCount = 0;
  while ((nRet = Replace (sTarget, sReplaced)) == 1) {
    nCount++;
  }
  return (nCount);
}

///
/// \brief  Separate string
///
/// \param  sSeparator [I ] Separator character list
///
/// \return Separated string list
///
LIBALT_API altWStrList altWStr::Separate(const altWStr & sSeparator) const
{
  altWStrList oList;
  altInt      nBegin = 0;
  altInt      nEnd;
  altStr      sTmp;

  while ((nEnd = this->FindChar (nBegin, sSeparator)) > 0) {
	  sTmp = this->SubStr (nBegin, nEnd - 1);
	  oList.push_back (sTmp);
    nBegin = nEnd + 1;
  }
  if (nBegin > 0) {
    sTmp = this->SubStr (nBegin);
    oList.push_back (sTmp);
  }

  return (oList);
}

///
/// \brief  Format
///
/// \param  szFormat  [I ] Format
///
/// \return Format string
///
LIBALT_API const altWChar * altWStr::Format(const wchar_t * szFormat, ...)
{
  altWChar *  pBuf = NULL;
  va_list     oArgs;
  altUInt     nNowSize = 0;
  altUInt     nNewSize = BUFSIZ;
  altInt      nRet;

  while (1) {
    nRet = ALT_BASE_EXALLOC (pBuf, nNowSize * sizeof (* pBuf), nNewSize * sizeof (* pBuf));
    if (nRet != ALT_SUCCESS) {
      ALT_BASE_FREE (pBuf);
      return NULL;
    }
    nNowSize = nNewSize;

    va_start (oArgs, szFormat);
    nRet = altVswnprintf (pBuf, nNowSize, szFormat, oArgs);
    va_end (oArgs);
    if (nRet > -1 && static_cast<altUInt>(nRet) < nNowSize) {
      break;
    }

    if (static_cast<altUInt>(nRet) >= nNowSize) {
      nNewSize = nRet + 1;
    }
    else {
      nNewSize = nNowSize * 2;
    }
  }

  ALT_BASE_FREE (m_szStr);
  ALT_BASE_WSTRALLOC (m_szStr, pBuf);
  ALT_BASE_FREE (pBuf);
  return (m_szStr);
}

///
/// \brief  Delete last character
///
/// \return true  success
/// \return false failed
///
LIBALT_API altBool altWStr::DeleteLastChar()
{
  if (m_nLen == 0) {
    return (false);
  }
  (* this) = SubStr(0, m_nLen - 1);
  return (true);
}

///
/// \brief  Check space or not
///
/// \return true ... space
///
LIBALT_API altBool altWStr::IsWSpace(altWChar wc)
{
#ifdef ALT_DOTNET
  return Char::IsWhiteSpace(wc);
#else
  return ::iswspace (wc);
#endif
}

///
/// \brief  Left trim
///
/// \return Wide string
///
LIBALT_API altWStr altWStr::LTrim()
{
  altUInt nLen = 0;
  for (altUInt i = 0; IsWSpace (m_szStr[i]); i++) {
    nLen++;
  }
  
  return (this->SubStr(0, nLen));
}

///
/// \brief  Right trim
///
/// \return Wide string
///
LIBALT_API altWStr altWStr::RTrim()
{
  altUInt nLen = 0;
  for (altUInt i = m_nLen; IsWSpace (m_szStr[i]); i--) {
    nLen++;
  }
  return (this->SubStr (nLen, m_nLen));
}

///
/// \brief  Trim
///
/// \return Wide string
///
LIBALT_API altWStr altWStr::Trim()
{
  this->LTrim();
  this->RTrim();
  return (this->m_szStr);
}

#if 0
///
/// \brief  Regex search
///
/// \param  sRegex  [I ] Regex string
///
/// \return Find string
///
LIBALT_API altWStr altWStr::RegexSearch(const altWStr & sRegex) const
{
  try {
    boost::basic_regex<wchar_t>  oRegex((const wchar_t *)sRegex.m_szStr);
    boost::match_results<const wchar_t *> oMatch;

    std::wstring sStr = (const wchar_t *)m_szStr;
    altBool bFound = boost::regex_search ((const wchar_t *)sStr.c_str(), oMatch, oRegex);
    if (bFound) {
      return (oMatch.str().c_str());
    }
  }
  catch (std::exception & e) {
    ALT_LOG_P (ALT_E_ERROR, e.what());
  }

  return (L"");
}
#endif

///
/// \brief  Get Boolean Value
///
/// \return Boolean Value
///
LIBALT_API altBool altWStr::ToBool() const
{
  return ((* this) == L"true");
}

///
/// \brief  Get Byte Value
///
/// \return Byte Value
///
LIBALT_API altByte altWStr::ToByte() const
{
  altByte nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%hhu", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%hhu", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Short Value
///
/// \return Short Value
///
LIBALT_API altShort altWStr::ToShort() const
{
  altShort nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%hd", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%hd", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Unsigned Short Value
///
/// \return Unsigned Short Value
///
LIBALT_API altUShort altWStr::ToUShort() const
{
  altUShort nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%hu", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%hu", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Int Value
///
/// \return Int Value
///
LIBALT_API altInt altWStr::ToInt() const
{
  altInt nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%d", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%d", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Unsigned Int Value
///
/// \return Unsinged Int Value
///
LIBALT_API altUInt altWStr::ToUInt() const
{
  altUInt nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%u", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%u", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Long Value
///
/// \return Long Value
///
LIBALT_API altLong altWStr::ToLong() const
{
  altLong nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%ld", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%ld", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Unsigned Long Value
///
/// \return Unsinged Long Value
///
LIBALT_API altULong altWStr::ToULong() const
{
  altULong nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%lu", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%lu", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get DWORD Value
///
/// \return DWORD Value
///
LIBALT_API DWORD altWStr::ToDWORD() const
{
  return (this->ToULong());
}

///
/// \brief  Get Long Long Value
///
/// \return Long Long Value
///
LIBALT_API altLongLong altWStr::ToLongLong() const
{
  altLongLong nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%lld", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%lld", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get Unsigned Long Long Value
///
/// \return Unsigned Long Long Value
///
LIBALT_API altULongLong altWStr::ToULongLong() const
{
  altULongLong nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%llu", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%llu", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get float Value
///
/// \return float Value
///
LIBALT_API altFloat altWStr::ToFloat() const
{
  altFloat nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%f", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%f", & nValue);
#endif
  return (nValue);
}

///
/// \brief  Get double Value
///
/// \return double Value
///
LIBALT_API altDouble altWStr::ToDouble() const
{
  altDouble nValue = 0;
#ifdef ALT_WIN
  ::swscanf_s ((const wchar_t *)m_szStr, L"%lf", & nValue);
#endif
#ifdef ALT_LINUX
  ::swscanf ((const wchar_t *)m_szStr, L"%lf", & nValue);
#endif
  return (nValue);
}

///
/// \brief  + operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
LIBALT_API altWStr operator + (const altWStr & oLeftStr, const altWStr & oRightStr)
{
  altWStr oStr = oLeftStr;
  oStr += oRightStr;
  return oStr;
}

///
/// \brief  == operator
///
/// \param  szLeftStr [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true same string
/// \return false not same string
///
LIBALT_API altBool operator == (const altWChar * szLeftStr, const altWStr & oRightStr)
{
  return (oRightStr == szLeftStr);
}

///
/// \brief  != operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true not same string
/// \return false same string
///
LIBALT_API altBool operator != (const altWChar * szLeftStr, const altWStr & oRightStr)
{
  return (oRightStr != szLeftStr);
}

///
/// \brief  < operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String <  Right String
/// \return false Left String >= Right String
///
LIBALT_API altBool operator < (const altWChar * szLeftStr, const altWStr & oRightStr)
{
  return (oRightStr > szLeftStr);
}

///
/// \brief  <= operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String <=  Right String
/// \return false Left String > Right String
///
LIBALT_API altBool operator <= (const altWChar * szLeftStr, const altWStr & oRightStr)
{
  return (oRightStr >= szLeftStr);
}

///
/// \brief  > operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String >  Right String
/// \return false Left String <= Right String
///
LIBALT_API altBool operator > (const altWChar * szLeftStr, const altWStr & oRightStr)
{
  return (oRightStr < szLeftStr);
}

///
/// \brief  >= operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String >=  Right String
/// \return false Left String < Right String
///
LIBALT_API altBool operator >= (const altWChar * szLeftStr, const altWStr & oRightStr)
{
  return (oRightStr <= szLeftStr);
}
