#ifndef ALTWSTR_H
#define ALTWSTR_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altWStr.h 1425 2009-09-08 21:15:17Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altWStr.h
 * \brief   Wide String Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altBase/altStr.h>
#include <altBase/altBase.h>

#ifdef ALT_WIN
#include <altBase/iconv/iconv.h>
#endif

#ifdef ALT_LINUX
#include <iconv.h>
#endif

#include <vector>

/*----------------------------------------------------------------*
 * Define
 *----------------------------------------------------------------*/
#ifdef ALT_WIN
///
/// \brief  CString to altWStr
///
/// \param  wStr  [IO] altWStr
/// \param  cStr  [I ] CString
///
#define ALT_WCOPY_CSTRING(wStr, cStr) \
{ \
  alt_t status; \
  altWChar * pBuf = NULL; \
  status = ALT_MALLOC (pBuf, ((cStr).GetLength() + 1) * sizeof (altWChar)); \
  ALT_LOG (status); \
  wsprintf ((LPWSTR)pBuf, L"%s", cStr); \
  wStr = pBuf; \
}
#endif

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
class altStr;
class altWStr;
typedef std::vector<altWStr> altWStrList;

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  String
///
class altWStr : public altBase
{
public:
  ///
  /// \brief  Constructor
  ///
  LIBALT_API altWStr();

  ///
  /// \brief  Constructor
  ///
  /// \param  c [I ] Character
  ///
  LIBALT_API altWStr(const altWChar c);

  ///
  /// \brief  Constructor
  ///
  /// \param  szStr [I ] string
  ///
  LIBALT_API altWStr(const altWChar * szStr);

  ///
  /// \brief  Constructor
  ///
  /// \param  szStr [I ] string
  /// \param  nLen  [I ] length
  ///
  LIBALT_API altWStr(const altWChar * szStr, const altUInt nLen);

  ///
  /// \brief  Constructor
  ///
  /// \param  szStr [I ] string
  ///
  LIBALT_API altWStr(const wchar_t * szStr);

  ///
  /// \brief  Copy Constructor
  ///
  /// \param  oStr [I ] string
  ///
  LIBALT_API altWStr(const altWStr & oStr);

  ///
  /// \brief  Copy Constructor
  ///
  /// \param  oStr [I ] string
  ///
  LIBALT_API altWStr(const altStr & oStr);

  ///
  /// \brief  Destructor
  ///
  LIBALT_API virtual ~altWStr();
  
  ///
  /// \brief  = operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return this object
  ///
  LIBALT_API altWStr &  operator =(const altWStr & oStr);

  ///
  /// \brief  = operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return this object
  ///
  LIBALT_API altWStr &  operator =(const altStr & oStr);

  ///
  /// \brief  + operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return this object
  ///
  LIBALT_API const altWStr operator +(const altWStr & oStr);

  ///
  /// \brief  += operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return this object
  ///
  LIBALT_API const altWStr & operator +=(const altWStr & oStr);

  ///
  /// \brief  == operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return true  same string
  /// \return false other string
  ///
  LIBALT_API const altBool operator ==(const altWStr & oStr) const;

  ///
  /// \brief  != operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return true  other string
  /// \return false same string
  ///
  LIBALT_API const altBool operator !=(const altWStr & oStr) const;

  ///
  /// \brief  < operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return true  
  /// \return false 
  ///
  LIBALT_API const altBool operator <(const altWStr & oStr) const;

  ///
  /// \brief  <= operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return true  
  /// \return false 
  ///
  LIBALT_API const altBool operator <=(const altWStr & oStr) const;

  ///
  /// \brief  > operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return true  
  /// \return false 
  ///
  LIBALT_API const altBool operator >(const altWStr & oStr) const;
  
  ///
  /// \brief  >= operator
  ///
  /// \param  oStr  [I ] string object
  ///
  /// \return true  
  /// \return false 
  ///
  LIBALT_API const altBool operator >=(const altWStr & oStr) const;

  ///
  /// \brief  [] operator
  ///
  /// \param  nIndex [I ] index
  ///
  /// \return character of index position
  ///
  LIBALT_API altWChar & operator [](const altUInt nIndex) const;

  ///
  /// \brief  Convert Encoding
  ///
  /// \param  szFrom [I ] From
  /// \param  szTo   [I ] To
  ///
  /// \return true success  
  /// \return false error
  ///
  LIBALT_API altBool Convert(const altChar * szFrom = "UTF-16", const altChar * szTo = "CP932");

  ///
  /// \brief  Get C styte string
  ///
  /// \return pointer of string
  ///
  LIBALT_API const altWChar * GetCStr() const;

  ///
  /// \brief  Get string length
  ///
  /// \return string length
  ///
  LIBALT_API const altUInt GetLen() const;

  ///
  /// \brief  Get buffer size
  ///
  /// \return buffer size
  ///
  LIBALT_API const altUInt GetSize() const;
  
  ///
  /// \brief  Search string
  ///
  /// \param  sStr  [I ] search string
  ///
  /// \return found index
  /// \return -1  not found
  ///
  LIBALT_API altInt Find(const altWStr & sStr) const;

  ///
  /// \brief  Search String from last
  ///
  /// \param  sStr  [I ] search string
  ///
  /// \return found index
  /// \return -1    not found
  ///
  LIBALT_API altInt FindLastOf(const altWStr & sStr) const;

  ///
  /// \brief  Search character
  ///
  /// \param  sCharList  [I ] character list
  ///
  /// \return found index
  /// \return -1  not found
  ///
  LIBALT_API altInt FindChar(const altWStr & sCharList) const;

  ///
  /// \brief  Search character
  ///
  /// \param  nBegin     [I ] begin index
  /// \param  sCharList  [I ] character list
  ///
  /// \return found index
  /// \return -1  not found
  ///
  LIBALT_API altInt FindChar(const altUInt nBegin, const altWStr & sCharList) const;

  ///
  /// \brief  Search character from last
  ///
  /// \param  sCharList  [I ] character list
  ///
  /// \return found index
  /// \return -1  not found
  ///
  LIBALT_API altInt FindLastOfChar(const altWStr & sCharList) const;

  ///
  /// \brief  sub string
  ///
  /// \param  nIndex  [I ] begin index
  ///
  /// \return string
  ///
  LIBALT_API altWStr SubStr(const altUInt nIndex) const;

  ///
  /// \brief  sub string
  ///
  /// \param  nBegin  [I ] begin index
  /// \param  nEnd    [I ] end index
  ///
  /// \return string
  ///
  LIBALT_API altWStr SubStr(const altUInt nBegin, const altUInt nEnd) const;

  ///
  /// \brief  replace
  ///
  /// \param  sTarget   [I ] target string
  /// \param  sReplaced [I ] replace string
  ///
  /// \return 0 target not found
  /// \return 1 target replaced
  ///
  LIBALT_API altUInt Replace(const altWStr & sTarget, const altWStr & sReplaced);

  ///
  /// \brief  replace all
  ///
  /// \param  sTarget   [I ] target string
  /// \param  sReplaced [I ] replace string
  ///
  /// \return replaced count
  ///
  LIBALT_API altUInt ReplaceAll(const altWStr & sTarget, const altWStr & sReplaced);

  ///
  /// \brief  Separate string
  ///
  /// \param  sSeparator [I ] Separator character list
  ///
  /// \return Separated string list
  ///
  LIBALT_API altWStrList Separate(const altWStr & sSeparator) const;

  ///
  /// \brief  Format
  ///
  /// \param  szFormat  [I ] Format
  ///
  /// \return Format string
  ///
  LIBALT_API const altWChar * Format(const wchar_t * szFormat, ...);

  ///
  /// \brief  Delete last character
  ///
  /// \return true  success
  /// \return false failed
  ///
  LIBALT_API altBool DeleteLastChar();

  ///
  /// \brief  Check space or not
  ///
  /// \return true ... space
  ///
  LIBALT_API altBool altWStr::IsWSpace(altWChar wc);

  ///
  /// \brief  Left trim
  ///
  /// \return Wide string
  ///
  LIBALT_API altWStr LTrim();

  ///
  /// \brief  Right trim
  ///
  /// \return Wide string
  ///
  LIBALT_API altWStr RTrim();

  ///
  /// \brief  Trim
  ///
  /// \return Wide string
  ///
  LIBALT_API altWStr Trim();

#if 0
  ///
  /// \brief  Regex search
  ///
  /// \param  sRegex  [I ] Regex string
  ///
  /// \return Find string
  ///
  LIBALT_API altWStr RegexSearch(const altWStr & sRegex) const;
#endif

  ///
  /// \brief  Get Boolean Value
  ///
  /// \return Boolean Value
  ///
  LIBALT_API altBool ToBool() const;

  ///
  /// \brief  Get Byte Value
  ///
  /// \return Byte Value
  ///
  LIBALT_API altByte ToByte() const;

  ///
  /// \brief  Get Short Value
  ///
  /// \return Short Value
  ///
  LIBALT_API altShort ToShort() const;

  ///
  /// \brief  Get Unsigned Short Value
  ///
  /// \return Unsigned Short Value
  ///
  LIBALT_API altUShort ToUShort() const;

  ///
  /// \brief  Get Int Value
  ///
  /// \return Int Value
  ///
  LIBALT_API altInt ToInt() const;

  ///
  /// \brief  Get Unsigned Int Value
  ///
  /// \return Unsinged Int Value
  ///
  LIBALT_API altUInt ToUInt() const;

  ///
  /// \brief  Get Long Value
  ///
  /// \return Long Value
  ///
  LIBALT_API altLong ToLong() const;

  ///
  /// \brief  Get Unsigned Long Value
  ///
  /// \return Unsinged Long Value
  ///
  LIBALT_API altULong ToULong() const;

  ///
  /// \brief  Get DWORD Value
  ///
  /// \return DWORD Value
  ///
  LIBALT_API DWORD ToDWORD() const;

  ///
  /// \brief  Get Long Long Value
  ///
  /// \return Long Long Value
  ///
  LIBALT_API altLongLong ToLongLong() const;

  ///
  /// \brief  Get Unsigned Long Long Value
  ///
  /// \return Unsigned Long Long Value
  ///
  LIBALT_API altULongLong ToULongLong() const;

  ///
  /// \brief  Get float Value
  ///
  /// \return float Value
  ///
  LIBALT_API altFloat ToFloat() const;

  ///
  /// \brief  Get double Value
  ///
  /// \return double Value
  ///
  LIBALT_API altDouble ToDouble() const;

private:
  altWChar *  m_szStr;  ///< string buffer
  altUInt     m_nLen;   ///< string length
};

///
/// \brief  + operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return + Operated String
///
LIBALT_API altWStr operator + (const altWStr & oLeftStr, const altWStr & oRightStr);

///
/// \brief  == operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true same string
/// \return false not same string
///
LIBALT_API altBool operator == (const altWChar * szLeftStr, const altWStr & oRightStr);

///
/// \brief  != operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true not same string
/// \return false same string
///
LIBALT_API altBool operator != (const altWChar * szLeftStr, const altWStr & oRightStr);

///
/// \brief  < operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String <  Right String
/// \return false Left String >= Right String
///
LIBALT_API altBool operator < (const altWChar * szLeftStr, const altWStr & oRightStr);

///
/// \brief  <= operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String <=  Right String
/// \return false Left String > Right String
///
LIBALT_API altBool operator <= (const altWChar * szLeftStr, const altWStr & oRightStr);

///
/// \brief  > operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String >  Right String
/// \return false Left String <= Right String
///
LIBALT_API altBool operator > (const altWChar * szLeftStr, const altWStr & oRightStr);

///
/// \brief  >= operator
///
/// \param  oLeftStr  [I ] Left String Parameter
/// \param  oRightStr [I ] Right String Parameter
///
/// \return true  Left String >=  Right String
/// \return false Left String < Right String
///
LIBALT_API altBool operator >= (const altWChar * szLeftStr, const altWStr & oRightStr);


#endif
