/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altSQLLoader.cpp 1347 2008-04-20 20:00:21Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altSQLLoader.cpp
 * \brief   SQL Loader Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altSQLLoader.h"
#include "altMisc/altFile.h"
#include "altBase/altMem.h"

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altSQLLoader::altSQLLoader() :
m_oSQLContainer(),
m_oWSQLContainer(),
m_sBaseDir(),
m_bUnicode (false)
{
}

///
/// \brief  Destructor
///
LIBALT_API altSQLLoader::~altSQLLoader()
{
  for (altSQLContainer::iterator i = m_oSQLContainer.begin(); i != m_oSQLContainer.end(); i++) {
    ALT_FREE (i->second);
  }
  m_oSQLContainer.clear();
  for (altWSQLContainer::iterator i = m_oWSQLContainer.begin(); i != m_oWSQLContainer.end(); i++) {
    ALT_FREE (i->second);
  }
  m_oWSQLContainer.clear();
}

///
/// \brief  Set Directory
///
/// \param  sDir  [I ] Directory Path
///
/// \return ALT_S_SUCCESS   Success
///
LIBALT_API alt_t altSQLLoader::SetDir(const altStr & sDir)
{
  m_sBaseDir = sDir;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Unicode mode
///
/// \param  bUnicode  [I ] true...unicode false...not unicode
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altSQLLoader::SetUnicode(const altBool bUnicode)
{
  m_bUnicode = bUnicode;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get SQL
///
/// \param  sFilePath   [I ] File Path
/// \param  oSQL        [ O] SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQL(const altStr & sFilePath, altSQL & oSQL)
{
  alt_t status;
  if (m_bUnicode) {
    status = GetSQLW (sFilePath, oSQL);
  }
  else {
    status = GetSQLA (sFilePath, oSQL);
  }
  ALT_RET (status);
}

///
/// \brief  Get SQL
///
/// \param  sFilePath   [I ] File Path
/// \param  oSQL        [ O] SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQLA(const altStr & sFilePath, altSQL & oSQL)
{
  altStr  sPath;
  altStr  sSQL;

  if (m_sBaseDir.GetLen() > 0) {
    sPath += m_sBaseDir;
    sPath += "/";
  }
  sPath += sFilePath;

  alt_t status;
  altChar * szSQL = m_oSQLContainer[sPath];
  if (szSQL != NULL) {
    status = oSQL.SetSQL (szSQL, sFilePath);
    ALT_ERR_RET (status);
    ALT_RET (ALT_S_SUCCESS);
  }

  altFile oFile;

  status = oFile.Open (sPath, "r");
  ALT_ERR_RET (status);

  altChar szBuf[BUFSIZ];
  while ((status = oFile.ReadLine (szBuf, sizeof (szBuf))) == ALT_S_SUCCESS) {
    // comment
    if (szBuf[0] == '#') {
      continue;
    }
    sSQL += szBuf;
    sSQL + " ";
  }

  altStr  sBuf;
  ALT_LOG_P (ALT_D_DEBUG, sBuf.Format ("SQL=%s", sSQL.GetCStr()));

  status = oFile.Close();
  ALT_ERR_RET (status);

  altChar * pSQL = NULL;
  status = ALT_STRALLOC (pSQL, sSQL.GetCStr());
  ALT_ERR_RET (status);
  m_oSQLContainer[sPath] = pSQL;

  status = oSQL.SetSQL (pSQL, sFilePath);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get SQL
///
/// \param  sFilePath   [I ] File Path
/// \param  oSQL        [ O] SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQLW(const altStr & sFilePath, altSQL & oSQL)
{
  altStr  sPath;
  altWStr sSQL;

  if (m_sBaseDir.GetLen() > 0) {
    sPath += m_sBaseDir;
    sPath += "/";
  }
  sPath += sFilePath;

  alt_t status;
  altWChar * szSQL = m_oWSQLContainer[sPath];
  if (szSQL != NULL) {
    status = oSQL.SetSQL (szSQL, sFilePath);
    ALT_ERR_RET (status);
    ALT_RET (ALT_S_SUCCESS);
  }

  altFile oFile;

  status = oFile.Open (sPath, "r");
  ALT_ERR_RET (status);

  altUInt   nLine = 0;
  altWChar  szBuf[BUFSIZ];
  memset (szBuf, 0x00, sizeof (szBuf));
  while ((status = oFile.ReadLine (szBuf, (sizeof (szBuf) - sizeof (altWChar)))) == ALT_S_SUCCESS) {
    if (nLine == 0) {
      altWChar wUTF16;
      memcpy (& wUTF16, szBuf, sizeof (wUTF16));
      if (wUTF16 == 0xFEFF) {
        memmove (& szBuf[0], & szBuf[1], sizeof (szBuf));
      }
      else {
        ALT_RET_P (ALT_E_ERROR, "Support UTF-16 only");
      }
    }
    nLine++;
    // comment
    if (szBuf[0] == L'#') {
      continue;
    }
    sSQL += szBuf;
    sSQL += L" ";
    memset (szBuf, 0x00, sizeof (szBuf));
  }

  status = oFile.Close();
  ALT_ERR_RET (status);

  altWChar * pSQL = NULL;
  status = ALT_STRWALLOC (pSQL, sSQL.GetCStr());
  ALT_ERR_RET (status);
  m_oWSQLContainer[sPath] = pSQL;

  status = oSQL.SetSQL (pSQL, sFilePath);
  ALT_ERR_RET (status);

  altStr  sBuf;
  ALT_LOG_P (ALT_D_DEBUG, sBuf.Format ("%d", oSQL.GetSQLW().GetLen()));

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Prepared Statement SQL
///
/// \param  sFilePath     [I ] File Path
/// \param  oDBConnection [I ] DB Connection
/// \param  pStmt         [ O] Prepared Statement SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQL(const altStr & sFilePath, altDBConnection & oDBConnection, altPreparedStatement * & pStmt){
  alt_t status;
  if (Unicode()) {
    status = GetSQLW (sFilePath, oDBConnection, pStmt);
  }
  else {
    status = GetSQLA (sFilePath, oDBConnection, pStmt);
  }
  return (status);
}

///
/// \brief  Get Prepared Statement SQL
///
/// \param  sFilePath     [I ] File Path
/// \param  oDBConnection [I ] DB Connection
/// \param  pStmt         [ O] Prepared Statement SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQLA(const altStr & sFilePath, altDBConnection & oDBConnection, altPreparedStatement * & pStmt)
{
  alt_t   status;
  altSQL  oSQL;

  status = GetSQL (sFilePath, oSQL);
  ALT_ERR_RET (status);

  status = oDBConnection.GetPreparedStatement (oSQL, pStmt);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Prepared Statement SQL
///
/// \param  sFilePath     [I ] File Path
/// \param  oDBConnection [I ] DB Connection
/// \param  pStmt         [ O] Prepared Statement SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQLW(const altStr & sFilePath, altDBConnection & oDBConnection, altPreparedStatement * & pStmt)
{
  alt_t   status;
  altSQL  oSQL;

  status = GetSQL (sFilePath, oSQL);
  ALT_ERR_RET (status);

  status = oDBConnection.GetPreparedStatement (oSQL, pStmt);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Prepared Statement SQL
///
/// \param  sFilePath     [I ] File Path
/// \param  oDBConnection [I ] DB Connection
/// \param  pStmt         [ O] Prepared Statement SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQL(const altStr & sFilePath, altDBConnectionPtr & pDBConnection, altPreparedStatement * & pStmt)
{
  alt_t status;
  if (Unicode()) {
    status = GetSQLW (sFilePath, pDBConnection, pStmt);
  }
  else {
    status = GetSQLA (sFilePath, pDBConnection, pStmt);
  }
  ALT_RET (status);
}

///
/// \brief  Get Prepared Statement SQL
///
/// \param  sFilePath     [I ] File Path
/// \param  oDBConnection [I ] DB Connection
/// \param  pStmt         [ O] Prepared Statement SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQLA(const altStr & sFilePath, altDBConnectionPtr & pDBConnection, altPreparedStatement * & pStmt)
{
  alt_t   status;
  altSQL  oSQL;

  status = GetSQL (sFilePath, oSQL);
  ALT_ERR_RET (status);

  status = pDBConnection->GetPreparedStatement (oSQL, pStmt);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Prepared Statement SQL
///
/// \param  sFilePath     [I ] File Path
/// \param  oDBConnection [I ] DB Connection
/// \param  pStmt         [ O] Prepared Statement SQL
///
/// \return ALT_S_SUCCESS   Success
/// \return ALT_E_ERROR     Error
///
LIBALT_API alt_t altSQLLoader::GetSQLW(const altStr & sFilePath, altDBConnectionPtr & oDBConnection, altPreparedStatement * & pStmt)
{
  alt_t   status;
  altSQL  oSQL;

  status = GetSQL (sFilePath, oSQL);
  ALT_ERR_RET (status);

  status = oDBConnection->GetPreparedStatement (oSQL, pStmt);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Unicode or not
///
/// \return true  Unicode
/// \return false Not Unicode
///
LIBALT_API altBool altSQLLoader::Unicode()
{
  return (m_bUnicode);
}
