/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id$

  Copyright (C) 2009 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altSQLServerDBConnection.cpp
 * \brief   SQL SERVER DB Connection Class
 * \date    2009
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altSQLServerDBConnection.h"
#include "altSQLLoader.h"
#include "altMisc/altFile.h"
#include "altBase/altUtil.h"

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altSQLServerDBConnection::altSQLServerDBConnection()
{
}

///
/// \brief  Destructor
///
LIBALT_API altSQLServerDBConnection::~altSQLServerDBConnection()
{
}

///
/// \brief  Connect
///
/// \param  sHost       [I ] Host Name
/// \param  nPort       [I ] Port Number
/// \param  sDB         [I ] DB Name
/// \param  sUser       [I ] User ID
/// \param  sPass       [I ] Password
/// \param  sCharset    [I ] Charcter Set
/// \param  bAutoCommit [I ] Auto commit or not
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altSQLServerDBConnection::Connect(const altStr & sHost, const altUInt nPort, const altStr & sDB, const altStr & sUser, const altStr & sPass, const altStr & sCharset, const altBool bAutoCommit)
{
  alt_t status;

  RETCODE rc;

  SQLAllocEnv (& henv);
  SQLAllocConnect (henv, & hdbc);

  rc = SQLConnectA(hdbc, (SQLCHAR *)sHost.GetCStr(), SQL_NTS, (SQLCHAR *)sUser.GetCStr(), SQL_NTS, (SQLCHAR *)sPass.GetCStr(), SQL_NTS);

  if (rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO) {
    ALT_RET (ALT_ERROR);
  }

  status = ClearPreparedStatementCache();
  ALT_ERR_RET (status);

  altStr  sBuf;
  ALT_LOG_P (ALT_I_INFO, sBuf.Format ("Connect DB %s", sHost.GetCStr());

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Disconnect
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altSQLServerDBConnection::Disconnect()
{
  alt_t status;

  SQLFreeStmt(hstmt, SQL_DROP);
  SQLDisconnect(hdbc);
  SQLFreeConnect(hdbc);
  SQLFreeEnv(henv);

  status = ClearPreparedStatementCache();
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Execute SQL
///
/// \param  oSQL    [I ] SQL
/// \param  oResult [ O] Result Set
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::ExecuteSQL(const altSQL & oSQL, altDBResultSet & oResult)
{
  alt_t status;

  status = oResult.Clear();
  ALT_ERR_RET (status);

  if (aiSQLLoader.Unicode()) {
    status = ExecuteSQL (oSQL.GetSQLW(), oResult);
  }
  else {
    status = ExecuteSQL (oSQL.GetSQLA(), oResult);
  }
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Execute SQL
///
/// \param  oSQL          [I ] SQL
/// \param  pAffectedRow  [ O] Affected Row
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::ExecuteSQL(const altSQL & oSQL, altULongLong * pAffectedRow)
{
  alt_t status;

  // Execute Query
  if (aiSQLLoader.Unicode()) {
    status = ExecuteSQL (oSQL.GetSQLW(), pAffectedRow);
  }
  else {
    ALT_LOG_P (ALT_D_DEBUG, oSQL.GetSQLA().GetCStr());
    status = ExecuteSQL (oSQL.GetSQLA(), pAffectedRow);
  }
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Prepared Statement
///
/// \param  oSQL        [I ] SQL
/// \param  pStatement  [ O] Prepared Statement
///
/// \return ALT_S_SUCCESS Success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::GetPreparedStatement(const altSQL & oSQL, altPreparedStatement * & pStatement)
{
  pStatement = m_oPreparedStatementMap[oSQL.GetSQLName()];
  if (pStatement != NULL) {
    ALT_RET (ALT_S_SUCCESS);
  }

  alt_t status;
  altMySQLPreparedStatement * pNewStmt = ALT_NEW altMySQLPreparedStatement (oSQL, m_oMySQL);

  status = pNewStmt->Prepare();
  ALT_ERR_RET (status);

  m_oPreparedStatementMap[oSQL.GetSQLName()] = pNewStmt;
  pStatement = pNewStmt;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Execute SQL
///
/// \param  oStmt       [IO] Prepared Statement
/// \param  oResultSet  [ O] ResultSet
///
/// \return ALT_S_SUCCESS Success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::ExecuteSQL(altPreparedStatement & oStmt, altDBResultSet & oResultSet)
{
  alt_t status;

  status = oResultSet.Clear();
  ALT_ERR_RET (status);

  status = oStmt.ExcecuteSQL (oResultSet);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Execute SQL
///
/// \param  oStmt         [IO] Prepared Statement
/// \param  pAffectedRow  [ O] ResultSet
///
/// \return ALT_S_SUCCESS Success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::ExecuteSQL(altPreparedStatement & oStmt, altULongLong * pAffectedRow)
{
  alt_t status;

  status = oStmt.ExcecuteSQL (pAffectedRow);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Begin
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altSQLServerDBConnection::Begin()
{
  alt_t   status;
  altSQL  oSQL;
  oSQL.SetSQL ("Begin", "");

  status = this->ExecuteSQL (oSQL, NULL);
  ALT_LOG (status);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Commit
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altSQLServerDBConnection::Commit()
{
  alt_t status;

  status = m_oMySQL.Commit();
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Rollback
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altSQLServerDBConnection::Rollback()
{
  alt_t status;

  status = m_oMySQL.Rollback();
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Execute SQL
///
/// \param  sSQL    [I ] SQL
/// \param  oResult [ O] Result Set
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::ExecuteSQL(const altStr & sSQL, altDBResultSet & oResult)
{
  alt_t status;

  // Execute Query
  SQLAllocStmt (hdbc, & hstmt);

  if (SQLExecDirect (hstmt, sSQL.GetCStr(), SQL_NTS) != SQL_SUCCESS) {
    ALT_RET_P (ALT_E_SQL, sSQL.GetCStr());
  }

  while ((rc = SQLFetch (hstmt)) == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO) {


  }

  SQLBindCol (hstmt, 1, );


  // Get Result
  altMySQLRes oRes = m_oMySQL.StoreResult();

  // Get Row Count
  altULongLong  nRowCount = oRes.NumRows();
  MYSQL_ROW     ppRow = oRes.FetchRow();
  altULong *    pLen = oRes.FetchLengths();
  altUInt       nFieldCount = oRes.NumFields();
  MYSQL_FIELD * pField = oRes.FetchField();

  if (nRowCount > 0) {
    for (altULongLong i = 0; i < nRowCount; i++) {
      altDBRow * pRow = ALT_NEW altDBRow();
      for (altUInt j = 0; j < nFieldCount; j++) {
        pRow->Add (pField[j].name, ppRow[j]);
      }
      oResult.Add (pRow);
      ppRow = oRes.FetchRow();
      pLen = oRes.FetchLengths();
    }
  }

  oRes.FreeResult();

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Execute SQL
///
/// \param  sSQL          [I ] SQL
/// \param  pAffectedRow  [ O] Affected Row
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::ExecuteSQL(const altStr & sSQL, altULongLong * pAffectedRow)
{
  alt_t status;

  // Execute Query
  status = m_oMySQL.Query (sSQL);
  if (ALT_IS_ERR (status)) {
    ALT_RET_P (ALT_E_SQL, sSQL.GetCStr());
  }

  if (pAffectedRow != NULL) {
    (* pAffectedRow) = m_oMySQL.GetAffectedRow();
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write Explain Log
///
/// \param  oStmt [I ] SQL
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_SQL     SQL Error
///
LIBALT_API alt_t altSQLServerDBConnection::WriteExplainLog(const altPreparedStatement & oStmt)
{
  alt_t           status;
  altSQL          oSQL;
  altDBResultSet  oResultSet;
  altStr          sSQL;

  sSQL += "explain ";
  sSQL += oStmt.toString();

  status = ExecuteSQL (sSQL, oResultSet);
  ALT_ERR_RET (status);

  status = oSQL.SetSQL (sSQL, oStmt.GetSQLName());
  ALT_ERR_RET (status);

  status = WriteExplainLog (oSQL, oResultSet);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write Explain Log
///
/// \param  oSQL        [I ] SQL
/// \param  oResultSet  [I ] ResultSet
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altSQLServerDBConnection::WriteExplainLog(const altSQL & oSQL, const altDBResultSet oResultSet)
{
  if (oResultSet.Size() == 0) {
    ALT_RET (ALT_S_SUCCESS);
  }

  altStr  sPath;

  if (m_bExplainLogDir.GetLen() > 0) {
    sPath += m_bExplainLogDir;
    sPath += "/";
  }
  sPath += oSQL.GetSQLName();
  sPath += ".log";

  altInt nIndex = sPath.FindLastOfChar("/\\");
  if (nIndex >= 0) {
    altStr sDir = sPath.SubStr (0, nIndex - 1);
    altMkDirR (sDir.GetCStr());
  }

  altFile oFile;

  alt_t status = oFile.Open (sPath, "a");
  ALT_ERR_RET (status);
  if (aiSQLLoader.Unicode()) {
    oFile.WriteLine (oSQL.GetSQLW());
  }
  else {
    oFile.WriteLine (oSQL.GetSQLA());
  }
  for (altUInt i = 0; i < oResultSet.Size(); i++) {
    for (altUInt j = 0; j < oResultSet[i].Size(); j++) {
      if (j > 0) {
        oFile.Write ("\t");
      }

      if (oResultSet[i][j] != NULL) {
        oFile.Write (oResultSet[i][j]);
      }
      else {
        oFile.Write ("\t");
      }
    }
    oFile.Write ("\n");
  }
  oFile.Close();

  ALT_RET (ALT_S_SUCCESS);
}

