/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altFile.cpp 1423 2008-10-31 23:26:11Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altFile.cpp
 * \brief   File Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altFile.h"
#include "altMiscUtil.h"
#include "altBase/altUtil.h"

#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef ALT_WIN
  #include <io.h>
#endif

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altFile::altFile() :
m_pFile (NULL),
m_sFileName (""),
m_pNotifyTotalSizeFunc (NULL),
m_pNotifyWriteSizeFunc (NULL),
m_pNotifyReadSizeFunc (NULL)
{
}

///
/// \brief  Destructor
///
LIBALT_API altFile::~altFile()
{
  alt_t status = Close();
  ALT_LOG (status);
}

///
/// \brief  Open file
///
/// \param  sFileName [I ] file name
/// \param  sMode     [I ] open mode
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_FILE_OPEN error
///
LIBALT_API alt_t altFile::Open(const altStr & sFileName, const altStr & sMode)
{
  altStr  sFilePath = sFileName;
  altStr  sOpenMode = sMode;
  altStr  sDirName;
  altInt  nIndex = sFileName.FindLastOfChar ("\\/");

  if (nIndex >= 0) {
    sFilePath = sFileName.SubStr (nIndex + 1);
    sDirName = sFileName.SubStr (0, nIndex - 1);
  }
  else {
    sFilePath = sFileName;
  }
  
  if (sDirName.GetLen() > 0) {
    // create directory
    altMkDirR (sDirName.GetCStr());
  }

  altFOpen (m_pFile, sFileName.GetCStr(), sMode.GetCStr());
  if (m_pFile == NULL) {
    altInt  nErrNo = errno;
    altStr  sBuf;
#ifdef ALT_WIN
    altChar szBuf[ALT_BUFSIZ];
    strerror_s (szBuf, sizeof (szBuf), nErrNo);
    ALT_RET_P (ALT_E_FILE_OPEN, sBuf.Format ("Error Code=%d (%s)[%s]", nErrNo, szBuf, sFileName.GetCStr()));
#endif
#ifdef ALT_LINUX
    ALT_RET_P (ALT_E_FILE_OPEN, sBuf.Format ("Error Code=%d (%s)[%s]", nErrNo, strerror (nErrNo), sFileName.GetCStr()));
#endif
  }
  m_sFileName = sFileName;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Open file
///
/// \param  sFileName [I ] file name
/// \param  sMode     [I ] open mode
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_FILE_OPEN error
///
LIBALT_API alt_t altFile::WOpen(const altWStr & swFileName, const altWStr & swMode)
{
  alt_t   status;
  altStr  sFileName = swFileName;
  altStr  sMode = swMode;
  status = Open (sFileName.GetCStr(), sMode.GetCStr());
  ALT_RET (status);
}

///
/// \brief  Close file
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::Close()
{
  if (m_pFile == NULL) {
    ALT_RET (ALT_S_SUCCESS);
  }
  altInt nRet = fclose (m_pFile);
  if (nRet == EOF) {
    altInt  nErrNo = errno;
    altStr  sBuf;
#ifdef ALT_WIN
    altChar szBuf[ALT_BUFSIZ];
    strerror_s (szBuf, sizeof (szBuf), nErrNo);
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, szBuf, m_sFileName.GetCStr()));
#endif
#ifdef ALT_LINUX
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, strerror (nErrNo), m_sFileName.GetCStr()));
#endif
  }
  m_pFile = NULL;
  m_sFileName = (altStr)"";
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read Line from File
///
/// \param  pBuf      [IO] Read Buffer
/// \param  nBufSize  [I ] Read Buffer Size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_EOF       end of file
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::ReadLine(altChar * pBuf, const altUInt nBufSize)
{
  if (m_pFile == NULL) {
    ALT_RET_P (ALT_E_ERROR, "file is not opened");
  }
  memset (pBuf, 0x00, nBufSize);
  if (fgets (pBuf, nBufSize, m_pFile) == NULL) {
    if (feof (m_pFile)) {
      clearerr (m_pFile);
      ALT_RET (ALT_S_EOF);
    }
    else{
      altInt  nErrNo = errno;
      altStr  sBuf;
#ifdef ALT_WIN
      altChar szBuf[ALT_BUFSIZ];
      strerror_s (szBuf, sizeof (szBuf), nErrNo);
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, szBuf, m_sFileName.GetCStr()));
#endif
#ifdef ALT_LINUX
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, strerror (nErrNo), m_sFileName.GetCStr()));
#endif
    }
  }
  
  if (m_pNotifyReadSizeFunc != NULL) {
    m_pNotifyReadSizeFunc (strlen (pBuf));
  }

  altChar * p = strpbrk (pBuf, "\r\n");
  if (p != NULL) {
    (* p) = 0x00;
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read Line from File
///
/// \param  pBuf      [IO] Read Buffer
/// \param  nBufSize  [I ] Read Buffer Size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_EOF       end of file
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::ReadLine(altWChar * pBuf, const altUInt nBufSize)
{
  if (m_pFile == NULL) {
    ALT_RET_P (ALT_E_ERROR, "file is not opened");
  }
  memset (pBuf, 0x00, nBufSize);

  for (altUInt i = 0; i < nBufSize; i++) {
    errno = 0;
    if (fread (& pBuf[i], sizeof (pBuf[i]), 1, m_pFile) == 0) {
      if (errno == 0) {
        ALT_RET (ALT_S_EOF);
      }
      else{
        altInt  nErrNo = errno;
        altStr  sBuf;
#ifdef ALT_WIN
        altChar szBuf[ALT_BUFSIZ];
        strerror_s (szBuf, sizeof (szBuf), nErrNo);
        ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, szBuf, m_sFileName.GetCStr()));
#endif
#ifdef ALT_LINUX
        ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, strerror (nErrNo), m_sFileName.GetCStr()));
#endif
      }
    }
    
    if (m_pNotifyReadSizeFunc != NULL) {
      m_pNotifyReadSizeFunc (sizeof (pBuf[i]));
    }

    altWChar w;
    memcpy (& w, & pBuf[i], sizeof (w));
    if (w == 0x000A) {
      break;
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read Line from File
///
/// \param  sLine [ O] Line
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_EOF       end of file
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::ReadLine(altStr & sLine)
{
  alt_t   status;
  altChar szBuf[ALT_BUFSIZ * 10];

  status = ReadLine (szBuf, ALT_ARRAY_SIZE (szBuf));
  sLine = (altStr)szBuf;

  ALT_RET (status);
}

///
/// \brief  Read Line from File
///
/// \param  sLine [ O] Line
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_EOF       end of file
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::ReadLine(altWStr & sLine)
{
  alt_t     status;
  altWChar  szBuf[ALT_BUFSIZ * 10];

  status = ReadLine (szBuf, ALT_ARRAY_SIZE (szBuf));
  sLine = szBuf;

  ALT_RET (status);
}

///
/// \brief  Read
///
/// \param  pBuf      [IO] Read Buffer
/// \param  nBufSize  [I ] Read Buffer Size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_EOF       end of file
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::Read(altChar * pBuf, const altUInt nBufSize)
{
  altUInt nTotalReadSize = 0;
  alt_t   status;

  status = Read (pBuf, nBufSize, nTotalReadSize);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read
///
/// \param  pBuf            [IO] Read Buffer
/// \param  nBufSize        [I ] Read Buffer Size
/// \param  nTotalReadSize  [ O] Read Size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_EOF       end of file
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altFile::Read(altChar * pBuf, const altUInt nBufSize, altUInt & nTotalReadSize)
{
  if (m_pFile == NULL) {
    ALT_RET_P (ALT_E_ERROR, "file is not opened");
  }
  nTotalReadSize = 0;

  alt_t status;

  altInt nFd = 0;
  status = GetFileNo (nFd);
  ALT_ERR_RET (status);

  while (nTotalReadSize != nBufSize) {
    altInt nReadSize = static_cast<altUInt>(read (nFd, pBuf + nTotalReadSize, (nBufSize - nTotalReadSize)));
    if (nReadSize == 0) {
      if (feof (m_pFile)) {
        clearerr (m_pFile);
        ALT_RET (ALT_S_EOF);
      }
      ALT_RET (ALT_S_EOF);
    }
    else if (nReadSize < 0) {
      altInt  nErrNo = errno;
      altStr  sBuf;
#ifdef ALT_WIN
      altChar szBuf[ALT_BUFSIZ];
      strerror_s (szBuf, sizeof (szBuf), nErrNo);
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, szBuf, m_sFileName.GetCStr()));
#endif
#ifdef ALT_LINUX
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%d (%s)[%s]", nErrNo, strerror (nErrNo), m_sFileName.GetCStr()));
#endif
    }
    nTotalReadSize += nReadSize;
    if (m_pNotifyReadSizeFunc != NULL) {
      m_pNotifyReadSizeFunc (nReadSize);
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read
///
/// \param  sText [ O] Text
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::Read(altWStr & sText)
{
  if (m_pFile == NULL) {
    ALT_RET_P (ALT_E_ERROR, "file is not opened");
  }

  altWStr   sBuf;
  altWChar  wChar;

  altUInt i = 0;
  while (fread (& wChar, sizeof (wChar), 1, m_pFile) != 0) {
    if (i == 0 && wChar == 0xFEFF) {
      continue;
    }
    sText += wChar;
    i++;
    
    if (m_pNotifyReadSizeFunc != NULL) {
      m_pNotifyReadSizeFunc (sizeof (wChar));
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write String
///
/// \param  sStr  [I ] String
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
/// \return ALT_E_WRITE     write error
///
LIBALT_API alt_t altFile::WriteLine(const altStr & sStr)
{
  if (m_pFile == NULL) {
    ALT_RET_P (ALT_E_ERROR, "file is not opened");
  }
  alt_t status;

  status = altFPuts (sStr.GetCStr(), m_pFile);
  ALT_ERR_RET (status);

#ifdef ALT_WIN
  status = altFPuts ("\r\n", m_pFile);
#endif
#ifdef ALT_LINUX
  status = altFPuts ("\n", m_pFile);
#endif
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write
///
/// \param  sStr  [I ] String
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
/// \return ALT_E_WRITE     write error
///
LIBALT_API alt_t altFile::Write(const altStr & sStr)
{
  if (m_pFile == NULL) {
    ALT_RET (ALT_E_ERROR);
  }
  fwrite (sStr.GetCStr(), sizeof (altChar), sStr.GetLen(), m_pFile);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write
///
/// \param  sStr  [I ] String
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
/// \return ALT_E_WRITE     write error
///
LIBALT_API alt_t altFile::WriteW(const altWStr & sStr)
{
  if (m_pFile == NULL) {
    ALT_RET (ALT_E_ERROR);
  }
  alt_t status;
  status = Write (sStr.GetCStr(), sStr.GetLen());
  ALT_ERR_RET (status);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write
///
/// \param  pData [I ] Data
/// \param  nSize [I ] Data size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
/// \return ALT_E_INVAL     invalid paramter
/// \return ALT_E_WRITE     write error
///
LIBALT_API alt_t altFile::Write(const altChar * pData, const altUInt nSize)
{
  if (m_pFile == NULL) {
    ALT_RET (ALT_E_ERROR);
  }

  ALT_P_ASSERT (pData != NULL);

  alt_t   status;
  altInt  nErrorCount = 0;
  size_t  nTotalWriteSize = 0;

  if (m_pNotifyTotalSizeFunc != NULL) {
    status = m_pNotifyTotalSizeFunc (nSize);
    ALT_LOG (status);
  }

  while (nTotalWriteSize != nSize) {
    altUInt nBufSize = ALT_BUFSIZ;
    if ((nSize - nTotalWriteSize) < nBufSize) {
      nBufSize = (nSize - nTotalWriteSize);
    }
    size_t nWriteSize = fwrite (& pData[nTotalWriteSize], sizeof (* pData), nBufSize, m_pFile);
    if (nWriteSize > 0) {
      nTotalWriteSize += nWriteSize;
      nErrorCount = 0;
      if (m_pNotifyWriteSizeFunc != NULL) {
        status = m_pNotifyWriteSizeFunc (nWriteSize);
        ALT_LOG (status);
      }
    }
    else {
      nErrorCount++;
    }
    if (nErrorCount > 10) {
      ALT_RET (ALT_E_WRITE);
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Write
///
/// \param  pData [I ] Data
/// \param  nSize [I ] Data size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
/// \return ALT_E_INVAL     invalid paramter
/// \return ALT_E_WRITE     write error
///
LIBALT_API alt_t altFile::Write(const altWChar * pData, const altUInt nSize)
{
  if (m_pFile == NULL) {
    ALT_RET (ALT_E_ERROR);
  }

  ALT_P_ASSERT (pData != NULL);

  altInt  nErrorCount = 0;
  size_t  nTotalWriteSize = 0;

  while (nTotalWriteSize != nSize) {
    size_t nWriteSize = fwrite (& pData[nTotalWriteSize], sizeof (* pData), nSize - nTotalWriteSize, m_pFile);
    if (nWriteSize > 0) {
      nTotalWriteSize += nWriteSize;
      nErrorCount = 0;
    }
    else {
      nErrorCount++;
    }
    if (nErrorCount > 10) {
      ALT_RET (ALT_E_WRITE);
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Delete file
///
/// \param  sFileName [I ] File name
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::Delete(const altStr & sFileName)
{
#ifdef ALT_WIN
  altInt nRet = _unlink (sFileName.GetCStr());
  if (nRet < 0) {
    altInt  nErrNo = errno;
    altChar szBuf[ALT_BUFSIZ];
    strerror_s (szBuf, sizeof (szBuf), nErrNo);
    altStr  sBuf;
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("file delete error (%s) errno=%d(%s)", sFileName.GetCStr(), nErrNo, szBuf));
  }
#endif
#ifdef ALT_LINUX
  altInt nRet = unlink (sFileName.GetCStr());
  if (nRet < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("file delete error (%s) errno=%d(%s)", sFileName.GetCStr(), nErrNo, strerror (nErrNo)));
  }
#endif
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get file descriptor
///
/// \param  nFileNo [I ] File descriptor
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::GetFileNo(altInt & nFileNo) const
{
  if (m_pFile == NULL) {
    ALT_RET_P (ALT_E_ERROR, "m_pFile is NULL");
  }
#ifdef ALT_WIN
  nFileNo = _fileno (m_pFile);
  if (nFileNo < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    altChar szBuf[BUFSIZ];
    strerror_s (szBuf, sizeof (szBuf), nErrNo);
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%s:%s", m_sFileName.GetCStr(), szBuf));
  }
#endif
#ifdef ALT_LINUX
  nFileNo = fileno (m_pFile);
  if (nFileNo < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%s:%s", m_sFileName.GetCStr(), strerror (nErrNo)));
  }
#endif
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get file size
///
/// \param  nFileSize [ O] File size
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::GetFileSize(altLong & nFileSize) const
{
  alt_t   status;
  altInt  nFileNo;

  status = GetFileNo (nFileNo);
  ALT_ERR_RET (status);

#ifdef ALT_WIN
  struct _stat  oBuf;
  altInt        nRet;

  nRet = _fstat (nFileNo, & oBuf);
  if (nRet < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    altChar szBuf[BUFSIZ];
    strerror_s (szBuf, sizeof (szBuf), nErrNo);
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%s:%s", m_sFileName.GetCStr(), szBuf));
  }
  nFileSize = oBuf.st_size;
#endif
#ifdef ALT_LINUX
  struct stat oBuf;
  altInt      nRet;

  nRet = fstat (nFileNo, & oBuf);
  if (nRet < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%s:%s", m_sFileName.GetCStr(), strerror (nErrNo)));
  }
  nFileSize = oBuf.st_size;
#endif

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get last file modification time 
///
/// \param  nModTime [ O] last file modification time
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::GetModTime(altULong & nModTime) const
{
  alt_t   status;
  altInt  nFileNo;

  status = GetFileNo (nFileNo);
  ALT_ERR_RET (status);

#ifdef ALT_WIN
  struct _stat  oBuf;
  altInt        nRet;

  nRet = _fstat (nFileNo, & oBuf);
  if (nRet < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    altChar szBuf[BUFSIZ];
    strerror_s (szBuf, sizeof (szBuf), nErrNo);
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%s:%s", m_sFileName.GetCStr(), szBuf));
  }
  nModTime = oBuf.st_mtime;
#endif
#ifdef ALT_LINUX
  struct stat oBuf;
  altInt      nRet;

  nRet = fstat (nFileNo, & oBuf);
  if (nRet < 0) {
    altInt  nErrNo = errno;
    altStr  sBuf;
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("%s:%s", m_sFileName.GetCStr(), strerror (nErrNo)));
  }
  nModTime = oBuf.st_mtime;
#endif

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Save
///
/// \param  pData     [ O] Data
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::Save(const altDataStream & oData)
{
  alt_t       status;
  altCharPtr  pData;

  status = oData.GetData (pData);
  ALT_ERR_RET (status);

  status = Write (pData, oData.GetDataSize());
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Load
///
/// \param  pData     [ O] Data
/// \param  nDataSize [ O] Data size
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::Load(altCharPtr & pData, altLong & nDataSize)
{
  alt_t     status;
  altChar * pBuf = NULL;

  status = GetFileSize (nDataSize);
  ALT_ERR_RET (status);

  status = ALT_MALLOC (pBuf, nDataSize);
  ALT_ERR_RET (status);

  status = Read (pBuf, nDataSize);
  ALT_ERR_RET (status);

  pData = pBuf;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Load
///
/// \param  oDataStream [ O] Data
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altFile::Load(altDataStream & oDataStream)
{
  alt_t   status;
  altChar aBuf[BUFSIZ];
  altUInt nReadSize = 0;

  memset (aBuf, 0x00, sizeof (aBuf));
  while ((status = Read (aBuf, sizeof (aBuf), nReadSize)) == ALT_S_SUCCESS) {
    if (nReadSize == 0) {
      break;
    }
    oDataStream.AddData (aBuf, nReadSize);
    memset (aBuf, 0x00, sizeof (aBuf));
    nReadSize = 0;
  }
  oDataStream.AddData (aBuf, nReadSize);

  status = oDataStream.ResetIndex();
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set notify total size function
///
/// \param  pNotifyTotalSizeFunc  [I ] Notify total size function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altFile::SetNotifyTotalSizeFunc(altNotifySizeFunc pNotifyTotalSizeFunc)
{
  m_pNotifyTotalSizeFunc = pNotifyTotalSizeFunc;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set notify write size function
///
/// \param  pNotifyWriteSizeFunc  [I ] Notify write size function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altFile::SetNotifyWriteSizeFunc(altNotifySizeFunc pNotifyWriteSizeFunc)
{
  m_pNotifyWriteSizeFunc = pNotifyWriteSizeFunc;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set notify read size function
///
/// \param  pNotifyWriteSizeFunc  [I ] Notify write size function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altFile::SetNotifyReadSizeFunc(altNotifySizeFunc pNotifySizeFunc)
{
  m_pNotifyReadSizeFunc = pNotifySizeFunc;
  ALT_RET (ALT_S_SUCCESS);
}
