#ifndef ALTFILE_H
#define ALTFILE_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altFile.h 1424 2008-11-17 14:21:35Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altProperty.h
 * \brief   property
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altBase/altDataStream.h>
#include <altBase/altWStr.h>
#include <altBase/altStr.h>
#include <altBase/altCharPtr.h>
#include <altBase/altBaseLog.h>
#include <altBase/altUncopyable.h>

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
///
/// \brief  Size notify function
///
typedef alt_t (* altNotifySizeFunc)(altUInt nSize);

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  File
///
class altFile : private altUncopyable
{
public:
  ///
  /// \brief  Constructor
  ///
  LIBALT_API altFile();

  ///
  /// \brief  Destructor
  ///
  LIBALT_API virtual ~altFile();

  ///
  /// \brief  Open file
  ///
  /// \param  sFileName [I ] file name
  /// \param  sMode     [I ] open mode
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_FILE_OPEN error
  ///
  LIBALT_API alt_t Open(const altStr & sFileName, const altStr & sMode);

  ///
  /// \brief  Open file
  ///
  /// \param  sFileName [I ] file name
  /// \param  sMode     [I ] open mode
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_FILE_OPEN error
  ///
  LIBALT_API alt_t WOpen(const altWStr & swFileName, const altWStr & swMode);

  ///
  /// \brief  Close file
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Close();

  ///
  /// \brief  Read Line from File
  ///
  /// \param  pBuf      [IO] Read Buffer
  /// \param  nBufSize  [I ] Read Buffer Size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_EOF       end of file
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t ReadLine(altChar * pBuf, const altUInt nBufSize);

  ///
  /// \brief  Read Line from File
  ///
  /// \param  pBuf      [IO] Read Buffer
  /// \param  nBufSize  [I ] Read Buffer Size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_EOF       end of file
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t ReadLine(altWChar * pBuf, const altUInt nBufSize);

  ///
  /// \brief  Read Line from File
  ///
  /// \param  sLine [ O] Line
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_EOF       end of file
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t ReadLine(altStr & sLine);

  ///
  /// \brief  Read Line from File
  ///
  /// \param  sLine [ O] Line
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_EOF       end of file
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t ReadLine(altWStr & sLine);

  ///
  /// \brief  Read
  ///
  /// \param  pBuf      [IO] Read Buffer
  /// \param  nBufSize  [I ] Read Buffer Size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_EOF       end of file
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Read(altChar * pBuf, const altUInt nBufSize);

  ///
  /// \brief  Read
  ///
  /// \param  pBuf            [IO] Read Buffer
  /// \param  nBufSize        [I ] Read Buffer Size
  /// \param  nTotalReadSize  [ O] Read Size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_EOF       end of file
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Read(altChar * pBuf, const altUInt nBufSize, altUInt & nTotalReadSize);

  ///
  /// \brief  Read
  ///
  /// \param  sText [ O] Text
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t Read(altWStr & sText);

  ///
  /// \brief  Write String
  ///
  /// \param  sStr  [I ] String
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  /// \return ALT_E_WRITE     write error
  ///
  LIBALT_API alt_t WriteLine(const altStr & sStr);

  ///
  /// \brief  Write
  ///
  /// \param  sStr  [I ] String
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  /// \return ALT_E_WRITE     write error
  ///
  LIBALT_API alt_t Write(const altStr & sStr);

  ///
  /// \brief  Write
  ///
  /// \param  sStr  [I ] String
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  /// \return ALT_E_WRITE     write error
  ///
  LIBALT_API alt_t WriteW(const altWStr & sStr);

  ///
  /// \brief  Write
  ///
  /// \param  pData [I ] Data
  /// \param  nSize [I ] Data size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  /// \return ALT_E_INVAL     invalid paramter
  /// \return ALT_E_WRITE     write error
  ///
  LIBALT_API alt_t Write(const altChar * pData, const altUInt nSize);

  ///
  /// \brief  Write
  ///
  /// \param  pData [I ] Data
  /// \param  nSize [I ] Data size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  /// \return ALT_E_INVAL     invalid paramter
  /// \return ALT_E_WRITE     write error
  ///
  LIBALT_API alt_t Write(const altWChar * pData, const altUInt nSize);

  ///
  /// \brief  Delete file
  ///
  /// \param  sFileName [I ] File name
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t Delete(const altStr & sFileName);

  ///
  /// \brief  Get file descriptor
  ///
  /// \param  nFileNo [I ] File descriptor
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t GetFileNo(altInt & nFileNo) const;

  ///
  /// \brief  Get file size
  ///
  /// \param  nFileSize [ O] File size
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t GetFileSize(altLong & nFileSize) const;

  ///
  /// \brief  Get last file modification time 
  ///
  /// \param  nModTime [ O] last file modification time
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t GetModTime(altULong & nModTime) const;

  ///
  /// \brief  Load
  ///
  /// \param  pData     [ O] Data
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t Save(const altDataStream & oData);

  ///
  /// \brief  Load
  ///
  /// \param  pData     [ O] Data
  /// \param  nDataSize [ O] Data size
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t Load(altCharPtr & pData, altLong & nDataSize);

  ///
  /// \brief  Load
  ///
  /// \param  oDataStream [ O] Data
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t Load(altDataStream & oDataStream);

  ///
  /// \brief  Set notify total size function
  ///
  /// \param  pNotifyTotalSizeFunc  [I ] Notify total size function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyTotalSizeFunc(altNotifySizeFunc pNotifyTotalSizeFunc);

  ///
  /// \brief  Set notify write size function
  ///
  /// \param  pNotifyWriteSizeFunc  [I ] Notify write size function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyWriteSizeFunc(altNotifySizeFunc pNotifyWriteSizeFunc);

  ///
  /// \brief  Set notify read size function
  ///
  /// \param  pNotifyWriteSizeFunc  [I ] Notify write size function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyReadSizeFunc(altNotifySizeFunc pNotifyWriteSizeFunc);

protected:
  FILE *                  m_pFile;
  altStr                  m_sFileName;
  altNotifySizeFunc       m_pNotifyTotalSizeFunc;
  altNotifySizeFunc       m_pNotifyWriteSizeFunc;
  altNotifySizeFunc       m_pNotifyReadSizeFunc;
};

#endif  // ALTFILE_H

