/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altProperty.cpp 1384 2008-04-28 12:39:40Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altProperty.cpp
 * \brief   Property Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altProperty.h"
#include "altBlowFish.h"
#include "altFile.h"
#include "altBase/altLog.h"
#include "altBase/altStatus.h"

/*----------------------------------------------------------------*
 * Define
 *----------------------------------------------------------------*/
///
/// \brief  Get method template
///
/// \param  method  [I ] method name
///
#define ALT_PROPERTY_TO_X(method) \
  altStr * pValue = m_oPropertyDataMap[sKey]; \
  if (pValue == NULL) { \
    return (nDefault); \
  } \
  return (pValue->method());

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altProperty::altProperty()
{
}

///
/// \brief  Destructor
///
LIBALT_API altProperty::~altProperty()
{
}

///
/// \brief  Load Property File
///
/// \param  sFileName [I ] file name
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altProperty::Load(const altStr & sFileName)
{
  alt_t   status;
  altFile oFile;
  altStr  sBuf;
  altChar szBuf[ALT_BUFSIZ];

  status = oFile.Open (sFileName, "r");
  ALT_ERR_RET (status);

  while (true) {
    status = oFile.ReadLine (szBuf, sizeof (szBuf));

    // end of file
    if (status == ALT_S_EOF) {
      status = oFile.Close();
      ALT_LOG (status);
      break;
    }

    // error caused
    if (ALT_IS_ERR (status)) {
      ALT_LOG (status);
      oFile.Close();
      ALT_RET (status);
    }

    sBuf = (altStr)szBuf;

    // delete comment
    altInt nCommentIndex = sBuf.Find("#");
    if (nCommentIndex > 0 && sBuf[nCommentIndex - 1] != '\\') {
      sBuf = sBuf.SubStr(0, nCommentIndex - 1);
    }

    altInt nEqualIndex = sBuf.Find("=");
    if (nEqualIndex > 0) {
      altStr  sKey = sBuf.SubStr (0, nEqualIndex - 1);
      altStr  sValue = sBuf.SubStr (nEqualIndex + 1);
      sKey.Trim();
      sValue.Trim();
      ALT_LOG_P (ALT_I_INFO, sBuf.Format ("%s=%s", sKey.GetCStr(), sValue.GetCStr()));
      m_oPropertyDataMap[sKey] = ALT_NEW altStr (sValue);
    }
  }
  oFile.Close();

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Save Property File
///
/// \param  sFileName [I ] file name
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altProperty::Save(const altStr & sFileName)
{
  alt_t   status;
  altFile oFile;

  status = oFile.Open (sFileName, "w");
  ALT_ERR_RET (status);

  for (altPropertyData::iterator i = m_oPropertyDataMap.begin(); i != m_oPropertyDataMap.end(); i++) {
    altStr  sBuf;
    if (i->second != NULL) {
      sBuf += i->first;
      sBuf += "=";
      sBuf += (* i->second);

      oFile.WriteLine (sBuf);
    }
  }

  oFile.Close();

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Load Property File
///
/// \param  sFileName [I ] file name
/// \param  pKey      [I ] key
/// \param  nSize     [I ] key size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altProperty::DecodeLoad(const altStr & sFileName, const altByte * pKey, altInt nSize)
{
  altBlowFish oBlowFish;
  altFile     oFile;
  alt_t       status;
  altCharPtr  pData;
  altLong     nDataSize = 0;
  altUInt     nDecodeSize = 0;

  status = oBlowFish.Initialize (pKey, nSize);
  ALT_ERR_RET (status);

  status = oFile.Open (sFileName, "r+b");
  ALT_ERR_RET (status);

  status = oFile.Load (pData, nDataSize);
  ALT_ERR_RET (status);

  status = oFile.Close();
  ALT_ERR_RET (status);

  nDecodeSize = nDataSize;

  status = oBlowFish.Decode (pData, nDecodeSize);
  ALT_ERR_RET (status);

  altStr  sBuf = (altChar *)pData;

  altStrList  oLineList = sBuf.Separate ("\r\n");

  for (altStrList::iterator i = oLineList.begin(); i != oLineList.end(); i++) {
    altStr      sLine = (* i);
    altStrList  oElementList = sLine.Separate ("=");
    if (oElementList.size() == 2) {
      altStr      sKey = oElementList[0];
      altStr *    pValue = ALT_NEW altStr (oElementList[1]);
      m_oPropertyDataMap[sKey] = pValue;
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Save Property File
///
/// \param  sFileName [I ] file name
/// \param  pKey      [I ] key
/// \param  nSize     [I ] key size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altProperty::EncodeSave(const altStr & sFileName, const altByte * pKey, altInt nSize)
{
  altBlowFish oBlowFish;
  altFile     oFile;
  alt_t       status;
  altCharPtr  pData = NULL;
  altChar *   pBuf = NULL;
  altUInt     nEncodeSize = 0;

  status = oBlowFish.Initialize (pKey, nSize);
  ALT_ERR_RET (status);

  altStr  sData;
  for (altPropertyData::iterator i = m_oPropertyDataMap.begin(); i != m_oPropertyDataMap.end(); i++) {
    sData += i->first;
    sData += "=";
    sData += (* i->second);
    sData += "\n";
  }

  status = ALT_STRALLOC (pBuf, sData.GetCStr());
  ALT_ERR_RET (status);

  pData = pBuf;

  nEncodeSize = sData.GetLen();
  status = oBlowFish.Encode (pData, nEncodeSize);
  ALT_ERR_RET (status);

  status = oFile.Open (sFileName, "wb");
  ALT_ERR_RET (status);

  status = oFile.Write (pData, nEncodeSize);
  ALT_ERR_RET (status);

  status = oFile.Close();
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Boolean Value
///
/// \param  sKey      [I ] Key Name
/// \param  bDefault  [I ] Default Value
///
/// \return Boolean Value
///
LIBALT_API altBool altProperty::GetBool(const altStr & sKey, const altBool nDefault)
{
  ALT_PROPERTY_TO_X (ToBool);
}

///
/// \brief  Get Short Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Short Value
///
LIBALT_API altShort altProperty::GetShort(const altStr & sKey, const altShort nDefault)
{
  ALT_PROPERTY_TO_X (ToShort);
}

///
/// \brief  Get Unsigned Short Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Unsigned Short Value
///
LIBALT_API altUShort altProperty::GetUShort(const altStr & sKey, const altUShort nDefault)
{
  ALT_PROPERTY_TO_X (ToUShort);
}

///
/// \brief  Get Int Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Int Value
///
LIBALT_API altInt altProperty::GetInt(const altStr & sKey, const altInt nDefault)
{
  ALT_PROPERTY_TO_X (ToInt);
}

///
/// \brief  Get Unsigned Int Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Unsigned Int Value
///
LIBALT_API altUInt altProperty::GetUInt(const altStr & sKey, const altUInt nDefault)
{
  ALT_PROPERTY_TO_X (ToUInt);
}
  
///
/// \brief  Get Long Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Long Value
///
LIBALT_API altLong altProperty::GetLong(const altStr & sKey, const altLong nDefault)
{
  ALT_PROPERTY_TO_X (ToLong);
}

///
/// \brief  Get Unsigned Long Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Unsigned Long Value
///
LIBALT_API altULong altProperty::GetULong(const altStr & sKey, const altULong nDefault)
{
  ALT_PROPERTY_TO_X (ToULong);
}

///
/// \brief  Get Long Long Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Long Long Value
///
LIBALT_API altLongLong altProperty::GetLongLong(const altStr & sKey, const altLongLong nDefault)
{
  ALT_PROPERTY_TO_X (ToLongLong);
}

///
/// \brief  Get Unsigned Long Long Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Unsigned Long Long Value
///
LIBALT_API altULongLong altProperty::GetULongLong(const altStr & sKey, const altULongLong nDefault)
{
  ALT_PROPERTY_TO_X (ToULongLong);
}

///
/// \brief  Get Float Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Float Value
///
LIBALT_API altFloat altProperty::GetFloat(const altStr & sKey, const altFloat nDefault)
{
  ALT_PROPERTY_TO_X (ToFloat);
}

///
/// \brief  Get Double Value
///
/// \param  sKey      [I ] Key Name
/// \param  nDefault  [I ] Default Value
///
/// \return Double Value
///
LIBALT_API altDouble altProperty::GetDouble(const altStr & sKey, const altDouble nDefault)
{
  ALT_PROPERTY_TO_X (ToDouble);
}

///
/// \brief  Get String Value
///
/// \param  sKey      [I ] Key Name
/// \param  sDefault  [I ] Default Value
///
/// \return String Value
///
LIBALT_API const altStr & altProperty::GetStr(const altStr & sKey, const altStr & sDefault)
{
  altStr * pValue = m_oPropertyDataMap[sKey];
  if (pValue == NULL) {
    return sDefault;
  }
  return (* pValue);
}

///
/// \brief  Set Boolean Value
///
/// \param  sKey    [I ] Key Name
/// \param  bValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetBool(const altStr & sKey, const altBool bValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetBool (bValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Short Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetShort(const altStr & sKey, const altShort nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetShort (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Unsigned Short Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetUShort(const altStr & sKey, const altUShort nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetUShort (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Int Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetInt(const altStr & sKey, const altInt nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetInt (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Unsigned Int Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetUInt(const altStr & sKey, const altUInt nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetUInt (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Long Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetLong(const altStr & sKey, const altLong nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetLong (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Unsigned Long Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetULong(const altStr & sKey, const altULong nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetULong (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Long Long Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetLongLong(const altStr & sKey, const altLongLong nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetLongLong (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Unsigned Long Long Value
///
/// \param  sKey    [I ] Key Name
/// \param  nValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetULongLong(const altStr & sKey, const altULongLong nValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetULongLong (nValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Float Value
///
/// \param  sKey    [I ] Key Name
/// \param  fValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetFloat(const altStr & sKey, const altFloat fValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetFloat (fValue);
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Double Value
///
/// \param  sKey    [I ] Key Name
/// \param  fValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetDouble(const altStr & sKey, const altDouble fValue)
{
  altStr * pVal = ALT_NEW altStr();
  pVal->SetDouble (fValue);
  m_oPropertyDataMap[sKey] = pVal; 
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set String Value
///
/// \param  sKey    [I ] Key Name
/// \param  sValue  [I ] Value
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altProperty::SetStr(const altStr & sKey, const altStr & sValue)
{
  altStr * pVal = ALT_NEW altStr();
  (* pVal) = sValue;
  m_oPropertyDataMap[sKey] = pVal;
  ALT_RET (ALT_S_SUCCESS);
}
