#ifndef ALTPROPERTY_H
#define ALTPROPERTY_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altProperty.h 1312 2008-04-14 13:28:10Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altProperty.h
 * \brief   property
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altBase/altLog.h>
#include <altBase/altStr.h>
#include <altBase/altCommon.h>

#include <map>

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
typedef std::map<altStr, altStr *>  altPropertyData;  ///< setting data

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  Property
///
class altProperty
{
public:
  ///
  /// \brief  Constructor
  ///
  LIBALT_API altProperty();

  ///
  /// \brief  Destructor
  ///
  LIBALT_API virtual ~altProperty();

  ///
  /// \brief  Load Property File
  ///
  /// \param  sFileName [I ] file name
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Load(const altStr & sFileName);

  ///
  /// \brief  Save Property File
  ///
  /// \param  sFileName [I ] file name
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Save(const altStr & sFileName);

  ///
  /// \brief  Load Property File
  ///
  /// \param  sFileName [I ] file name
  /// \param  pKey      [I ] key
  /// \param  nSize     [I ] key size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t DecodeLoad(const altStr & sFileName, const altByte * pKey, altInt nSize);

  ///
  /// \brief  Save Property File
  ///
  /// \param  sFileName [I ] file name
  /// \param  pKey      [I ] key
  /// \param  nSize     [I ] key size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t EncodeSave(const altStr & sFileName, const altByte * pKey, altInt nSize);

  ///
  /// \brief  Get Boolean Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  bDefault  [I ] Default Value
  ///
  /// \return Boolean Value
  ///
  LIBALT_API altBool GetBool(const altStr & sKey, const altBool nDefault = false);

  ///
  /// \brief  Get Short Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Short Value
  ///
  LIBALT_API altShort GetShort(const altStr & sKey, const altShort nDefault = 0);

  ///
  /// \brief  Get Unsigned Short Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Unsigned Short Value
  ///
  LIBALT_API altUShort GetUShort(const altStr & sKey, const altUShort nDefault = 0);

  ///
  /// \brief  Get Int Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Int Value
  ///
  LIBALT_API altInt GetInt(const altStr & sKey, const altInt nDefault = 0);

  ///
  /// \brief  Get Unsigned Int Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Unsigned Int Value
  ///
  LIBALT_API altUInt GetUInt(const altStr & sKey, const altUInt nDefault = 0);

  ///
  /// \brief  Get Long Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Long Value
  ///
  LIBALT_API altLong GetLong(const altStr & sKey, const altLong nDefault = 0);

  ///
  /// \brief  Get Unsigned Long Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Unsigned Long Value
  ///
  LIBALT_API altULong GetULong(const altStr & sKey, const altULong nDefault = 0);

  ///
  /// \brief  Get Long Long Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Long Long Value
  ///
  LIBALT_API altLongLong GetLongLong(const altStr & sKey, const altLongLong nDefault = 0);

  ///
  /// \brief  Get Unsigned Long Long Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Unsigned Long Long Value
  ///
  LIBALT_API altULongLong GetULongLong(const altStr & sKey, const altULongLong nDefault = 0);

  ///
  /// \brief  Get Float Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Float Value
  ///
  LIBALT_API altFloat GetFloat(const altStr & sKey, const altFloat nDefault = 0);

  ///
  /// \brief  Get Double Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  nDefault  [I ] Default Value
  ///
  /// \return Double Value
  ///
  LIBALT_API altDouble GetDouble(const altStr & sKey, const altDouble nDefault = 0);

  ///
  /// \brief  Get String Value
  ///
  /// \param  sKey      [I ] Key Name
  /// \param  sDefault  [I ] Default Value
  ///
  /// \return String Value
  ///
  LIBALT_API const altStr & GetStr(const altStr & sKey, const altStr & sDefault = "");

  ///
  /// \brief  Set Boolean Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  bValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetBool(const altStr & sKey, const altBool bValue);

  ///
  /// \brief  Set Short Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetShort(const altStr & sKey, const altShort nValue);

  ///
  /// \brief  Set Unsigned Short Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetUShort(const altStr & sKey, const altUShort nValue);

  ///
  /// \brief  Set Int Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetInt(const altStr & sKey, const altInt nValue);

  ///
  /// \brief  Set Unsigned Int Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetUInt(const altStr & sKey, const altUInt nValue);

  ///
  /// \brief  Set Long Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetLong(const altStr & sKey, const altLong nValue);

  ///
  /// \brief  Set Unsigned Long Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetULong(const altStr & sKey, const altULong nValue);

  ///
  /// \brief  Set Long Long Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetLongLong(const altStr & sKey, const altLongLong nValue);

  ///
  /// \brief  Set Unsigned Long Long Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  nValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetULongLong(const altStr & sKey, const altULongLong nValue);

  ///
  /// \brief  Set Float Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  fValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetFloat(const altStr & sKey, const altFloat fValue);

  ///
  /// \brief  Set Double Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  fValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetDouble(const altStr & sKey, const altDouble fValue);

  ///
  /// \brief  Set String Value
  ///
  /// \param  sKey    [I ] Key Name
  /// \param  sValue  [I ] Value
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetStr(const altStr & sKey, const altStr & sValue);

protected:
  altPropertyData   m_oPropertyDataMap;   ///< property data map
};

#endif  // ALTPROPERTY_H

