/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altWinMutex.cpp 1268 2008-04-11 00:45:39Z nekosys $

  Copyright (C) 2008 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altWinMutex.cpp
 * \brief   Windows Mutex
 * \date    2008
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altWinMutex.h"

#ifdef ALT_WIN
/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altWinMutex::altWinMutex() :
m_hMutex (NULL)
{
}

///
/// \brief  Destructor
///
LIBALT_API altWinMutex::~altWinMutex()
{
  Close();
}

///
/// \brief  Create Mutex
///
/// \param  sName   [I ] Mutex Name
/// \param  bOwner  [I ] Owner
///
/// \return ALT_S_SUCCESS     success
/// \return ALT_S_MUTEX_EXIST exist
/// \return ALT_E_ERROR       error
///
LIBALT_API alt_t altWinMutex::Create(const altStr & sName, altBool bOwner)
{
  if (m_hMutex != NULL) {
    Close();
  }
  m_hMutex = CreateMutexA (NULL, bOwner, sName.GetCStr());
  if (GetLastError() == ERROR_ALREADY_EXISTS) {
    ALT_RET (ALT_S_MUTEX_EXIST);
  }
  if (m_hMutex == NULL) {
    ALT_RET (ALT_E_ERROR);
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Open Mutex
///
/// \param  sName     [I ] Mutex Name
/// \param  dwAccess  [I ] Access (SYNCHRONIZE or MUTEX_ALL_ACCESS)
/// \param  bInherit  [I ] Inherit
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altWinMutex::Open(const altStr & sName, DWORD dwAccess, altBool bInherit)
{
  if (m_hMutex != NULL) {
    Close();
  }
  m_hMutex = OpenMutexA (dwAccess, bInherit, sName.GetCStr());
  if (m_hMutex == NULL) {
    altStr  sBuf;
    ALT_ERR_RET_P (ALT_E_ERROR, sBuf.Format ("%d", GetLastError()));
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Close Mutex
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altWinMutex::Close()
{
  if (m_hMutex != NULL) {
    CloseHandle (m_hMutex);
    m_hMutex = NULL;
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Release Mutex
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altWinMutex::Release()
{
  if (m_hMutex == NULL) {
    ALT_RET (ALT_E_ERROR);
  }
  if (! ReleaseMutex (m_hMutex)) {
    ALT_RET (ALT_E_ERROR);
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Wait
///
/// \param  dwMilliseconds  [I ] Timeout
///
/// \return ALT_S_SUCCESS success
/// \return ALT_S_TIMEOUT timeout
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altWinMutex::Wait(DWORD dwMilliseconds)
{
  if (m_hMutex == NULL) {
    ALT_RET (ALT_E_ERROR);
  }
  DWORD dwRet = WaitForSingleObject (m_hMutex, dwMilliseconds);
  if (dwRet == WAIT_ABANDONED) {
    ALT_RET (ALT_S_SUCCESS);
  }
  else if (dwRet == WAIT_OBJECT_0) {
    ALT_RET (ALT_S_SUCCESS);
  }
  else if (dwRet == WAIT_TIMEOUT) {
    ALT_RET (ALT_S_TIMEOUT);
  }
  ALT_RET (ALT_E_ERROR);
}

#endif
