/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOAcceptor.cpp 1393 2008-05-01 18:23:12Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOServer.cpp
 * \brief   NEKO Network Server Acceptor
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOAcceptor.h"
#include "altNEKOConnection.h"
#include "altNEKOConnectionManager.h"

#include <fcntl.h>

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
/// \param  pReceiveCallBack    [I ] receive call back function
/// \param  nThreadCount        [I ] Reqeust Thread Count
/// \param  nAcceptThreadCount  [I ] Accept Thread Count
/// \param  nSSLVersion         [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
LIBALT_API altNEKOAcceptor::altNEKOAcceptor(const altNEKOReceiveCallBack pReceiveCallBack, const altUInt nThreadCount, const altUInt nAcceptThreadCount, const altByte nSSLVersion) :
m_oReceiver (pReceiveCallBack, nThreadCount),
m_oSender(),
m_nSSLVersion (nSSLVersion),
m_nListenSocket (),
m_nPort (0),
m_nAcceptThreadCount (nAcceptThreadCount),
m_oAcceptThreadContainer()
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOAcceptor::~altNEKOAcceptor()
{
  ClosePort();
}

///
/// \brief  Open Port
///
/// \param  nPort               [I ] Port NO
/// \param  nListenQueSize      [I ] Listen Queue Size
/// \param  pAcceptCallBack     [I ] Accept Call Back Function
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_INVAL     parameter error
/// \return ALT_E_SOCKET    create socket error
/// \return ALT_E_BIND      bind error
/// \return ALT_E_LISTEN    listen error
///
LIBALT_API alt_t altNEKOAcceptor::OpenPort(const altInt nPort, const altInt nListenQueSize, const altNEKOAcceptCallBack pAcceptCallBack)
{
  alt_t   status;
  altStr  sBuf;

  // create listen socket
  if (m_nListenSocket.GetSocket() == 0) {
    status = m_nListenSocket.Init (AF_INET, SOCK_STREAM, 0);
    ALT_ERR_RET_P (status, sBuf.Format ("nPort=%d nListenQueSize=%d", nPort, nListenQueSize));
  }

  // listen
  status = m_nListenSocket.Listen (nPort, nListenQueSize);
  ALT_LOG_P (ALT_D_DEBUG, sBuf.Format ("Open port [%d]", nPort));
  ALT_ERR_RET_P (status, sBuf.Format ("nPort=%d nListenQueSize=%d", nPort, nListenQueSize));

  m_pAcceptCallBackFunc = pAcceptCallBack;
  m_nPort = nPort;

  // Add TCP Receiver Broker to Connection Manager
  aiNEKOConnectionManager.Add (nPort, & m_oReceiver);
  
  // start accept thread
  for (altUInt i = 0; i < m_nAcceptThreadCount; i++) {
    altLoopThread * pLoopThread = ALT_NEW altLoopThread (20);
    if (pLoopThread == NULL) {
      ALT_RET (ALT_E_NOMEM);
    }
    if (m_nSSLVersion == ALT_NO_SSL) {
      pLoopThread->Start (altNEKOAcceptor::TCPAcceptThread, this);
    }
    else {
      pLoopThread->Start (altNEKOAcceptor::SSLAcceptThread, this);
    }
    m_oAcceptThreadContainer.push_back (pLoopThread);
  }

  status = m_oReceiver.Start();
  ALT_ERR_RET (status);

  status = m_oSender.Start();
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Close Port
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOAcceptor::ClosePort()
{
  alt_t status = m_nListenSocket.Close();
  ALT_LOG (status);
  for (altUInt i = 0; i < m_oAcceptThreadContainer.size(); i++) {
    m_oAcceptThreadContainer[i]->Stop();
    delete m_oAcceptThreadContainer[i];
    m_oAcceptThreadContainer[i] = NULL;
  }
  m_nPort = 0;
  m_pAcceptCallBackFunc = NULL;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get port number
///
/// \return Port number
///
LIBALT_API altInt altNEKOAcceptor::GetPortNo() const
{
  return (m_nPort);
}

///
/// \brief  Get sender packet count
///
/// \return Sender packet count
///
LIBALT_API altUInt altNEKOAcceptor::GetSenderPacketCount() const
{
  return (m_oSender.GetPacketCount());
}

///
/// \brief  Get receiver packet count
///
/// \return Receiver packet count
///
LIBALT_API altUInt altNEKOAcceptor::GetReceiverPacketCount() const
{
  return (m_oReceiver.GetPacketCount());
}

///
/// \brief  Accept Thread
///
/// \param  pParam  [I ] parameter
///
LIBALT_API alt_t altNEKOAcceptor::TCPAcceptThread(void * pParam)
{
  altNEKOAcceptor *   pAcceptor = static_cast<altNEKOAcceptor *>(pParam);
	fd_set              oRfds;
	struct timeval      oTimeout;
  alt_t               status;
  SOCKET              nServerSocket = pAcceptor->m_nListenSocket.GetSocket();

  FD_ZERO (& oRfds);
  FD_SET (nServerSocket, & oRfds);

  oTimeout.tv_sec = 0;
  oTimeout.tv_usec = 100;

  status = altNetUtil::Select (nServerSocket, & oRfds, NULL, NULL, & oTimeout);
  ALT_LOG (status);

  if (status == ALT_S_SUCCESS) {

    SOCKADDR_IN oClientSockAddr;
    altInt      nClientSockAddrLen = sizeof (oClientSockAddr);
    SOCKET      nClientSocket;

    // accept
    status = altNetUtil::Accept (nServerSocket, & oClientSockAddr, & nClientSockAddrLen, nClientSocket);
    ALT_ERR_RET (status);


    // Set non block
#ifdef ALT_LINUX
    altInt nFlag = fcntl (nClientSocket, F_GETFL, 0);
    fcntl (nClientSocket, F_SETFL, nFlag | O_NONBLOCK);
#endif
#ifdef ALT_WIN
    u_long nFlag = 1;
    ioctlsocket (nClientSocket, FIONBIO, & nFlag);
#endif

    altInetAddress oInetAddress (oClientSockAddr);
    
    altStr  sBuf;
    ALT_LOG_P (ALT_I_INFO, sBuf.Format ("Accept %s:%u", oInetAddress.GetIP().GetCStr(), oInetAddress.GetPort()));

    altCNEKOConnectionPtr  pNEKOConnection;
    // create new connection
    pNEKOConnection = ALT_NEW altNEKOConnection (NULL, NULL, nClientSocket, oClientSockAddr, pAcceptor->m_nPort, pAcceptor->m_oSender, pAcceptor->m_nSSLVersion);
    if (pNEKOConnection == NULL) {
      ALT_RET (ALT_E_NOMEM);
    }

    // add to connection manager
    status = aiNEKOConnectionManager.Add (pNEKOConnection);
    if (ALT_IS_ERR (status)) {
      SOCKET nClientSocket = pNEKOConnection->GetSocket();
      altNetUtil::CloseSocket (nClientSocket);
      ALT_RET (status);
    }

    // add to receive broker
    status = pAcceptor->m_oReceiver.Add (pNEKOConnection);
    if (ALT_IS_ERR (status)) {
      aiNEKOConnectionManager.Del (pNEKOConnection);
      ALT_RET (status);
    }

    // Call Accept call back function
    if (pAcceptor->m_pAcceptCallBackFunc != NULL) {
      pAcceptor->m_pAcceptCallBackFunc (pAcceptor->m_nPort, pNEKOConnection);
    }
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Accept Thread
///
/// \param  pParam  [I ] parameter
///
LIBALT_API alt_t altNEKOAcceptor::SSLAcceptThread(void * pParam)
{
  altNEKOAcceptor * pAcceptor = static_cast<altNEKOAcceptor *>(pParam);
  fd_set            oRfds;
  struct timeval    oTimeout;
  alt_t             status;
  SOCKET            nServerSocket = pAcceptor->m_nListenSocket.GetSocket();

  FD_ZERO (& oRfds);
  FD_SET (nServerSocket, & oRfds);

  oTimeout.tv_sec = 0;
  oTimeout.tv_usec = 100;

  status = altNetUtil::Select (nServerSocket, & oRfds, NULL, NULL, & oTimeout);
  ALT_ERR_RET (status);

  if (status == ALT_S_SUCCESS) {
    SOCKADDR_IN oClientSockAddr;
    altInt      nClientSockAddrLen = sizeof (oClientSockAddr);
    SOCKET      nClientSocket;

    // accept
    status = altNetUtil::Accept (nServerSocket, & oClientSockAddr, & nClientSockAddrLen, nClientSocket);
    ALT_ERR_RET (status);

    SSL_CTX * pSSLContext = NULL;
    
    if (pAcceptor->m_nSSLVersion == ALT_SSL_V23) {
      ALT_LOG_P (ALT_D_DEBUG, "use SSLv23_server_method()");
      pSSLContext = SSL_CTX_new (SSLv23_server_method ());
    }
    else if (pAcceptor->m_nSSLVersion == ALT_SSL_V2) {
      ALT_LOG_P (ALT_D_DEBUG, "use SSLv2_server_method()");
      pSSLContext = SSL_CTX_new (SSLv2_server_method ());
    }
    else if (pAcceptor->m_nSSLVersion == ALT_SSL_V3) {
      ALT_LOG_P (ALT_D_DEBUG, "use SSLv3_server_method()");
      pSSLContext = SSL_CTX_new (SSLv3_server_method ());
    }
    else if (pAcceptor->m_nSSLVersion == ALT_TSL_V1) {
      ALT_LOG_P (ALT_D_DEBUG, "use TLSv1_server_method()");
      pSSLContext = SSL_CTX_new (TLSv1_server_method ());
    }

    if (pSSLContext == NULL) {
      
      altInt  nErrNo = errno;
      DWORD   dwSSLErrNo = ERR_get_error();
      altStr  sBuf;

      altNetUtil::CloseSocket (nClientSocket);

#ifdef ALT_WIN
      altChar szBuf[BUFSIZ];
      strerror_s (szBuf, sizeof (szBuf), nErrNo);
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_CTX_new error (%d:%s) SSL_ERR_NO=(%lu:%s)", nErrNo, szBuf, dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
#endif
#ifdef ALT_LINUX
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_CTX_new error (%d:%s) SSL_ERR_NO=(%lu:%s)", nErrNo, strerror (nErrNo), dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
#endif
    }

    SSL * pSSL = SSL_new (pSSLContext);
    if (pSSL == NULL) {
      altNetUtil::CloseSocket (nClientSocket);
      ALT_RET_P (ALT_E_ERROR, "SSL_new error");
    }

    altInt nRet = SSL_set_fd (pSSL, (int)nClientSocket);
    if (nRet == 0) {
      altNetUtil::CloseSocket (nClientSocket);
      ALT_RET_P (ALT_E_ERROR, "SSL_set_fd error");
    }

    nRet = SSL_use_certificate_file (pSSL, "public.key", SSL_FILETYPE_PEM);
    if (nRet != 1) {
      altInt  nErrNo = errno;
      altInt  nSSLErrNo = SSL_get_error (pSSL, nRet);
      DWORD   dwSSLErrNo = ERR_get_error();
      altStr  sBuf;
      
      altNetUtil::CloseSocket (nClientSocket);

#ifdef ALT_WIN
      altChar szBuf[BUFSIZ];
      strerror_s (szBuf, sizeof (szBuf), nErrNo);
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_use_certificate_file error (%d:%s) SSL_ERR_NO=%d(%lu:%s)", nErrNo, szBuf, nSSLErrNo, dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
#endif
#ifdef ALT_LINUX
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_use_certificate_file error (%d:%s) SSL_ERR_NO=%d(%lu:%s)", nErrNo, strerror (nErrNo), nSSLErrNo, dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
#endif
    }

    nRet = SSL_use_PrivateKey_file (pSSL, "private.key", SSL_FILETYPE_PEM);
    if (nRet != 1) {
      altInt  nErrNo = errno;
      altInt  nSSLErrNo = SSL_get_error (pSSL, nRet);
      DWORD   dwSSLErrNo = ERR_get_error();
      altStr  sBuf;
      
      altNetUtil::CloseSocket (nClientSocket);

#ifdef ALT_WIN
      altChar szBuf[BUFSIZ];
      strerror_s (szBuf, sizeof (szBuf), nErrNo);
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_use_certificate_file error (%d:%s) SSL_ERR_NO=%d(%lu:%s)", nErrNo, szBuf, nSSLErrNo, dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
#endif
#ifdef ALT_LINUX
      ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_use_PrivateKey_file error (%d:%s) SSL_ERR_NO=%d(%lu:%s)", nErrNo, strerror (nErrNo), nSSLErrNo, dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
#endif
    }

retry_ssl_accept:
    nRet = SSL_accept (pSSL);
    if (nRet != 1) {
      altInt nErrNo = SSL_get_error (pSSL, nRet);
      if (nErrNo == SSL_ERROR_WANT_ACCEPT) {
        goto retry_ssl_accept;
      }
      else if (nErrNo == SSL_ERROR_SYSCALL) {
#ifdef ALT_WIN
        altInt nErrorNo = WSAGetLastError();
        altStr sBuf;
        ALT_LOG_P (ALT_E_ERROR, sBuf.Format ("SSL_accept error (%d) %d", nErrNo, nErrorNo));
#endif
#ifdef ALT_LINUX
        altInt nErrorNo = errno;
        altStr sBuf;
        ALT_LOG_P (ALT_E_ERROR, sBuf.Format ("SSL_accept error (%d) %d:%s", nErrNo, nErrorNo, strerror (nErrorNo)));
#endif
        altNetUtil::CloseSocket (nClientSocket);
        ALT_RET (ALT_E_ERROR);
      }
      else {
        DWORD   dwSSLErrNo = ERR_get_error();
        altStr  sBuf;
        altNetUtil::CloseSocket (nClientSocket);
        ALT_RET_P (ALT_E_ERROR, sBuf.Format ("SSL_accept error (%d) (%lu:%s)", nErrNo, dwSSLErrNo, ERR_reason_error_string (dwSSLErrNo)));
      }
    }

    // Set non block
#ifdef ALT_LINUX
    altInt nFlag = fcntl (nClientSocket, F_GETFL, 0);
    fcntl (nClientSocket, F_SETFL, nFlag | O_NONBLOCK);
#endif
#ifdef ALT_WIN
    u_long nFlag = 1;
    ioctlsocket (nClientSocket, FIONBIO, & nFlag);
#endif

    altInetAddress oInetAddress(oClientSockAddr);

    altCNEKOConnectionPtr       pNEKOConnection;
    // create new connection
    pNEKOConnection = ALT_NEW altNEKOConnection (pSSL, pSSLContext, nClientSocket, oClientSockAddr, pAcceptor->m_nPort, pAcceptor->m_oSender, pAcceptor->m_nSSLVersion);
    if (pNEKOConnection == NULL) {
      ALT_RET (ALT_E_NOMEM);
    }

    // add to connection manager
    status = aiNEKOConnectionManager.Add (pNEKOConnection);
    if (ALT_IS_ERR (status)) {
      SOCKET nClientSocket = pNEKOConnection->GetSocket();
      altNetUtil::CloseSocket (nClientSocket);
      ALT_RET (status);
    }

    // add to receive
    status = pAcceptor->m_oReceiver.Add (pNEKOConnection);
    if (ALT_IS_ERR (status)) {
      aiNEKOConnectionManager.Del (pNEKOConnection);
      ALT_RET (status);
    }

    // Call Accept call back function
    if (pAcceptor->m_pAcceptCallBackFunc != NULL) {
      pAcceptor->m_pAcceptCallBackFunc (pAcceptor->m_nPort, pNEKOConnection);
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}
