/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOClient.cpp 1388 2008-04-29 04:02:11Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOClient.cpp
 * \brief   NEKO Network Client
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOClient.h"
#include "altBase/altDataStream.h"
#include "altNet/altNetUtil.h"
#include "altBase/altUtil.h"

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
/// \param  pCallBackFunc       [I ] Recv Call Back Function
/// \param  pDisconnectCallBack [I ] Disconnect Call Back Function
/// \param  nSSLVersion         [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
LIBALT_API altNEKOClient::altNEKOClient(altNEKOClientCallBackFunc pCallBackFunc, altNEKOClientDisconnectCallBack pDisconnectCallBack, const altByte nSSLVersion) :
m_oSocket (0, nSSLVersion),
m_dwSessionID (0),
m_dwRecvSessionID (0),
m_pCallBackFunc (pCallBackFunc),
m_pDisconnectCallBack (pDisconnectCallBack),
m_oSendThread(50),
m_oRecvThread(50),
m_oInetAddr(),
m_nSSLVersion (nSSLVersion),
m_oSendQue(),
m_oRecvQue(),
m_nSendPacketID (0),
m_nRecvPacketID (0),
m_oRemoteFuncMap(),
m_bRemoteFuncImported (false),
m_pNotifyErrorFunc (NULL)
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOClient::~altNEKOClient()
{
  alt_t status = DisconnectNoWait();
  ALT_LOG (status);
  altSleep (100);
}

///
/// \brief  Connect
///
/// \param  sIP   [I ] IP Address
/// \param  nPort [I ] Port NO
///
/// \return ALT_S_SUCCESS       success
/// \return ALT_E_UNKNOWN_HOST  unknown host error
/// \return ALT_E_CONNECT       connect error
/// \return ALT_E_ERROR         error
///
LIBALT_API alt_t altNEKOClient::Connect(const altStr & sIP, const altUInt nPort)
{
  alt_t status;

  status = m_oInetAddr.Init (sIP.GetCStr(), nPort);
  ALT_ERR_RET (status);

  status = m_oSocket.Connect (m_oInetAddr);
  ALT_ERR_RET (status);

  m_oSendThread.Start (altNEKOClient::SendThread, this);
  m_oRecvThread.Start (altNEKOClient::RecvThread, this);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Disconnect
///
/// \return ALT_S_SUCCESS       success
///
LIBALT_API alt_t altNEKOClient::Disconnect()
{
  alt_t status;

  m_oRecvThread.StopNoWait();
  m_oSendThread.StopNoWait();

  m_oRecvThread.Stop();
  m_oSendThread.Stop();

  status = m_oSocket.Close();
  ALT_ERR_RET (status);
  m_nSendPacketID = 0;
  m_nRecvPacketID = 0;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Disconnect
///
/// \return ALT_S_SUCCESS       success
///
LIBALT_API alt_t altNEKOClient::DisconnectNoWait()
{
  alt_t status;

  m_oRecvThread.StopNoWait();
  m_oSendThread.StopNoWait();
  
  status = m_oSocket.Close();
  ALT_ERR_RET (status);
  m_nSendPacketID = 0;
  m_nRecvPacketID = 0;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send
///
/// \param  sFuncName   [I ] Function Name
/// \param  pData       [I ] Send Data
/// \param  nSize       [I ] Send Data Size
/// \paraam dwSessionID [ O] Session ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_NOMEM     out of memory error
///
LIBALT_API alt_t altNEKOClient::Send(const altStr & sFuncName, const altCharPtr & pData, const altUInt nSize, DWORD & dwSessionID)
{
  alt_t status;

  m_dwSessionID++;
  dwSessionID = m_dwSessionID;

  altClientPacket * pPacket = ALT_NEW altClientPacket();
  if (pPacket == NULL) {
    ALT_RET (ALT_E_NOMEM);
  }

  altDataStream oDataStream;

  status = oDataStream.Write (sFuncName);
  ALT_ERR_RET (status);

  status = oDataStream.Write (m_dwSessionID);
  ALT_ERR_RET (status);

  status = oDataStream.AddData (pData, nSize);
  ALT_ERR_RET (status);

  altCharPtr  pSendData;
  altUInt     nSendDataSize;

  status = oDataStream.GetData (pSendData);
  ALT_ERR_RET (status);

  nSendDataSize = oDataStream.GetDataSize();

  status = pPacket->Set (pSendData, nSendDataSize);
  ALT_ERR_RET (status);

  status = m_oSendQue.Push (pPacket);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send
///
/// \param  sFuncName   [I ] Function Name
/// \param  pData       [I ] Send Data
/// \param  nSize       [I ] Send Data Size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_NOMEM     out of memory error
///
LIBALT_API alt_t altNEKOClient::Send(const altStr & sFuncName, const altCharPtr & pData, const altUInt nSize)
{
  alt_t status;
  DWORD dwSessionID;
  status = Send (sFuncName, pData, nSize, dwSessionID);
  ALT_ERR_RET (status);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send
///
/// \param  sFuncName   [I ] Function Name
/// \param  oDataStream [I ] Send Data
/// \paraam dwSessionID [ O] Session ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_NOMEM     out of memory error
///
LIBALT_API alt_t altNEKOClient::Send(const altStr & sFuncName, const altDataStream & oDataStream, DWORD & dwSessionID)
{
  alt_t status;

  altClientPacket * pPacket = ALT_NEW altClientPacket();
  if (pPacket == NULL) {
    ALT_RET (ALT_E_NOMEM);
  }

  m_dwSessionID++;
  dwSessionID = m_dwSessionID;

  altDataStream oSendDataStream;

  status = oSendDataStream.Write (sFuncName);
  ALT_ERR_RET (status);

  status = oSendDataStream.Write (m_dwSessionID);
  ALT_ERR_RET (status);

  status = oSendDataStream.AddData (oDataStream);
  ALT_ERR_RET (status);

  altCharPtr  pSendData;
  altUInt     nSendDataSize;

  status = oSendDataStream.GetData (pSendData);
  ALT_ERR_RET (status);

  nSendDataSize = oSendDataStream.GetDataSize();

  status = pPacket->Set (pSendData, nSendDataSize);
  ALT_ERR_RET (status);

  status = m_oSendQue.Push (pPacket);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send
///
/// \param  sFuncName   [I ] Function Name
/// \param  oDataStream [I ] Send Data
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_NOMEM     out of memory error
///
LIBALT_API alt_t altNEKOClient::Send(const altStr & sFuncName, const altDataStream & oDataStream)
{
  alt_t status;
  DWORD dwSessionID;

  status = Send (sFuncName, oDataStream, dwSessionID);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get IP
///
/// \return IP Address
///
LIBALT_API altStr altNEKOClient::GetIP() const
{
  return (m_oInetAddr.GetIP());
}

///
/// \brief Get Port NO
///
/// \return Port NO
///
LIBALT_API altInt altNEKOClient::GetPort() const
{
  return (m_oInetAddr.GetPort());
}

///
/// \brief  Disconnected
///
/// \return ALT_S_SUCCESS       success
///
LIBALT_API alt_t altNEKOClient::Disconnected()
{
  alt_t status;

  m_oRecvThread.StopNoWait();
  m_oSendThread.StopNoWait();

  status = m_oSocket.Close();
  ALT_ERR_RET (status);

  m_nSendPacketID = 0;
  m_nRecvPacketID = 0;

  if (m_pDisconnectCallBack != NULL) {
    (m_pDisconnectCallBack) (* this);
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief Get is connected
///
/// \return true    connecting
/// \return false   not connecting
///
LIBALT_API altBool altNEKOClient::Connected()
{
  return (m_oSocket.Connected());
}

///
/// \brief  Get session ID
///
/// \return session ID
///
LIBALT_API DWORD altNEKOClient::GetSessionID()
{
  return (m_dwSessionID);
}

///
/// \brief  Get receive session ID
///
/// \return Receive session ID
///
LIBALT_API DWORD altNEKOClient::GetRecvSessionID()
{
  return (m_dwRecvSessionID);
}

///
/// \brief  Check sessioning or not
///
/// \return true...sessioning false...no session
///
LIBALT_API altBool altNEKOClient::IsWaitSession()
{
  return (m_dwSessionID > m_dwRecvSessionID);
}

///
/// \brief  Import remote function table
///
/// \param  pTable  [I ] Remote function table
/// \param  nSize   [I ] Remote function table size
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altNEKOClient::ImportRemoteFuncTable(altNEKORemoteFuncTable * pTable, altUInt nSize)
{
  try {
    for (altUInt i = 0; i < nSize; i++) {
      m_oRemoteFuncMap[pTable[i].sFuncName] = pTable[i].pFunc;
    }
  }
  catch (std::exception & e) {
    ALT_RET_P (ALT_E_ERROR, e.what());
  }
  catch (...) {
    ALT_RET (ALT_E_ERROR);
  }
  m_bRemoteFuncImported = true;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Receive Total Size Notify Function
///
/// \param  pNotifyTotalSizeCallBack  [I ] Total Size Notify Function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOClient::SetNotifyTotalSizeFunc(altNotifyTotalSizeFunc pNotifyTotalSizeCallBack)
{
  alt_t status;
  status = m_oSocket.SetNotifyTotalSizeFunc (pNotifyTotalSizeCallBack);
  ALT_RET (status);
}

///
/// \brief  Set Receive Receive Size Notify Function
///
/// \param  pNotifyRecvSizeCallBack  [I ] Receive Size Notify Function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOClient::SetNotifyRecvSizeFunc(altNotifyRecvSizeFunc pNotifyRecvSizeCallBack)
{
  alt_t status;
  status = m_oSocket.SetNotifyRecvSizeFunc (pNotifyRecvSizeCallBack);
  ALT_RET (status);
}

///
/// \brief  Set Notify Error Function
///
/// \param  pNotifyErrorFunc  [I ] Notify Error Function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOClient::SetNotifyErrorFunc(altNEKOClientCallBackFunc pNotifyErrorFunc)
{
  m_pNotifyErrorFunc = pNotifyErrorFunc;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set receive thread interval
///
/// \param  msec  [I ] interval(msec)
///
/// \return ALT_S_SUCCESS
///
LIBALT_API alt_t altNEKOClient::SetRecvThreadInterval(const DWORD msec)
{
  m_oRecvThread.SetInterval (msec);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set send thread interval
///
/// \param  msec  [I ] interval(msec)
///
/// \return ALT_S_SUCCESS
///
LIBALT_API alt_t altNEKOClient::SetSendThreadInterval(const DWORD msec)
{
  m_oSendThread.SetInterval (msec);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send Thread
///
/// \param  pParam  [I ] this Client Object
///
LIBALT_API alt_t altNEKOClient::SendThread(void * pParam)
{
  altNEKOClient * pClient = (altNEKOClient *)pParam;
  alt_t status;

  altClientPacket * pPacket = NULL;
  status = pClient->m_oSendQue.Pop (pPacket);
  if (pPacket != NULL) {

    status = pPacket->Compress ();
    if (ALT_IS_ERR (status)) {
      ALT_LOG (status);
    }

    status = pClient->m_oSocket.Send (pPacket->GetData(), pPacket->GetSize(), pClient->m_nSendPacketID);
    delete pPacket;
    pClient->m_nSendPacketID++;
    if (ALT_IS_ERR (status)) {
      ALT_LOG (status);
      // Disconnect
      status = pClient->Disconnected();
      ALT_RET (status);
    }
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Recv Thread
///
/// \param  pParam  [I ] this Client Object
///
LIBALT_API alt_t altNEKOClient::RecvThread(void * pParam)
{
  altNEKOClient * pClient = (altNEKOClient *)pParam;
  alt_t status;

  altClientPacket * pPacket = NULL;
  altCharPtr        pBuf;
  altUInt           nRecvSize = ALT_BUFSIZ;
  fd_set            oRfds;
  fd_set            oEfds;
  struct timeval    oTimeout;
  SOCKET            nSocket = pClient->m_oSocket.GetSocket();
  
  oTimeout.tv_sec = 0;
  oTimeout.tv_usec = 10;

  FD_ZERO (& oRfds);
  FD_ZERO (& oEfds);

  FD_SET (nSocket, & oRfds);
  FD_SET (nSocket, & oEfds);
  
  status = altNetUtil::Select (nSocket, & oRfds, NULL, & oEfds, & oTimeout);
  if (ALT_IS_ERR (status)) {
    ALT_LOG (status);
    // Disconnect
    status = pClient->Disconnected();
    ALT_RET (status);
  }

  if (status == ALT_S_SUCCESS) {

    if (FD_ISSET (nSocket, & oRfds)) {
      altULongLong  nPacketID;
      altStr        sBuf;

      status = pClient->m_oSocket.Recv (pBuf, nRecvSize, nPacketID);
      if (nPacketID != pClient->m_nRecvPacketID) {
        ALT_RET_P (ALT_E_ERROR, sBuf.Format ("Invalid packet Remote[%llu] != Local[%llu]", nPacketID, pClient->m_nRecvPacketID));
      }
      pClient->m_nRecvPacketID++;

      if (status == ALT_S_SUCCESS) {
        pPacket = ALT_NEW altClientPacket ();
        
        status = pPacket->Set (pBuf, nRecvSize);
        if (ALT_IS_ERR (status)) {
          delete pPacket;
          ALT_RET (status);
        }
        
        status = pPacket->Decompress();
        if (ALT_IS_ERR (status)) {
          delete pPacket;
          ALT_RET (status);
        }

        status = pClient->m_oRecvQue.Push (pPacket);
        if (ALT_IS_ERR (status)) {
          delete pPacket;
          ALT_RET (status);
        }
        altThread oThread;
        oThread.Start (altNEKOClient::RequestThread, pClient);
      }
      else {
        ALT_LOG (status);
        // Disconnect
        status = pClient->Disconnected();
        ALT_RET (status);
      }
    }

    if (FD_ISSET (nSocket, & oEfds)) {
      // Disconnect
      status = pClient->Disconnected();
      ALT_RET (status);
    }
  }
  else if (status == ALT_E_ERROR) {
    // Disconnect
    status = pClient->Disconnected();
    ALT_RET (status);
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Request Thread
///
/// \param  pParam  [I ] this Client Object
///
LIBALT_API ALT_THREAD altNEKOClient::RequestThread(void * pParam)
{
  altNEKOClient * pClient = (altNEKOClient *)pParam;
  alt_t status;

  altClientPacket * pPacket = NULL;
  status = pClient->m_oRecvQue.Pop (pPacket);
  if (pPacket != NULL) {

    altDataStream oDataStream;
    altStr        sFuncName;

    status = oDataStream.SetData (pPacket->GetData(), pPacket->GetSize());
    ALT_LOG (status);

    DWORD dwRecvSessionID;
    oDataStream.Read (sFuncName);
    oDataStream.Read (dwRecvSessionID);

    if (pClient->m_bRemoteFuncImported) {
      altNEKOClientCallBackFunc pFunc = pClient->m_oRemoteFuncMap[sFuncName];
      if (pFunc != NULL) {
        status = pFunc (* pClient, sFuncName, dwRecvSessionID, oDataStream);
        if (ALT_IS_ERR (status) && pClient->m_pNotifyErrorFunc != NULL) {
          pClient->m_pNotifyErrorFunc (* pClient, sFuncName, dwRecvSessionID, oDataStream);
        }
      }
    }
    else if (pClient->m_pCallBackFunc != NULL) {
      status = pClient->m_pCallBackFunc (* pClient, sFuncName, dwRecvSessionID, oDataStream);
      if (ALT_IS_ERR (status) && pClient->m_pNotifyErrorFunc != NULL) {
        pClient->m_pNotifyErrorFunc (* pClient, sFuncName, dwRecvSessionID, oDataStream);
      }
    }
    pClient->m_dwRecvSessionID = dwRecvSessionID;
    delete pPacket;
  }

  ALT_THREAD_END (0);
}

