#ifndef ALTNEKOCLIENT_H
#define ALTNEKOCLIENT_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOClient.h 1378 2008-04-25 20:57:23Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOClient.h
 * \brief   NEKO Network Client
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altNeko/altNEKOSocket.h>
#include <altNet/altSSLClient.h>
#include <altBase/altDataStream.h>
#include <map>

/*----------------------------------------------------------------*
 * #define
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
class altNEKOClient;

///
/// \brief Client call back function
///
/// \param  oClient     [IO] Client object
/// \param  sFuncName   [I ] Function name
/// \param  dwSessionID [I ] Session ID
/// \param  oDataStream [I ] Data stream
/// \param  pParam      [I ] Call back paramter
///
typedef alt_t (* altNEKOClientCallBackFunc)(altNEKOClient & oClient, const altStr & sFuncName, const DWORD dwSessionID, altDataStream & oDataStream);

///
/// \brief  Disconnect call back function
///
/// \param  oClient     [IO] Client object
///
typedef alt_t (* altNEKOClientDisconnectCallBack)(altNEKOClient & oClient);

///
/// \brief  Call back table element
///
typedef struct {
  altStr                      sFuncName;  ///< Function name
  altNEKOClientCallBackFunc   pFunc;      ///< Function
} altNEKORemoteFuncTable;

///
/// \brief  Call back function map
///
typedef std::map<altStr, altNEKOClientCallBackFunc> altNEKORemoteFuncMap;

///
/// \brief  Session result map
///
typedef std::map<DWORD, alt_t> altNEKOSessionResultMap;

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  NEKO client class
///
class altNEKOClient : public altBase, private altUncopyable, public altSyncThread
{
public:
  ///
  /// \brief  Constructor
  ///
  /// \param  pCallBackFunc       [I ] Recv Call Back Function
  /// \param  pDisconnectCallBack [I ] Disconnect Call Back Function
  /// \param  nSSLVersion         [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
  ///
  LIBALT_API altNEKOClient(altNEKOClientCallBackFunc pCallBackFunc, altNEKOClientDisconnectCallBack pDisconnectCallBack, const altByte nSSLVersion = ALT_SSL_V3);

  ///
  /// \brief  Destructor
  ///
  LIBALT_API virtual ~altNEKOClient();

  ///
  /// \brief  Connect
  ///
  /// \param  sIP   [I ] IP Address
  /// \param  nPort [I ] Port NO
  ///
  /// \return ALT_S_SUCCESS       success
  /// \return ALT_E_UNKNOWN_HOST  unknown host error
  /// \return ALT_E_CONNECT       connect error
  /// \return ALT_E_ERROR         error
  ///
  LIBALT_API virtual alt_t Connect(const altStr & sIP, const altUInt nPort);

  ///
  /// \brief  Disconnect
  ///
  /// \return ALT_S_SUCCESS       success
  ///
  LIBALT_API alt_t Disconnect();

  ///
  /// \brief  Disconnect
  ///
  /// \return ALT_S_SUCCESS       success
  ///
  LIBALT_API alt_t DisconnectNoWait();

  ///
  /// \brief  Send
  ///
  /// \param  sFuncName   [I ] Function Name
  /// \param  pData       [I ] Send Data
  /// \param  nSize       [I ] Send Data Size
  /// \paraam dwSessionID [ O] Session ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_NOMEM     out of memory error
  ///
  LIBALT_API alt_t Send(const altStr & sFuncName, const altCharPtr & pData, const altUInt nSize, DWORD & dwSessionID);

  ///
  /// \brief  Send
  ///
  /// \param  sFuncName   [I ] Function Name
  /// \param  pData       [I ] Send Data
  /// \param  nSize       [I ] Send Data Size
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_NOMEM     out of memory error
  ///
  LIBALT_API alt_t Send(const altStr & sFuncName, const altCharPtr & pData, const altUInt nSize);

  ///
  /// \brief  Send
  ///
  /// \param  sFuncName   [I ] Function Name
  /// \param  oDataStream [I ] Send Data
  /// \paraam dwSessionID [ O] Session ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_NOMEM     out of memory error
  ///
  LIBALT_API alt_t Send(const altStr & sFuncName, const altDataStream & oDataStream, DWORD & dwSessionID);

  ///
  /// \brief  Send
  ///
  /// \param  sFuncName   [I ] Function Name
  /// \param  oDataStream [I ] Send Data
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_NOMEM     out of memory error
  ///
  LIBALT_API alt_t Send(const altStr & sFuncName, const altDataStream & oDataStream);

  ///
  /// \brief  Get IP
  ///
  /// \return IP Address
  ///
  LIBALT_API altStr GetIP() const;

  ///
  /// \brief Get Port NO
  ///
  /// \return Port NO
  ///
  LIBALT_API altInt GetPort() const;

  ///
  /// \brief  Disconnected
  ///
  /// \return ALT_S_SUCCESS       success
  ///
  LIBALT_API alt_t Disconnected();

  ///
  /// \brief Get is connected
  ///
  /// \return true    connecting
  /// \return false   not connecting
  ///
  LIBALT_API altBool Connected();

  ///
  /// \brief  Get session ID
  ///
  /// \return session ID
  ///
  LIBALT_API DWORD GetSessionID();

  ///
  /// \brief  Get receive session ID
  ///
  /// \return Receive session ID
  ///
  LIBALT_API DWORD GetRecvSessionID();

  ///
  /// \brief  Check sessioning or not
  ///
  /// \return true...sessioning false...no session
  ///
  LIBALT_API altBool IsWaitSession();

  ///
  /// \brief  Import remote function table
  ///
  /// \param  pTable  [I ] Remote function table
  /// \param  nSize   [I ] Remote function table size
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API alt_t ImportRemoteFuncTable(altNEKORemoteFuncTable * pTable, altUInt nSize);

  ///
  /// \brief  Get session result
  ///
  /// \param  dwSessionID [I ] Session ID
  ///
  /// \return Session result
  ///
  LIBALT_API alt_t GetSessionResult(const DWORD dwSessionID);
  
  ///
  /// \brief  Set Receive Total Size Notify Function
  ///
  /// \param  pNotifyTotalSizeCallBack  [I ] Total Size Notify Function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyTotalSizeFunc(altNotifyTotalSizeFunc pNotifyTotalSizeCallBack);

  ///
  /// \brief  Set Receive Receive Size Notify Function
  ///
  /// \param  pNotifyRecvSizeCallBack  [I ] Receive Size Notify Function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyRecvSizeFunc(altNotifyRecvSizeFunc pNotifyRecvSizeCallBack);

  ///
  /// \brief  Set Notify Error Function
  ///
  /// \param  pNotifyErrorFunc  [I ] Notify Error Function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyErrorFunc(altNEKOClientCallBackFunc pNotifyErrorFunc);

  ///
  /// \brief  Set receive thread interval
  ///
  /// \param  msec  [I ] interval(msec)
  ///
  /// \return ALT_S_SUCCESS
  ///
  LIBALT_API alt_t SetRecvThreadInterval(const DWORD msec);

  ///
  /// \brief  Set send thread interval
  ///
  /// \param  msec  [I ] interval(msec)
  ///
  /// \return ALT_S_SUCCESS
  ///
  LIBALT_API alt_t SetSendThreadInterval(const DWORD msec);

protected:
  ///
  /// \brief  Send Thread
  ///
  /// \param  pParam  [I ] this Client Object
  ///
  LIBALT_API static alt_t SendThread(void * pParam);

  ///
  /// \brief  Recv Thread
  ///
  /// \param  pParam  [I ] this Client Object
  ///
  LIBALT_API static alt_t RecvThread(void * pParam);

  ///
  /// \brief  Request Thread
  ///
  /// \param  pParam  [I ] this Client Object
  ///
  LIBALT_API static ALT_THREAD RequestThread(void * pParam);

  altNEKOSocket                     m_oSocket;              ///< Socket
  DWORD                             m_dwSessionID;          ///< Session ID
  DWORD                             m_dwRecvSessionID;      ///< Receive session ID
  altNEKOClientCallBackFunc         m_pCallBackFunc;        ///< Call back function
  altNEKOClientDisconnectCallBack   m_pDisconnectCallBack;  ///< Disconnect call back function
  altLoopThread                     m_oSendThread;          ///< Send Thread
  altLoopThread                     m_oRecvThread;          ///< Recv Thread
  altInetAddress                    m_oInetAddr;            ///< Internet Address
  altByte                           m_nSSLVersion;          ///< SSL Version
  altQue<altClientPacket *>         m_oSendQue;             ///< Send Que
  altQue<altClientPacket *>         m_oRecvQue;             ///< Recv Que
  altULongLong                      m_nSendPacketID;        ///< Send packet ID
  altULongLong                      m_nRecvPacketID;        ///< Recv packet ID
  altNEKORemoteFuncMap              m_oRemoteFuncMap;       ///< Remote function map
  altBool                           m_bRemoteFuncImported;  ///< Remote function table imported or not
  altNEKOClientCallBackFunc         m_pNotifyErrorFunc;     ///< Notify error function
};
#endif  //ALTNEKOCLIENT_H
