/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOConnection.cpp 1391 2008-05-01 18:07:37Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOConnection.cpp
 * \brief   NEKO Network Connection Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOConnection.h"

/*----------------------------------------------------------------*
 * Define
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
/// \param  pSSL            [I ] Pointer of SSL Object
/// \param  m_pSSLContext   [I ] Pointer of SSL Context Object
/// \param  nSocket         [I ] Socket
/// \param  oSockAddrIn     [I ] Internet Address of Socket
/// \param  nAcceptedPortNo [I ] Accepted Port Number
/// \param  oSender         [I ] Sender
/// \param  nSSLVersion     [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
LIBALT_API altNEKOConnection::altNEKOConnection(SSL * pSSL, SSL_CTX * pSSLContext, const SOCKET nSocket, const SOCKADDR_IN & oSockAddrIn, const altInt nAcceptedPortNo, altNEKOSender & oSender, const altByte nSSLVersion) :
m_bServer (false),
m_oSocket (nSocket, nSSLVersion),
m_oSockAddrIn (oSockAddrIn),
m_nAcceptedPortNo (nAcceptedPortNo),
m_oSender (oSender),
m_bServerConnection (false),
m_nSendPacketID (0),
m_nRecvPacketID (0)
{
  m_oSocket.SetSSL (pSSL, pSSLContext);
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOConnection::~altNEKOConnection()
{
}

///
/// \brief  Get Socket
///
/// \return Socket
///
LIBALT_API const SOCKET altNEKOConnection::GetSocket() const
{
  return (m_oSocket.GetSocket());
}
  
///
/// \brief Get IP Address
///
/// \return IP Address
///
LIBALT_API const altStr altNEKOConnection::GetIP() const
{
  return (m_oSockAddrIn.GetIP());
}

///
/// \brief Get Port NO
///
/// \return Port NO
///
LIBALT_API altInt altNEKOConnection::GetPort() const
{
  return (m_oSockAddrIn.GetPort());
}

///
/// \brief  Get accepted port number
///
/// \return Accepted port number
///
LIBALT_API const altInt altNEKOConnection::GetAcceptedPort() const
{
  return (m_nAcceptedPortNo);
}

///
/// \brief  Close Socket
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOConnection::Close()
{
  alt_t status = m_oSocket.Close();
  ALT_ERR_RET (status);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read Data
///
/// \param  pData [ O] receive data
/// \param  nSize [IO] receive data size
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_NET_EOF   connection closed
/// \return ALT_S_TIMEOUT   timeout
/// \return ALT_E_NOMEM     out of memory error
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOConnection::Recv(altCharPtr & pData, altUInt & nSize) const
{
  alt_t         status;
  altULongLong  nPacketID;

  status = m_oSocket.Recv (pData, nSize, nPacketID);
  ALT_ERR_RET (status);

  if (status == ALT_S_NET_EOF) {
    ALT_RET (status);
  }

  if (nPacketID != m_nRecvPacketID) {
    altStr  sBuf;
    ALT_RET_P (ALT_E_ERROR, sBuf.Format ("Invalid Packet ID Remote[%llu] != Local[%llu]", nPacketID, m_nRecvPacketID));
  }

  m_nRecvPacketID++;

  ALT_RET (status);
}

///
/// \brief  Send Data
///
/// \param  pData [I ] send data
/// \param  nSize [I ] send data size
///
/// \return ALT_S_SUCCESS   successs
/// \return ALT_E_NOMEM     out of memory
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOConnection::Send(const altCharPtr & pData, const altInt nSize) const
{
  alt_t status;

  status = m_oSender.Send (this, pData, nSize);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send Data
///
/// \param  sRemoteFuncName [I ] Remote function name
/// \param  oDataStream     [I ] Send data
///
/// \return ALT_S_SUCCESS   successs
/// \return ALT_E_NOMEM     out of memory
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOConnection::Send(const altStr & sRemoteFuncName, const altDataStream & oDataStream) const
{
  alt_t         status;
  altDataStream oSendDataStream;
  DWORD         dwSessionID = 0;
  
  status = oSendDataStream.Write (sRemoteFuncName);
  ALT_ERR_RET (status);

  status = oSendDataStream.Write (dwSessionID);
  ALT_ERR_RET (status);

  altCharPtr  pSendData;
  status = oDataStream.GetData (pSendData);
  ALT_ERR_RET (status);
  altUInt nSize;
  nSize = oDataStream.GetDataSize();

  status = oSendDataStream.AddData (pSendData, nSize);
  ALT_ERR_RET (status);

  status = oSendDataStream.GetData (pSendData);
  ALT_ERR_RET (status);
  nSize = oSendDataStream.GetDataSize();

  status = Send (pSendData, nSize);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Real Send Data
///
/// \param  pData [I ] send data
/// \param  nSize [I ] send data size
///
/// \return ALT_S_SUCCESS   successs
/// \return ALT_E_SEND      send error
///
LIBALT_API alt_t altNEKOConnection::RealSend(const altCharPtr & pData, const altInt nSize) const
{
  alt_t   status;

  status = m_oSocket.Send (pData, nSize, m_nSendPacketID);
  ALT_ERR_RET (status);

  m_nSendPacketID++;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get connecting or not
///
/// \return true  connecting
/// \return false not connecting
///
LIBALT_API altBool altNEKOConnection::Connected() const
{
  return (m_oSocket.Connected());
}

///
/// \brief  Get server connection or not
///
/// \return true  server connection
/// \return false client connection
///
LIBALT_API altBool altNEKOConnection::IsServerConnection() const
{
  return (m_bServerConnection);
}
