/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOConnectionManager.cpp 1394 2008-05-01 19:10:17Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOConnectionManager.cpp
 * \brief   NEKO Network Connection Manager
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOConnectionManager.h"

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altNEKOConnectionManager::altNEKOConnectionManager() :
m_oClientConnectionMap(),
m_oServerConnectionMap(),
m_oNEKOReceiverMap(),
m_pDisconnectedCallBackFunc (NULL),
m_oServerInfoMap()
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOConnectionManager::~altNEKOConnectionManager()
{
  m_pDisconnectedCallBackFunc = NULL;
}

///
/// \brief  Set Disconnected Call Back Function
///
/// \param  pDisconnectedFunc [I ] Disconnected Call Back Function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOConnectionManager::SetDisconnectedCallBackFunc(altNEKODisconnectCallBackFunc pDisconnectedFunc)
{
  m_pDisconnectedCallBackFunc = pDisconnectedFunc;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Add Connection
///
/// \param  pConnection   [I ] Pointer of NET Connection
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altNEKOConnectionManager::Add(const altCNEKOConnectionPtr & pConnection)
{
  BeginLock();
  try {
    m_oClientConnectionMap[pConnection->GetSocket()] = pConnection;
  }
  catch (std::exception & e) {
    EndLock();
    ALT_RET_P (ALT_E_ERROR, e.what());
  }
  catch (...) {
    EndLock();
    ALT_RET (ALT_E_ERROR);
  }
  EndLock();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Delete Connection
///
/// \param  pConnection   [I ] Pointer of NET Connection
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOConnectionManager::Del(const altCNEKOConnectionPtr & pConnection)
{
  alt_t status;
  BeginLock();

  for (altNEKOReceiverMap::iterator i = m_oNEKOReceiverMap.begin(); i != m_oNEKOReceiverMap.end(); i++) {
    status = i->second->Del (pConnection);
    if (status == ALT_S_SUCCESS) {
      break;
    }
  }

  altNEKOConnectionMap::iterator i = m_oClientConnectionMap.find (pConnection->GetSocket());
  if (i != m_oClientConnectionMap.end()) {

    // Call Disconnected Call Back Function
    if (m_pDisconnectedCallBackFunc != NULL) {
      m_pDisconnectedCallBackFunc (i->second);
    }

    // Close Connection
    altNEKOConnection & oNEKOConnection = const_cast<altNEKOConnection &>(* (i->second));
    status = oNEKOConnection.Close();
    ALT_LOG (status);

    m_oClientConnectionMap.erase (i);
  }

  EndLock();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set to Server Connection
///
/// \param  sServerName   [I : Server Name
/// \param  pConnection   [I ] Pointer of NET Connection
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altNEKOConnectionManager::SetToServerConnection(const altStr & sServerName, const altCNEKOConnectionPtr & pConnection)
{
  altNEKOConnectionMap::iterator i = m_oClientConnectionMap.find (pConnection->GetSocket());
  if (i == m_oClientConnectionMap.end()) {
    ALT_RET (ALT_E_ERROR);
  }
  m_oClientConnectionMap.erase (i);

  pConnection->m_bServer = true;
  m_oServerConnectionMap[sServerName] = pConnection;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get server connection
///
/// \param  sServerName   [I ] Server Name
///
/// \return Get server connection
///
LIBALT_API const altCNEKOConnectionPtr & altNEKOConnectionManager::GetServerConnection(const altStr & sServerName)
{
  return (m_oServerConnectionMap[sServerName]);
}

///
/// \brief  Get Connection Count
///
/// \return Connection Count
///
LIBALT_API altUInt altNEKOConnectionManager::Size() const
{
  return static_cast<altUInt>(m_oClientConnectionMap.size());
}

///
/// \brief  Add NET Receiver Broker
///
/// \param  nPort               [I ] Port Number
/// \param  pTCPReceiverBroker  [I ] Pointer of TCP Receiver Broker
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altNEKOConnectionManager::Add(const altInt nPort, altNEKOReceiver * pNEKOReceiver)
{
  try {
    m_oNEKOReceiverMap[nPort] = pNEKOReceiver;
  }
  catch (std::exception & e) {
    ALT_RET_P (ALT_E_ERROR, e.what());
  }
  catch (...) {
    ALT_RET (ALT_E_ERROR);
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Delete TCP Receiver Broker
///
/// \param  nPort [I ] Port Number
///
LIBALT_API void altNEKOConnectionManager::DeletePort(const altInt nPort)
{
  BeginLock();
  m_oNEKOReceiverMap.erase (nPort);
  EndLock();
}

///
/// \brief  Close all client connection
///
LIBALT_API void altNEKOConnectionManager::CloseAllClientConnection()
{
  BeginLock();
  for (altNEKOReceiverMap::iterator i = m_oNEKOReceiverMap.begin(); i != m_oNEKOReceiverMap.end(); i++) {
    i->second->DelAllClient();
  }
  for (altNEKOConnectionMap::iterator i = m_oClientConnectionMap.begin(); i != m_oClientConnectionMap.end(); i++) {
    // Close Connection
    altNEKOConnection & oNEKOConnection = const_cast<altNEKOConnection &>(* (i->second));
    alt_t status = oNEKOConnection.Close();
    ALT_LOG (status);

    m_oClientConnectionMap.erase (i);
  }
  EndLock();
}

