/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOReceiver.cpp 1391 2008-05-01 18:07:37Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOReceiver.cpp
 * \brief   NEKO Network Data Receiver
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOReceiver.h"
#include "altNet/altNETConnectionManager.h"
#include "altNet/altNETReceiverBroker.h"
#include "altNet/altNetUtil.h"
#include "altBase/altMem.h"
#include "altBase/altUtil.h"

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
/// \param  pReceiveCallBackFunc  [I ] Receive Call Back Function
/// \param  nThreadCount          [I ] Request Thread Count
///
LIBALT_API altNEKOReceiver::altNEKOReceiver(const altNEKOReceiveCallBack pReceiveCallBackFunc, const altUInt nThreadCount) :
m_oPacketQue(),
m_oConnectionMap(),
m_oReceiveThreaed (50),
m_oRequestInvoker (m_oPacketQue, pReceiveCallBackFunc, nThreadCount)
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOReceiver::~altNEKOReceiver()
{
  alt_t status;
  m_oRequestInvoker.~altNEKORequestInvoker();
  status = Stop();
  ALT_LOG (status);
}

///
/// \brief  Add Connection
///
/// \param  pConnection [I ] NET Connection
///
/// \return ALT_S_SUCCESS   success
///
LIBALT_API alt_t altNEKOReceiver::Add(const altCNEKOConnectionPtr & pConnection)
{
  BeginLock();
  m_oConnectionMap[pConnection->GetSocket()] = pConnection;
  EndLock();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Del Connection
///
/// \param  pConnection [I ] NET Connection
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_NOT_FOUND not found
///
LIBALT_API alt_t altNEKOReceiver::Del(const altCNEKOConnectionPtr & pConnection)
{
  BeginLock();
  altNEKOConnectionMap::iterator i = m_oConnectionMap.find (pConnection->GetSocket());
  if (i != m_oConnectionMap.end()) {
    m_oConnectionMap.erase (i);
  }
  else {
    EndLock();
    ALT_RET (ALT_S_NOT_FOUND);
  }
  EndLock();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get Size
///
/// \return content size
///
LIBALT_API altUInt altNEKOReceiver::Size() const
{
  return (static_cast<altUInt>(m_oConnectionMap.size()));
}

///
/// \brief  Start Receive Thread
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOReceiver::Start()
{
  m_oReceiveThreaed.Start (altNEKOReceiver::ReceiveThread, this);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Stop Receiver Thread
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOReceiver::Stop()
{
  m_oReceiveThreaed.Stop();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Pop received packet
///
/// \param  pPacket [ O] pointer of packet
///
/// \return ALT_S_SUCCESS     success
/// \return ALT_S_NOT_FOUND   not data
///
LIBALT_API alt_t altNEKOReceiver::Pop (altNEKOPacket * & pPacket)
{
  alt_t status;
  status = m_oPacketQue.Pop (pPacket);
  ALT_RET (status);
}

///
/// \brief  Delete all client connection
///
LIBALT_API void altNEKOReceiver::DelAllClient()
{
  BeginLock();
  for (altNEKOConnectionMap::iterator i = m_oConnectionMap.begin(); i != m_oConnectionMap.end(); i++) {
    if (! i->second->m_bServer) {
      m_oConnectionMap.erase (i);
    }
  }
  EndLock();
}

///
/// \brief  Get packet count
///
/// \return Packet count
///
LIBALT_API altUInt altNEKOReceiver::GetPacketCount() const
{
  return (m_oPacketQue.Size());
}

///
/// \brief  receive thread
///
/// \param  pParam  [I ] altNETReceiver object
///
LIBALT_API alt_t altNEKOReceiver::ReceiveThread(void * pParam)
{
  alt_t                               status;
  altNEKOReceiver *                   pReceiver = static_cast<altNEKOReceiver *>(pParam);
  
  pReceiver->BeginLock();

  altNEKOPacketQue &                  oPacketQue = pReceiver->m_oPacketQue;
  altNEKOConnectionMap                oConnectionMap = pReceiver->m_oConnectionMap;
  fd_set                              oRfds;
  fd_set                              oEfds;
  struct timeval                      oTimeout;
  std::vector<altCNEKOConnectionPtr>  oDisconnectList;

  oTimeout.tv_sec = 0;
  oTimeout.tv_usec = 10;

  for (altNEKOConnectionMap::iterator i = oConnectionMap.begin(); i != oConnectionMap.end(); i++) {
    altCNEKOConnectionPtr pNEKOConnection = i->second;
    SOCKET nSocket = pNEKOConnection->GetSocket();

    FD_ZERO (& oRfds);
    FD_ZERO (& oEfds);

    FD_SET (nSocket, & oRfds);
    FD_SET (nSocket, & oEfds);

    status = altNetUtil::Select (nSocket, & oRfds, NULL, & oEfds, & oTimeout);
    if (ALT_IS_ERR (status)) {
      ALT_LOG (status);
      oDisconnectList.push_back (pNEKOConnection);
      continue;
    }

    if (status == ALT_S_SUCCESS) {
      for (altNEKOConnectionMap::iterator i = oConnectionMap.begin(); i != oConnectionMap.end(); i++) {
        
        altCNEKOConnectionPtr pNEKOConnection = i->second;
        SOCKET                nSocket = pNEKOConnection->GetSocket();
        altCharPtr            pData;
        altUInt               nSize;

        if (FD_ISSET (nSocket, & oRfds)) {
          status = pNEKOConnection->Recv (pData, nSize);
          if (status == ALT_S_SUCCESS) {
            altNEKOPacket * pPacket = ALT_NEW altNEKOPacket();
            pPacket->Set (pNEKOConnection, pData, nSize);
            status = pPacket->Decompress();
            if (ALT_IS_ERR (status)) {
              ALT_LOG (status);
              continue;
            }
            oPacketQue.Push (pPacket);
          }
          else if (status == ALT_S_NET_EOF) {
            oDisconnectList.push_back (pNEKOConnection);
          }
          else if (ALT_IS_ERR (status)) {
            ALT_LOG (status);
            oDisconnectList.push_back (pNEKOConnection);
          }
        }
        else if (FD_ISSET (pNEKOConnection->GetSocket(), & oEfds)) {
          oDisconnectList.push_back (pNEKOConnection);
        }
      }
    }
  }
  pReceiver->EndLock();

  // remove diconnected connection
  for (altUInt i = 0; i < oDisconnectList.size(); i++) {
    pReceiver->Del (oDisconnectList[i]);
    aiNEKOConnectionManager.Del (oDisconnectList[i]);
  }

  ALT_RET (ALT_S_SUCCESS);
}

