/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOSender.cpp 1352 2008-04-20 21:21:20Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOSender.cpp
 * \brief   NEKO Network Sender
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOSender.h"
#include "altNEKOPacket.h"
#include "altNEKOConnectionManager.h"
#include "altNet/altNetUtil.h"
#include "altBase/altUtil.h"

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altNEKOSender::altNEKOSender() :
m_oPacketQue (),
m_oSendThread (50)
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOSender::~altNEKOSender()
{
  alt_t status = Stop();
  ALT_LOG (status);
}

///
/// \brief  Send
///
/// \param  pConnection [I ] Connection
/// \param  pData       [I ] Data
/// \param  nSize       [I ] Data Size
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_NOMEM   out of memory
///
LIBALT_API alt_t altNEKOSender::Send(const altCNEKOConnectionPtr & pConnection, const altCharPtr & pData, const altInt nSize) const
{
  alt_t status;
  altNEKOPacket * pPacket = ALT_NEW altNEKOPacket();
  
  if (pPacket == NULL) {
    ALT_RET (ALT_E_NOMEM);
  }

  status = pPacket->Set (pConnection, pData, nSize);
  ALT_ERR_RET (status);

  m_oPacketQue.Push (pPacket);
#ifdef ALT_WIN
  m_oPacketQue.SetSignal();
#endif

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Start Send Thread
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOSender::Start()
{
  m_oSendThread.Start (altNEKOSender::SendThread, this);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Stop Send Thread
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOSender::Stop()
{
  m_oSendThread.Stop();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get packet count
///
/// \return Packet count
///
LIBALT_API altUInt altNEKOSender::GetPacketCount() const
{
  return (m_oPacketQue.Size());
}

///
/// \brief  Send Thread
///
/// \param  pParam  [I ] altNETSender object
///
LIBALT_API alt_t altNEKOSender::SendThread(void * pParam)
{
  altNEKOSender *           pSender = (altNEKOSender *)pParam;
  altQue<altNEKOPacket *> & oQue = pSender->m_oPacketQue;
  alt_t                     status;

#ifdef ALT_WIN
  oQue.WaitSignal(100);
#endif
#ifdef ALT_LINUX
  altSleep (1);
#endif
  if (oQue.Empty()) {
    ALT_RET (ALT_S_SUCCESS);
  }

  altNEKOPacket * pPacket = NULL;
  status = oQue.Pop (pPacket);
  if (status == ALT_S_SUCCESS) {
    altCNEKOConnectionPtr pConnection = NULL;
    altCharPtr            pData;
    altUInt               nSize = 0;

    status = pPacket->Compress();
    if (ALT_IS_ERR (status)) {
      delete pPacket;
      ALT_RET (status);
    }

    status = pPacket->Get (pConnection, pData, nSize);
    if (ALT_IS_ERR (status)) {
      delete pPacket;
      ALT_RET (status);
    }

    if (! pConnection->Connected()) {
      delete pPacket;
      ALT_RET (ALT_S_SUCCESS);
    }

    pConnection->BeginLock();
    status = pConnection->RealSend (pData, nSize);
    pConnection->EndLock();
    if (ALT_IS_ERR (status)) {
      delete pPacket;
      aiNEKOConnectionManager.Del (pConnection);
      ALT_RET (status);
    }

    delete pPacket;
  }

  ALT_RET (ALT_S_SUCCESS);
}
