/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOServer.cpp 1395 2008-05-01 20:50:30Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOServer.cpp
 * \brief   NEKO Network Server
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOServer.h"
#include "altMisc/altClock.h"
#include "altBase/altUtil.h"

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altNEKOServer::altNEKOServer() :
m_oRemoteFuncMap(),
m_oPortMap(),
m_oAcceptorMap(),
m_oClientMap()
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOServer::~altNEKOServer()
{
}

///
/// \brief  Open Port
///
/// \param  nPort               [I ] Port NO
/// \param  oRemoteFunc         [I ] Remote function
/// \param  nRemoteFuncSize     [I ] Remote function count
/// \param  pAcceptCallBack     [I ] Accept Call Back Function
/// \param  pDisconnectCallBack [I ] Disconnected Call Back Function
/// \param  nThreadCount        [I ] Request Thread Count
/// \param  nListenQueSize      [I ] Listen Que Size
/// \param  nAcceptThreadCount  [I ] Accept Thread Count
/// \param  nSSLVersion         [I ] SSL Version Number(ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
/// \return ALT_S_SUCCESS     success
/// \return ALT_E_WSASTARTUP  start up error
/// \return ALT_E_WSAVERSION  WinSock version error
/// \return ALT_E_NOMEM       Out Of Memory error
/// \return ALT_E_INVAL       Invalid Paramter
///
LIBALT_API alt_t altNEKOServer::OpenPort(const altInt nPort, const altRemoteFuncDef oRemoteFunc[], const altUInt nRemoteFuncSize, const altNEKOAcceptCallBack pAcceptCallBack, const altNEKODisconnectCallBackFunc pDisconnectCallBack, const altUInt nThreadCount, const altInt nListenQueSize, const altUInt nAcceptThreadCount, const altByte nSSLVersion)
{
  alt_t status;

  for (altUInt i = 0; i < nRemoteFuncSize; i++) {
    m_oRemoteFuncMap[oRemoteFunc[i].sFuncName] = oRemoteFunc[i].pRemoteFunc;
  }
  m_oRemoteFuncMap["altNEKOServer::ServerConnect"] = altNEKOServer::ServerConnect;

  status = altNetUtil::WSAStartup();
  ALT_ERR_RET (status);

  status = altNetUtil::SSLLibraryInit();
  ALT_ERR_RET (status);

  status = aiNEKOConnectionManager.SetDisconnectedCallBackFunc (pDisconnectCallBack);
  ALT_ERR_RET (status);

  altNEKOAcceptor * pAcceptor = ALT_NEW altNEKOAcceptor(ReceiveCallBack, nThreadCount, nAcceptThreadCount, nSSLVersion);
  if (pAcceptor == NULL) {
    ALT_RET (ALT_E_NOMEM);
  }

  status = pAcceptor->OpenPort (nPort, nListenQueSize, pAcceptCallBack);
  ALT_ERR_RET (status);
  
  m_oPortMap[nPort] = nSSLVersion;
  m_oAcceptorMap[nPort] = pAcceptor;

  altStr  sBuf;
  ALT_LOG_P (ALT_I_INFO, sBuf.Format ("Open Port [%u] SSL.ver[%u]", nPort, nSSLVersion));

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Connect to other server
///
/// \param  sIP                 [I ] IP Address
/// \param  nPort               [I ] Port NO
/// \param  pTable              [I ] Remote function table
/// \param  nSize               [I ] Remote function table size
/// \param  sMyServerName       [I ] My server name
/// \param  pDisconnectCallBack [I ] Disconnect Call Back Function
/// \param  nSSLVersion         [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
/// \return ALT_S_SUCCESS       success
/// \return ALT_E_UNKNOWN_HOST  unknown host error
/// \return ALT_E_CONNECT       connect error
/// \return ALT_E_NOMEM         out of memory error
/// \return ALT_E_ERROR         error
///
LIBALT_API alt_t altNEKOServer::Connect(const altStr & sIP, const altUInt nPort, altNEKORemoteFuncTable * pTable, altUInt nSize, const altStr & sMyServerName, altNEKOClientDisconnectCallBack pDisconnectCallBack, const altByte nSSLVersion)
{
  alt_t   status;
  altStr  sKey;
  altStr  sPort;
  altStr  sBuf;

  sPort.SetUInt (nPort);

  sKey += sIP;
  sKey += ":";
  sKey += sPort;

  ALT_LOG_P (ALT_I_INFO, sBuf.Format ("Connect to %s:%u ....", sIP.GetCStr(), nPort));

  if (m_oClientMap[sKey] != NULL) {
    status = m_oClientMap[sKey]->Connect (sIP, nPort);
    ALT_RET (status);
  }

  altNEKOClient * pClient = ALT_NEW altNEKOClient(NULL, pDisconnectCallBack, nSSLVersion);
  if (pClient == NULL) {
    ALT_RET (ALT_E_NOMEM);
  }

  status = pClient->ImportRemoteFuncTable (pTable, nSize);
  if (ALT_IS_ERR (status)) {
    delete pClient;
    ALT_RET (status);
  }
  status = pClient->Connect (sIP, nPort);
  if (ALT_IS_ERR (status)) {
    delete pClient;
    ALT_RET (status);
  }

  BeginLock();
  m_oClientMap[sKey] = pClient;
  EndLock();

  altDataStream oDataStream;
  status = oDataStream.Write (sMyServerName);
  ALT_ERR_RET (status);

  status = pClient->Send ("altNEKOServer::ServerConnect", oDataStream);
  ALT_ERR_RET (status);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get connection
///
/// \param  sIP   [I ] IP
/// \param  nPort [I ] Port
///
/// \return Connection
///
LIBALT_API altNEKOClient * altNEKOServer::GetConnection(const altStr & sIP, const altUInt nPort)
{
  altNEKOClient * pClient = NULL;
  altStr          sKey;

  BeginLock();
  pClient = m_oClientMap[sKey.Format ("%s:%u", sIP.GetCStr(), nPort)];
  EndLock();

  return (pClient);
}

///
/// \brief  Disconnect from other server
///
/// \param  sIP   [I ] IP Address
/// \param  nPort [I ] Port NO
///
/// \return ALT_S_SUCCESS       success
/// \return ALT_E_ERROR         error
///
LIBALT_API alt_t altNEKOServer::Disconnect(const altStr & sIP, const altUInt nPort)
{
  alt_t   status;
  altStr  sKey;
  altStr  sPort;

  sPort.SetUInt (nPort);

  sKey += sIP;
  sKey += ":";
  sKey += sPort;

  BeginLock();
  altNEKOClient * pClient = m_oClientMap[sKey];
  if (pClient == NULL) {
    EndLock();
    ALT_RET (ALT_E_ERROR);
  }
  status = pClient->Disconnect ();
  ALT_LOG (status);
  m_oClientMap.erase (sKey);
  delete pClient;
  EndLock();

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get server connection
///
/// \param  sServerName   [I ] Server Name
///
/// \return Get server connection
///
LIBALT_API const altCNEKOConnectionPtr & altNEKOServer::GetServerConnection(const altStr & sServerName)
{
  return (aiNEKOConnectionManager.GetServerConnection (sServerName));
}

///
/// \brief  Close Port
///
/// \param  nPort   [I ] Port NO
///
/// \return ALT_S_SUCCESS   success
///
LIBALT_API alt_t altNEKOServer::ClosePort(const altInt nPort)
{
  altNEKOAcceptor * pAcceptor = m_oAcceptorMap[nPort];
  if (pAcceptor == NULL) {
    EndLock();
    ALT_RET (ALT_S_SUCCESS);
  }
  altNEKOAcceptorMap::iterator i = m_oAcceptorMap.find (nPort);
  m_oAcceptorMap.erase (i);
  delete pAcceptor;
  m_oPortMap.erase (nPort);
  EndLock();
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Close All Port
///
/// \return ALT_S_SUCCESS   success
///
LIBALT_API alt_t altNEKOServer::ClosePortAll()
{
  BeginLock();
  for (altNEKOAcceptorMap::iterator i = m_oAcceptorMap.begin(); i != m_oAcceptorMap.end(); i++) {
    i->second->ClosePort();
  }
  m_oAcceptorMap.clear();
  m_oPortMap.clear();
  EndLock();
  
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get connection count
///
/// \return Connection count
///
LIBALT_API altUInt altNEKOServer::GetConnectionCount()
{
  return (aiNEKOConnectionManager.Size());
}

///
/// \brief  Get sender packet count
///
/// \param  nPort [I ] Port number
///
/// \return Sender packet count
///
LIBALT_API altUInt altNEKOServer::GetSenerPacketCount(const altInt nPort)
{
  altNEKOAcceptor * pAcceptor = m_oAcceptorMap[nPort];
  if (pAcceptor == NULL) {
    return (0);
  }
  return (pAcceptor->GetSenderPacketCount());
}

///
/// \brief  Get receiver packet count
///
/// \param  nPort [I ] Port number
///
/// \return Receiver packet count
///
LIBALT_API altUInt altNEKOServer::GetReceiverPacketCount(const altInt nPort)
{
  altNEKOAcceptor * pAcceptor = m_oAcceptorMap[nPort];
  if (pAcceptor == NULL) {
    return (0);
  }
  return (pAcceptor->GetReceiverPacketCount());
}

///
/// \brief  Get Instance
///
/// \return Instance
///
LIBALT_API altNEKOServer & altNEKOServer::GetInstance()
{
  return (altSingleton<altNEKOServer>::GetInstance());
}

///
/// \brief  Receive call back function
///
/// \param  oConnection [I ] Connection
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_NOMEM   out of memory
///
LIBALT_API alt_t altNEKOServer::ReceiveCallBack(const altCNEKOConnectionPtr & pConnection, const altCharPtr & pData, const altUInt nSize)
{
  alt_t         status;
  altDataStream oDataStream;

  status = oDataStream.SetData (pData, nSize);
  ALT_ERR_RET (status);

  altStr  sFuncName;

  status = oDataStream.Read (sFuncName);
  ALT_ERR_RET (status);

  DWORD   dwSessionID;
  status = oDataStream.Read (dwSessionID);
  ALT_ERR_RET (status);

  altRemoteFunction pRemoteFunc = aiNEKOServer.m_oRemoteFuncMap[sFuncName];
  if (pRemoteFunc != NULL) {
    altDataStream oReturnDataStream;
    
    status = oReturnDataStream.Write (sFuncName);
    ALT_ERR_RET (status);

    status = oReturnDataStream.Write (dwSessionID);
    ALT_ERR_RET (status);

    altStr    sBuf;
    altClock  oClock;
    pRemoteFunc (pConnection, oDataStream, oReturnDataStream);
    ALT_LOG_P (ALT_I_INFO, sBuf.Format ("[%s](%4.1f) SessionID=%lu", sFuncName.GetCStr(), oClock.Elapsed(), dwSessionID));

    altCharPtr  pReturnData;
    status = oReturnDataStream.GetData (pReturnData);
    ALT_ERR_RET (status);

    status = pConnection->Send (pReturnData, oReturnDataStream.GetDataSize());
    ALT_ERR_RET (status);
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Default disconnect call back function
///
/// \param  oClient [I ] Server client object
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOServer::ServerDisconnectCallBack(altNEKOClient & oClient)
{
  altThread oReconnectThread;
  oReconnectThread.Start (altNEKOServer::ReconnectThread, & oClient);
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Reconnect Thread
///
/// \param  pParam  [I ] this Client Object
///
LIBALT_API ALT_THREAD altNEKOServer::ReconnectThread(void * pParam)
{
  altNEKOClient * pClient = (altNEKOClient *)(pParam);
  alt_t status = ALT_E_ERROR;
  while (status != ALT_S_SUCCESS) {
    status = pClient->Connect (pClient->GetIP(), pClient->GetPort());
    altSleep (1000);
  }
  ALT_THREAD_END (0);
}

///
/// \brief  Server connect
///
/// \param  pConnection       [I ] Connection
/// \param  oDataStream       [I ] Receive data
/// \param  oReturnDataStream [I ] Send data
///
/// \return ALT_S_SUCCESS success
/// \return ALT_E_ERROR   error
///
LIBALT_API alt_t altNEKOServer::ServerConnect(const altCNEKOConnectionPtr & pConnection, altDataStream & oDataStream, altDataStream & oReturnDataStream)
{
  alt_t   status;
  altStr  sServerName;

  try {
    status = oDataStream.Read (sServerName);
    if (ALT_IS_ERR (status)) {
      ALT_LOG (status);
      throw status;
    }

    status = aiNEKOConnectionManager.SetToServerConnection (sServerName, pConnection);
    if (ALT_IS_ERR (status)) {
      ALT_LOG (status);
      throw status;
    }
  }
  catch (alt_t e) {
    ALT_LOG (e);
    oReturnDataStream.Write (false);
  }

  altStr  sBuf;
  ALT_LOG_P (ALT_I_INFO, sBuf.Format ("Accept from Server [%s](%s:%u)", sServerName.GetCStr(), pConnection->GetIP().GetCStr(), pConnection->GetPort()));

  ALT_RET (ALT_S_SUCCESS);
}
