#ifndef ALTNEKOSERVER_H
#define ALTNEKOSERVER_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOServer.h 1395 2008-05-01 20:50:30Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOServer.h
 * \brief   NEKO Network Server
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altNeko/altNEKOClient.h>
#include <altNeko/altNEKOAcceptor.h>
#include <altNeko/altNEKOConnection.h>
#include <altNet/altNETConnectionManager.h>
#include <altBase/altDataStream.h>

/*----------------------------------------------------------------*
 * Define
 *----------------------------------------------------------------*/
#define aiNEKOServer altNEKOServer::GetInstance()

///
/// \brief  Entry remote function to table
///
/// \param  function  [I ] Function name
///
#define ALT_REMOTE_FUNC_ENT(function) {#function, function}

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
///
/// \brief  Remote function
///
/// \param  pConnection       [I ] Client connection
/// \param  oDataStream       [I ] Receive data stream
/// \param  oReturnDataStream [I ] Send data stream
///
typedef alt_t (* altRemoteFunction)(const altCNEKOConnectionPtr & pConnection, altDataStream & oDataStream, altDataStream & oReturnDataStream);

typedef std::map<altStr, altRemoteFunction> altRemoteFunctionMap;

typedef struct {
  altStr            sFuncName;      ///< Remote function name
  altRemoteFunction pRemoteFunc;    ///< Remote function
} altRemoteFuncDef;

typedef std::map<altInt, altByte>           altPortMap;
typedef std::map<altInt, altNEKOAcceptor *> altNEKOAcceptorMap;
typedef std::map<altStr, altNEKOClient *>   altNEKOClientMap;

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  NEKO Server Class
///
class altNEKOServer : public altSingleton<altNEKOServer>, protected altSyncThread
{
friend class altSingleton<altNEKOServer>;
protected:
  ///
  /// \brief  Constructor
  ///
  LIBALT_API altNEKOServer();

  ///
  /// \brief  Destructor
  ///
  LIBALT_API virtual ~altNEKOServer();

public:
  ///
  /// \brief  Open Port
  ///
  /// \param  nPort               [I ] Port NO
  /// \param  oRemoteFunc         [I ] Remote function
  /// \param  nRemoteFuncSize     [I ] Remote function count
  /// \param  pReceiveCallBack    [I ] Receive Call Back Function
  /// \param  pAcceptCallBack     [I ] Accept Call Back Function
  /// \param  pDisconnectCallBack [I ] Disconnected Call Back Function
  /// \param  nThreadCount        [I ] Request Thread Count
  /// \param  nListenQueSize      [I ] Listen Que Size
  /// \param  nAcceptThreadCount  [I ] Accept Thread Count
  /// \param  nSSLVersion         [I ] SSL Version Number(ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
  ///
  /// \return ALT_S_SUCCESS     success
  /// \return ALT_E_WSASTARTUP  start up error
  /// \return ALT_E_WSAVERSION  WinSock version error
  /// \return ALT_E_NOMEM       Out Of Memory error
  /// \return ALT_E_INVAL       Invalid Paramter
  ///
  LIBALT_API alt_t OpenPort(const altInt nPort, const altRemoteFuncDef oRemoteFunc[], const altUInt nRemoteFuncSize, const altNEKOAcceptCallBack pAcceptCallBack = NULL, const altNEKODisconnectCallBackFunc pDisconnectCallBack = NULL, const altUInt nThreadCount = 10, const altInt nListenQueSize = 5, const altUInt nAcceptThreadCount = 1, const altByte nSSLVersion = ALT_SSL_V3);

  ///
  /// \brief  Connect to other server
  ///
  /// \param  sIP                 [I ] IP Address
  /// \param  nPort               [I ] Port NO
  /// \param  pTable              [I ] Remote function table
  /// \param  nSize               [I ] Remote function table size
  /// \param  sMyServerName       [I ] My server name
  /// \param  pDisconnectCallBack [I ] Disconnect Call Back Function
  /// \param  nSSLVersion         [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
  ///
  /// \return ALT_S_SUCCESS       success
  /// \return ALT_E_UNKNOWN_HOST  unknown host error
  /// \return ALT_E_CONNECT       connect error
  /// \return ALT_E_NOMEM         out of memory error
  /// \return ALT_E_ERROR         error
  ///
  LIBALT_API alt_t Connect(const altStr & sIP, const altUInt nPort, altNEKORemoteFuncTable * pTable, altUInt nSize, const altStr & sMyServerName, altNEKOClientDisconnectCallBack pDisconnectCallBack = altNEKOServer::ServerDisconnectCallBack, const altByte nSSLVersion = ALT_SSL_V3);

  ///
  /// \brief  Get connection
  ///
  /// \param  sIP   [I ] IP
  /// \param  nPort [I ] Port
  ///
  /// \return Connection
  ///
  LIBALT_API altNEKOClient * GetConnection(const altStr & sIP, const altUInt nPort);

  ///
  /// \brief  Disconnect from other server
  ///
  /// \param  sIP   [I ] IP Address
  /// \param  nPort [I ] Port NO
  ///
  /// \return ALT_S_SUCCESS       success
  /// \return ALT_E_ERROR         error
  ///
  LIBALT_API alt_t Disconnect(const altStr & sIP, const altUInt nPort);

  ///
  /// \brief  Get server connection
  ///
  /// \param  sServerName   [I ] Server Name
  ///
  /// \return Get server connection
  ///
  LIBALT_API const altCNEKOConnectionPtr & GetServerConnection(const altStr & sServerName);

  ///
  /// \brief  Close Port
  ///
  /// \param  nPort   [I ] Port NO
  ///
  /// \return ALT_S_SUCCESS   success
  ///
  LIBALT_API alt_t ClosePort(const altInt nPort);

  ///
  /// \brief  Close All Port
  ///
  /// \return ALT_S_SUCCESS   success
  ///
  LIBALT_API alt_t ClosePortAll();

  ///
  /// \brief  Get connection count
  ///
  /// \return Connection count
  ///
  LIBALT_API altUInt GetConnectionCount();

  ///
  /// \brief  Get sender packet count
  ///
  /// \param  nPort [I ] Port number
  ///
  /// \return Sender packet count
  ///
  LIBALT_API altUInt GetSenerPacketCount(const altInt nPort);

  ///
  /// \brief  Get receiver packet count
  ///
  /// \param  nPort [I ] Port number
  ///
  /// \return Receiver packet count
  ///
  LIBALT_API altUInt GetReceiverPacketCount(const altInt nPort);

  ///
  /// \brief  Get Instance
  ///
  /// \return Instance
  ///
  LIBALT_API static altNEKOServer & GetInstance();

protected:
  ///
  /// \brief  Receive call back function
  ///
  /// \param  oConnection [I ] Connection
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_NOMEM   out of memory
  ///
  LIBALT_API static alt_t ReceiveCallBack(const altCNEKOConnectionPtr & pConnection, const altCharPtr & pData, const altUInt nSize);

  ///
  /// \brief  Default disconnect call back function
  ///
  /// \param  oClient [I ] Server client object
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API static alt_t ServerDisconnectCallBack(altNEKOClient & oClient);

  ///
  /// \brief  Reconnect Thread
  ///
  /// \param  pParam  [I ] this Client Object
  ///
  LIBALT_API static ALT_THREAD ReconnectThread(void * pParam);

  ///
  /// \brief  Server connect
  ///
  /// \param  pConnection       [I ] Connection
  /// \param  oDataStream       [I ] Receive data
  /// \param  oReturnDataStream [I ] Send data
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  LIBALT_API static alt_t ServerConnect(const altCNEKOConnectionPtr & pConnection, altDataStream & oDataStream, altDataStream & oReturnDataStream);

protected:
  altRemoteFunctionMap  m_oRemoteFuncMap; ///< Remote function map
  altPortMap            m_oPortMap;       ///< Port map
  altNEKOAcceptorMap    m_oAcceptorMap;   ///< Acceptor map
  altNEKOClientMap      m_oClientMap;     ///< Client map
};

#endif
