/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOSocket.cpp 1041 2008-03-29 23:33:10Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOSocket.cpp
 * \brief   NEKO Network Socket Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altNEKOSocket.h"
#include "altNet/altNetUtil.h"
#include "altBase/altMem.h"

/*----------------------------------------------------------------*
 * Define
 *----------------------------------------------------------------*/

/*----------------------------------------------------------------*
 * Class variables
 *----------------------------------------------------------------*/
const altUShort altNEKOSocket::HEADER = 0xACBD;
const altUInt   altNEKOSocket::HEADER_SIZE = sizeof (HEADER) + sizeof (altUInt) + sizeof (altULongLong);

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
/// \param  nSocket   [I ] Socket Descriptor
/// \param  nSSLVersion [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
LIBALT_API altNEKOSocket::altNEKOSocket(const SOCKET nSocket, const altByte nSSLVersion) :
m_oTCPSocket (nSocket),
m_oSSLSocket (nSocket),
m_nSSLVersion (nSSLVersion),
m_pNotifyTotalSizeFunc (NULL),
m_pNotifyRecvSizeFunc (NULL)
{
}

///
/// \brief  Destructor
///
LIBALT_API altNEKOSocket::~altNEKOSocket()
{
  alt_t status;
  status = Close();
  ALT_LOG (status);
}

///
/// \brief  Connect
///
/// \param  oInetAddr   [I ] Internet Socket Address
/// \param  nSSLVersion [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
///
/// \return ALT_S_SUCCESS   success
/// \retrun ALT_E_CONNECT   connect error
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOSocket::Connect(const altInetAddress & oInetAddr)
{
  alt_t status;
  
  if (m_nSSLVersion == ALT_NO_SSL) {
    if (m_oTCPSocket.GetSocket() == 0) {
      status = m_oTCPSocket.Init (AF_INET, SOCK_STREAM, 0);
      ALT_ERR_RET (status);
    }
    status = m_oTCPSocket.Connect (oInetAddr);
    ALT_ERR_RET (status);
  }
  else {
    if (m_oSSLSocket.GetSocket() == 0) {
      status = m_oSSLSocket.Init (AF_INET, SOCK_STREAM, 0);
      ALT_ERR_RET (status);
    }
    status = m_oSSLSocket.Connect (oInetAddr, m_nSSLVersion);
    ALT_ERR_RET (status);
  }
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Close Socket
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOSocket::Close()
{
  alt_t status;
  
  if (m_nSSLVersion == ALT_NO_SSL) {
    status = m_oTCPSocket.Close();
    ALT_ERR_RET (status);
  }
  else {
    status = m_oSSLSocket.Close();
    ALT_ERR_RET (status);
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read Data
///
/// \param  pData     [ O] Receive data
/// \param  nSize     [IO] Receive data size
/// \param  nPacketID [ O] Packet ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_NET_EOF   connection closed
/// \return ALT_S_TIMEOUT   timeout
/// \return ALT_E_NOMEM     out of memory error
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOSocket::Recv(altCharPtr & pData, altUInt & nSize, altULongLong & nPacketID) const
{
  alt_t     status;

  if (m_nSSLVersion == ALT_NO_SSL) {
    status = RecvAll (m_oTCPSocket.GetSocket(), pData, nSize, nPacketID);
    ALT_ERR_RET (status);
  }
  else {
    status = RecvAllSSL (m_oSSLSocket.GetSSL(), pData, nSize, nPacketID);
    ALT_ERR_RET (status);
  }

  ALT_RET (status);
}

///
/// \brief  Send Data
///
/// \param  pData     [I ] Send data
/// \param  nSize     [I ] Send data size
/// \param  nPacketID [I ] Packet ID
///
/// \return ALT_S_SUCCESS   successs
/// \return ALT_E_INVAL     invalid parameter
/// \return ALT_E_SEND      send error
///
LIBALT_API alt_t altNEKOSocket::Send(const altChar * pData, const altInt nSize, const altULongLong nPacketID) const
{
  alt_t   status;
  
  ALT_P_ASSERT (pData != NULL);

  if (m_nSSLVersion == ALT_NO_SSL) {
    status = SendAll (m_oTCPSocket.GetSocket(), pData, nSize, nPacketID);
    ALT_ERR_RET (status);
  }
  else {
    status = SendAllSSL (m_oSSLSocket.GetSSL(), pData, nSize, nPacketID);
    ALT_ERR_RET (status);
  }

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Get connecting or not
///
/// \return true  connecting
/// \return false not connecting
///
LIBALT_API altBool altNEKOSocket::Connected() const
{
  if (m_nSSLVersion == ALT_NO_SSL) {
    return (m_oTCPSocket.Connected());
  }
  return (m_oSSLSocket.Connected());
}

///
/// \brief  Get Socket
///
/// \return Socket
///
LIBALT_API const SOCKET altNEKOSocket::GetSocket() const
{
  if (m_nSSLVersion == ALT_NO_SSL) {
    return (m_oTCPSocket.GetSocket());
  }
  return (m_oSSLSocket.GetSocket());
}

///
/// \breif  Set SSL Object
///
/// \param  pSSL        [I ] Pointer of SSL Object
/// \param  pSSLContext [I ] Pointer of SSL Context Object
///
LIBALT_API void altNEKOSocket::SetSSL(SSL * pSSL, SSL_CTX * pSSLContext)
{
  m_oSSLSocket.SetSSL (pSSL, pSSLContext);
}

///
/// \brief  Set Receive Total Size Notify Function
///
/// \param  pNotifyTotalSizeCallBack  [I ] Total Size Notify Function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOSocket::SetNotifyTotalSizeFunc(altNotifyTotalSizeFunc pNotifyTotalSizeCallBack)
{
  m_pNotifyTotalSizeFunc = pNotifyTotalSizeCallBack;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Set Receive Receive Size Notify Function
///
/// \param  pNotifyRecvSizeCallBack  [I ] Receive Size Notify Function
///
/// \return ALT_S_SUCCESS success
///
LIBALT_API alt_t altNEKOSocket::SetNotifyRecvSizeFunc(altNotifyRecvSizeFunc pNotifyRecvSizeCallBack)
{
  m_pNotifyRecvSizeFunc = pNotifyRecvSizeCallBack;
  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read All Data
///
/// \param  nSocket   [I ] Socket
/// \param  pData     [ O] Receive data
/// \param  nSize     [ O] Receive data size
/// \param  nPacketID [ O] Packet ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_NET_EOF   connection closed
/// \return ALT_S_TIMEOUT   timeout
/// \return ALT_E_NOMEM     out of memory error
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOSocket::RecvAll(SOCKET nSocket, altCharPtr & pData, altUInt & nSize, altULongLong & nPacketID) const
{
  alt_t   status;
  altInt  nRecvSize = 0;
  altChar pBuf[ALT_BUFSIZ];

  status = altNetUtil::Recv (nSocket, pBuf, HEADER_SIZE, nRecvSize);
  ALT_ERR_RET (status);

  if (status == ALT_S_NET_EOF) {
    ALT_RET (status);
  }

  ALT_P_ASSERT (nRecvSize == (altInt)HEADER_SIZE);

  altUShort nHeader = 0x0000;
  altUInt   nPacketSize = 0;

  memcpy (& nHeader, pBuf, sizeof (nHeader));
  memcpy (& nPacketSize, & pBuf[sizeof (nHeader)], sizeof (nPacketSize));
  memcpy (& nPacketID, & pBuf[sizeof (nHeader) + sizeof (nPacketSize)], sizeof (nPacketID));

  ALT_P_ASSERT (nHeader == HEADER);
  
  if (m_pNotifyTotalSizeFunc != NULL) {
    m_pNotifyTotalSizeFunc (nPacketSize);
  }

  altChar * pDataBuf = NULL;
  altUInt   nTotalRecvSize = 0;
  while (nTotalRecvSize != nPacketSize) {
    status = altNetUtil::Recv (nSocket, pBuf, sizeof (pBuf), nRecvSize);
    ALT_ERR_RET (status);

    if (m_pNotifyRecvSizeFunc != NULL) {
      m_pNotifyRecvSizeFunc (nRecvSize);
    }

    status = ALT_EXALLOC (pDataBuf, nTotalRecvSize, nTotalRecvSize + nRecvSize);
    ALT_ERR_RET (status);
    memcpy (& pDataBuf[nTotalRecvSize], pBuf, nRecvSize);

    nTotalRecvSize += nRecvSize;
  }

  pData = pDataBuf;
  nSize = nPacketSize;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Read All Data
///
/// \param  pSSL      [IO] Pointer of SSL
/// \param  pData     [ O] Receive data
/// \param  nSize     [IO] Receive data size
/// \param  nPacketID [ O] Packet ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_S_NET_EOF   connection closed
/// \return ALT_S_TIMEOUT   timeout
/// \return ALT_E_NOMEM     out of memory error
/// \return ALT_E_ERROR     error
///
LIBALT_API alt_t altNEKOSocket::RecvAllSSL(SSL * pSSL, altCharPtr & pData, altUInt & nSize, altULongLong & nPacketID) const
{
  alt_t   status;
  altInt  nRecvSize = 0;
  altChar pBuf[ALT_BUFSIZ];

  status = altNetUtil::RecvSSL (pSSL, pBuf, HEADER_SIZE, nRecvSize);
  ALT_ERR_RET (status);

  if (status == ALT_S_NET_EOF) {
    ALT_RET (status);
  }

  ALT_P_ASSERT (nRecvSize == (altInt)HEADER_SIZE);

  altUShort nHeader = 0x0000;
  altUInt   nPacketSize = 0;

  memcpy (& nHeader, pBuf, sizeof (nHeader));
  memcpy (& nPacketSize, & pBuf[sizeof (nHeader)], sizeof (nPacketSize));
  memcpy (& nPacketID, & pBuf[sizeof (nHeader) + sizeof (nPacketSize)], sizeof (nPacketID));

  ALT_P_ASSERT (nHeader == HEADER);

  if (m_pNotifyTotalSizeFunc != NULL) {
    m_pNotifyTotalSizeFunc (nPacketSize);
  }

  altChar * pDataBuf = NULL;
  altUInt   nTotalRecvSize = 0;
  while (nTotalRecvSize != nPacketSize) {
    status = altNetUtil::RecvSSL (pSSL, pBuf, sizeof (pBuf), nRecvSize);
    ALT_ERR_RET (status);
    
    if (m_pNotifyRecvSizeFunc != NULL) {
      m_pNotifyRecvSizeFunc (nRecvSize);
    }

    status = ALT_EXALLOC (pDataBuf, nTotalRecvSize, nTotalRecvSize + nRecvSize);
    ALT_ERR_RET (status);
    memcpy (& pDataBuf[nTotalRecvSize], pBuf, nRecvSize);

    nTotalRecvSize += nRecvSize;
  }

  pData = pDataBuf;
  nSize = nPacketSize;

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send All
///
/// \param  nSocket   [I ] Socket
/// \param  pData     [I ] Data
/// \param  nSize     [I ] Data size
/// \param  nPacketID [I ] Packet ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_INVAL     invalid parameter
/// \return ALT_E_SEND      send error
///
LIBALT_API alt_t altNEKOSocket::SendAll(const SOCKET nSocket, const altChar * pData, const altInt nSize, const altULongLong nPacketID) const
{
  alt_t     status;
  altChar * pBuf = NULL;

  ALT_P_ASSERT (pData != NULL);

  status = ALT_MALLOC (pBuf, nSize + HEADER_SIZE);
  ALT_ERR_RET (status);

  memcpy (pBuf, & HEADER, sizeof (HEADER));
  memcpy (& pBuf[sizeof (HEADER)], & nSize, sizeof (nSize));
  memcpy (& pBuf[sizeof (HEADER) + sizeof (nSize)], & nPacketID, sizeof (nPacketID));
  memcpy (& pBuf[HEADER_SIZE], pData, nSize);

  altInt  nTotalSendSize = 0;
  altInt  nSendSize = nSize + HEADER_SIZE;
  altInt  nSendByte = 0;

  while (nTotalSendSize != nSendSize) {
    status = altNetUtil::Send (nSocket, & pBuf[nTotalSendSize], nSendSize - nTotalSendSize, nSendByte);
    if (ALT_IS_ERR (status)) {
      ALT_FREE (pBuf);
      ALT_RET (status);
    }
    nTotalSendSize += nSendByte;
  }

  ALT_FREE (pBuf);

  ALT_RET (ALT_S_SUCCESS);
}

///
/// \brief  Send All
///
/// \param  pSSL      [I ] Pointer of SSL
/// \param  pData     [I ] Data
/// \param  nSize     [I ] Data size
/// \param  nPacketID [I ] Packet ID
///
/// \return ALT_S_SUCCESS   success
/// \return ALT_E_INVAL     invalid parameter
/// \return ALT_E_SEND      send error
///
LIBALT_API alt_t altNEKOSocket::SendAllSSL(SSL * pSSL, const altChar * pData, const altInt nSize, const altULongLong nPacketID) const
{
  alt_t     status;
  altChar * pBuf = NULL;

  ALT_P_ASSERT (pData != NULL);

  status = ALT_MALLOC (pBuf, nSize + HEADER_SIZE);
  ALT_ERR_RET (status);

  memcpy (pBuf, & HEADER, sizeof (HEADER));
  memcpy (& pBuf[sizeof (HEADER)], & nSize, sizeof (nSize));
  memcpy (& pBuf[sizeof (HEADER) + sizeof (nSize)], & nPacketID, sizeof (nPacketID));
  memcpy (& pBuf[HEADER_SIZE], pData, nSize);

  altInt  nTotalSendSize = 0;
  altInt  nSendSize = nSize + HEADER_SIZE;
  altInt  nSendByte = 0;

  while (nTotalSendSize != nSendSize) {
    status = altNetUtil::SendSSL (pSSL, & pBuf[nTotalSendSize], nSendSize - nTotalSendSize, nSendByte);
    if (ALT_IS_ERR (status)) {
      ALT_FREE (pBuf);
      ALT_RET (status);
    }
    nTotalSendSize += nSendByte;
  }

  ALT_FREE (pBuf);

  ALT_RET (ALT_S_SUCCESS);
}

