#ifndef ALTNEKOSOCKET_H
#define ALTNEKOSOCKET_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altNEKOSocket.h 1249 2008-04-08 03:48:39Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
 *---------------------------------------------------------------------------*/
/**
 * \file    altNEKOSocket.h
 * \brief   NEKO Socket
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altNet/altSocket.h>
#include <altNet/altSSLSocket.h>

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
///
/// \brief  Notify total size call back function
///
/// \param  nTotalSize  [I ] Total size
///
typedef alt_t (* altNotifyTotalSizeFunc)(altUInt nTotalSize);

///
/// \brief  Notify receive size call back function
///
/// \param  nRecvSize  [I ] receive size
///
typedef alt_t (* altNotifyRecvSizeFunc)(altInt nRecvSize);

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  SSL Socket Class
///
class altNEKOSocket : public altBase
{
public:
  ///
  /// \brief  Constructor
  ///
  /// \param  nSocket   [I ] Socket Descriptor
  /// \param  nSSLVersion [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
  ///
  LIBALT_API altNEKOSocket(const SOCKET nSocket, const altByte nSSLVersion);

  ///
  /// \brief  Destructor
  ///
  LIBALT_API virtual ~altNEKOSocket();

  ///
  /// \brief  Connect
  ///
  /// \param  oInetAddr   [I ] Internet Socket Address
  /// \param  nSSLVersion [I ] SSL Version (ALT_NO_SSL or ALT_SSL_V23 or ALT_SSL_V2 or ALT_SSL_V3 or ALT_TSL_V1)
  ///
  /// \return ALT_S_SUCCESS   success
  /// \retrun ALT_E_CONNECT   connect error
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Connect(const altInetAddress & oInetAddr);

  ///
  /// \brief  Close Socket
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Close();

  ///
  /// \brief  Read Data
  ///
  /// \param  pData     [ O] Receive data
  /// \param  nSize     [IO] Receive data size
  /// \param  nPacketID [ O] Packet ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_NET_EOF   connection closed
  /// \return ALT_S_TIMEOUT   timeout
  /// \return ALT_E_NOMEM     out of memory error
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t Recv(altCharPtr & pData, altUInt & nSize, altULongLong & nPacketID) const;

  ///
  /// \brief  Send Data
  ///
  /// \param  pData     [I ] Send data
  /// \param  nSize     [I ] Send data size
  /// \param  nPacketID [I ] Packet ID
  ///
  /// \return ALT_S_SUCCESS   successs
  /// \return ALT_E_INVAL     invalid parameter
  /// \return ALT_E_SEND      send error
  ///
  LIBALT_API alt_t Send(const altChar * pData, const altInt nSize, const altULongLong nPacketID) const;

  ///
  /// \brief  Get connecting or not
  ///
  /// \return true  connecting
  /// \return false not connecting
  ///
  LIBALT_API altBool Connected() const;

  ///
  /// \brief  Get Socket
  ///
  /// \return Socket
  ///
  LIBALT_API const SOCKET GetSocket() const;

  ///
  /// \breif  Set SSL Object
  ///
  /// \param  pSSL        [I ] Pointer of SSL Object
  /// \param  pSSLContext [I ] Pointer of SSL Context Object
  ///
  LIBALT_API void SetSSL(SSL * pSSL, SSL_CTX * pSSLContext);

  ///
  /// \brief  Set Receive Total Size Notify Function
  ///
  /// \param  pNotifyTotalSizeCallBack  [I ] Total Size Notify Function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyTotalSizeFunc(altNotifyTotalSizeFunc pNotifyTotalSizeCallBack);

  ///
  /// \brief  Set Receive Receive Size Notify Function
  ///
  /// \param  pNotifyRecvSizeCallBack  [I ] Receive Size Notify Function
  ///
  /// \return ALT_S_SUCCESS success
  ///
  LIBALT_API alt_t SetNotifyRecvSizeFunc(altNotifyRecvSizeFunc pNotifyRecvSizeCallBack);

protected:
  ///
  /// \brief  Read All Data
  ///
  /// \param  nSocket   [I ] Socket
  /// \param  pData     [ O] Receive data
  /// \param  nSize     [ O] Receive data size
  /// \param  nPacketID [ O] Packet ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_NET_EOF   connection closed
  /// \return ALT_S_TIMEOUT   timeout
  /// \return ALT_E_NOMEM     out of memory error
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t RecvAll(SOCKET nSocket, altCharPtr & pData, altUInt & nSize, altULongLong & nPacketID) const;

  ///
  /// \brief  Read All Data
  ///
  /// \param  pSSL      [IO] Pointer of SSL
  /// \param  pData     [ O] Receive data
  /// \param  nSize     [IO] Receive data size
  /// \param  nPacketID [ O] Packet ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_S_NET_EOF   connection closed
  /// \return ALT_S_TIMEOUT   timeout
  /// \return ALT_E_NOMEM     out of memory error
  /// \return ALT_E_ERROR     error
  ///
  LIBALT_API alt_t RecvAllSSL(SSL * pSSL, altCharPtr & pData, altUInt & nSize, altULongLong & nPacketID) const;

  ///
  /// \brief  Send All
  ///
  /// \param  nSocket   [I ] Socket
  /// \param  pData     [I ] Data
  /// \param  nSize     [I ] Data size
  /// \param  nPacketID [I ] Packet ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_INVAL     invalid parameter
  /// \return ALT_E_SEND      send error
  ///
  LIBALT_API alt_t SendAll(const SOCKET nSocket, const altChar * pData, const altInt nSize, const altULongLong nPacketID) const;

  ///
  /// \brief  Send All
  ///
  /// \param  pSSL      [I ] Pointer of SSL
  /// \param  pData     [I ] Data
  /// \param  nSize     [I ] Data size
  /// \param  nPacketID [I ] Packet ID
  ///
  /// \return ALT_S_SUCCESS   success
  /// \return ALT_E_INVAL     invalid parameter
  /// \return ALT_E_SEND      send error
  ///
  LIBALT_API alt_t SendAllSSL(SSL * pSSL, const altChar * pData, const altInt nSize, const altULongLong nPacketID) const;

  altSocket               m_oTCPSocket;           ///< TCP socket
  altSSLSocket            m_oSSLSocket;           ///< SSL socket
  altByte                 m_nSSLVersion;          ///< SSL version
  altNotifyTotalSizeFunc  m_pNotifyTotalSizeFunc; ///< Notify receive total size function
  altNotifyRecvSizeFunc   m_pNotifyRecvSizeFunc;  ///< Notify receive size function
  static const altUShort  HEADER;                 ///< Packet header
  static const altUInt    HEADER_SIZE;            ///< Packet header size
};

#endif
