#!/usr/bin/python
# -*- coding: utf-8 -*-

# お守りスキルのSeed特定ツールのコントロールクラス
# 2013/12/04 written by kei9

import threading
import time
import os.path
import sys
import pickle
import sqlite3

import wx

import view
import model
import convertcoding

SETTING_FILE = u"settings"
SETTING_THRESHOLD1 = u"threshold1"
SETTING_THRESHOLD2 = u"threshold2"
SETTING_SKILLS = u"skills"

def _get_base_dir():
    u""" for pyinstaller 2.1 """
    if getattr(sys, 'frozen', False):
        # we are running in a |PyInstaller| bundle
        basedir = sys._MEIPASS
    else:
        # we are running in a normal Python environment
        #1basedir = os.path.dirname(__file__)
        basedir = os.path.dirname(os.path.abspath(__file__))
    return convertcoding.convert_unicode(basedir)


class AmuletToolController(wx.App):
    u""" アプリケーションの制御クラス """
    def OnInit(self):
        self._read_settings()
        self.frame_view = view.MainFrameView(os.path.join(_get_base_dir(), u"view", view.XRC_MAIN_FRAME))

        self._init_events()

        self.frame_view.Show()

        self.frame_view.DisableNoteBook()

        self._init_database()
        self._init_notebook_seed1()
        self._init_notebook_seed2()
        self._init_notebook_skill2()
        self._init_notebook_setting()
        self._init_notebook_amulet()

        self.frame_view.EnableNoteBook()
        return True

    def _init_events(self):
        u"""イベント登録"""
        frame = self.frame_view.frame

        # menu event
        frame.Bind(wx.EVT_MENU, self.OnClose, id=self.frame_view.ID_MENU_ITEM_EXIT)
        frame.Bind(wx.EVT_MENU, self.OnAboutBox, id=self.frame_view.ID_MENU_ITEM_ABOUT)
        frame.Bind(wx.EVT_CLOSE, self.CloseHandler)

        # button event

    def _init_database(self):
        u""" DBの初期設定 """

        if not os.path.exists(model.DB_FILE_NAME):
            u""" DBが存在しない時は再生成する """
            frame = self.frame_view.frame
            try:
                generator = model.DataBaseGenerator(model.DB_FILE_NAME)
                #generator = model.DataBaseGenerator()

                dlg_view = view.GaugeDialogView(os.path.join(_get_base_dir(), u"view", view.XRC_GAUGE_DIALOG))
                def _loop():
                    while t1.is_alive():
                        dlg_view.gauge.Pulse()
                        time.sleep(0.2)
                    dlg_view.finish_generation()

                t1 = threading.Thread(target=generator.generate_db)
                t2 = threading.Thread(target=_loop)
                t1.start()
                t2.start()

                dlg_view.ShowModal()
                t1.join()
                t2.join()
                dlg_view.Destroy()
            except sqlite3.Error as e:
                self._show_error_dialog(u"データベース生成中にエラーが発生しました")

        # access to db
        try:
            self.db_accessor = model.DataBaseAccessor(model.DB_FILE_NAME)
        except sqlite3.Error as e:
            self._show_error_dialog(u"データベースが壊れています")

        # get dictionaries
        (self._skill_id2name_dict, 
            self._skill_name2id_dict,
            self._amulet_id2name_dict, 
            self._amulet_name2id_dict,
            self._amulet_id2skill1_id_dict,
            self._amulet_id2skill2_id_dict) = self.db_accessor.get_dicts()

    def _init_notebook_seed1(self):
        u""" Seed1特定タブのviewの初期化 """
        self.notebook_seed1_view = self.frame_view.notebook_seed1_view
        self.notebook_seed1_view.bind_button_search(self.OnClickSeed1Search)
        self.notebook_seed1_view.bind_button_clear(self.OnClickSeed1Clear)

        u" 各種お守りの第1スキル選択のセット "
        amu_names, name2skill1s = [], {}
        for amu_id in sorted(self._amulet_id2name_dict.keys()):
            name = self._amulet_id2name_dict[amu_id]
            amu_names.append(name)
            name2skill1s[name] = [self._skill_id2name_dict[skill_id]
                    for skill_id in self._amulet_id2skill1_id_dict[amu_id]]
        self.notebook_seed1_view.set_amuletname_skillnames_dict(amu_names, name2skill1s)

    def _init_notebook_seed2(self):
        u""" Seed2特定タブのviewの初期化 """
        self.notebook_seed2_view = self.frame_view.notebook_seed2_view
        self.notebook_seed2_view.bind_button_search(self.OnClickSeed2Search)
        self.notebook_seed2_view.bind_button_clear(self.OnClickSeed2Clear)
        self.notebook_seed2_view.bind_button_skill2(self.OnClickSkill2SearchFromSeed2)

        u" 各種お守りの第2スキル選択 "
        for amu_key, amu_name in zip([view.KEY_AMULET1, view.KEY_AMULET2, view.KEY_AMULET3], 
                [view.NAME_AMULET1, view.NAME_AMULET2, view.NAME_AMULET3]):
            amu_id = self._amulet_name2id_dict[amu_name]
            skill_ids = self._amulet_id2skill2_id_dict[amu_id]
            skill_names = [view.VAL_NO_SKILL] + [self._skill_id2name_dict[x] for x in skill_ids]
            self.notebook_seed2_view.set_skill_names(amu_key, skill_names)
        self.notebook_seed2_view.set_skill_selected_idx(0)

    def _init_notebook_skill2(self):
        u""" Seed2によるSkill2一覧タブのviewの初期化 """
        self.notebook_skill2_view = self.frame_view.notebook_skill2_view
        self.notebook_skill2_view.bind_button_search(self.OnClickSkillSearch)
        self.notebook_skill2_view.bind_button_clear(self.OnClickSkillClear)

    def _update_notebook_amulet(self, amulet_id=None):
        u""" お守り検索タブのviewの更新
        amulet_id が Noneでない場合、お守りの種類に応じてComboboxを再設定する
        また、選択中のスキルに応じて最大最小を設定する"""
        #min1, max1, min2, max2 = self.db_accessor.get_skill_minmax()
        #self.notebook_amulet_view.set_skill_minmax(min1, max1, min2, max2)

        if amulet_id is not None:
            for i, skill_dict in enumerate([self._amulet_id2skill1_id_dict, self._amulet_id2skill2_id_dict]):
                skill_ids = skill_dict[amulet_id]
                skill_names = [self._skill_id2name_dict[_id] for _id in skill_ids]
                if i == 0:
                    self.notebook_amulet_view.set_items_skill1_combobox(skill_names)
                    skill1_selected = skill_ids[0]
                    self.notebook_amulet_view.set_selection_skill1_combobox(0)
                elif i == 1:
                    self.notebook_amulet_view.set_items_skill2_combobox([view.VAL_NO_SKILL] + skill_names)
                    self.notebook_amulet_view.set_selection_skill2_combobox(0)  # select No Skill
        else:
            amulet_name = self.notebook_amulet_view.get_selected_amulet()
            amulet_id = self._amulet_name2id_dict[amulet_name]

        # settting min max
        skill1_str = self.notebook_amulet_view.get_string_selection_skill1_combobox()
        skill2_str = self.notebook_amulet_view.get_string_selection_skill2_combobox()
        skill1_id = self._skill_name2id_dict[skill1_str] if skill1_str in self._skill_name2id_dict else None
        skill2_id = self._skill_name2id_dict[skill2_str] if skill2_str in self._skill_name2id_dict else None

        if skill2_id is None:
            minmax_dict = self.db_accessor.select_minmax_from_skill_ids(amulet_id, [skill1_id])
            min1, max1 = minmax_dict[skill1_id][0], minmax_dict[skill1_id][1]
            min2, max2 = 0, 0
        else:
            minmax_dict = self.db_accessor.select_minmax_from_skill_ids(amulet_id, [skill1_id, skill2_id])
            min1, max1 = minmax_dict[skill1_id][0], minmax_dict[skill1_id][1]
            min2, max2 = minmax_dict[skill2_id][2], minmax_dict[skill2_id][3]
        self.notebook_amulet_view.set_skill_minmax(min1, max1, min2, max2)

    def _update_notebook_amulet_seed2s(self):
        u"""お守り検索タブのSeed2リストを更新する"""
        set_items = []
        threshold_type = self.notebook_amulet_view.get_selected_threshold_type()
        if threshold_type == view.THRESHOLD_TYPE_ALL:
            include_th1, include_th2, include_ini = True, True, True
        elif threshold_type == view.THRESHOLD_TYPE_THRESHOLD1:
            include_th1, include_th2, include_ini = True, False, False
        elif threshold_type == view.THRESHOLD_TYPE_THRESHOLD2:
            include_th1, include_th2, include_ini = False, True, False
        elif threshold_type == view.THRESHOLD_TYPE_INISHIE:
            include_th1, include_th2, include_ini = False, False, True

        seed2_set = self.threshold1_seed2_set | self.threshold2_seed2_set | self.threshold_inishie_set
        for seed in seed2_set:
            type_set, include = [], False
            if seed in self.threshold1_seed2_set:
                type_set.append(u"判定値1")
                if include_th1:
                    include = True
            if seed in self.threshold2_seed2_set:
                type_set.append(u"判定値2")
                if include_th2:
                    include = True
            if seed in self.threshold_inishie_set: # seed in inishie_set
                type_set.append(u"いにしえ")
                if include_ini:
                    include = True
            if include:
                set_items.append(u"{0:06d}: (".format(seed) + u",".join(type_set) + u")")
        self.notebook_amulet_view.set_listbox_items(set_items)
        if len(set_items) > 0:
            self.notebook_amulet_view.set_selection_listbox(0)
            self.notebook_amulet_view.set_skill_button_enable(True)
        else:
            self.notebook_amulet_view.set_skill_button_enable(False)

    def _init_notebook_amulet(self):
        u""" お守り検索タブの初期設定 """
        self.notebook_amulet_view = self.frame_view.notebook_amulet_view
        self.notebook_amulet_view.bind_radiobox_amulet(self.OnClickAmuletRadioAmulet)
        self.notebook_amulet_view.bind_radiobox_threshold_type(self.OnClickAmuletRadioThresholdType)
        self.notebook_amulet_view.bind_combobox(self.OnClickAmuletCombo)
        self.notebook_amulet_view.bind_button_search(self.OnClickAmuletSearch)
        self.notebook_amulet_view.bind_button_clear(self.OnClickAmuletClear)
        self.notebook_amulet_view.bind_button_skill(self.OnClickSkillSearchFromAmulet)

        amu_id = self._amulet_name2id_dict[view.NAME_AMULET1]
        self.notebook_amulet_view.set_radio_value(True, view.NAME_AMULET1)
        self._update_notebook_amulet(amu_id)

        self.threshold1_seed2_set, self.threshold2_seed2_set, self.threshold_inishie_set = set(), set(), set()

    def _init_notebook_setting(self):
        u""" 設定タブの初期設定 """
        self.notebook_setting_view = self.frame_view.notebook_setting_view
        self.notebook_setting_view.bind_button_ok(self.OnClickSettingOK)
        self.notebook_setting_view.bind_button_clear(self.OnClickSettingClear)
        self.notebook_setting_view.set_skill_strings(self._skill_name2id_dict.keys())
        self._update_notebook_setting()

    def _update_notebook_setting(self):
        u"""設定タブの値更新"""
        self.notebook_setting_view.set_threshold(self._highlight_threshold1, 
                self._highlight_threshold2)
        self.notebook_setting_view.set_checked_strings([self._skill_id2name_dict[x] for x in self._highlight_skills])

    def _update_highlight(self):
        u""" update highlight cells """
        skill_names = [self._skill_id2name_dict[x] for x in self._highlight_skills]
        self.notebook_skill2_view.set_skill2_highlight(
                skill_names, self._highlight_threshold1, self._highlight_threshold2)
        self.notebook_seed1_view.update_highlight(skill_names)

    u""" Seed1 view's event"""
    def OnClickSeed1Search(self, evt):
        u""" search seed1s from selected skill1s """
        tenun_key = self.notebook_seed1_view.get_tenun_radio_key()
        amu_skill_name_list = self.notebook_seed1_view.get_selected_amulets_and_names()
        if tenun_key == view.KEY_TENUN555:
            tenun_key = model.KEY_TENUN555
        elif tenun_key == view.KEY_TENUN888:
            tenun_key = model.KEY_TENUN888
        else:
            raise KeyError(u"key '{0}' not found in TENUN KEYS".format(tenun_key))
        amu_skill_id_list = [(self._amulet_name2id_dict[x], self._skill_name2id_dict[y]) for x, y in amu_skill_name_list]
        seed1s = self.db_accessor.select_seed1s_tenun(amu_skill_id_list, tenun_key)
        if len(seed1s) == 0:
            self.notebook_seed1_view.set_text_result(u"条件に一致するSeed1は見つかりません")
            self.notebook_seed1_view.clear_seed1_grid()
        elif len(seed1s) == 1:
            seed1 = [x for x in seed1s][0]
            table_no, no = self.db_accessor.select_table_nos_from_seed1(seed1s, tenun_key)[seed1]
            self.notebook_seed1_view.set_text_result(
                    u"Seed1: {0}, テーブルNo: {1}, 通し番号: {2}".format(seed1, table_no, no))
            # list of (no, seed1)
            near_num = view.NEAR_SEED1_NUMBERS
            no_seed1_list = self.db_accessor.select_near_seed1s_from_seed1(
                    seed1, near_num, near_num, tenun_key)
            seed1_list = [x[1] for x in no_seed1_list]

            # seed1 -> (no, table_no, result_num, amu_ids, skill_id2)
            result_dict = self.db_accessor.select_all_from_seed1s(seed1_list, tenun_key)
            for key, val in result_dict.items():
                _no, _table_no, _result_num, _amu_ids, _skill_ids = val
                _amu_names = [self._amulet_id2name_dict[x] 
                        if x in self._amulet_id2name_dict else None for x in _amu_ids]
                _skill_names = [self._skill_id2name_dict[x] 
                        if x in self._skill_id2name_dict else None for x in _skill_ids]
                result_dict[key] = (_no, _table_no, _result_num, _amu_names, _skill_names)
            self.notebook_seed1_view.set_seed1_dict(result_dict, seed1)
            skill_names = [self._skill_id2name_dict[x] for x in self._highlight_skills]
            self.notebook_seed1_view.update_highlight(skill_names)
        else:
            self.notebook_seed1_view.set_text_result(
                    u"Seed1は{0}件あります。条件を絞ってください".format(len(seed1s)))
            self.notebook_seed1_view.clear_seed1_grid()

    def OnClickSeed1Clear(self, evt):
        u""" clear seed1s from selected skill1s """
        self.notebook_seed1_view.clear_combobox()
        self.notebook_seed1_view.clear_seed1_grid()
        self.notebook_seed1_view.clear_text_result()

    u""" Seed2 view's event """
    def OnClickSeed2Search(self, evt):
        u""" search seed2s from selected skill2s """
        amu_id2skill_id_list_dict = {}
        for amu_key, amu_name in zip([view.KEY_AMULET1, view.KEY_AMULET2, view.KEY_AMULET3], 
                [view.NAME_AMULET1, view.NAME_AMULET2, view.NAME_AMULET3]):
            amu_id = self._amulet_name2id_dict[amu_name]
            names = self.notebook_seed2_view.get_selected_skill_names(amu_key)
            ls = [self._skill_name2id_dict[name] if name in self._skill_name2id_dict else None 
                    for name in names]
            amu_id2skill_id_list_dict[amu_id] = ls
        seed_sets = self.db_accessor.select_seed2s(amu_id2skill_id_list_dict)
        self.notebook_seed2_view.set_result_text(u"""Seedの候補は{0}個です。""".format(len(seed_sets)))

        if len(seed_sets) > 0:
            self.notebook_seed2_view.set_seed_lists([u"{0:06d}".format(seed) for seed in seed_sets])
            self.notebook_seed2_view.set_skill2_button_enable(True)
        else:
            self.notebook_seed2_view.clear_seed_list()
            self.notebook_seed2_view.set_skill2_button_enable(False)

    def OnClickSeed2Clear(self, evt):
        u""" reset seed2 search settings of combo box"""
        self.notebook_seed2_view.set_skill_selected_idx(0)
        self.notebook_seed2_view.clear_seed_list()
        self.notebook_seed2_view.set_result_text(u"")
        self.notebook_seed2_view.set_skill2_button_enable(False)

    def OnClickSkill2SearchFromSeed2(self, evt):
        u""" change page to skill2 search from seed2"""
        seed2 = self.notebook_seed2_view.get_selected_seed2()
        if seed2 is not None:
            self.notebook_skill2_view.set_seed2_value(seed2)
            self.frame_view.note_book.SetSelection(view.SKILL_SEARCH_PAGE)
            self.OnClickSkillSearch(evt)

    u""" Skill2 search from Seed2's event """
    def OnClickSkillSearch(self, evt):
        u""" skill search from seed2"""
        seed2 = self.notebook_skill2_view.get_seed2_value()
        if seed2 is not None:
            skill_dict, threshold1_dict, threshold2_dict = self.db_accessor.select_skill2s_from_seed2s([seed2])
            try:
                for amu_key, amu_name in zip([view.KEY_AMULET1, view.KEY_AMULET2, view.KEY_AMULET3], 
                        [view.NAME_AMULET1, view.NAME_AMULET2, view.NAME_AMULET3]):
                    amu_id = self._amulet_name2id_dict[amu_name]
                    skill_names = [self._skill_id2name_dict[x] for x in skill_dict[amu_id][seed2]]
                    self.notebook_skill2_view.set_skill2_by_col_key(amu_key, skill_names)
                th_vals = [u"{0}".format(x) for x in threshold1_dict[seed2]]
                self.notebook_skill2_view.set_skill2_by_col_key(view.KEY_THRESHOLD1, th_vals)
                th_vals = [u"{0}".format(x) for x in threshold2_dict[seed2]]
                self.notebook_skill2_view.set_skill2_by_col_key(view.KEY_THRESHOLD2, th_vals)
                #inishie
                skill_id, th1, th2 = self.db_accessor.select_inishie_skill2_from_seed2([seed2])[seed2]
                self.notebook_skill2_view.set_inishie(self._skill_id2name_dict[skill_id], th1, th2)
                # explanation
                table_no, no = self.db_accessor.select_table_nos_from_seed2([seed2])[seed2]
                self.notebook_skill2_view.set_result_text(
                        u"SEED2:{2}は通し番号{1}、テーブルNo{0}です".format(table_no, no, seed2))
            except KeyError, e:
                self._show_message_dialog(message=u"指定されたSeed値は存在しません")
            finally:
                self._update_highlight()

        else:
            self._show_message_dialog(message=u"Seed値には数字を入力してください")

    def OnClickSkillClear(self, evt):
        u""" clear skills from seed """
        self.notebook_skill2_view.clear_skill2_grid()
        self.notebook_skill2_view.clear_skill2_highlight()
        self.notebook_skill2_view.clear_items_amulet_prospects()
        self.notebook_skill2_view.clear_result_text()
        self.notebook_skill2_view.clear_inishie()

    u""" amulet search event """
    def OnClickAmuletRadioAmulet(self, evt):
        u""" switch skill lists by amulet id """
        #btn_id = evt.GetId()
        amulet_name = self.notebook_amulet_view.get_selected_amulet()
        amu_id = self._amulet_name2id_dict[amulet_name]
        self._update_notebook_amulet(amu_id)
        self.notebook_amulet_view.set_result_text_ctrl_value(u"")
        self.notebook_amulet_view.set_skill_button_enable(False)
        self.notebook_amulet_view.clear_listbox_items()

    def OnClickAmuletRadioThresholdType(self, evt):
        u""" switch seed lists by threshold_type """
        self._update_notebook_amulet_seed2s()

    def OnClickAmuletCombo(self, evt):
        u""" switch skill minmax by amulet id and skill_id"""
        self._update_notebook_amulet()

    def OnClickAmuletSearch(self, evt):
        u""" search seeds from amulet condition """
        skill1_name, skill2_name = self.notebook_amulet_view.get_string_selection_skill_names()
        skill1_val, skill2_val = self.notebook_amulet_view.get_skill_values()
        slot_val = self.notebook_amulet_view.get_slot_value()
        amulet_name = self.notebook_amulet_view.get_selected_amulet()
        amu_id = self._amulet_name2id_dict[amulet_name]

        if skill1_name == skill2_name:
            self._show_message_dialog(message=u"異なるスキルを選択してください")
        elif skill1_val == 0:
            self._show_message_dialog(message=u"第1スキルの値には0以外を指定してください")
        else:
            skill1_id = self._skill_name2id_dict[skill1_name]
            if skill2_name in self._skill_name2id_dict:
                skill2_id = self._skill_name2id_dict[skill2_name]
            else:
                skill2_id = None # for skill2 is 0
            suff_dict = {}
            tup = self.db_accessor.get_sufficient_value(
                amu_id, skill1_id, skill2_id, skill1_val, skill2_val)
            if tup is not None:
                suff_val = tup[0]
                (threshold, self.threshold1_seed2_set, self.threshold2_seed2_set, 
                    self.threshold_inishie_set) = self.db_accessor.select_seed2s_from_sufficient_val(
                        amu_id, suff_val, slot_val, skill2_id)
                seed2_set = self.threshold1_seed2_set | self.threshold2_seed2_set | self.threshold_inishie_set
                self._update_notebook_amulet_seed2s()
                if len(seed2_set) > 0:
                    self.notebook_amulet_view.set_result_text_ctrl_value(
u"""指定されたお守りの充足値:\t{0}
必要な判定値:\t{1}\n判定値1:\t{2}個のSeed2で出現\n判定値2:\t{3}個のSeed2で出現\nいにしえ:\t{4}個のSeed2で出現
全体:\t{5}個のSeed2で出現""".format(
                            suff_val, threshold, len(self.threshold1_seed2_set), len(self.threshold2_seed2_set), len(self.threshold_inishie_set), len(seed2_set)))
                else:
                    self.notebook_amulet_view.set_result_text_ctrl_value(
u"""指定されたお守りの充足値:\t{0}\n必要な判定値:\t{1}
指定されたお守りは見つかりませんでした""".format(suff_val, threshold))
                    self.notebook_amulet_view.set_skill_button_enable(False)
                    self.notebook_amulet_view.clear_listbox_items()
            else:
                self.threshold1_seed2_set, self.threshold2_seed2_set, self.threshold_inishie_set = set(), set(), set()
                self.notebook_amulet_view.set_result_text_ctrl_value(
                    u"エラー。充足値が計算できません")
                self.notebook_amulet_view.set_skill_button_enable(False)
                self.notebook_amulet_view.clear_listbox_items()

    def OnClickAmuletClear(self, evt):
        u""" clear amulet conditions """
        self.notebook_amulet_view.set_radio_value(True, view.NAME_AMULET1)
        amu_id = self._amulet_name2id_dict[view.NAME_AMULET1]
        self._update_notebook_amulet(amu_id)
        self.notebook_amulet_view.set_skill_button_enable(False)
        self.notebook_amulet_view.set_result_text_ctrl_value(u"")
        self.notebook_amulet_view.clear_listbox_items()

    def OnClickSkillSearchFromAmulet(self, evt):
        u""" change page to skill search from amulet"""
        seed = self.notebook_amulet_view.get_string_selection_listbox().split(u":")[0]
        if seed.isdigit():
            self.notebook_skill2_view.set_seed2_value(seed)
            self.frame_view.note_book.SetSelection(view.SKILL_SEARCH_PAGE)
            self.OnClickSkillSearch(evt)

    u""" settings' event """
    def OnClickSettingOK(self, evt):
        u""" get settings of setting tab """
        (self._highlight_threshold1, 
                self._highlight_threshold2) = self.frame_view.notebook_setting_view.get_threshold()
        self._highlight_skills = set([self._skill_name2id_dict[x] for x in 
                self.frame_view.notebook_setting_view.get_checked_strings()
                if x in self._skill_name2id_dict])
        self._update_highlight()

    def OnClickSettingClear(self, evt):
        u""" reset settings of setting tab """
        self._highlight_threshold1 = view.HIGHLIGHT_THRESHOLD1
        self._highlight_threshold2 = view.HIGHLIGHT_THRESHOLD2
        self._highlight_skills = set()
        self._update_notebook_setting()
        self._update_highlight()


    def _show_error_dialog(self, message=u"予期せぬエラーが発生しました", caption=u"エラー"):
        u""" エラーダイアログを表示し、
        OKボタンが押されたらアプリケーションを終了する
        """
        dlg = wx.MessageDialog(self.frame_view.frame, 
            message,
            caption, wx.OK | wx.ICON_ERROR)
        dlg.ShowModal()
        dlg.Destroy()
        wx.Exit()
    
    def _show_message_dialog(self, message, caption=u"メッセージ"):
        u""" メッセージダイアログを表示する
        """
        dlg = wx.MessageDialog(self.frame_view.frame, 
            message,
            caption, wx.OK | wx.ICON_INFORMATION)
        dlg.ShowModal()
        dlg.Destroy()

    def CloseHandler(self, evt):
        dlg = wx.MessageDialog(parent = self.frame_view.frame,
                message = u"終了します。よろしいですか？", 
                caption = u"終了確認", 
                style = wx.YES_NO)
        result = dlg.ShowModal()
        dlg.Destroy()
        if result == wx.ID_YES:
            self._write_settings()
            wx.Exit()

    def OnClose(self, evt):
        self.frame_view.Close()

    def OnAboutBox(self, evt):
        info = self.frame_view.GetAboutInfo()
        wx.AboutBox(info)

    def _write_settings(self):
        with open(SETTING_FILE, mode="w") as f:
            data = {SETTING_THRESHOLD1:self._highlight_threshold1, 
                    SETTING_THRESHOLD2:self._highlight_threshold2, 
                    SETTING_SKILLS:self._highlight_skills}
            pickle.dump(data, f)

    def _read_settings(self):
        if os.path.exists(SETTING_FILE):
            with open(SETTING_FILE, mode="r") as f:
                try:
                    data = pickle.load(f)
                    self._highlight_threshold1 = data[SETTING_THRESHOLD1] 
                    self._highlight_threshold2 = data[SETTING_THRESHOLD2] 
                    self._highlight_skills = data[SETTING_SKILLS]
                except EOFError, e:
                    self._highlight_threshold1 = view.HIGHLIGHT_THRESHOLD1
                    self._highlight_threshold2 = view.HIGHLIGHT_THRESHOLD2
                    self._highlight_skills = set()
        else:
            self._highlight_threshold1 = view.HIGHLIGHT_THRESHOLD1
            self._highlight_threshold2 = view.HIGHLIGHT_THRESHOLD2
            self._highlight_skills = set()

if __name__ == "__main__":
    app = AmuletToolController(False)
    app.MainLoop()

