/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javax.sql;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.Ref;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Map;
import java.io.InputStream;
import java.io.Reader;
import java.util.Calendar;
import java.math.BigDecimal;

/**
 * An interface which provides means to access data which
 * persists on a database. It extends the functionality of
 * {@link java.sql.ResultSet ResultSet} into a form that it can be used as a
 * JavaBean component, suited for a visual programming environment.
 * <p>
 * {@code RowSet} provides getters and setters for properties relating to the
 * general database environment together with the getters and setters for
 * distinct data values which constitute the row set. The {@code RowSet} class
 * supports JavaBean events so that other components in an application can be
 * informed when changes happen such as changes in data values.
 * <p>
 * {@code RowSet} is a facility implemented on top of the remainder of the JDBC
 * API. It may be <i>connected</i>, maintaining a connection to the database
 * throughout its lifecycle. The changes made on a <i>disconnected</i> {@code
 * RowSet} on the other hand can be persisted only establishing a new connection
 * with the database each time.
 * <p>
 * Disconnected {@code RowSets} make use of {@code RowSetReaders} to populate
 * the {@code RowSet} with data, possibly from a non-relational database source.
 * They may also use {@code RowSetWriters} to send data back to the underlying
 * data store. There is considerable freedom in the way that {@code
 * RowSetReaders} and {@code RowSetWriters} may be implemented to retrieve and
 * store data.
 *
 * @see RowSetReader
 * @see RowSetWriter
 */
public interface RowSet extends ResultSet {

    /**
     * Registers the supplied {@link RowSetListener} with this {@code RowSet}.
     * Once registered, the {@link RowSetListener} is notified of events
     * generated by the {@code RowSet}.
     * 
     * @param theListener
     *            an object which implements the {@code rowSetListener}
     *            interface.
     */
    public void addRowSetListener(RowSetListener theListener);

    /**
     * Clears the parameters previously set for this {@code RowSet}.
     * <p>
     * The {@code RowSet} object retains its value until either a new value for
     * a parameter is set or its value is actively reset. {@code
     * clearParameters} provides a facility to clear the values for all
     * parameters with one method call.
     *
     * @throws SQLException
     *             if a problem occurs accessing the database.
     */
    public void clearParameters() throws SQLException;

    /**
     * Fetches data for this {@code RowSet} from the database. If successful,
     * any existing data for the {@code RowSet} is discarded and its metadata is
     * overwritten.
     * <p>
     * Data is retrieved connecting to the database and executing an
     * according SQL statement. This requires some or all of the following
     * properties to be set: URL, database name, user name, password,
     * transaction isolation, type map; plus some or all of the properties:
     * command, read only, maximum field size, maximum rows, escape processing,
     * and query timeout.
     * <p>
     * The {@code RowSet} may use a {@code RowSetReader} to access the database
     * it will then invoke the {@link RowSetReader#readData} method on the
     * reader to fetch the data. When the new data is fetched all the listeners
     * are notified to take appropriate measures.
     *
     * @throws SQLException
     *             if a problem occurs accessing the database or if the
     *             properties needed to access the database have not been set.
     * @see RowSetMetaData
     * @see RowSetReader
     */
    public void execute() throws SQLException;

    /**
     * Gets the {@code RowSet}'s command property.
     * 
     * @return a string containing the {@code RowSet}'s command property. A
     *         command is a SQL statement which is executed to fetch required
     *         data into the {@code RowSet}.
     */
    public String getCommand();

    /**
     * Gets the ODBC Data Source Name property associated with this {@code
     * RowSet}. The database name can be used to find a {@link DataSource}
     * which has been registered with a naming service - the {@link DataSource}
     * can then be used to create a connection to the database.
     * 
     * @return the name of the database.
     */
    public String getDataSourceName();

    /**
     * Reports if escape processing is enabled for this {@code RowSet}. If
     * escape processing is on, the driver performs a substitution of the escape
     * syntax with the applicable code before sending an SQL command to the
     * database. The default value for escape processing is {@code true}.
     * 
     * @return {@code true} if escape processing is enabled, {@code
     *         false} otherwise.
     * @throws SQLException
     *             if a problem occurs accessing the database.
     */
    public boolean getEscapeProcessing() throws SQLException;

    /**
     * Gets the maximum number of bytes that can be returned for column values
     * which are of type {@code BINARY}, {@code VARBINARY}, {@code
     * LONGVARBINARYBINARY}, {@code CHAR}, {@code VARCHAR}, or {@code
     * LONGVARCHAR}. Excess data is silently discarded if the number is
     * exceeded.
     * 
     * @return the current maximum size in bytes. 0 implies no size limit.
     * @throws SQLException
     *             if a problem occurs accessing the database.
     */
    public int getMaxFieldSize() throws SQLException;

    /**
     * Gets the maximum number of rows for this {@code RowSet}. Excess rows are
     * discarded silently if the limit is exceeded.
     * 
     * @return the previous maximum number of rows. 0 implies no row limit.
     * @throws SQLException
     *             if a problem occurs accessing the database.
     */
    public int getMaxRows() throws SQLException;

    /**
     * Gets the value of the password property for this {@code RowSet}. This
     * property is used when a connection to the database is established.
     * Therefore it should be set prior to invoking the {@link #execute} method.
     * 
     * @return the value of the password property.
     */
    public String getPassword();

    /**
     * Gets the timeout for the driver when a query operation is executed. If a
     * query takes longer than the timeout then a {@code SQLException} is
     * thrown.
     * 
     * @return the timeout value in seconds.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public int getQueryTimeout() throws SQLException;

    /**
     * Gets the transaction isolation level property set for this 
     * {@code RowSet}. The transaction isolation level defines the 
     * policy implemented on the database for maintaining the data 
     * values consistent.
     * 
     * @return the current transaction isolation level. Must be one of:
     *         <ul>
     *         <li>{@code Connection.TRANSACTION_READ_UNCOMMITTED}</li>
     *         <li>{@code Connection.TRANSACTION_READ_COMMITTED}</li>
     *         <li>{@code Connection.TRANSACTION_REPEATABLE_READ}</li>
     *         <li>{@code Connection.TRANSACTION_SERIALIZABLE}</li>
     *         </ul>
     * @see java.sql.Connection
     */
    public int getTransactionIsolation();

    /**
     * Gets the custom mapping of SQL User-Defined Types (UDTs) and Java classes
     * for this {@code RowSet}, if applicable.
     * 
     * @return the custom mappings of SQL types to Java classes.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public Map<String, Class<?>> getTypeMap() throws SQLException;

    /**
     * Gets the URL property value for this {@code RowSet}. If there is no
     * {@code DataSource} object specified, the {@code RowSet} uses the URL to
     * establish a connection to the database. The default value for the URL is
     * {@code null}.
     * 
     * @return a String holding the value of the URL property.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public String getUrl() throws SQLException;

    /**
     * Gets the value of the {@code username} property for this {@code RowSet}.
     * The {@code username} is used when establishing a connection to the
     * database and should be set before the {@code execute} method is invoked.
     * 
     * @return a {@code String} holding the value of the {@code username}
     *         property.
     */
    public String getUsername();

    /**
     * Indicates if this {@code RowSet} is read-only.
     * 
     * @return {@code true} if this {@code RowSet} is read-only, {@code false}
     *         if it is updatable.
     */
    public boolean isReadOnly();

    /**
     * Removes a specified {@link RowSetListener} object from the set of
     * listeners which will be notified of events by this {@code RowSet}.
     * 
     * @param theListener
     *            the {@link RowSetListener} to remove from the set of listeners
     *            for this {@code RowSet}.
     */
    public void removeRowSetListener(RowSetListener theListener);

    /**
     * Sets the specified {@code ARRAY} parameter in the {@code RowSet} command
     * with the supplied {@code java.sql.Array} value.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theArray
     *            the {@code Array} data value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setArray(int parameterIndex, Array theArray)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the ASCII data in the supplied {@code java.io.InputStream} value.
     * Data is read from the {@code InputStream} until end-of-file is reached.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theInputStream
     *            the ASCII data value to which the parameter is set.
     * @param length
     *            the length of the data in bytes.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setAsciiStream(int parameterIndex, InputStream theInputStream,
            int length) throws SQLException;

    /**
     * Sets the value of the specified SQL {@code NUMERIC} parameter in the
     * {@code RowSet} command with the data in the supplied {@code
     * java.math.BigDecimal} value.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theBigDecimal
     *            the big decimal value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBigDecimal(int parameterIndex, BigDecimal theBigDecimal)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to the binary data in the supplied input stream. Data is read from the
     * input stream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theInputStream
     *            the binary data stream to which the parameter is set.
     * @param length
     *            the length of the data in bytes.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBinaryStream(int parameterIndex, InputStream theInputStream,
            int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to the supplied {@code Blob} value.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theBlob
     *            the {@code Blob} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(int parameterIndex, Blob theBlob) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to the supplied boolean.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theBoolean
     *            the {@code boolean} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBoolean(int parameterIndex, boolean theBoolean)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to the supplied byte value.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theByte
     *            the {@code byte} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setByte(int parameterIndex, byte theByte) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to the supplied byte array value.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theByteArray
     *            the {@code Array} of {@code bytes} to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBytes(int parameterIndex, byte[] theByteArray)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to the sequence of Unicode characters carried by the supplied {@code
     * java.io.Reader}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theReader
     *            the {@code Reader} which contains the Unicode data to set the
     *            parameter.
     * @param length
     *            the length of the data in the {@code Reader} in characters.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCharacterStream(int parameterIndex, Reader theReader,
            int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the value of a supplied {@code java.sql.Clob}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theClob
     *            the {@code Clob} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(int parameterIndex, Clob theClob) throws SQLException;

    /**
     * Sets the Command property for this {@code RowSet} - the command is an SQL
     * query which runs when the {@code execute} method is invoked. This
     * property is optional for databases that do not support commands.
     * 
     * @param cmd
     *            the SQL query. Can be {@code null}.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCommand(String cmd) throws SQLException;

    /**
     * Sets the concurrency property of this {@code RowSet}. The default value
     * is {@code ResultSet.CONCUR_READ_ONLY}.
     * 
     * @param concurrency
     *            the concurrency value. One of:
     *            <ul>
     *            <li>{@code ResultSet.CONCUR_READ_ONLY}</li>
     *            <li>{@code ResultSet.CONCUR_UPDATABLE}</li>
     *            </ul>
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.sql.ResultSet
     */
    public void setConcurrency(int concurrency) throws SQLException;

    /**
     * Sets the database name property for the {@code RowSet}.
     * <p>
     * The database name can be used to find a {@link DataSource} which has been
     * registered with a naming service - the {@link DataSource} can then be
     * used to create a connection to the database.
     *
     * @param name
     *            the database name.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDataSourceName(String name) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the value of a supplied {@code java.sql.Date}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theDate
     *            the date value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDate(int parameterIndex, Date theDate) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the value of a supplied {@code java.sql.Date}, where the conversion
     * of the date to an SQL {@code DATE} value is calculated using a supplied
     * {@code Calendar}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theDate
     *            the date to which the parameter is set.
     * @param theCalendar
     *            the {@code Calendar} to use in converting the Date to an SQL
     *            {@code DATE} value.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDate(int parameterIndex, Date theDate, Calendar theCalendar)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the supplied {@code double}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theDouble
     *            the {@code double} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDouble(int parameterIndex, double theDouble)
            throws SQLException;

    /**
     * Sets the escape processing status for this {@code RowSet}. If escape
     * processing is on, the driver performs a substitution of the escape syntax
     * with the applicable code before sending an SQL command to the database.
     * The default value for escape processing is {@code true}.
     * 
     * @param enable
     *            {@code true} to enable escape processing, {@code false} to
     *            turn it off.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setEscapeProcessing(boolean enable) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the supplied {@code float}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theFloat
     *            the {@code float} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setFloat(int parameterIndex, float theFloat)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the supplied {@code integer}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theInteger
     *            the {@code integer} value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setInt(int parameterIndex, int theInteger) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * with the supplied {@code long}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theLong
     *            the {@code long} value value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setLong(int parameterIndex, long theLong) throws SQLException;

    /**
     * Sets the maximum number of bytes which can be returned for a column value
     * where the column type is one of {@code BINARY}, {@code VARBINARY},
     * {@code LONGVARBINARYBINARY}, {@code CHAR}, {@code VARCHAR}, or {@code
     * LONGVARCHAR}. Data which exceeds this limit is silently discarded. For
     * portability, a value greater than 256 is recommended.
     * 
     * @param max
     *            the maximum size of the returned column value in bytes. 0
     *            implies no size limit.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setMaxFieldSize(int max) throws SQLException;

    /**
     * Sets the maximum number of rows which can be held by the {@code RowSet}.
     * Any additional rows are silently discarded.
     * 
     * @param max
     *            the maximum number of rows which can be held in the {@code
     *            RowSet}. 0 means no limit.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setMaxRows(int max) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to SQL {@code NULL}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param sqlType
     *            the type of the parameter, as defined by {@code
     *            java.sql.Types}.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNull(int parameterIndex, int sqlType) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to SQL {@code NULL}. This form of the {@code setNull} method should be
     * used for User Defined Types and {@code REF} parameters.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param sqlType
     *            the type of the parameter, as defined by {@code
     *            java.sql.Types}.
     * @param typeName
     *            the fully qualified name of an SQL user defined type or the
     *            name of the SQL structured type referenced by a {@code REF}
     *            type. Ignored if the sqlType is not a UDT or REF type.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNull(int parameterIndex, int sqlType, String typeName)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied Java object.
     * <p>
     * The JDBC specification provides a standard mapping for Java objects to
     * SQL data types. Database specific types can be mapped by JDBC driver
     * specific Java types.
     *
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theObject
     *            the Java object containing the data value to which the
     *            parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(int parameterIndex, Object theObject)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied Java object.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theObject
     *            the Java object containing the data value.
     * @param targetSqlType
     *            the SQL type to send to the database, as defined in {@code
     *            java.sql.Types}.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(int parameterIndex, Object theObject,
            int targetSqlType) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied Java object.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theObject
     *            the Java object containing the data value.
     * @param targetSqlType
     *            the SQL type to send to the database, as defined in {@code
     *            java.sql.Types}.
     * @param scale
     *            the number of digits after the decimal point, for {@code
     *            java.sql.Types.DECIMAL} and {@code java.sql.Types.NUMERIC}
     *            types. Ignored for all other types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(int parameterIndex, Object theObject,
            int targetSqlType, int scale) throws SQLException;

    /**
     * Sets the database Password for this {@code RowSet}. This property is used
     * when a connection to the database is established. Therefore it should be
     * set prior to invoking the {@link #execute} method.
     * 
     * @param password
     *            a {@code String} holding the password.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setPassword(String password) throws SQLException;

    /**
     * Gets the timeout for the driver when a query operation is executed. If a
     * query takes longer than the timeout, a {@code SQLException} is thrown.
     * 
     * @param seconds
     *            the number of seconds for the timeout.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setQueryTimeout(int seconds) throws SQLException;

    /**
     * Sets whether the {@code RowSet} is read-only or updatable.
     * 
     * @param readOnly
     *            {@code true} to set the {@code RowSet} to read-only state,
     *            {@code false} to allow updates.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setReadOnly(boolean readOnly) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code java.sql.Ref}. This is sent to the database as an
     * SQL {@code REF} value.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theRef
     *            the value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.sql.Ref
     */
    public void setRef(int parameterIndex, Ref theRef) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code short integer}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theShort
     *            the value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setShort(int parameterIndex, short theShort)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code String}. The string is placed into the database as a
     * {@code VARCHAR} or {@code LONGVARCHAR} SQL value, depending on the
     * database limits for the length of {@code VARCHAR} values.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theString
     *            the value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setString(int parameterIndex, String theString)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code java.sql.Time}, converting it to an SQL {@code TIME}
     * value using the system default {@code Calendar}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theTime
     *            the value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.util.Calendar
     * @see java.sql.Time
     */
    public void setTime(int parameterIndex, Time theTime) throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code java.sql.Time}, converting it to an SQL {@code TIME}
     * value using a supplied {@code Calendar}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theTime
     *            the value to which the parameter is set.
     * @param theCalendar
     *            the {@code Calendar} to use in the conversion operation.
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.util.Calendar
     * @see java.sql.Time
     */
    public void setTime(int parameterIndex, Time theTime, Calendar theCalendar)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code java.sql.Timestamp}, converting it to an SQL {@code
     * TIMESTAMP} value using the system default {@code Calendar}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theTimestamp
     *            the value to which the parameter is set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.util.Calendar
     * @see java.sql.Timestamp
     */
    public void setTimestamp(int parameterIndex, Timestamp theTimestamp)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the {@code RowSet} command
     * to a supplied {@code java.sql.Timestamp}, converting it to an SQL {@code
     * TIMESTAMP} value using a supplied {@code Calendar}.
     * 
     * @param parameterIndex
     *            the index of the parameter to set; the first parameter's index
     *            is 1.
     * @param theTimestamp
     *            the value to which the parameter is set.
     * @param theCalendar
     *            the {@code Calendar} to use in the conversion operation
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.util.Calendar
     * @see java.sql.Timestamp
     */
    public void setTimestamp(int parameterIndex, Timestamp theTimestamp,
            Calendar theCalendar) throws SQLException;

    /**
     * Sets the target instance's transaction isolation level to one of a
     * discrete set of possible values. The transaction isolation level defines
     * the policy implemented on the database for maintaining the data values
     * consistent.
     * <p>
     * Keep in mind that setting a transaction isolation level has no effect
     * unless your driver and DBMS support it.
     *
     * @param level
     *            the transaction isolation level. One of:
     *            <ul>
     *            <li>{@code Connection.TRANSACTION_READ_UNCOMMITTED}</li> 
     *            <li>{@code Connection.TRANSACTION_READ_COMMITTED}</li> 
     *            <li>{@code Connection.TRANSACTION_REPEATABLE_READ}</li> 
     *            <li>{@code Connection.TRANSACTION_SERIALIZABLE}</li>
     *            </ul>
     * @throws SQLException
     *             if an error occurs accessing the database.
     * @see java.sql.Connection
     */
    public void setTransactionIsolation(int level) throws SQLException;

    /**
     * Sets the type of this {@code RowSet}. By default, the type is
     * non-scrollable.
     * 
     * @param type
     *            the type for the {@code RowSet}. One of:
     *            <ul>
     *            <li>{@code ResultSet.TYPE_FORWARD_ONLY}</li> 
     *            <li>{@code ResultSet.TYPE_SCROLL_INSENSITIVE}</li> 
     *            <li>{@code ResultSet.TYPE_SCROLL_SENSITIVE}</li>
     *            </ul>
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setType(int type) throws SQLException;

    /**
     * Sets the mapping of SQL User Defined Types (UDTs) to Java classes. The
     * Java classes must all implement the {@link java.sql.SQLData SQLData}
     * interface.
     * 
     * @param theTypeMap
     *            the names of SQL UDTs and the Java classes to which they are
     *            mapped.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTypeMap(Map<String, Class<?>> theTypeMap)
            throws SQLException;

    /**
     * Sets the URL used by this {@code RowSet} to access the database via a
     * {@code DriverManager}. The URL is optional - an alternative is to use a
     * database name to create a connection.
     * 
     * @param theURL
     *            the URL for the database. Can be {@code null}.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setUrl(String theURL) throws SQLException;

    /**
     * Sets the {@code Username} property for the {@code RowSet}, used to
     * authenticate a connection to the database.
     * 
     * @param theUsername
     *            the new user name for this row set.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setUsername(String theUsername) throws SQLException;
}
