/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.tools.lint.checks;

import com.android.tools.lint.detector.api.Context;
import com.android.tools.lint.detector.api.Issue;
import com.android.tools.lint.detector.api.LayoutDetector;
import com.android.tools.lint.detector.api.Scope;
import com.android.tools.lint.detector.api.Severity;
import com.android.tools.lint.detector.api.Speed;

import org.w3c.dom.Attr;
import org.w3c.dom.Element;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * Check which looks at the children of ScrollViews and ensures that they fill/match
 * the parent width instead of setting wrap_content.
 */
public class ScrollViewChildDetector extends LayoutDetector {
    /** The main issue discovered by this detector */
    public static final Issue ISSUE = Issue.create(
            "ScrollViewSize", //$NON-NLS-1$
            "Checks that ScrollViews use wrap_content in scrolling dimension",
            // TODO add a better explanation here!
            "ScrollView children must set their layout_width or layout_height attributes " +
            "to wrap_content rather than fill_parent or match_parent in the scrolling " +
            "dimension",
            CATEGORY_LAYOUT, 7, Severity.WARNING);

    /** Constructs a new {@link ScrollViewChildDetector} */
    public ScrollViewChildDetector() {
    }

    @Override
    public Issue[] getIssues() {
        return new Issue[] { ISSUE };
    }

    @Override
    public Speed getSpeed() {
        return Speed.FAST;
    }

    @Override
    public Scope getScope() {
        return Scope.SINGLE_FILE;
    }

    @Override
    public Collection<String> getApplicableElements() {
        return Arrays.asList(new String[] {
                SCROLL_VIEW,
                HORIZONTAL_SCROLL_VIEW
        });
    }

    @Override
    public void visitElement(Context context, Element element) {
        List<Element> children = getChildren(element);
        boolean isHorizontal = HORIZONTAL_SCROLL_VIEW.equals(element.getTagName());
        String attributeName = isHorizontal ? ATTR_LAYOUT_WIDTH : ATTR_LAYOUT_HEIGHT;
        for (Element child : children) {
            Attr sizeNode = child.getAttributeNodeNS(ANDROID_URI, attributeName);
            String value = sizeNode != null ? sizeNode.getValue() : null;
            if (VALUE_FILL_PARENT.equals(value) || VALUE_MATCH_PARENT.equals(value)) {
                String msg = String.format("This %1$s should use android:%2$s=\"wrap_content\"",
                        child.getTagName(), attributeName);
                context.toolContext.report(ISSUE, context.getLocation(sizeNode),
                        msg);
            }
        }
    }
}
