title: Definition of Initial Conditions

Navigate: [&larr; Restarting Simulations](tut_5_restart.html)
| [Overview](index.html)
| [Setting up a Convergence Criterium &rarr;](tut_7_convergence.html)

In this tutorial, we cover the definition of initial conditions. 
They can be defined either in lattice units or in physical units.
Lattice units can be confusing, especially to people just starting in this field.
The relationships between lattice and physical units can be found here.

> [Here](http://wiki.palabos.org/_media/howtos:lbunits.pdf), you can have a look at a paper about unit conversion by Jonas Latt.

In this tutorial, a two-dimensional plain channel is set up. 
Not only are the boundaries specified to obtain a defined pressure drop over the channel length, 
but also are the initial conditions set in a consistent manner.

For the viscous, laminar two-dimensional plain channel flow, an analytical solution
of the incompressible Navier-Stokes equation can be derived. From the analytical solution
the pressure drop, the velocity profile and the shear stress distribution can be computed.

Before starting, we need to define the flow regime and physical reference values.

```lua
u0Phys    = 1.
viscPhys  = u0Phys*heightPhys/Re
csPhys    = 300    -- m/s
rho0Phys  = 1.
```

For the lattice Boltzmann simulation, basic simulation parameters such as a characteristic velocity, 
a resolution and a relaxation parameter omega need to be specified.

```lua
csLB    = 1/math.sqrt(3)
cs2     = 1./3.
u0LB    = 0.05
rho0LB  = 1.
rho0LB0 = 0.
omega0  = 1.7 
```

Depending on the model used, the reference pressure differs. For the incompressible model, 
the reference pressure is 0, while for the compressible model the reference pressure is `rho0*cs2`

```lua
if model == 'lbm_incomp' then
  p0 = 0.
else
  p0 = rho0LB*cs2*dx*dx/dt/dt
end 
```

From the solution of the Navier-Stokes equation, the following relations for the
velocity distribution across the channel height can be obtained

```lua
function uX(x,y,z)
    uPhys = u0Phys*(1-(2.*y/heightPhys)^2)
    return uPhys
  end
```

Similarly for the pressure drop along the channel length

```lua
function pressureRef(x,y,z)
  dp = u0Phys*8.*viscPhys*rho0Phys/heightPhys^2*length
  return p0 + dp*0.5 - dp/length*x
  end
```

and the shear stress across the channel height

```lua
function Sxy(x,y,z)
  tauxy= -viscPhys*rho0Phys*8./heightPhys^2*u0Phys*y
  S_xy = tauxy/viscPhys/rho0Phys
  return S_xy
  end
```

Now the physics table establishes the connection between the lattice reference values 
and the physical values and gives Musubi means of 
transferring between these two unit systems.

```lua
physics = { dt = dt, rho0 = rho0Phys }
```
> [[mus_physics_module]]

For the Lattice Boltzmann algorithm, a reference density and the relaxation rate omega need to be defined.

```lua
fluid = { omega = omega }
```
> [[mus_fluid_module]]

Now the initial conditions for each element in the simulation domain is defined by setting
each physical quantity and connecting it to a lua function, which we defined above.

```lua
initial_condition = { pressure  = ic_pressure,
                      velocityX = ic_uX,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                      Sxx = 0.,    
                      Syy = 0.,    
                      Sxy = ic_Sxy 
                      }
```                      
@note The whole code of musubi.lua is shown in the tutorial [The full Toolchain](tut_2_mus_toolchain.html).

Next chapter: [Set up a Convergence Criterium &rarr;](tut_7_convergence.html)
