! See copyright notice in the COPYRIGHT file.
?? include 'header/lbm_macros.inc'
?? include 'header/lbm_interfaceMacros.inc'
?? include 'treelm/source/logMacros.inc'
! ****************************************************************************** !
!> Boundary condition wall treatment routines
!!
!! This module contains higher order wall treatments
!! A detailed description on the implementation details are given 
!! in [[tem_bc_module]].
!!
module mus_bc_fluid_wall_module

  ! include treelm modules
  use env_module,               only: rk
  use tem_param_module,         only: cs2inv, rho0
  use tem_time_module,          only: tem_time_type
  use treelmesh_module,         only: treelmesh_type
  use tem_varSys_module,        only: tem_varSys_type
  use tem_debug_module,         only: dbgUnit
  use tem_property_module,      only: prp_solid
  use tem_construction_module,  only: tem_levelDesc_type

  ! include musubi modules
  use mus_bc_header_module,      only: boundary_type, glob_boundary_type
  use mus_scheme_layout_module,  only: mus_scheme_layout_type
  use mus_field_prop_module,     only: mus_field_prop_type
  use mus_derVarPos_type_module, only: mus_derVarPos_type
  use mus_param_module,          only: mus_param_type
  use mus_physics_module,        only: mus_physics_type
  use mus_mixture_module,        only: mus_mixture_type

  implicit none

  private

  ! public :: wall_multiReflection
  public :: slip_wall, spc_slip_wall
  public :: wall_libb
  public :: do_nothing

contains

! ****************************************************************************** !
  !> slip-wall boundary condition. Slip defined by a slip factor
  !!
  !! \li Normal velocity,\f$ u_n = 0 \f$
  !! \li Tangential velocity, \f$ \frac{\partial u_t}{\partial n} = 0 \f$
  !! \li Pressure, \f$ \frac{\partial P}{\partial n} = 0 \f$
  !! For slip-wall boundary, the slip factor will be multiplied by the velocity
  !! if slip factor = 1, then it is full/free-slip and if slip factor = 0, then
  !! it is no-slip  
  !!
  !! @todo KM: Currently, free-slip boundary works only for axis-parallel planes.
  !!           Need to extend it for arbitary geometries
  !!
  !  subroutine slip_wall
?? copy :: bc_routineHeader( slip_wall )
    ! --------------------------------------------------------------------------- 
    ! defining local variables
    real(kind=rk) :: fTmp( layout%fStencil%QQ*globBC%nElems(iLevel) )
    real(kind=rk) :: vel(3*globBC%nElems(iLevel)) ! Velocity on boundary element
    real(kind=rk) :: velTmp(3), rho
    integer :: iELem, iDir, bndNormalDir, QQ, posInBuffer
    ! --------------------------------------------------------------------------- 

    QQ = layout%fStencil%QQ

    do iElem = 1, globBC%nElems(iLevel)
      posInBuffer = globBC%elemLvl( iLevel )%posInBcElemBuf%val( iElem )
      fTmp( (iElem-1)*QQ+1: (iElem-1)*QQ+QQ ) &
        &       = bcBuffer( (posInBuffer-1)*nScalars+varPos(1) : &
        &                   (posInBuffer-1)*nScalars+varPos(1)+QQ-1 )
    end do

    ! Get local velocity
    call derVarPos%velFromState( state  = fTmp ,                 &
      &                          iField = iField,                &
      &                          nElems = globBC%nElems(iLevel), &
      &                          varSys = varSys,                &
      &                          layout = layout,                &
      &                          res    = vel                    )

    do iElem = 1, globBC%nElems(iLevel)
      velTmp = vel((iElem-1)*3+1 : iELem*3) * me%slip_fac
      rho = sum(fTmp( (iElem-1)*QQ+1: (iElem-1)*QQ+QQ ))
      bndNormalDir = layout%fStencil%cxDirInv( globBC%elemLvl( iLevel )%       &
        &                                                normalInd%val( iElem ))
      !write(dbgUnit(1),*) 'bndNormalDir ',  bndNormalDir
      if( abs(layout%fStencil%cxDir( 1, bndNormalDir )) == 1) velTmp(1) = 0.0_rk
      if( abs(layout%fStencil%cxDir( 2, bndNormalDir )) == 1) velTmp(2) = 0.0_rk
      if( abs(layout%fStencil%cxDir( 3, bndNormalDir )) == 1) velTmp(3) = 0.0_rk
      !write(dbgUnit(1),*) 'velTmp ', velTmp

      do iDir = 1, layout%fStencil%QQN
        ! Write the values
        if( globBC%elemLvl(iLevel)%bitmask%val(iDir, iElem )) then
          ! Depending on PUSH or pull, use + or - for cxDir, because directions
          ! are inverted
          state(                                                               &
& ?FETCH?(iDir, iField, globBC%elemLvl(iLevel)%elem%val(iElem), QQ, nScalars, nSize,neigh))=&
          ! We need to get post-collision pdf in direction
          ! alpha- outgoing direction, which is the inverse direction of bitmask
          ! For PULL this means, get the outgoing one, as this is the one which
          ! will be bounced back
          ! For PUSH this means, get the already bounced back pdf back, so take
          ! the incoming
            & fTmp(?IDX?(layout%fStencil%cxDirInv(iDir),iElem,QQ,globBC%nElems(iLevel)))&
            &       - layout%weight( iDir )*6._rk*rho                          &
            &       * ( layout%fStencil%cxDir( 1, layout%fStencil%             &
            &                                       cxDirInv( iDir ))*velTmp(1)&
            &       +   layout%fStencil%cxDir( 2, layout%fStencil%             &
            &                                       cxDirInv( iDir ))*velTmp(2)&
            &       +   layout%fStencil%cxDir( 3, layout%fStencil%             &
            &                                       cxDirInv( iDir ))*velTmp(3))
        end if
      end do
    end do !iElem

  end subroutine slip_wall
! ****************************************************************************** !


! ****************************************************************************** !
  !> slip-wall boundary condition. Slip defined by a slip factor
  !!
  !! * Normal velocity,\( u_n = 0 \)
  !! * Tangential velocity, \( \frac{\partial u_t}{\partial n} = 0 \)
  !! * Pressure, \( \frac{\partial P}{\partial n} = 0 \)
  !! 
  !! For slip-wall boundary, the slip factor will be multiplied by the velocity
  !! if slip factor = 1, then it is full/free-slip and if slip factor = 0, then
  !! it is no-slip  
  !!
  !! @todo KM: Currently, free-slip boundary works only for axis-parallel planes.
  !!           Need to extend it for arbitary geometries
  !!
  !  subroutine slip_wall
?? copy :: bc_routineHeader( spc_slip_wall )
    ! ------------------------------------------------------------------------ 
    real(kind=rk) :: fTmp_all( layout%fStencil%QQ*globBC%nElems(iLevel) &
      &              * varSys%nStateVars )
    real(kind=rk) :: fTmp(layout%fStencil%QQ)
    real(kind=rk) :: mom(3*globBC%nElems(iLevel)), momTmp(3)
    integer :: iELem, iDir, bndNormalDir, pos, iFieldLoc, QQ, posInBuffer
    ! ------------------------------------------------------------------------ 
    QQ = layout%fStencil%QQ

    do iElem = 1, globBC%nElems(iLevel)
      posInBuffer = globBC%elemLvl( iLevel )%posInBcElemBuf%val( iElem )
      do iFieldLoc = 1, varSys%nStateVars
        do iDir = 1, QQ
          pos = varSys%method%val(iFieldLoc)%state_varPos(iDir)
          fTmp_all( pos+(iElem-1)*nScalars ) &
            &  = bcBuffer( pos+(posInBuffer-1)*nScalars )
        end do
      end do
    end do

    call derVarPos%momFromState( state  = fTmp_all,                &
      &                          iField = iField,                  &
      &                          nElems = globBC%nElems( iLevel ), &
      &                          varSys = varSys,                  &
      &                          layout = layout,                  &
      &                          res    = mom                      )


    do iElem = 1, globBC%nElems(iLevel)
      if( .not. btest( levelDesc%property(                        &
        &              globBC%elemLvl(iLevel)%elem%val(iElem)), prp_solid))then

      momTmp(1) = mom((iElem-1)*3+1) * me%slip_fac
      momTmp(2) = mom((iElem-1)*3+2) * me%slip_fac
      momTmp(3) = mom((iElem-1)*3+3) * me%slip_fac

      bndNormalDir = layout%fStencil%cxDirInv( globBC%elemLvl( iLevel )%       &
        &                                                normalInd%val( iElem ))
      if( abs(layout%fStencil%cxDir( 1, bndNormalDir )) == 1) momTmp(1) = 0.0_rk
      if( abs(layout%fStencil%cxDir( 2, bndNormalDir )) == 1) momTmp(2) = 0.0_rk
      if( abs(layout%fStencil%cxDir( 3, bndNormalDir )) == 1) momTmp(3) = 0.0_rk

      posInBuffer = globBC%elemLvl( iLevel )%posInBcElemBuf%val( iElem )
      fTmp(1:QQ) = bcBuffer( (posInBuffer-1)*nScalars+varPos(1) : &
        &                    (posInBuffer-1)*nScalars+varPos(1)+QQ-1 )

      do iDir = 1, layout%fStencil%QQN
        if( globBC%elemLvl(iLevel)%bitmask%val( iDir, iElem )) then
          ! Depending on PUSH or pull, use + or - for cxDir, because directions
          ! are inverted
          state(                                                               &
&?FETCH?( iDir, iField, globBC%elemLvl(iLevel)%elem%val(iElem), QQ, nScalars, nSize,neigh))&
              & = fTmp(layout%fStencil%cxDirInv( iDir ))  &
              &    + layout%weight( iDir )*2._rk*cs2inv &
              &    * ( layout%fStencil%cxDir( 1, iDir )*momTmp(1) &
              &    +   layout%fStencil%cxDir( 2, iDir )*momTmp(2) &
              &    +   layout%fStencil%cxDir( 3, iDir )*momTmp(3) )
        end if
      end do
      end if
    end do

  end subroutine spc_slip_wall
! ****************************************************************************** !

! ****************************************************************************** !
?? copy :: bc_routineHeader( wall_libb )
    ! ---------------------------------------------------------------------------
    real(kind=rk) :: fIn, fOut, fNgh
    real(kind=rk) :: cIn, cOut, cNgh
    integer :: iLink
    ! ---------------------------------------------------------------------------

    !NEC$ ivdep
    !DIR$ ivdep
    !IBM* independent
    do iLink = 1, me%links(iLevel)%nVals

      cIn  = me%bouzidi(iLevel)% cIn( iLink )
      cOut = me%bouzidi(iLevel)%cOut( iLink )
      cNgh = me%bouzidi(iLevel)%cNgh( iLink )

      fIn  = bcBuffer( me%bouzidi(iLevel)% inPos(iLink) )
      fOut = bcBuffer( me%bouzidi(iLevel)%outPos(iLink) )
      fNgh = me%neigh(iLevel)%computeNeighBuf(me%bouzidi(iLevel)%nghPos(iLink))

      state( me%links(iLevel)%val(iLink) ) = cIn*fIn + cOut*fOut + cNgh*fNgh

    end do ! iLink

  end subroutine wall_libb
! ****************************************************************************** !

! ****************************************************************************** !
?? copy :: bc_routineHeader( do_nothing )
  end subroutine do_nothing
! ****************************************************************************** !

end module mus_bc_fluid_wall_module
! ****************************************************************************** !
