! See copyright notice in the COPYRIGHT file.
?? include 'header/lbm_macros.inc'
?? include 'header/lbm_d3q19Macros.inc'
?? include 'header/lbm_interfaceMacros.inc'
! ****************************************************************************** !
!> author: Jiaxing Qi
!! This module provides the definition and methods for
!! MRT advection relaxation scheme.
!! The LB equaton using MRT is
!!    f(t+dt,x+dx) = f - M^(-1) * S * ( (M*f) - m^(eq) )
!!
!! The moments m(1:19) = M * f(1:19) are labeled as
!!  m( 1) = rho
!!  m( 2) = e = rho * (ux^2 + uy^2 + uz^2)
!!  m( 3) = epsilon
!!  m( 4) = jx = rho * ux
!!  m( 5) = qx
!!  m( 6) = jy = rho * uy
!!  m( 7) = qy
!!  m( 8) = jz = rho * uz
!!  m( 9) = qz
!!  m(10) = 3 * pxx = rho * (2ux^2 - uy^2 - uz^2)
!!  m(11) = 3 * Pixx
!!  m(12) = pzz  = rho * (uy^2 - uz^2)
!!  m(13) = Piww
!!  m(14) = pxy  = rho * ux * uy
!!  m(15) = pyz  = rho * uy * uz
!!  m(16) = pzx  = rho * uz * ux
!!  m(17) = mx
!!  m(18) = my
!!  m(19) = mz
!!
!! The non-zero equilibirium moments are given by
!!  meq( 1) = rho
!!  meq( 2) = rho0 * ( ux^2 + uy^2 + uz^2 )
!!  meq( 4) = rho0 * ux
!!  meq( 6) = rho0 * uy
!!  meq( 8) = rho0 * uz
!!  meq(10) = rho0 * ( 2*ux^2 - uy^2 - uz^2 )
!!  meq(12) = rho0 * ( uy^2 - uz^2 )
!!  meq(14) = rho0 * ux * uy
!!  meq(15) = rho0 * uy * uz
!!  meq(16) = rho0 * ux * uz
!!
!! Density (rho) and velocity (ux, uy, uz) are conserved during collision.
!!  i.e. m(1) = meq(1) --> mneq(1) = 0
!!       m(4) = meq(4) --> mneq(4) = 0
!!       m(6) = meq(6) --> mneq(6) = 0
!!       m(8) = meq(8) --> mneq(8) = 0
!!
!! The collision parameters S correspondes to the omega in BGK model.
!!
!! The MRT implementation here is taken from:\n
!! J. Toelke, S. Freudiger, and M. Krafczyk,
!! "An adaptive scheme using hierarchical grids for lattice Boltzmann
!! multi-phase flow simulations," Comput. Fluids, vol. 35, pp. 820–830,
!! 2006. \n
!! Notice that the collision matrix S used in this papar corresponds to
!! -omega in BGK model, because it express the LB equation is slightly
!! different way.
!! In this paper, the following notions are used:\n
!!  s(a) = s(2)
!!  s(b) = s(3)
!!  s(c) = s(5) = s(7) = s(9)
!!  s(d) = s(11) = s(13
!!  s(e) = s(17) = s(18) = s(19)
!!  s(w) = s(10) = s(12) = s(14) = s(15) = s(16)
!! It is suggested that, for D3Q19,
!!  s(a) = s(b) = s(c) = s(d) = s(e) = max( s(w), -1.0 )
!!
!! SubGrid Stress model (SGS)
!! The implementation here is taken from:\n
!! M. Stiebler, M. Krafczyk, S. Freudiger, M. Geier
!! "Lattice Boltzmann large eddy simulation of subcritical flows around a sphere
!! on non-uniform grids", Computers and Mathematics with Applications, vol. 61
!! (2011), pp. 3475-3484
!! Equation 12:\n
!! tau_{total} = 3 * nu0 + dt * 0.5
!!               + 0.5 * ( sqrt( tau0*tau0  + 18 * Cs * Cs * dt * dt * Q ) - tau0 )
!!             = 0.5 * ( tau0 + sqrt( tau0 * tau0 + 18 * Cs * Cs * dt * dt * Q) )
!! Q = sqrt( 2.0 * sum( Pi^{neq} * Pi^{neq} ) )
!!
!! For single field LBM: QQ=nScalars
!! 
module mus_mrt_module

  ! include treelm modules
  use env_module,               only: rk
  use tem_varSys_module,        only: tem_varSys_type, tem_varSys_op_type
  use tem_param_module,         only: div1_2, div1_3, div1_4, div1_6, div1_8, &
    &                                 div1_12, div1_16, div1_18, div1_24, div1_36,&
    &                                 div1_48, div1_72, &
    &                                 cs2inv, cs4inv, t2cs2inv, t2cs4inv, rho0

  ! include musubi modules
  use mus_field_prop_module,    only: mus_field_prop_type
  use mus_scheme_layout_module, only: mus_scheme_layout_type
  use mus_scheme_type_module,   only: mus_scheme_type
  use mus_param_module,         only: mus_param_type

  implicit none

  private

  public :: mrt_advRel_d3q19
  public :: mrt_advRel_d3q19_explicit
  public :: mrt_advRel_d3q19_les
  public :: mrt_advRel_d3q19_les_explicit

  public :: mrt_advRel_d3q19_incomp
  public :: mrt_advRel_d3q19_incomp_explicit
  public :: mrt_advRel_d3q19_incomp_les
  public :: mrt_advRel_d3q19_incomp_les_explicit


  public :: mrt_advRel_explicit
  public :: mrt_advRel_incomp_explicit

  !=============================================================================
  ! D3Q19 flow model
  !=============================================================================
  !> Definition of the discrete velocity set

  integer,parameter :: QQ   = 19   !< number of pdf directions

  integer,parameter :: qN00 = 1     !< west             x-
  integer,parameter :: q0N0 = 2     !< south            y-
  integer,parameter :: q00N = 3     !< bottom           z-
  integer,parameter :: q100 = 4     !< east             x+
  integer,parameter :: q010 = 5     !< north            y+
  integer,parameter :: q001 = 6     !< top              z+
  integer,parameter :: q0NN = 7     !<                  z-,y-
  integer,parameter :: q0N1 = 8     !<                  z+,y-
  integer,parameter :: q01N = 9     !<                  z-,y+
  integer,parameter :: q011 = 10    !<                  z+,y+
  integer,parameter :: qN0N = 11    !<                  x-,z-
  integer,parameter :: q10N = 12    !<                  x+,z-
  integer,parameter :: qN01 = 13    !<                  x-,z+
  integer,parameter :: q101 = 14    !<                  x+,z+
  integer,parameter :: qNN0 = 15    !<                  y-,x-
  integer,parameter :: qN10 = 16    !<                  y+,x-
  integer,parameter :: q1N0 = 17    !<                  y-,x+
  integer,parameter :: q110 = 18    !<                  y+,x+
  integer,parameter :: q000 = 19    !< rest density is last

  ! D3Q19 MRT pdf -> moment transformation matrix
  ! How to use:
  ! do iDir = 1, QQ
  !   moment(iDir) = sum( PDF(:) * MMtrD3Q19(iDir,:) )
  ! end do
  !  W      S     B     E     N     T     BS    TS   BN    TN    BW    BE    TW    TE    SW    NW    SE    NE     0
  real(kind=rk), dimension(19,19),parameter,public  :: MMtrD3Q19 = &
  reshape((/ &
   1.0,   1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0, -1.0, &
  -2.0,  -2.0, -2.0, -2.0, -2.0, -2.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0, &
  -1.0,   0.0,  0.0,  1.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0, -1.0,  1.0, -1.0,  1.0, -1.0, -1.0,  1.0,  1.0,  0.0, &
   2.0,   0.0,  0.0, -2.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0, -1.0,  1.0, -1.0,  1.0, -1.0, -1.0,  1.0,  1.0,  0.0, &
   0.0,  -1.0,  0.0,  0.0,  1.0,  0.0, -1.0, -1.0,  1.0,  1.0,  0.0,  0.0,  0.0,  0.0, -1.0,  1.0, -1.0,  1.0,  0.0, &
   0.0,   2.0,  0.0,  0.0, -2.0,  0.0, -1.0, -1.0,  1.0,  1.0,  0.0,  0.0,  0.0,  0.0, -1.0,  1.0, -1.0,  1.0,  0.0, &
   0.0,   0.0, -1.0,  0.0,  0.0,  1.0, -1.0,  1.0, -1.0,  1.0, -1.0, -1.0,  1.0,  1.0,  0.0,  0.0,  0.0,  0.0,  0.0, &
   0.0,   0.0,  2.0,  0.0,  0.0, -2.0, -1.0,  1.0, -1.0,  1.0, -1.0, -1.0,  1.0,  1.0,  0.0,  0.0,  0.0,  0.0,  0.0, &
   2.0,  -1.0, -1.0,  2.0, -1.0, -1.0, -2.0, -2.0, -2.0, -2.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  0.0, &
  -2.0,   1.0,  1.0, -2.0,  1.0,  1.0, -2.0, -2.0, -2.0, -2.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  1.0,  0.0, &
   0.0,   1.0, -1.0,  0.0,  1.0, -1.0,  0.0,  0.0,  0.0,  0.0, -1.0, -1.0, -1.0, -1.0,  1.0,  1.0,  1.0,  1.0,  0.0, &
   0.0,  -1.0,  1.0,  0.0, -1.0,  1.0,  0.0,  0.0,  0.0,  0.0, -1.0, -1.0, -1.0, -1.0,  1.0,  1.0,  1.0,  1.0,  0.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  1.0, -1.0, -1.0,  1.0,  0.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0,  1.0, -1.0, -1.0,  1.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  1.0, -1.0, -1.0,  1.0,  0.0,  0.0,  0.0,  0.0,  0.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  0.0,  1.0, -1.0,  1.0, -1.0, -1.0, -1.0,  1.0,  1.0,  0.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0, -1.0, -1.0,  1.0,  1.0,  0.0,  0.0,  0.0,  0.0,  1.0, -1.0,  1.0, -1.0,  0.0, &
   0.0,   0.0,  0.0,  0.0,  0.0,  0.0,  1.0, -1.0,  1.0, -1.0, -1.0, -1.0,  1.0,  1.0,  0.0,  0.0,  0.0,  0.0,  0.0  &
  /),(/19,19/), order=(/ 2,1 /) )

!   real(kind=rk), dimension(19,19),parameter,public  :: MMivD3Q19 = &
!   reshape((/ &
!   div1_18,        0,  -div1_18,   -div1_6,    div1_6,         0,         0,         0,         0,   div1_12,  -div1_12, &
!         0,         0,       0,      0,      0,      0,      0,      0, &
!   div1_18,        0,  -div1_18,         0,         0,   -div1_6,    div1_6,         0,         0,  -div1_24,   div1_24, &
!    div1_8,   -div1_8,       0,      0,      0,      0,      0,      0, &
!   div1_18,        0,  -div1_18,         0,         0,         0,         0,   -div1_6,    div1_6,  -div1_24,   div1_24, &
!   -div1_8,    div1_8,       0,      0,      0,      0,      0,      0, &
!   div1_18,        0,  -div1_18,    div1_6,   -div1_6,         0,         0,         0,         0,   div1_12,  -div1_12, &
!         0,         0,       0,      0,      0,      0,      0,      0, &
!   div1_18,        0,  -div1_18,         0,         0,    div1_6,   -div1_6,         0,         0,  -div1_24,   div1_24, &
!    div1_8,   -div1_8,       0,      0,      0,      0,      0,      0, &
!   div1_18,        0,  -div1_18,         0,         0,         0,         0,    div1_6,   -div1_6,  -div1_24,   div1_24, &
!   -div1_8,    div1_8,       0,      0,      0,      0,      0,      0, &
!   div1_36,  div1_24,   div1_72,         0,         0,  -div1_12,  -div1_24,  -div1_12,  -div1_24,  -div1_24,  -div1_24, &
!         0,         0,       0,    div1_4,      0,      0,   -div1_8,    div1_8, &
!   div1_36,  div1_24,   div1_72,         0,         0,  -div1_12,  -div1_24,   div1_12,   div1_24,  -div1_24,  -div1_24, &
!         0,         0,       0,   -div1_4,      0,      0,   -div1_8,   -div1_8, &
!   div1_36,  div1_24,   div1_72,         0,         0,   div1_12,   div1_24,  -div1_12,  -div1_24,  -div1_24,  -div1_24, &
!         0,         0,       0,   -div1_4,      0,      0,    div1_8,    div1_8, &
!   div1_36,  div1_24,   div1_72,         0,         0,   div1_12,   div1_24,   div1_12,   div1_24,  -div1_24,  -div1_24, &
!         0,         0,       0,    div1_4,      0,      0,    div1_8,   -div1_8, &
!   div1_36,  div1_24,   div1_72,  -div1_12,  -div1_24,         0,         0,  -div1_12,  -div1_24,   div1_48,   div1_48, &
!  -div1_16,  -div1_16,       0,      0,    div1_4,    div1_8,      0,   -div1_8, &
!   div1_36,  div1_24,   div1_72,   div1_12,   div1_24,         0,         0,  -div1_12,  -div1_24,   div1_48,   div1_48, &
!  -div1_16,  -div1_16,       0,      0,   -div1_4,   -div1_8,      0,   -div1_8, &
!   div1_36,  div1_24,   div1_72,  -div1_12,  -div1_24,         0,         0,   div1_12,   div1_24,   div1_48,   div1_48, &
!  -div1_16,  -div1_16,       0,      0,   -div1_4,    div1_8,      0,    div1_8, &
!   div1_36,  div1_24,   div1_72,   div1_12,   div1_24,         0,         0,   div1_12,   div1_24,   div1_48,   div1_48, &
!  -div1_16,  -div1_16,       0,      0,    div1_4,   -div1_8,      0,    div1_8, &
!   div1_36,  div1_24,   div1_72,  -div1_12,  -div1_24,  -div1_12,  -div1_24,         0,         0,   div1_48,   div1_48, &
!   div1_16,   div1_16,  div1_4,      0,      0,   -div1_8,    div1_8,      0, &
!   div1_36,  div1_24,   div1_72,  -div1_12,  -div1_24,   div1_12,   div1_24,         0,         0,   div1_48,   div1_48, &
!   div1_16,   div1_16, -div1_4,      0,      0,   -div1_8,   -div1_8,      0, &
!   div1_36,  div1_24,   div1_72,   div1_12,   div1_24,  -div1_12,  -div1_24,         0,         0,   div1_48,   div1_48, &
!   div1_16,   div1_16, -div1_4,      0,      0,    div1_8,    div1_8,      0, &
!   div1_36,  div1_24,   div1_72,   div1_12,   div1_24,   div1_12,   div1_24,         0,         0,   div1_48,   div1_48, &
!   div1_16,   div1_16,  div1_4,      0,      0,    div1_8,   -div1_8,      0, &
!    div1_3,  -div1_2,    div1_6,         0,         0,         0,         0,         0,         0,         0,         0, &
!         0,         0,       0,      0,      0,      0,      0,      0 &
!   /),(/19,19/))

  ! D3Q19 MRT moment --> PDF transformation matrix
  ! How to use:
  ! do iDir = 1, QQ
  !   fneq(iDir) = sum( MMIvD3Q19(iDir,:) * mneq(:) )
  ! end do
  real(kind=rk), dimension(19,19),parameter,public  :: MMivD3Q19 = &
  reshape((/ &
   1/18.0,     0.0, -1/18.0,  -1/6.0,   1/6.0,     0.0,      0.0,     0.0,     0.0,  1/12.0, -1/12.0,     0.0,     0.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0,&
   1/18.0,     0.0, -1/18.0,     0.0,     0.0,  -1/6.0,    1/6.0,     0.0,     0.0, -1/24.0,  1/24.0,   1/8.0,  -1/8.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0,&
   1/18.0,     0.0, -1/18.0,     0.0,     0.0,     0.0,      0.0,  -1/6.0,   1/6.0, -1/24.0,  1/24.0,  -1/8.0,   1/8.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0,&
   1/18.0,     0.0, -1/18.0,   1/6.0,  -1/6.0,     0.0,      0.0,     0.0,     0.0,  1/12.0, -1/12.0,     0.0,     0.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0,&
   1/18.0,     0.0, -1/18.0,     0.0,     0.0,   1/6.0,   -1/6.0,     0.0,     0.0, -1/24.0,  1/24.0,   1/8.0,  -1/8.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0,&
   1/18.0,     0.0, -1/18.0,     0.0,     0.0,     0.0,      0.0,   1/6.0,  -1/6.0, -1/24.0,  1/24.0,  -1/8.0,   1/8.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0,&
   1/36.0,  1/24.0,  1/72.0,     0.0,     0.0, -1/12.0,  -1/24.0, -1/12.0, -1/24.0, -1/24.0, -1/24.0,     0.0,     0.0,    0.0,&
 1/4.0,    0.0,    0.0, -1/8.0,  1/8.0,&
   1/36.0,  1/24.0,  1/72.0,     0.0,     0.0, -1/12.0,  -1/24.0,  1/12.0,  1/24.0, -1/24.0, -1/24.0,     0.0,     0.0,    0.0,&
-1/4.0,    0.0,    0.0, -1/8.0, -1/8.0,&
   1/36.0,  1/24.0,  1/72.0,     0.0,     0.0,  1/12.0,   1/24.0, -1/12.0, -1/24.0, -1/24.0, -1/24.0,     0.0,     0.0,    0.0,&
-1/4.0,    0.0,    0.0,  1/8.0,  1/8.0,&
   1/36.0,  1/24.0,  1/72.0,     0.0,     0.0,  1/12.0,   1/24.0,  1/12.0,  1/24.0, -1/24.0, -1/24.0,     0.0,     0.0,    0.0,&
 1/4.0,    0.0,    0.0,  1/8.0, -1/8.0,&
   1/36.0,  1/24.0,  1/72.0, -1/12.0, -1/24.0,     0.0,      0.0, -1/12.0, -1/24.0,  1/48.0,  1/48.0, -1/16.0, -1/16.0,    0.0,&
   0.0,  1/4.0,  1/8.0,    0.0, -1/8.0,&
   1/36.0,  1/24.0,  1/72.0,  1/12.0,  1/24.0,     0.0,      0.0, -1/12.0, -1/24.0,  1/48.0,  1/48.0, -1/16.0, -1/16.0,    0.0,&
   0.0, -1/4.0, -1/8.0,    0.0, -1/8.0,&
   1/36.0,  1/24.0,  1/72.0, -1/12.0, -1/24.0,     0.0,      0.0,  1/12.0,  1/24.0,  1/48.0,  1/48.0, -1/16.0, -1/16.0,    0.0,&
   0.0, -1/4.0,  1/8.0,    0.0,  1/8.0,&
   1/36.0,  1/24.0,  1/72.0,  1/12.0,  1/24.0,     0.0,      0.0,  1/12.0,  1/24.0,  1/48.0,  1/48.0, -1/16.0, -1/16.0,    0.0,&
   0.0,  1/4.0, -1/8.0,    0.0,  1/8.0,&
   1/36.0,  1/24.0,  1/72.0, -1/12.0, -1/24.0, -1/12.0,  -1/24.0,     0.0,     0.0,  1/48.0,  1/48.0,  1/16.0,  1/16.0,  1/4.0,&
   0.0,    0.0, -1/8.0,  1/8.0,    0.0,&
   1/36.0,  1/24.0,  1/72.0, -1/12.0, -1/24.0,  1/12.0,   1/24.0,     0.0,     0.0,  1/48.0,  1/48.0,  1/16.0,  1/16.0, -1/4.0,&
   0.0,    0.0, -1/8.0, -1/8.0,    0.0,&
   1/36.0,  1/24.0,  1/72.0,  1/12.0,  1/24.0, -1/12.0,  -1/24.0,     0.0,     0.0,  1/48.0,  1/48.0,  1/16.0,  1/16.0, -1/4.0,&
   0.0,    0.0,  1/8.0,  1/8.0,    0.0,&
   1/36.0,  1/24.0,  1/72.0,  1/12.0,  1/24.0,  1/12.0,   1/24.0,     0.0,     0.0,  1/48.0,  1/48.0,  1/16.0,  1/16.0,  1/4.0,&
   0.0,    0.0,  1/8.0, -1/8.0,    0.0,&
    1/3.0,  -1/2.0,   1/6.0,     0.0,     0.0,     0.0,      0.0,     0.0,     0.0,     0.0,     0.0,     0.0,     0.0,    0.0,&
   0.0,    0.0,    0.0,    0.0,    0.0 &
  /),(/19,19/), order=(/ 2,1 /) )

contains

! ****************************************************************************** !
  !> Advection relaxation routine for the MRT model.
  !! This routine has roughly 260 FLOPS per elements.
  !!
?? copy :: compute_routineHeader( mrt_advRel_d3q19 )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem
?? copy :: pdfTmp19( f )
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: inv_rho ! inverse local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    ! MRT Variables
    real(kind=rk) :: meq2, meq10, meq12
    real(kind=rk) :: m2, m6, m8, m14, m15, m16
    ! m6, m8 are temporary var
?? copy :: var19( s_mrt )
    ! mout1 are temp var
?? copy :: var19( mout )
?? copy :: var19( sum )
    real(kind=rk) :: c0, c1, c2, c3, c4, c5, c6
    real(kind=rk) :: mout5_4, mout7_4, mout9_4
    real(kind=rk) :: sum_c1_c2, sub_c1_c2
    real(kind=rk) :: sum_5_17, sub_7_18, d1,d2,d3,d4
    real(kind=rk) :: sum_9_19, sub_5_17, e1,e2,e3,e4
    real(kind=rk) :: sum_7_18, sub_9_19, g1,g2,g3,g4
    ! ---------------------------------------------------------------------------

    s_mrt2  = fieldProp(1)%fluid%mrt( level )%s_mrt( 2) * div1_24
    s_mrt3  = fieldProp(1)%fluid%mrt( level )%s_mrt( 3) * div1_72
    s_mrt5  = fieldProp(1)%fluid%mrt( level )%s_mrt( 5) * div1_24
    s_mrt7  = fieldProp(1)%fluid%mrt( level )%s_mrt( 7) * div1_24
    s_mrt9  = fieldProp(1)%fluid%mrt( level )%s_mrt( 9) * div1_24
    s_mrt10 = fieldProp(1)%fluid%mrt( level )%s_mrt(10)
    s_mrt11 = fieldProp(1)%fluid%mrt( level )%s_mrt(11)
    s_mrt12 = fieldProp(1)%fluid%mrt( level )%s_mrt(12)
    s_mrt13 = fieldProp(1)%fluid%mrt( level )%s_mrt(13)
    s_mrt14 = fieldProp(1)%fluid%mrt( level )%s_mrt(14) * div1_4
    s_mrt15 = fieldProp(1)%fluid%mrt( level )%s_mrt(15) * div1_4
    s_mrt16 = fieldProp(1)%fluid%mrt( level )%s_mrt(16) * div1_4
    s_mrt17 = fieldProp(1)%fluid%mrt( level )%s_mrt(17) * div1_8
    s_mrt18 = fieldProp(1)%fluid%mrt( level )%s_mrt(18) * div1_8
    s_mrt19 = fieldProp(1)%fluid%mrt( level )%s_mrt(19) * div1_8

!$omp do schedule(static)
    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1, nSolve

      !> First load all local values into temp array
      fN00 = inState(?FETCH?( qN00, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f0N0 = inState(?FETCH?( q0N0, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f00N = inState(?FETCH?( q00N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f100 = inState(?FETCH?( q100, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f010 = inState(?FETCH?( q010, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f001 = inState(?FETCH?( q001, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f0NN = inState(?FETCH?( q0NN, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f0N1 = inState(?FETCH?( q0N1, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f01N = inState(?FETCH?( q01N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f011 = inState(?FETCH?( q011, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fN0N = inState(?FETCH?( qN0N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f10N = inState(?FETCH?( q10N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fN01 = inState(?FETCH?( qN01, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f101 = inState(?FETCH?( q101, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fNN0 = inState(?FETCH?( qNN0, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fN10 = inState(?FETCH?( qN10, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f1N0 = inState(?FETCH?( q1N0, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f110 = inState(?FETCH?( q110, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f000 = inState(?FETCH?( q000, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))

      m6     = f101 + fN0N + f10N + fN01
      m8     = f011 + f0NN + f01N + f0N1
      sum1   = f110 + fNN0 + f1N0 + fN10
      m2     =-f000 + sum1 + m6 + m8

      sum2 = f010 + f0N0
      sum3 = f001 + f00N
      sum4 = 2._rk * ( f100+ fN00 )
      sum5 = sum2 + sum3

      ! epsilon
      mout3 = ( 2._rk*(f000 - sum5) - sum4 + m2 ) * s_mrt3

      rho = m2 + 2.0d0 * f000 + sum5 + sum4 * div1_2
      inv_rho = 1.0d0 / rho

?? copy :: ux_d3q19( u_x, f, inv_rho )
?? copy :: uy_d3q19( u_y, f, inv_rho )
?? copy :: uz_d3q19( u_z, f, inv_rho )

      ! Equilibrium moments
      ! non zero meq are only: 2, 10, 12, 14, 15, 16
      meq2  = rho * ( u_x*u_x + u_y*u_y + u_z*u_z )
      meq10 = rho * 3.0_rk*u_x*u_x - meq2
      meq12 = rho * ( u_y*u_y - u_z*u_z )

      mout2  = s_mrt2 * (m2 - meq2)

      !pxy
      m14 = f110 + fNN0 - f1N0 - fN10
      mout14 = s_mrt14 * (m14 - rho*u_x*u_y)
      !pyz
      m15 = f011 + f0NN - f01N - f0N1
      mout15 = s_mrt15 * (m15 - rho*u_y*u_z)
      !pxz
      m16 = f101 + fN0N - f10N - fN01
      mout16 = s_mrt16 * (m16 - rho*u_x*u_z)

      sum6 = sum1 + m6 - m8 * 2.0d0
      sum7 = sum4 - sum5
      !3pxx
      mout10 = ( sum7 + sum6 - meq10 ) * s_mrt10
      !3pixx
      mout11 = ( - sum7 + sum6 ) * s_mrt11

      sum8 = sum1 - m6
      sum9 = sum2 - sum3
      !pww
      mout12 = ( sum8 + sum9 - meq12 ) * s_mrt12
      !piww
      mout13 = ( sum8 - sum9 ) * s_mrt13

c1 = f110 - fNN0
c2 = f1N0 - fN10
c3 = f101 - fN0N
c4 = f10N - fN01

      sum10 = c1 + c2
      sum11 = c3 + c4
      !qx
      mout5  = ( sum10 + sum11 - 2.0_rk*(f100 - fN00) ) * s_mrt5
      !mx
      mout17 = ( sum10 - sum11 ) * s_mrt17

c5 = f011 - f0NN
c6 = f01N - f0N1

      sum12 = c1 - c2
      sum13 = c5 + c6
      !qy
      mout7  = ( sum12 + sum13 - 2.0_rk*(f010 - f0N0) ) * s_mrt7
      !my
      mout18 = ( - sum12 + sum13 ) * s_mrt18

      sum14 = c3 - c4
      sum15 = c5 - c6
      !qz
      mout9  = ( sum14 + sum15 - 2.0_rk*(f001 - f00N) ) * s_mrt9
      !mz
      mout19 = ( sum14 - sum15 ) * s_mrt19

      ! Transformation back to PDF
      outstate( ?SAVE?( 19,1,iElem,QQ,QQ,nElems,neigh )) = f000 + 12._rk*(mout2-mout3)

! -------------------------------------------------------------------------------
      c0 = - 4._rk*mout3 + div1_12*(mout10 - mout11)
      mout5_4 = mout5 * 4.0d0
      outstate( ?SAVE?(  4,1,iElem,QQ,QQ,nElems,neigh )) = f100 - ( c0 - mout5_4 )
      outstate( ?SAVE?(  1,1,iElem,QQ,QQ,nElems,neigh )) = fN00 - ( c0 + mout5_4 )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      c1 = - 4._rk*mout3 - div1_24*(mout10 - mout11)
      c2 = div1_8 *(mout12 - mout13)

      sum_c1_c2 = c1 + c2
      mout7_4 = mout7 * 4.0d0
      outstate( ?SAVE?(  5,1,iElem,QQ,QQ,nElems,neigh)) = f010 - (sum_c1_c2 - mout7_4)
      outstate( ?SAVE?(  2,1,iElem,QQ,QQ,nElems,neigh)) = f0N0 - (sum_c1_c2 + mout7_4)

      sub_c1_c2 = c1 - c2
      mout9_4 = mout9 * 4.0d0
      outstate( ?SAVE?(  6,1,iElem,QQ,QQ,nElems,neigh)) = f001 - (sub_c1_c2 - mout9_4)
      outstate( ?SAVE?(  3,1,iElem,QQ,QQ,nElems,neigh)) = f00N - (sub_c1_c2 + mout9_4)
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      mout1 = mout2 + mout3
      c3 =   mout1  + div1_48 * (mout10 + mout11)   &
       &            + div1_16 * (mout12 + mout13)
      sum_5_17 = mout5 + mout17
      sub_7_18 = mout7 - mout18

      d1 = c3 + mout14
      d2 = sum_5_17 + sub_7_18
      outstate( ?SAVE?( 18,1,iElem,QQ,QQ,nElems,neigh)) = f110-(d1+d2)
      outstate( ?SAVE?( 15,1,iElem,QQ,QQ,nElems,neigh)) = fNN0-(d1-d2)

      d3 = c3 - mout14
      d4 = sum_5_17 - sub_7_18
      outstate( ?SAVE?( 17,1,iElem,QQ,QQ,nElems,neigh)) = f1N0-(d3+d4)
      outstate( ?SAVE?( 16,1,iElem,QQ,QQ,nElems,neigh)) = fN10-(d3-d4)
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      c4 = c3 - div1_8*(mout12+mout13)

      sum_9_19 = mout9 + mout19
      sub_5_17 = mout5 - mout17

      e1 = c4 + mout16
      e2 = sum_9_19 + sub_5_17
      outstate( ?SAVE?( 14,1,iElem,QQ,QQ,nElems,neigh )) = f101 - ( e1 + e2 )
      outstate( ?SAVE?( 11,1,iElem,QQ,QQ,nElems,neigh )) = fN0N - ( e1 - e2 )

      e3 = c4 - mout16
      e4 = sum_9_19 - sub_5_17
      outstate( ?SAVE?( 12,1,iElem,QQ,QQ,nElems,neigh )) = f10N - ( e3 - e4 )
      outstate( ?SAVE?( 13,1,iElem,QQ,QQ,nElems,neigh )) = fN01 - ( e3 + e4 )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      c5 = mout1 - div1_24*(mout10+mout11)
      sum_7_18 = mout7 + mout18
      sub_9_19 = mout9 - mout19

      g1 = c5 + mout15
      g2 = sum_7_18 + sub_9_19
      outstate( ?SAVE?( 10,1,iElem,QQ,QQ,nElems,neigh )) = f011 - ( g1 + g2 )
      outstate( ?SAVE?(  7,1,iElem,QQ,QQ,nElems,neigh )) = f0NN - ( g1 - g2 )

      g3 = c5 - mout15
      g4 = sum_7_18 - sub_9_19
      outstate( ?SAVE?(  9,1,iElem,QQ,QQ,nElems,neigh )) = f01N - ( g3 + g4 )
      outstate( ?SAVE?(  8,1,iElem,QQ,QQ,nElems,neigh )) = f0N1 - ( g3 - g4 )
! -------------------------------------------------------------------------------

    enddo nodeloop
!$omp end do

  end subroutine mrt_advRel_d3q19
! ****************************************************************************** !


! ****************************************************************************** !
  !> Advection relaxation routine for the MRT model.
  !! This routine has roughly 392 FLOPS per element.
  !! 
?? copy :: compute_routineHeader( mrt_advRel_d3q19_les )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer :: iElem
?? copy :: pdfTmp19( f )
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: inv_rho ! inverse local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    real(kind=rk) :: usq
    ! MRT Variables
    real(kind=rk) :: meq10, meq12
    real(kind=rk) :: s_mrt( 2:QQ )
    real(kind=rk) :: m( QQ )
    real(kind=rk) :: mout( 2:QQ )
    real(kind=rk) :: c0, c1, c2, c3, c4, c5, c6, c7
    ! Variables required for LES
    real(kind=rk) :: Q
    real(kind=rk) :: cs2_dt2_18
    real(kind=rk) :: tau0, tau_total
    ! variables for fNeq calculation
    real(kind=rk) :: usqn, usqn_o2
    real(kind=rk) :: coeff_1, coeff_2
    real(kind=rk) :: ui1,  fac_1,  sum1_2
    real(kind=rk) ::       fac_2,  sum2_2
    real(kind=rk) :: ui3,  fac_3,  sum3_2
    real(kind=rk) ::       fac_4,  sum4_2
    real(kind=rk) ::       fac_9,  sum9_2
    real(kind=rk) :: ui10, fac_10, sum10_2
    real(kind=rk) :: ui11, fac_11, sum11_2
    real(kind=rk) :: ui12, fac_12, sum12_2
    real(kind=rk) :: ui13, fac_13, sum13_2
    real(kind=rk) :: fNeq_NE_SW
    real(kind=rk) :: fNeq_NW_SE
    real(kind=rk) :: fNeq_BW_TE
    real(kind=rk) :: fNeq_BE_TW
    real(kind=rk) :: fNeq_BS_TN
    real(kind=rk) :: fNeq_BN_TS
    real(kind=rk) :: fNeq_N_S
    real(kind=rk) :: fNeq_W_E
    real(kind=rk) :: fNeq_T_B
    real(kind=rk) :: s(6)
    integer :: nScalars
    ! ---------------------------------------------------------------------------
    nScalars = varSys%nScalars

    s_mrt( 2)     = fieldProp(1)%fluid%mrt( level )%s_mrt( 2)
    s_mrt( 3)     = fieldProp(1)%fluid%mrt( level )%s_mrt( 3)
    s_mrt( 5)     = fieldProp(1)%fluid%mrt( level )%s_mrt( 5) * div1_24
    s_mrt( 7)     = fieldProp(1)%fluid%mrt( level )%s_mrt( 7) * div1_24
    s_mrt( 9)     = fieldProp(1)%fluid%mrt( level )%s_mrt( 9) * div1_24
    s_mrt(11)     = fieldProp(1)%fluid%mrt( level )%s_mrt(11)
    s_mrt(13)     = fieldProp(1)%fluid%mrt( level )%s_mrt(13)
    s_mrt(17:19)  = fieldProp(1)%fluid%mrt( level )%s_mrt(17:19) * div1_8

    tau0     = 1._rk / fieldProp(1)%fluid%omLvl( level )
    cs2_dt2_18 = 18._rk &
      &    * fieldProp(1)%fluid%les%c_s * fieldProp(1)%fluid%les%c_s &
      &    * fieldProp(1)%fluid%dtLvl(level) * fieldProp(1)%fluid%dtLvl(level)

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1, nSolve
      !> First load all local values into temp array
      fN00 = inState( ?FETCH?(  1, 1, iElem, QQ, nScalars, nElems,neigh ))
      f0N0 = inState( ?FETCH?(  2, 1, iElem, QQ, nScalars, nElems,neigh ))
      f00N = inState( ?FETCH?(  3, 1, iElem, QQ, nScalars, nElems,neigh ))
      f100 = inState( ?FETCH?(  4, 1, iElem, QQ, nScalars, nElems,neigh ))
      f010 = inState( ?FETCH?(  5, 1, iElem, QQ, nScalars, nElems,neigh ))
      f001 = inState( ?FETCH?(  6, 1, iElem, QQ, nScalars, nElems,neigh ))
      f0NN = inState( ?FETCH?(  7, 1, iElem, QQ, nScalars, nElems,neigh ))
      f0N1 = inState( ?FETCH?(  8, 1, iElem, QQ, nScalars, nElems,neigh ))
      f01N = inState( ?FETCH?(  9, 1, iElem, QQ, nScalars, nElems,neigh ))
      f011 = inState( ?FETCH?( 10, 1, iElem, QQ, nScalars, nElems,neigh ))
      fN0N = inState( ?FETCH?( 11, 1, iElem, QQ, nScalars, nElems,neigh ))
      f10N = inState( ?FETCH?( 12, 1, iElem, QQ, nScalars, nElems,neigh ))
      fN01 = inState( ?FETCH?( 13, 1, iElem, QQ, nScalars, nElems,neigh ))
      f101 = inState( ?FETCH?( 14, 1, iElem, QQ, nScalars, nElems,neigh ))
      fNN0 = inState( ?FETCH?( 15, 1, iElem, QQ, nScalars, nElems,neigh ))
      fN10 = inState( ?FETCH?( 16, 1, iElem, QQ, nScalars, nElems,neigh ))
      f1N0 = inState( ?FETCH?( 17, 1, iElem, QQ, nScalars, nElems,neigh ))
      f110 = inState( ?FETCH?( 18, 1, iElem, QQ, nScalars, nElems,neigh ))
      f000 = inState( ?FETCH?( 19, 1, iElem, QQ, nScalars, nElems,neigh ))

      !pxy
      m(14) = f110+ fNN0- f1N0- fN10
      !pyz
      m(15) = f011+ f0NN- f01N- f0N1
      !pxz
      m(16) = f101+ fN0N- f10N- fN01

      ! local x-, y- and z-velocity
?? copy :: ux_d3q19( u_x, f, 1._rk )
?? copy :: uy_d3q19( u_y, f, 1._rk )
?? copy :: uz_d3q19( u_z, f, 1._rk )

m(1) = f010 + f0N0
m(4) = f001 + f00N
c7   = 2._rk * ( f100+ fN00 )
c0   = f110 + fNN0 + f1N0 + fN10
m(6) = f101 + fN0N + f10N + fN01
m(8) = f011 + f0NN + f01N + f0N1

      ! local density
      rho = m(1) + m(4) + c7 * div1_2 + f000 + c0 + m(6) + m(8)

      ! inverse local density
      inv_rho = 1._rk / rho

      ! transfer moments to velocities
      u_x = u_x * inv_rho
      u_y = u_y * inv_rho
      u_z = u_z * inv_rho

      ! square velocity and derived constants
      usq = u_x*u_x + u_y*u_y + u_z*u_z

      ! meq2  = rho*usq
      meq10 = rho*(3.0_rk*u_x*u_x - usq)
      meq12 = rho*( u_y*u_y - u_z*u_z )

      m(2) =  - f000 + c0 + m(6) + m(8)

      !epsilon
      m(3) =  f000- 2._rk*( m(1) + m(4) ) -c7 + c0 + m(6) + m(8)

      !3pxx
      m(10) = - m(1) - m(4) + c0 + m(6) + c7 - m(8) * 2.0_rk
      !3pixx
      m(11) = + m(1) + m(4) + c0 + m(6) - c7 - m(8) * 2.0_rk
      !pww
      m(12) =   m(1) - m(4) + c0 - m(6)
      !piww
      m(13) = - m(1) + m(4) + c0 - m(6)

c1 = f110 - fNN0
c2 = f1N0 - fN10
c3 = f101 - fN0N
c4 = f10N - fN01

      !qx
      m(5) =  -2.0_rk*(f100- fN00) + c1 + c2 + c3 + c4
      !mx
      m(17) = c1 + c2 - c3 - c4

c5 = f011 - f0NN
c6 = f01N - f0N1

      !qy
      m(7) =  -2.0_rk*(f010- f0N0) + c1 - c2 + c5 + c6
      !my
      m(18) =-c1 + c2 + c5 + c6

      !qz
      m(9) =  -2.0_rk*(f001- f00N) + c3 - c4 + c5 - c6
      !mz
      m(19) = c3 - c4 - c5 + c6

      ! ---------------------- LES Part -----------------------------------------
      usqn = div1_36 * (1._rk - 1.5_rk * usq) * rho

      coeff_1 = div1_8 * rho

      ui1     = u_x + u_y
      fac_1   = coeff_1 * ui1
      sum1_2  = fac_1 * ui1 + usqn
      fNeq_NE_SW = f110 + fNN0 - sum1_2*2.0_rk

      ui3     = -u_x + u_y
      fac_3   = coeff_1 * ui3
      sum3_2  = fac_3 * ui3 + usqn
      fNeq_NW_SE = fN10 + f1N0 - sum3_2*2.0_rk

      ui10    =  u_x + u_z
      fac_10  = coeff_1 * ui10
      sum10_2 = fac_10 * ui10 + usqn
      fNeq_BW_TE = fN0N + f101 - sum10_2*2.0_rk

      ui12    = -u_x + u_z
      fac_12  = coeff_1 * ui12
      sum12_2 = fac_12 * ui12 + usqn
      fNeq_BE_TW = f10N + fN01 - sum12_2*2.0_rk

      ui11    =  u_y + u_z
      fac_11  = coeff_1 * ui11
      sum11_2 = fac_11 * ui11 + usqn
      fNeq_BS_TN = f0NN + f011 - sum11_2*2.0_rk

      ui13    = -u_y + u_z
      fac_13  = coeff_1 * ui13
      sum13_2 = fac_13 * ui13 + usqn
      fNeq_BN_TS = f01N + f0N1 - sum13_2*2.0_rk

      coeff_2 = div1_4 * rho
      usqn_o2 = 2.0_rk * usqn

      fac_2   = coeff_2 * u_y
      sum2_2  = fac_2 * u_y + usqn_o2
      fNeq_N_S = f010 + f0N0 - sum2_2*2.0_rk

      fac_4   = coeff_2 * u_x
      sum4_2  = fac_4 * u_x + usqn_o2
      fNeq_W_E = fN00 + f100 - sum4_2*2.0_rk

      fac_9   = coeff_2 * u_z
      sum9_2  = fac_9 * u_z + usqn_o2
      fNeq_T_B = f001 + f00N - sum9_2*2.0_rk

      ! Calculate second moment of fNeq
      ! Sxx = (W+E) + (SW+NE) + (SE+NW) + (BW+TE) + (BE+TW)
      s(1) = fNeq_W_E + fNeq_NE_SW + fNeq_NW_SE + fNeq_BW_TE + fNeq_BE_TW

      ! Syy = (S+N) + (NE+SW) + (NW+SE) + (BS+TN) + (BN+TS)
      s(2) = fNeq_N_S + fNeq_NE_SW + fNeq_NW_SE + fNeq_BS_TN + fNeq_BN_TS

      ! Szz = (B+T) + (BS+TN) + (BN+TS) + (BW+TE) + (BE+TW)
      s(3) = fNeq_T_B + fNeq_BS_TN + fNeq_BN_TS + fNeq_BW_TE + fNeq_BE_TW

      ! Sxy = (SW+NE) - (NW+SE)
      s(4) = fNeq_NE_SW - fNeq_NW_SE

      ! Syz = (BS+TN) - (TS+BN)
      s(5) = fNeq_BS_TN - fNeq_BN_TS

      ! Sxz = (BW+TE) - (BE+TW)
      s(6) = fNeq_BW_TE - fNeq_BE_TW

      Q = sqrt(2._rk*(  s(1)*s(1)         &
        &             + s(2)*s(2)         &
        &             + s(3)*s(3)         &
        &             + s(4)*s(4)*2._rk   &
        &             + s(5)*s(5)*2._rk   &
        &             + s(6)*s(6)*2._rk ) )

      ! cs2_dt2_18 = c_s * c_s * dt * dt * 18
      tau_total = 0.5_rk * ( tau0 + sqrt(tau0*tau0 + Q * cs2_dt2_18) )
      s_mrt(10) = 1.0_rk / tau_total
      ! s_mrt(12) = s_mrt(10)
      s_mrt(14) = div1_4 / tau_total
      ! s_mrt(15) = s_mrt(16) = s_mrt(14)
      ! Now the relaxation parameters are changed based on the strain rate
      ! ---------------------- LES Part -----------------------------------------

      mout( 2) = s_mrt( 2) * (m( 2) - rho*usq)
      mout( 3) = s_mrt( 3) *  m( 3)
      mout( 5) = s_mrt( 5) *  m( 5)
      mout( 7) = s_mrt( 7) *  m( 7)
      mout( 9) = s_mrt( 9) *  m( 9)
      mout(10) = s_mrt(10) * (m(10) - meq10)
      mout(11) = s_mrt(11) *  m(11)
      mout(12) = s_mrt(10) * (m(12) - meq12) ! s_mrt(12) = s_mrt(10)
      mout(13) = s_mrt(13) *  m(13)
      mout(14) = s_mrt(14) * (m(14) - rho*u_x*u_y)
      mout(15) = s_mrt(14) * (m(15) - rho*u_y*u_z)! s_mrt(15) = s_mrt(14)
      mout(16) = s_mrt(14) * (m(16) - rho*u_x*u_z)! s_mrt(16) = s_mrt(14)
      mout(17) = s_mrt(17) *  m(17)
      mout(18) = s_mrt(18) *  m(18)
      mout(19) = s_mrt(19) *  m(19)

      ! Transformation back to PDF
      outstate( ?SAVE?( 19,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &             f000 -( -div1_2*mout(2) + div1_6*mout(3))

! -------------------------------------------------------------------------------
c0 = - div1_18*mout(3) + div1_12*(mout(10) - mout(11))
      outstate( ?SAVE?(  4,1,iElem,QQ,nScalars,nElems,neigh )) = f100 -( c0 - 4.0_rk *mout(5) )
      outstate( ?SAVE?(  1,1,iElem,QQ,nScalars,nElems,neigh )) = fN00 -( c0 + 4.0_rk *mout(5) )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c1 =  - div1_18*mout(3)  - div1_24*(mout(10) - mout(11))
c2 =    div1_8 * (mout(12) - mout(13))

      outstate( ?SAVE?(  5,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &             f010 -( c1 + c2 - 4.0_rk*mout(7) )

      outstate( ?SAVE?(  2,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &             f0N0 -( c1 + c2 + 4.0_rk*mout(7) )

      outstate( ?SAVE?(  6,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &             f001 -( c1 - c2 - 4.0_rk*mout(9) )

      outstate( ?SAVE?(  3,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &             f00N -( c1 - c2 + 4.0_rk*mout(9) )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c3 =   div1_24*mout(2)  + div1_72*mout(3)  &
 &   + div1_48 * ( mout(10) + mout(11) )  &
 &   + div1_16 * ( mout(12) + mout(13) )

      outstate( ?SAVE?( 18,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  f110 -(c3 + mout(5) + mout(7) + mout(14) + mout(17) - mout(18))

      outstate( ?SAVE?( 15,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  fNN0 -(c3 - mout(5) - mout(7) + mout(14) - mout(17) + mout(18))

      outstate( ?SAVE?( 17,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  f1N0 -(c3 + mout(5) - mout(7) - mout(14) + mout(17) + mout(18))

      outstate( ?SAVE?( 16,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  fN10 -(c3 - mout(5) + mout(7) - mout(14) - mout(17) - mout(18))
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c4 =   div1_24*mout(2)  + div1_72*mout(3)  &
 &   + div1_48 * ( mout(10) + mout(11) ) &
 &   - div1_16 * ( mout(12) + mout(13) )

      outstate( ?SAVE?( 14,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  f101 -( c4 + mout(5) + mout(9) + mout(16) - mout(17)+mout(19))

      outstate( ?SAVE?( 11,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  fN0N -( c4 - mout(5) - mout(9) + mout(16) + mout(17)-mout(19))

      outstate( ?SAVE?( 12,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  f10N -( c4 + mout(5) - mout(9) - mout(16) - mout(17)-mout(19))

      outstate( ?SAVE?( 13,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        &  fN01 -( c4 - mout(5) + mout(9) - mout(16) + mout(17)+mout(19))
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c5 = div1_24*(mout(2)-mout(10)-mout(11)) + div1_72*mout(3)

      outstate( ?SAVE?( 10,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        & f011 -( c5 + mout(7) + mout(9) + mout(15) + mout(18) - mout(19))

      outstate( ?SAVE?(  7,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        & f0NN -( c5 - mout(7) - mout(9) + mout(15) - mout(18) + mout(19))

      outstate( ?SAVE?(  9,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        & f01N -( c5 + mout(7) - mout(9) - mout(15) + mout(18) + mout(19))

      outstate( ?SAVE?(  8,1,iElem,QQ,nScalars,nElems,neigh )) =                       &
        & f0N1 -( c5 - mout(7) + mout(9) - mout(15) - mout(18) - mout(19))
! -------------------------------------------------------------------------------

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_les
! ****************************************************************************** !


! ****************************************************************************** !
  !> Advection relaxation routine for the MRT model.
  !! This routine has roughly 205 FLOPS per element.
  !! 
?? copy :: compute_routineHeader( mrt_advRel_d3q19_incomp )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem
?? copy :: pdfTmp19( f )
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    ! MRT Variables
    real(kind=rk) :: meq2, meq10, meq12
    real(kind=rk) :: m2, m6, m8, m14, m15, m16
    ! m6, m8 are temporary var
?? copy :: var19( s_mrt )
    ! mout1 are temp var
?? copy :: var19( mout )
?? copy :: var19( sum )
    real(kind=rk) :: c0, c1, c2, c3, c4, c5, c6
    real(kind=rk) :: mout5_4, mout7_4, mout9_4
    real(kind=rk) :: sum_c1_c2, sub_c1_c2
    real(kind=rk) :: sum_5_17, sub_7_18, d1,d2,d3,d4
    real(kind=rk) :: sum_9_19, sub_5_17, e1,e2,e3,e4
    real(kind=rk) :: sum_7_18, sub_9_19, g1,g2,g3,g4
    ! ---------------------------------------------------------------------------

    s_mrt2  = fieldProp(1)%fluid%mrt( level )%s_mrt( 2) * div1_24
    s_mrt3  = fieldProp(1)%fluid%mrt( level )%s_mrt( 3) * div1_72
    s_mrt5  = fieldProp(1)%fluid%mrt( level )%s_mrt( 5) * div1_24
    s_mrt7  = fieldProp(1)%fluid%mrt( level )%s_mrt( 7) * div1_24
    s_mrt9  = fieldProp(1)%fluid%mrt( level )%s_mrt( 9) * div1_24
    s_mrt10 = fieldProp(1)%fluid%mrt( level )%s_mrt(10)
    s_mrt11 = fieldProp(1)%fluid%mrt( level )%s_mrt(11)
    s_mrt12 = fieldProp(1)%fluid%mrt( level )%s_mrt(12)
    s_mrt13 = fieldProp(1)%fluid%mrt( level )%s_mrt(13)
    s_mrt14 = fieldProp(1)%fluid%mrt( level )%s_mrt(14) * div1_4
    s_mrt15 = fieldProp(1)%fluid%mrt( level )%s_mrt(15) * div1_4
    s_mrt16 = fieldProp(1)%fluid%mrt( level )%s_mrt(16) * div1_4
    s_mrt17 = fieldProp(1)%fluid%mrt( level )%s_mrt(17) * div1_8
    s_mrt18 = fieldProp(1)%fluid%mrt( level )%s_mrt(18) * div1_8
    s_mrt19 = fieldProp(1)%fluid%mrt( level )%s_mrt(19) * div1_8

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1, nSolve

      !> First load all local values into temp array
      fN00 = inState(?FETCH?( qN00, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f0N0 = inState(?FETCH?( q0N0, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f00N = inState(?FETCH?( q00N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f100 = inState(?FETCH?( q100, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f010 = inState(?FETCH?( q010, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f001 = inState(?FETCH?( q001, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f0NN = inState(?FETCH?( q0NN, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f0N1 = inState(?FETCH?( q0N1, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f01N = inState(?FETCH?( q01N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f011 = inState(?FETCH?( q011, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fN0N = inState(?FETCH?( qN0N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f10N = inState(?FETCH?( q10N, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fN01 = inState(?FETCH?( qN01, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f101 = inState(?FETCH?( q101, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fNN0 = inState(?FETCH?( qNN0, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      fN10 = inState(?FETCH?( qN10, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f1N0 = inState(?FETCH?( q1N0, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f110 = inState(?FETCH?( q110, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))
      f000 = inState(?FETCH?( q000, 1, iElem, QQ, varSys%nScalars, nElems,neigh ))

?? copy :: ux_d3q19( u_x, f, 1._rk )
?? copy :: uy_d3q19( u_y, f, 1._rk )
?? copy :: uz_d3q19( u_z, f, 1._rk )

      ! Equilibrium moments
      ! non zero meq are only: 2, 10, 12, 14, 15, 16
      meq2  = u_x*u_x + u_y*u_y + u_z*u_z
      meq10 = 3.0_rk*u_x*u_x - meq2
      meq12 = u_y*u_y - u_z*u_z

      !pxy
      m14 = f110 + fNN0 - f1N0 - fN10
      mout14 = s_mrt14 * (m14 - u_x*u_y)
      !pyz
      m15 = f011 + f0NN - f01N - f0N1
      mout15 = s_mrt15 * (m15 - u_y*u_z)
      !pxz
      m16 = f101 + fN0N - f10N - fN01
      mout16 = s_mrt16 * (m16 - u_x*u_z)

      ! -------------------------------------------------------------------------
      ! now calculate mrt against the bgk
      m6 = f101 + fN0N + f10N + fN01
      m8 = f011 + f0NN + f01N + f0N1

      sum1   =   f110 + fNN0 + f1N0 + fN10
      m2     = - f000 + sum1 + m6 + m8
      mout2  =   s_mrt2  * (m2 - meq2)

      sum2 = f010 + f0N0
      sum3 = f001 + f00N
      sum4 = 2._rk * ( f100+ fN00 )
      sum5 = sum2 + sum3

      ! epsilon
      mout3 = ( 2._rk*(f000 - sum5) - sum4 + m2 ) * s_mrt3

      sum6 = sum1 + m6 - m8 * 2.0d0
      sum7 = sum4 - sum5
      !3pxx
      mout10 = ( sum7 + sum6 - meq10 ) * s_mrt10
      !3pixx
      mout11 = ( - sum7 + sum6 ) * s_mrt11

      sum8 = sum1 - m6
      sum9 = sum2 - sum3
      !pww
      mout12 = ( sum8 + sum9 - meq12 ) * s_mrt12
      !piww
      mout13 = ( sum8 - sum9 ) * s_mrt13

c1 = f110 - fNN0
c2 = f1N0 - fN10
c3 = f101 - fN0N
c4 = f10N - fN01

      sum10 = c1 + c2
      sum11 = c3 + c4
      !qx
      mout5  = ( sum10 + sum11 - 2.0_rk*(f100 - fN00) ) * s_mrt5
      !mx
      mout17 = ( sum10 - sum11 ) * s_mrt17

c5 = f011 - f0NN
c6 = f01N - f0N1

      sum12 = c1 - c2
      sum13 = c5 + c6
      !qy
      mout7  = ( sum12 + sum13 - 2.0_rk*(f010 - f0N0) ) * s_mrt7
      !my
      mout18 = ( - sum12 + sum13 ) * s_mrt18

      sum14 = c3 - c4
      sum15 = c5 - c6
      !qz
      mout9  = ( sum14 + sum15 - 2.0_rk*(f001 - f00N) ) * s_mrt9
      !mz
      mout19 = ( sum14 - sum15 ) * s_mrt19

      ! Transformation back to PDF
      outstate( ?SAVE?( 19,1,iElem,QQ,QQ,nElems,neigh )) = f000 + 12._rk*(mout2-mout3)

! -------------------------------------------------------------------------------
      c0 = - 4._rk*mout3 + div1_12*(mout10 - mout11)
      mout5_4 = mout5 * 4.0d0
      outstate( ?SAVE?(  4,1,iElem,QQ,QQ,nElems,neigh )) = f100 - ( c0 - mout5_4 )
      outstate( ?SAVE?(  1,1,iElem,QQ,QQ,nElems,neigh )) = fN00 - ( c0 + mout5_4 )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      c1 = - 4._rk*mout3 - div1_24*(mout10 - mout11)
      c2 = div1_8 *(mout12 - mout13)

      sum_c1_c2 = c1 + c2
      mout7_4 = mout7 * 4.0d0
      outstate( ?SAVE?(  5,1,iElem,QQ,QQ,nElems,neigh)) = f010 - (sum_c1_c2 - mout7_4)
      outstate( ?SAVE?(  2,1,iElem,QQ,QQ,nElems,neigh)) = f0N0 - (sum_c1_c2 + mout7_4)

      sub_c1_c2 = c1 - c2
      mout9_4 = mout9 * 4.0d0
      outstate( ?SAVE?(  6,1,iElem,QQ,QQ,nElems,neigh)) = f001 - (sub_c1_c2 - mout9_4)
      outstate( ?SAVE?(  3,1,iElem,QQ,QQ,nElems,neigh)) = f00N - (sub_c1_c2 + mout9_4)
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      mout1 = mout2 + mout3
      c3 =   mout1  + div1_48 * (mout10 + mout11)   &
       &            + div1_16 * (mout12 + mout13)
      sum_5_17 = mout5 + mout17
      sub_7_18 = mout7 - mout18

      d1 = c3 + mout14
      d2 = sum_5_17 + sub_7_18
      outstate( ?SAVE?( 18,1,iElem,QQ,QQ,nElems,neigh)) = f110-(d1+d2)
      outstate( ?SAVE?( 15,1,iElem,QQ,QQ,nElems,neigh)) = fNN0-(d1-d2)

      d3 = c3 - mout14
      d4 = sum_5_17 - sub_7_18
      outstate( ?SAVE?( 17,1,iElem,QQ,QQ,nElems,neigh)) = f1N0-(d3+d4)
      outstate( ?SAVE?( 16,1,iElem,QQ,QQ,nElems,neigh)) = fN10-(d3-d4)
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      c4 = c3 - div1_8*(mout12+mout13)

      sum_9_19 = mout9 + mout19
      sub_5_17 = mout5 - mout17

      e1 = c4 + mout16
      e2 = sum_9_19 + sub_5_17
      outstate( ?SAVE?( 14,1,iElem,QQ,QQ,nElems,neigh )) = f101 - ( e1 + e2 )
      outstate( ?SAVE?( 11,1,iElem,QQ,QQ,nElems,neigh )) = fN0N - ( e1 - e2 )

      e3 = c4 - mout16
      e4 = sum_9_19 - sub_5_17
      outstate( ?SAVE?( 12,1,iElem,QQ,QQ,nElems,neigh )) = f10N - ( e3 - e4 )
      outstate( ?SAVE?( 13,1,iElem,QQ,QQ,nElems,neigh )) = fN01 - ( e3 + e4 )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
      c5 = mout1 - div1_24*(mout10+mout11)
      sum_7_18 = mout7 + mout18
      sub_9_19 = mout9 - mout19

      g1 = c5 + mout15
      g2 = sum_7_18 + sub_9_19
      outstate( ?SAVE?( 10,1,iElem,QQ,QQ,nElems,neigh )) = f011 - ( g1 + g2 )
      outstate( ?SAVE?(  7,1,iElem,QQ,QQ,nElems,neigh )) = f0NN - ( g1 - g2 )

      g3 = c5 - mout15
      g4 = sum_7_18 - sub_9_19
      outstate( ?SAVE?(  9,1,iElem,QQ,QQ,nElems,neigh )) = f01N - ( g3 + g4 )
      outstate( ?SAVE?(  8,1,iElem,QQ,QQ,nElems,neigh )) = f0N1 - ( g3 - g4 )
! -------------------------------------------------------------------------------

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_incomp
! ****************************************************************************** !

! ****************************************************************************** !
  !> Advection relaxation routine for the incompressible MRT les model
  !! This routine has roughly 380 FLOPS per element.
  !!
?? copy :: compute_routineHeader( mrt_advRel_d3q19_incomp_les )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem
?? copy :: pdfTmp19( f )
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    ! MRT Variables
    real(kind=rk) :: meq2, meq10, meq12
    ! real(kind=rk) :: s_mrt( 2:QQ )
    real(kind=rk) :: s_mrt2, s_mrt3, s_mrt5, s_mrt7, s_mrt9, s_mrt10, s_mrt11
    real(kind=rk) :: s_mrt13, s_mrt17, s_mrt18, s_mrt19
    real(kind=rk) :: m( QQ )
    real(kind=rk) :: mout( 2:QQ )
    real(kind=rk) :: c0, c1, c2, c3, c4, c5, c6, c7
    ! Variables required for LES
    real(kind=rk) :: Q
    real(kind=rk) :: cs2_dt2_18
    real(kind=rk) :: tau0, tau_total
    ! variables for fNeq calculation
    real(kind=rk) :: usqn, usqn_o2
    ! real(kind=rk) :: coeff_1, coeff_2
    real(kind=rk) :: ui1,  fac_1,  sum1_2
    real(kind=rk) ::       fac_2,  sum2_2
    real(kind=rk) :: ui3,  fac_3,  sum3_2
    real(kind=rk) ::       fac_4,  sum4_2
    real(kind=rk) ::       fac_9,  sum9_2
    real(kind=rk) :: ui10, fac_10, sum10_2
    real(kind=rk) :: ui11, fac_11, sum11_2
    real(kind=rk) :: ui12, fac_12, sum12_2
    real(kind=rk) :: ui13, fac_13, sum13_2
    real(kind=rk) :: fNeq_NE_SW
    real(kind=rk) :: fNeq_NW_SE
    real(kind=rk) :: fNeq_BW_TE
    real(kind=rk) :: fNeq_BE_TW
    real(kind=rk) :: fNeq_BS_TN
    real(kind=rk) :: fNeq_BN_TS
    real(kind=rk) :: fNeq_N_S
    real(kind=rk) :: fNeq_W_E
    real(kind=rk) :: fNeq_T_B
    real(kind=rk) :: s(6)
    ! ---------------------------------------------------------------------------

    s_mrt2  = fieldProp(1)%fluid%mrt( level )%s_mrt( 2)
    s_mrt3  = fieldProp(1)%fluid%mrt( level )%s_mrt( 3)
    s_mrt5  = fieldProp(1)%fluid%mrt( level )%s_mrt( 5) * div1_24
    s_mrt7  = fieldProp(1)%fluid%mrt( level )%s_mrt( 7) * div1_24
    s_mrt9  = fieldProp(1)%fluid%mrt( level )%s_mrt( 9) * div1_24
    s_mrt11 = fieldProp(1)%fluid%mrt( level )%s_mrt(11)
    s_mrt13 = fieldProp(1)%fluid%mrt( level )%s_mrt(13)
    s_mrt17 = fieldProp(1)%fluid%mrt( level )%s_mrt(17) * div1_8
    s_mrt18 = fieldProp(1)%fluid%mrt( level )%s_mrt(18) * div1_8
    s_mrt19 = fieldProp(1)%fluid%mrt( level )%s_mrt(19) * div1_8

    tau0       = 1._rk / fieldProp(1)%fluid%omLvl( level )
    cs2_dt2_18 = 18._rk &
      &    * fieldProp(1)%fluid%les%c_s * fieldProp(1)%fluid%les%c_s &
      &    * fieldProp(1)%fluid%dtLvl(level) * fieldProp(1)%fluid%dtLvl(level)

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem=1,nSolve
      !> First load all local values into temp array
      fN00 = inState( ?FETCH?(  1, 1, iElem, QQ, QQ, nElems,neigh ))
      f0N0 = inState( ?FETCH?(  2, 1, iElem, QQ, QQ, nElems,neigh ))
      f00N = inState( ?FETCH?(  3, 1, iElem, QQ, QQ, nElems,neigh ))
      f100 = inState( ?FETCH?(  4, 1, iElem, QQ, QQ, nElems,neigh ))
      f010 = inState( ?FETCH?(  5, 1, iElem, QQ, QQ, nElems,neigh ))
      f001 = inState( ?FETCH?(  6, 1, iElem, QQ, QQ, nElems,neigh ))
      f0NN = inState( ?FETCH?(  7, 1, iElem, QQ, QQ, nElems,neigh ))
      f0N1 = inState( ?FETCH?(  8, 1, iElem, QQ, QQ, nElems,neigh ))
      f01N = inState( ?FETCH?(  9, 1, iElem, QQ, QQ, nElems,neigh ))
      f011 = inState( ?FETCH?( 10, 1, iElem, QQ, QQ, nElems,neigh ))
      fN0N = inState( ?FETCH?( 11, 1, iElem, QQ, QQ, nElems,neigh ))
      f10N = inState( ?FETCH?( 12, 1, iElem, QQ, QQ, nElems,neigh ))
      fN01 = inState( ?FETCH?( 13, 1, iElem, QQ, QQ, nElems,neigh ))
      f101 = inState( ?FETCH?( 14, 1, iElem, QQ, QQ, nElems,neigh ))
      fNN0 = inState( ?FETCH?( 15, 1, iElem, QQ, QQ, nElems,neigh ))
      fN10 = inState( ?FETCH?( 16, 1, iElem, QQ, QQ, nElems,neigh ))
      f1N0 = inState( ?FETCH?( 17, 1, iElem, QQ, QQ, nElems,neigh ))
      f110 = inState( ?FETCH?( 18, 1, iElem, QQ, QQ, nElems,neigh ))
      f000 = inState( ?FETCH?( 19, 1, iElem, QQ, QQ, nElems,neigh ))

      !pxy
      m(14) = f110+ fNN0- f1N0- fN10
      !pyz
      m(15) = f011+ f0NN- f01N- f0N1
      !pxz
      m(16) = f101+ fN0N- f10N- fN01

      ! local x-, y- and z-velocity
?? copy :: ux_d3q19( u_x, f, 1._rk )
?? copy :: uy_d3q19( u_y, f, 1._rk )
?? copy :: uz_d3q19( u_z, f, 1._rk )

      ! Equilibrium moments
      ! non zero meq are only: 2, 10, 12, 14, 15, 16
      meq2  = u_x*u_x + u_y*u_y + u_z*u_z
      meq10 = 3.0_rk*u_x*u_x - meq2
      meq12 = u_y*u_y - u_z*u_z

      ! -------------------------------------------------------------------------
      ! now calculate mrt against the bgk
m(6) = f101 + fN0N + f10N + fN01
m(8) = f011 + f0NN + f01N + f0N1
c0   = f110 + fNN0 + f1N0 + fN10

      m(2) =  - f000 + c0 + m(6) + m(8)

m(1) = f010 + f0N0
m(4) = f001 + f00N
c7   = 2._rk * ( f100+ fN00 )

      !epsilon
      m(3) =  f000 - 2._rk*( m(1) + m(4) ) - c7 + c0 + m(6) + m(8)

      !3pxx
      m(10) = - m(1) - m(4) + c0 + m(6) + c7 - m(8) * 2.0_rk
      !3pixx
      m(11) =   m(1) + m(4) + c0 + m(6) - c7 - m(8) * 2.0_rk
      !pww
      m(12) =   m(1) - m(4) + c0 - m(6)
      !piww
      m(13) = - m(1) + m(4) + c0 - m(6)

c1 = f110 - fNN0
c2 = f1N0 - fN10
c3 = f101 - fN0N
c4 = f10N - fN01

      !qx
      m( 5) = -2.0_rk*(f100- fN00) + c1 + c2 + c3 + c4
      !mx
      m(17) = c1 + c2 - c3 - c4

c5 = f011 - f0NN
c6 = f01N - f0N1

      !qy
      m( 7) = -2.0_rk*(f010- f0N0) + c1 - c2 + c5 + c6
      !my
      m(18) =-c1 + c2 + c5 + c6

      !qz
      m( 9) = -2.0_rk*(f001- f00N) + c3 - c4 + c5 - c6
      !mz
      m(19) = c3 - c4 - c5 + c6

      ! ---------------------- LES Part -----------------------------------------
      ! square velocity and derived constants
      ! rho0 is assume to be 1.0
      rho  =   m(1) + m(4) + c7 * div1_2 + f000 + c0 + m(6) + m(8)
      ! meq2 == usq
      usqn = div1_36 * (rho - 1.5_rk * meq2)

      ! coeff_1 = div1_8

      ui1     = u_x + u_y
      fac_1   = div1_8 * ui1
      sum1_2  = fac_1 * ui1 + usqn
      fNeq_NE_SW = f110 + fNN0 - sum1_2*2.0_rk

      ui3     = -u_x + u_y
      fac_3   = div1_8 * ui3
      sum3_2  = fac_3 * ui3 + usqn
      fNeq_NW_SE = fN10 + f1N0 - sum3_2*2.0_rk

      ui10    =  u_x + u_z
      fac_10  = div1_8 * ui10
      sum10_2 = fac_10 * ui10 + usqn
      fNeq_BW_TE = fN0N + f101 - sum10_2*2.0_rk

      ui12    = -u_x + u_z
      fac_12  = div1_8 * ui12
      sum12_2 = fac_12 * ui12 + usqn
      fNeq_BE_TW = f10N + fN01 - sum12_2*2.0_rk

      ui11    =  u_y + u_z
      fac_11  = div1_8 * ui11
      sum11_2 = fac_11 * ui11 + usqn
      fNeq_BS_TN = f0NN + f011 - sum11_2*2.0_rk

      ui13    = -u_y + u_z
      fac_13  = div1_8 * ui13
      sum13_2 = fac_13 * ui13 + usqn
      fNeq_BN_TS = f01N + f0N1 - sum13_2*2.0_rk

      ! coeff_2 = div1_4
      usqn_o2 = 2.0_rk * usqn

      fac_2   = div1_4 * u_y
      sum2_2  = fac_2 * u_y + usqn_o2
      fNeq_N_S = f010 + f0N0 - sum2_2*2.0_rk

      fac_4   = div1_4 * u_x
      sum4_2  = fac_4 * u_x + usqn_o2
      fNeq_W_E = fN00 + f100 - sum4_2*2.0_rk

      fac_9   = div1_4 * u_z
      sum9_2  = fac_9 * u_z + usqn_o2
      fNeq_T_B = f001 + f00N - sum9_2*2.0_rk

      ! Calculate second moment of fNeq
      ! Sxx = (W+E) + (SW+NE) + (SE+NW) + (BW+TE) + (BE+TW)
      s(1) = fNeq_W_E + fNeq_NE_SW + fNeq_NW_SE + fNeq_BW_TE + fNeq_BE_TW

      ! Syy = (S+N) + (NE+SW) + (NW+SE) + (BS+TN) + (BN+TS)
      s(2) = fNeq_N_S + fNeq_NE_SW + fNeq_NW_SE + fNeq_BS_TN + fNeq_BN_TS

      ! Szz = (B+T) + (BS+TN) + (BN+TS) + (BW+TE) + (BE+TW)
      s(3) = fNeq_T_B + fNeq_BS_TN + fNeq_BN_TS + fNeq_BW_TE + fNeq_BE_TW

      ! Sxy = (SW+NE) - (NW+SE)
      s(4) = fNeq_NE_SW - fNeq_NW_SE

      ! Syz = (BS+TN) - (TS+BN)
      s(5) = fNeq_BS_TN - fNeq_BN_TS

      ! Sxz = (BW+TE) - (BE+TW)
      s(6) = fNeq_BW_TE - fNeq_BE_TW

      Q = sqrt(2._rk*(  s(1)*s(1)         &
        &             + s(2)*s(2)         &
        &             + s(3)*s(3)         &
        &             + s(4)*s(4)*2._rk   &
        &             + s(5)*s(5)*2._rk   &
        &             + s(6)*s(6)*2._rk ) )

      ! cs2_dt2_18 = c_s * c_s * dt * dt * 18
      tau_total = 0.5d0 * ( tau0 + sqrt(tau0*tau0 + Q * cs2_dt2_18) )
      s_mrt10   = 1.0_rk / tau_total
      ! s_mrt(12) = s_mrt(10)
      ! s_mrt(14) = s_mrt(15) = s_mrt(16) == div1_4 * s_mrt(10)
      ! Now the relaxation parameters are changed based on the strain rate
      ! ---------------------- LES Part -----------------------------------------

      mout( 2) = s_mrt2  * (m( 2) - meq2)
      mout( 3) = s_mrt3  *  m( 3)
      mout( 5) = s_mrt5  *  m( 5)
      mout( 7) = s_mrt7  *  m( 7)
      mout( 9) = s_mrt9  *  m( 9)
      mout(10) = s_mrt10 * (m(10) - meq10)
      mout(11) = s_mrt11 *  m(11)
      mout(12) = s_mrt10 * (m(12) - meq12)
      mout(13) = s_mrt13 *  m(13)
      mout(14) = s_mrt10 * (m(14) - u_x*u_y) * div1_4
      mout(15) = s_mrt10 * (m(15) - u_y*u_z) * div1_4
      mout(16) = s_mrt10 * (m(16) - u_x*u_z) * div1_4
      mout(17) = s_mrt17 *  m(17)
      mout(18) = s_mrt18 *  m(18)
      mout(19) = s_mrt19 *  m(19)

      ! Transformation back to PDF
      outstate( ?SAVE?( 19,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &             f000 - ( -div1_2*mout(2) + div1_6*mout(3) )

! -------------------------------------------------------------------------------
c0 = - div1_18*mout(3) + div1_12*(mout(10) - mout(11))
      outstate( ?SAVE?(  4,1,iElem,QQ,QQ,nElems,neigh )) = f100 -( c0 - 4.0_rk *mout(5) )
      outstate( ?SAVE?(  1,1,iElem,QQ,QQ,nElems,neigh )) = fN00 -( c0 + 4.0_rk *mout(5) )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c1 =  - div1_18*mout(3) - div1_24*(mout(10) - mout(11))
c2 =    div1_8 * (mout(12) - mout(13))

      outstate( ?SAVE?(  5,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &             f010 -( c1 + c2 - 4.0_rk*mout(7) )

      outstate( ?SAVE?(  2,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &             f0N0 -( c1 + c2 + 4.0_rk*mout(7) )

      outstate( ?SAVE?(  6,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &             f001 -( c1 - c2 - 4.0_rk*mout(9) )

      outstate( ?SAVE?(  3,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &             f00N -( c1 - c2 + 4.0_rk*mout(9) )
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c3 =   div1_24*mout(2)  + div1_72*mout(3)  &
 &   + div1_48 * ( mout(10) + mout(11) )  &
 &   + div1_16 * ( mout(12) + mout(13) )

      outstate( ?SAVE?( 18,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  f110 -(c3 + mout(5) + mout(7) + mout(14) + mout(17) - mout(18))

      outstate( ?SAVE?( 15,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  fNN0 -(c3 - mout(5) - mout(7) + mout(14) - mout(17) + mout(18))

      outstate( ?SAVE?( 17,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  f1N0 -(c3 + mout(5) - mout(7) - mout(14) + mout(17) + mout(18))

      outstate( ?SAVE?( 16,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  fN10 -(c3 - mout(5) + mout(7) - mout(14) - mout(17) - mout(18))
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c4 =   div1_24*mout(2)  + div1_72*mout(3)  &
 &   + div1_48 * ( mout(10) + mout(11) ) &
 &   - div1_16 * ( mout(12) + mout(13) )

      outstate( ?SAVE?( 14,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  f101 -( c4 + mout(5) + mout(9) + mout(16) - mout(17)+mout(19))

      outstate( ?SAVE?( 11,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  fN0N -( c4 - mout(5) - mout(9) + mout(16) + mout(17)-mout(19))

      outstate( ?SAVE?( 12,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  f10N -( c4 + mout(5) - mout(9) - mout(16) - mout(17)-mout(19))

      outstate( ?SAVE?( 13,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        &  fN01 -( c4 - mout(5) + mout(9) - mout(16) + mout(17)+mout(19))
! -------------------------------------------------------------------------------

! -------------------------------------------------------------------------------
c5 = div1_24*(mout(2)-mout(10)-mout(11)) + div1_72*mout(3)

      outstate( ?SAVE?( 10,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        & f011 -( c5 + mout(7) + mout(9) + mout(15) + mout(18) - mout(19))

      outstate( ?SAVE?(  7,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        & f0NN -( c5 - mout(7) - mout(9) + mout(15) - mout(18) + mout(19))

      outstate( ?SAVE?(  9,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        & f01N -( c5 + mout(7) - mout(9) - mout(15) + mout(18) + mout(19))

      outstate( ?SAVE?(  8,1,iElem,QQ,QQ,nElems,neigh )) =                            &
        & f0N1 -( c5 - mout(7) + mout(9) - mout(15) - mout(18) - mout(19))
! -------------------------------------------------------------------------------

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_incomp_les
! ****************************************************************************** !


! ****************************************************************************** !
?? copy :: compute_routineHeader( mrt_advRel_d3q19_incomp_explicit )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem, iDir
    real(kind=rk) :: pdfTmp( QQ ) ! temporary local pdf values
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    ! MRT Variables
    real(kind=rk) :: s_mrt( QQ ), meq( QQ )
    real(kind=rk) :: mneq( QQ ), mom( QQ )
    real(kind=rk) :: fneq( QQ )
    ! ---------------------------------------------------------------------------

    s_mrt(:) = fieldProp(1)%fluid%mrt( level )%s_mrt(:)

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1,nSolve

      !> First load all local values into temp array
      do iDir = 1, QQ
        pdfTmp( iDir ) = inState( ?FETCH?( iDir, 1, iElem, QQ, QQ, nElems,neigh ))
      end do

      ! local density
      rho = sum( pdfTmp(1:QQ) )

      ! local x-, y- and z-velocity
      u_x = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(1,:)) )
      u_y = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(2,:)) )
      u_z = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(3,:)) )

      ! -------------------------------------------------------------------------
      ! Equilibrium moments
      meq(1:QQ) =  0.0_rk

      meq( 1) = rho
      meq( 2) = rho0 * ( u_x * u_x + u_y * u_y + u_z * u_z )
      meq( 4) = u_x*rho0
      meq( 6) = u_y*rho0
      meq( 8) = u_z*rho0
      meq(10) = rho0*(2.0_rk*u_x*u_x - ( u_y*u_y + u_z*u_z ))
      meq(12) = rho0*( u_y*u_y - u_z*u_z )
      meq(14) = rho0*u_x*u_y
      meq(15) = rho0*u_y*u_z
      meq(16) = rho0*u_x*u_z

      ! convert pdf into moment
      do iDir = 1, QQ
        mom(iDir) = sum( pdfTmp(:) * MMtrD3Q19(iDir,:) )
      end do

      ! compute neq moment
      mneq(:) = s_mrt(:) * ( mom(:) - meq(:) )

      ! compute fNeq
      do iDir = 1, QQ
        fneq(iDir) = sum( MMIvD3Q19(iDir,:) * mneq(:) )
        outState(?SAVE?(iDir,1,iElem,QQ,QQ,nElems,neigh)) = pdfTmp(iDir) - fneq(iDir)
      end do

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_incomp_explicit
! ****************************************************************************** !

! ****************************************************************************** !
?? copy :: compute_routineHeader( mrt_advRel_d3q19_explicit )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem, iDir
    real(kind=rk) :: pdfTmp( QQ ) ! temporary local pdf values
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    ! MRT Variables
    real(kind=rk) :: s_mrt( QQ ), meq( QQ )
    real(kind=rk) :: mneq( QQ ), mom( QQ )
    real(kind=rk) :: fneq( QQ )
    ! ---------------------------------------------------------------------------

    s_mrt(:) = fieldProp(1)%fluid%mrt( level )%s_mrt(:)

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1,nSolve

      !> First load all local values into temp array
      do iDir = 1, QQ
        pdfTmp( iDir ) = inState( ?FETCH?( iDir, 1, iElem, QQ, QQ, nElems,neigh ))
      end do

      ! local density
      rho = sum( pdfTmp(1:QQ) )

      ! local x-, y- and z-velocity
      u_x = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(1,:)) ) / rho
      u_y = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(2,:)) ) / rho
      u_z = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(3,:)) ) / rho

      ! -------------------------------------------------------------------------
      ! Equilibrium moments
      meq(1:QQ) =  0.0_rk

      meq( 1) = rho
      meq( 2) = rho * ( u_x * u_x + u_y * u_y + u_z * u_z )
      meq( 4) = u_x*rho
      meq( 6) = u_y*rho
      meq( 8) = u_z*rho
      meq(10) = rho*(2.0_rk*u_x*u_x - ( u_y*u_y + u_z*u_z ))
      meq(12) = rho*( u_y*u_y - u_z*u_z )
      meq(14) = rho*u_x*u_y
      meq(15) = rho*u_y*u_z
      meq(16) = rho*u_x*u_z

      ! convert pdf into moment
      do iDir = 1, QQ
        mom(iDir) = sum( pdfTmp(:) * MMtrD3Q19(iDir,:) )
      end do

      ! compute neq moment
      mneq(:) = s_mrt(:) * ( mom(:) - meq(:) )

      ! compute fNeq
      do iDir = 1, QQ
        fneq(iDir) = sum( MMIvD3Q19(iDir,:) * mneq(:) )
        outState(?SAVE?(iDir,1,iElem,QQ,QQ,nElems,neigh)) = pdfTmp(iDir) - fneq(iDir)
      end do

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_explicit
! ****************************************************************************** !

! ****************************************************************************** !
?? copy :: compute_routineHeader( mrt_advRel_d3q19_les_explicit )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem, iDir
    real(kind=rk) :: pdfTmp( QQ ) ! temporary local pdf values
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    real(kind=rk) :: usq, ucx
    ! MRT Variables
    real(kind=rk) :: s_mrt( QQ ), meq( QQ )
    real(kind=rk) :: mneq( QQ ), mom( QQ )
    real(kind=rk) :: fneq( QQ )
    real(kind=rk) :: feq( QQ ), neq( QQ )
    real(kind=rk) :: tau0, cs2_dt2_18, tau_total
    real(kind=rk) :: q_bgk(3,3), q_sum
    ! ---------------------------------------------------------------------------

    s_mrt(:) = fieldProp(1)%fluid%mrt( level )%s_mrt(:)
    tau0     = 1._rk / fieldProp(1)%fluid%omLvl( level )
    cs2_dt2_18 = 18._rk &
      &    * fieldProp(1)%fluid%les%c_s * fieldProp(1)%fluid%les%c_s &
      &    * fieldProp(1)%fluid%dtLvl(level) * fieldProp(1)%fluid%dtLvl(level)

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1,nSolve

      !> First load all local values into temp array
      do iDir = 1, QQ
        pdfTmp( iDir ) = inState( ?FETCH?( iDir, 1, iElem, QQ, QQ, nElems,neigh ))
      end do

      ! local density
      rho = sum( pdfTmp(1:QQ) )

      ! local x-, y- and z-velocity
      u_x = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(1,:)) ) / rho
      u_y = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(2,:)) ) / rho
      u_z = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(3,:)) ) / rho
      usq = u_x * u_x + u_y * u_y + u_z * u_z

      ! -------------------------------------------------------------------------
      ! Equilibrium moments
      meq(1:QQ) =  0.0_rk

      meq( 1) = rho
      meq( 2) = rho * usq
      meq( 4) = u_x*rho
      meq( 6) = u_y*rho
      meq( 8) = u_z*rho
      meq(10) = rho*(2.0_rk*u_x*u_x - ( u_y*u_y + u_z*u_z ))
      meq(12) = rho*( u_y*u_y - u_z*u_z )
      meq(14) = rho*u_x*u_y
      meq(15) = rho*u_y*u_z
      meq(16) = rho*u_x*u_z

      ! convert pdf into moment
      do iDir = 1, QQ
        mom(iDir) = sum( pdfTmp(:) * MMtrD3Q19(iDir,:) )
      end do

      ! ---------------------- LES Part -----------------------------------------
      do iDir = 1, QQ

        ! velocity times lattice unit velocity
        ucx = real( layout%fStencil%cxDir( 1, iDir ), kind=rk)*u_x           &
          & + real( layout%fStencil%cxDir( 2, iDir ), kind=rk)*u_y           &
          & + real( layout%fStencil%cxDir( 3, iDir ), kind=rk)*u_z

        ! calculate equilibrium density
        fEq( iDir ) = layout%weight( iDir ) * rho * ( 1.d0 + ucx*cs2inv &
          &         + ucx*ucx*t2cs4inv - usq*t2cs2inv )

      enddo

      ! Calculate the non-equilibrium part
      nEq(:) = pdfTmp(:) - fEq(:)

      ! Now calculate the stress tensor part
      q_bgk      = 0._rk
      q_bgk(1,1) = sum( real(layout%fStencil%cxDir( 1, :)                    &
        &                   *layout%fStencil%cxDir( 1, :),kind=rk)*nEq(:) )
      q_bgk(2,1) = sum( real(layout%fStencil%cxDir( 2, :)                    &
        &                   *layout%fStencil%cxDir( 1, :),kind=rk)*nEq(:) )
      q_bgk(1,2) = q_bgk(2,1)
      q_bgk(2,2) = sum( real(layout%fStencil%cxDir( 2, :)                    &
        &                   *layout%fStencil%cxDir( 2, :),kind=rk)*nEq(:) )
      q_bgk(3,3) = sum( real(layout%fStencil%cxDir( 3, :)                    &
        &                   *layout%fStencil%cxDir( 3, :),kind=rk)*nEq(:) )
      q_bgk(3,1) = sum( real(layout%fStencil%cxDir( 3, :)                    &
        &                   *layout%fStencil%cxDir( 1, :),kind=rk)*nEq(:) )
      q_bgk(1,3) = q_bgk(3,1)
      q_bgk(3,2) = sum( real(layout%fStencil%cxDir( 3, :)                    &
        &                   *layout%fStencil%cxDir( 2, :),kind=rk)*nEq(:) )
      q_bgk(2,3) = q_bgk(3,2)

      q_sum = sqrt(2._rk*(q_bgk(1,1)*q_bgk(1,1)                                &
        &               + q_bgk(2,1)*q_bgk(2,1)                                &
        &               + q_bgk(1,2)*q_bgk(1,2)                                &
        &               + q_bgk(2,2)*q_bgk(2,2)                                &
        &               + q_bgk(3,1)*q_bgk(3,1)                                &
        &               + q_bgk(3,2)*q_bgk(3,2)                                &
        &               + q_bgk(1,3)*q_bgk(1,3)                                &
        &               + q_bgk(2,3)*q_bgk(2,3)                                &
        &               + q_bgk(3,3)*q_bgk(3,3) ) )

      ! and the turbulent part of the viscosity
      ! cs2_dt2_18 = c_s * c_s * dt * dt * 18
      tau_total = 0.5_rk * ( tau0 + sqrt(tau0*tau0 + q_sum * cs2_dt2_18) )
      s_mrt( 10 )     = 1.0_rk / tau_total
      s_mrt( 12 )     = 1.0_rk / tau_total
      s_mrt( 14:16 )  = 1.0_rk / tau_total
      ! Now the relaxation parameters are changed based on the strain rate
      ! ---------------------- LES Part -----------------------------------------

      ! compute neq moment
      mneq(:) = s_mrt(:) * ( mom(:) - meq(:) )

      ! compute fNeq
      do iDir = 1, QQ
        fneq(iDir) = sum( MMIvD3Q19(iDir,:) * mneq(:) )
        outState(?SAVE?(iDir,1,iElem,QQ,QQ,nElems,neigh)) = pdfTmp(iDir) - fneq(iDir)
      end do

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_les_explicit
! ****************************************************************************** !

! ****************************************************************************** !
?? copy :: compute_routineHeader( mrt_advRel_d3q19_incomp_les_explicit )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem, iDir
    real(kind=rk) :: pdfTmp( QQ ) ! temporary local pdf values
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    real(kind=rk) :: usq, ucx
    ! MRT Variables
    real(kind=rk) :: s_mrt( QQ ), meq( QQ )
    real(kind=rk) :: mneq( QQ ), mom( QQ )
    real(kind=rk) :: fneq( QQ )
    real(kind=rk) :: feq( QQ ), neq( QQ )
    real(kind=rk) :: tau0, cs2_dt2_18, tau_total
    real(kind=rk) :: q_bgk(3,3), q_sum
    ! ---------------------------------------------------------------------------

    s_mrt(:) = fieldProp(1)%fluid%mrt( level )%s_mrt(:)
    tau0     = 1._rk / fieldProp(1)%fluid%omLvl( level )
    cs2_dt2_18 = 18._rk &
      &    * fieldProp(1)%fluid%les%c_s * fieldProp(1)%fluid%les%c_s &
      &    * fieldProp(1)%fluid%dtLvl(level) * fieldProp(1)%fluid%dtLvl(level)

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1,nSolve

      !> First load all local values into temp array
      do iDir = 1, QQ
        pdfTmp( iDir ) = inState( ?FETCH?( iDir, 1, iElem, QQ, QQ, nElems,neigh ))
      end do

      ! local density
      rho = sum( pdfTmp(1:QQ) )

      ! local x-, y- and z-velocity
      u_x = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(1,:)) )
      u_y = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(2,:)) )
      u_z = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(3,:)) )
      usq = u_x * u_x + u_y * u_y + u_z * u_z

      ! -------------------------------------------------------------------------
      ! Equilibrium moments
      meq(1:QQ) = 0.0_rk

      meq( 1) = rho
      meq( 2) = rho0 * usq
      meq( 4) = rho0 * u_x
      meq( 6) = rho0 * u_y
      meq( 8) = rho0 * u_z
      meq(10) = rho0 * ( 2.0_rk*u_x*u_x - ( u_y*u_y + u_z*u_z ) )
      meq(12) = rho0 * ( u_y*u_y - u_z*u_z )
      meq(14) = rho0 * u_x*u_y
      meq(15) = rho0 * u_y*u_z
      meq(16) = rho0 * u_x*u_z

      ! convert pdf into moment
      do iDir = 1, QQ
        mom(iDir) = sum( pdfTmp(:) * MMtrD3Q19(iDir,:) )
      end do

      ! ---------------------- LES Part -----------------------------------------
      do iDir = 1, QQ

        ! velocity times lattice unit velocity
        ucx =   dble( layout%fStencil%cxDir( 1, iDir ) ) * u_x &
          &   + dble( layout%fStencil%cxDir( 2, iDir ) ) * u_y &
          &   + dble( layout%fStencil%cxDir( 3, iDir ) ) * u_z

        ! calculate equilibrium density
        fEq( iDir ) = layout%weight( iDir ) * ( rho + rho0*( ucx*cs2inv      &
          &         + ucx*ucx*t2cs4inv - usq*t2cs2inv ) )

      enddo

      ! Calculate the non-equilibrium part
      nEq(:) = pdfTmp(:) - fEq(:)

      ! Now calculate the stress tensor part
      q_bgk      = 0._rk
      q_bgk(1,1) = sum( real(layout%fStencil%cxDir( 1, :)                    &
        &                   *layout%fStencil%cxDir( 1, :),kind=rk)*nEq(:) )
      q_bgk(2,1) = sum( real(layout%fStencil%cxDir( 2, :)                    &
        &                   *layout%fStencil%cxDir( 1, :),kind=rk)*nEq(:) )
      q_bgk(1,2) = q_bgk(2,1)
      q_bgk(2,2) = sum( real(layout%fStencil%cxDir( 2, :)                    &
        &                   *layout%fStencil%cxDir( 2, :),kind=rk)*nEq(:) )
      q_bgk(3,3) = sum( real(layout%fStencil%cxDir( 3, :)                    &
        &                   *layout%fStencil%cxDir( 3, :),kind=rk)*nEq(:) )
      q_bgk(3,1) = sum( real(layout%fStencil%cxDir( 3, :)                    &
        &                   *layout%fStencil%cxDir( 1, :),kind=rk)*nEq(:) )
      q_bgk(1,3) = q_bgk(3,1)
      q_bgk(3,2) = sum( real(layout%fStencil%cxDir( 3, :)                    &
        &                   *layout%fStencil%cxDir( 2, :),kind=rk)*nEq(:) )
      q_bgk(2,3) = q_bgk(3,2)

      q_sum = sqrt(2._rk*(q_bgk(1,1)*q_bgk(1,1)                                &
        &               + q_bgk(2,1)*q_bgk(2,1)                                &
        &               + q_bgk(1,2)*q_bgk(1,2)                                &
        &               + q_bgk(2,2)*q_bgk(2,2)                                &
        &               + q_bgk(3,1)*q_bgk(3,1)                                &
        &               + q_bgk(3,2)*q_bgk(3,2)                                &
        &               + q_bgk(1,3)*q_bgk(1,3)                                &
        &               + q_bgk(2,3)*q_bgk(2,3)                                &
        &               + q_bgk(3,3)*q_bgk(3,3) ) )

      ! and the turbulent part of the viscosity
      ! cs2_dt2_18 = c_s * c_s * dt * dt * 18
      tau_total = 0.5_rk * ( tau0 + sqrt(tau0*tau0 + q_sum * cs2_dt2_18) )
      s_mrt( 10 )     = 1.0_rk / tau_total
      s_mrt( 12 )     = 1.0_rk / tau_total
      s_mrt( 14:16 )  = 1.0_rk / tau_total
      ! Now the relaxation parameters are changed based on the strain rate
      ! ---------------------- LES Part -----------------------------------------

      ! compute neq moment
      mneq(:) = s_mrt(:) * ( mom(:) - meq(:) )

      ! compute fNeq
      do iDir = 1, QQ
        fneq(iDir) = sum( MMIvD3Q19(iDir,:) * mneq(:) )
        outState(?SAVE?(iDir,1,iElem,QQ,QQ,nElems,neigh)) = pdfTmp(iDir) - fneq(iDir)
      end do

    enddo nodeloop

  end subroutine mrt_advRel_d3q19_incomp_les_explicit
! ****************************************************************************** !


! **************************************************************************** !
!> Unoptimized explicit implementation
?? copy :: compute_routineHeader( mrt_advRel_explicit )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem, iDir
    real(kind=rk) :: pdfTmp( layout%fStencil%QQ ) ! temporary local pdf values
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    real(kind=rk) :: usq, ucx
    real(kind=rk) :: fEq( layout%fStencil%QQ ) !< equilibrium distribution
    ! MRT Variables
    real(kind=rk) :: fneq( layout%fStencil%QQ )
    integer :: QQ
    ! ---------------------------------------------------------------------------
    QQ = layout%fStencil%QQ

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1,nSolve

      !> First load all local values into temp array
      do iDir = 1, QQ
        pdfTmp( iDir ) = inState( ?FETCH?( iDir, 1, iElem, QQ, QQ, nElems,neigh ))
      end do

      ! local density
      rho = sum( pdfTmp(1:QQ) )

      ! local x-, y- and z-velocity
      u_x = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(1,:)) ) / rho
      u_y = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(2,:)) ) / rho
      u_z = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(3,:)) ) / rho
      usq = u_x*u_x + u_y*u_y + u_z*u_z

      ! -------------------------------------------------------------------------
      ! Calculate the equilibrium distribution function
      do iDir = 1, QQ

        ! velocity times lattice unit velocity
        ucx =   layout%fStencil%cxDirRK(1,iDir) * u_x       &
          &   + layout%fStencil%cxDirRK(2,iDir) * u_y       &
          &   + layout%fStencil%cxDirRK(3,iDir) * u_z

        ! calculate equilibrium density
        fEq( iDir ) = layout%weight( iDir ) * rho * ( 1.d0 + ucx*cs2inv        &
          &         + ucx*ucx*t2cs4inv - usq*t2cs2inv )

      enddo

      ! compute fNeq
      fneq = pdfTmp - fEq
      fneq = matmul( fieldProp(1)%fluid%mrt(level)%omegaMoments, fneq )

      do iDir = 1, QQ
        outState(?SAVE?(iDir,1,iElem,QQ,QQ,nElems,neigh)) &
          & = pdfTmp(iDir) - fneq(iDir)
      end do

    enddo nodeloop

  end subroutine mrt_advRel_explicit
! **************************************************************************** !

! **************************************************************************** !
!> Unoptimized explicit implementation
?? copy :: compute_routineHeader( mrt_advRel_incomp_explicit )
?? copy :: compute_routineParams
    ! ---------------------------------------------------------------------------
    integer       :: iElem, iDir
    real(kind=rk) :: pdfTmp( layout%fStencil%QQ ) ! temporary local pdf values
    real(kind=rk) :: rho     ! local density
    real(kind=rk) :: u_x     ! local x-velocity
    real(kind=rk) :: u_y     ! local y-velocity
    real(kind=rk) :: u_z     ! local z-velocity
    real(kind=rk) :: usq, ucx
    real(kind=rk) :: fEq( layout%fStencil%QQ ) !< equilibrium distribution
    ! MRT Variables
    real(kind=rk) :: fneq( layout%fStencil%QQ )
    integer :: QQ
    ! ---------------------------------------------------------------------------
    QQ = layout%fStencil%QQ

    !NEC$ ivdep
?? IF (SOA) THEN
?? copy :: dir_vector
?? ELSE
?? copy :: dir_novec
?? END IF
    nodeloop: do iElem = 1,nSolve

      !> First load all local values into temp array
      do iDir = 1, QQ
        pdfTmp( iDir ) = inState( ?FETCH?( iDir, 1, iElem, QQ, QQ, nElems,neigh ))
      end do

      ! local density
      rho = sum( pdfTmp(1:QQ) )

      ! local x-, y- and z-velocity
      u_x = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(1,:)) )
      u_y = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(2,:)) )
      u_z = sum( pdfTmp(1:QQ) * dble(layout%fStencil%cxDir(3,:)) )
      usq = u_x*u_x + u_y*u_y + u_z*u_z

      ! -------------------------------------------------------------------------
      ! Calculate the equilibrium distribution function
      do iDir = 1, QQ

        ! velocity times lattice unit velocity
        ucx =   layout%fStencil%cxDirRK(1,iDir) * u_x       &
          &   + layout%fStencil%cxDirRK(2,iDir) * u_y       &
          &   + layout%fStencil%cxDirRK(3,iDir) * u_z

        ! calculate equilibrium density
        fEq( iDir ) = layout%weight( iDir ) * ( rho + rho0*( ucx*cs2inv &
          &         + ucx*ucx*t2cs4inv - usq*t2cs2inv ) )

      enddo

      ! compute fNeq
      fneq = pdfTmp - fEq
      fneq = matmul( fieldProp(1)%fluid%mrt(level)%omegaMoments, fneq )

      do iDir = 1, QQ
        outState(?SAVE?(iDir,1,iElem,QQ,QQ,nElems,neigh)) &
          & = pdfTmp(iDir) - fneq(iDir)
      end do

    enddo nodeloop

  end subroutine mrt_advRel_incomp_explicit
! **************************************************************************** !

end module mus_mrt_module
! ****************************************************************************** !
