! See copyright notice in the COPYRIGHT file.
! ************************************************************************** !
!> author: Kannan Masilamani
!! author: Jiaxing Qi
!! This module provides the MUSUBI specific functions for calculating
!! macroscopic quantities from the state variables for incompressible LBM
!! models.\n
!! Notice that only those quantities that related to density should have a
!! formula which differs from normal LBM model.
!!
!! The depending common interface between MUSUBI and ATELES is defined in the
!! tem_derived_module. The functionality for accessing a variable from the state
!! and evaluating a lua function are also provided in the tem_derived module.
!!
!! Do not use get_Element or get_Point routines to update the state !
!!

?? include 'header/lbm_macros.inc'
?? include 'header/lbm_deriveMacros.inc'
?? include 'header/lbm_interfaceMacros.inc'
?? include 'header/lbm_d3q19Macros.inc'
?? include 'treelm/source/deriveMacros.inc'
module mus_derQuanIncomp_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer

  ! include treelm modules
  use tem_param_module,         only: div1_2, div1_3, div1_54, div1_9, div3_4, &
    &                                 div1_36, div3_4h,                        &
    &                                 sqrt3, cs2inv, cs2, t2cs4inv, t2cs2inv,  &
    &                                 cs4inv, rho0, rho0Inv, q000
  use env_module,               only: rk, long_k, labelLen
  use tem_float_module,         only: operator(.feq.), operator(.fge.), &
    &                                 operator(.fle.)
  use tem_varSys_module,        only: tem_varSys_type, tem_varSys_op_type,     &
    &                                 tem_varSys_append_derVar,                &
    &                                 tem_varSys_proc_point,                   &
    &                                 tem_varSys_proc_element,                 &
    &                                 tem_varSys_proc_setParams,               &
    &                                 tem_varSys_proc_getParams,               &
    &                                 tem_varSys_proc_setupIndices,            &
    &                                 tem_varSys_proc_getValOfIndex,           &
    &                                 tem_varSys_getPoint_dummy,               &
    &                                 tem_varSys_getElement_dummy,             &
    &                                 tem_varSys_setupIndices_dummy,           &
    &                                 tem_varSys_getValOfIndex_dummy,          &
    &                                 tem_varSys_setParams_dummy,              &
    &                                 tem_varSys_getParams_dummy
  use tem_variable_module,      only: tem_variable_type
  use tem_stencil_module,       only: tem_stencilHeader_type
  use tem_topology_module,      only: tem_levelOf
  use tem_time_module,          only: tem_time_type
  use treelmesh_module,         only: treelmesh_type
  use tem_subTree_type_module,  only: tem_subTree_type, tem_treeIDfrom_subTree
  use tem_aux_module,           only: tem_abort
  use tem_logging_module,       only: logUnit
  use tem_operation_var_module, only: tem_evalMag_forElement,      &
    &                                 tem_evalMag_forPoint,        &
    &                                 tem_evalMag_fromIndex,       &
    &                                 tem_opVar_setupIndices,      &
    &                                 tem_get_new_varSys_data_ptr, &
    &                                 tem_opVar_setParams,         &
    &                                 tem_opVar_getParams
  use tem_debug_module,         only: dbgUnit
  use tem_grow_array_module,    only: grw_labelarray_type, append


  ! include musubi modules
  use mus_source_var_module,         only: mus_source_op_type
  use mus_pdf_module,                only: pdf_data_type
  use mus_varSys_module,             only: mus_varSys_data_type,               &
    &                                      mus_varSys_solverData_type,         &
    &                                      mus_get_new_solver_ptr,             &
    &                                      mus_deriveVar_ForPoint,             &
    &                                      mus_generic_varFromPDF_fromIndex,   &
    &                                      mus_generic_fromPDF_forElement,     &
    &                                      mus_derive_fromPDF
  use mus_stateVar_module,           only: mus_accessVar_setupIndices,         &
    &                                      mus_stateVar_Fetch_fromIndex,       &
    &                                      mus_stateVar_Fetch_now_fromIndex,   &
    &                                      mus_access_stateFetch_ForElement,   &
    &                                      mus_access_stateFetch_now_ForElement
  use mus_scheme_header_module,      only: mus_scheme_header_type
  use mus_scheme_layout_module,      only: mus_scheme_layout_type
  use mus_scheme_type_module,        only: mus_scheme_type
  use mus_derivedQuantities_module2, only: secondMom
  use mus_derQuan_module,            only: deriveDensity,                      &
    &                                      deriveDensity_fromIndex,            &
    &                                      deriveMomentum,                     &
    &                                      deriveMomentum_fromIndex,           &
    &                                      deriveShearStress,                  &
    &                                      deriveShearMag,                     &
    &                                      deriveWSS2D,                        &
    &                                      deriveWSS3D,                        &
    &                                      deriveTemp,                         &
    &                                      deriveShearStressnNwtn,             &
    &                                      deriveShearRate,                    &
    &                                      deriveBndForce
  use mus_operation_var_module,      only: mus_opVar_setupIndices
  use mus_mrtRelaxation_module,      only: mus_set_mrtRelaxation, mus_mrt_type, &
    &                                      mus_alloc_mrt
  use mus_derVarPos_type_module,     only: mus_derVarPos_type
  use mus_physics_module,            only: mus_convertFac_type

  implicit none

  private

  public :: mus_append_derVar_lbmIncomp
  ! equilbrium from macro uses different interface defined in
  ! mus_variable_module
  public :: deriveEquilIncomp_FromMacro
  public :: deriveEquilIncomp_FromMacro_d3q19
  public :: deriveVelIncomp_FromState
  public :: deriveVelIncomp_FromState_d3q19
  public :: deriveVelIncomp_FromPreColState
  public :: deriveEqIncomp_FromState
  public :: deriveAuxIncomp_fromState
  public :: deriveEquilIncomp_fromAux

  ! source variables
  public :: derive_spongeOmegaIncomp
  public :: derive_forceIncomp
  public :: applySrc_spongeOmegaIncomp
  public :: applySrc_forceIncomp
  public :: applySrc_spongeViscIncomp
  public :: applySrc_spongeViscIncomp_MRT

contains


  ! ************************************************************************* !
  !> subroutine to add derive variables for incompressible LBM
  !! (schemekind = 'lbm_incomp') to the varsys.
  subroutine mus_append_derVar_lbmIncomp( varSys, solverData, schemeHeader, &
    &                                     stencil, fldLabel, derVarName )
    ! ---------------------------------------------------------------------- !
    !> global variable system
    type(tem_varSys_type), intent(inout)  :: varSys

    !> Contains pointer to solver data types
    type(mus_varSys_solverData_type), target, intent(in) :: solverData

    !> identifier of the scheme
    type(mus_scheme_header_type), intent(in)  :: schemeHeader

    !> compute stencil defintion
    type(tem_stencilHeader_type), intent(in)       :: stencil

    !> array of field label prefix. Size=nFields
    character(len=*), intent(in)              :: fldLabel

    !> array of derive physical variables
    type(grw_labelarray_type), intent(inout) :: derVarName
    ! ---------------------------------------------------------------------- !
    ! number of derive variables
    integer :: nDerVars, iVar, nComponents, addedPos, iIn
    logical :: wasAdded
    character(len=labelLen), allocatable ::  input_varname(:)
    character(len=labelLen)  ::  varName
    procedure(tem_varSys_proc_point), pointer :: get_point => NULL()
    procedure(tem_varSys_proc_element), pointer :: get_element => NULL()
    procedure(tem_varSys_proc_setParams), pointer :: set_params => null()
    procedure(tem_varSys_proc_getParams), pointer :: get_params => null()
    procedure(tem_varSys_proc_setupIndices), pointer :: &
      &                                      setup_indices => null()
    procedure(tem_varSys_proc_getValOfIndex), pointer :: &
      &                                       get_valOfIndex => null()
    type(c_ptr) :: method_data
    character(len=labelLen), allocatable :: derVarName_loc(:)
    ! ---------------------------------------------------------------------- !
    nullify(get_point, get_element, set_params, get_params, setup_indices, &
      &     get_valOfIndex)

    nDerVars = 16
    allocate(derVarName_loc(nDerVars))
    derVarName_loc = [ 'fetch_pdf      ', 'fetch_pdf_now  ', &
      &                'pressure       ', 'equilibrium    ', &
      &                'non_equilibrium', 'kinetic_energy ', &
      &                'shear_stress   ', 'strain_rate    ', &
      &                'shear_rate     ', 'wss            ', &
      &                'momentum       ', 'vel_mag        ', &
      &                'bnd_force      ', 'fetch_pdf      ', &
      &                'shear_mag      ', 'temperature    '  ]

    do iVar = 1, nDerVars
      call append(derVarName, derVarName_loc(iVar))
      ! set default pointers, overwrite if neccessary
      get_element => tem_varSys_getElement_dummy
      get_point => mus_deriveVar_ForPoint
      setup_indices => mus_opVar_setupIndices
      get_valOfIndex => tem_varSys_getvalOfIndex_dummy
      set_params => tem_varSys_setParams_dummy
      get_params => tem_varSys_getParams_dummy
      method_data  = mus_get_new_solver_ptr(solverData)

      select case(trim(adjustl(derVarName_loc(iVar))))
      case ('fetch_pdf')
        get_element => mus_access_stateFetch_ForElement
        get_valOfIndex => mus_stateVar_Fetch_fromIndex
        setup_indices => mus_accessVar_setupIndices
        nComponents = stencil%QQ
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('fetch_pdf_now')
        get_element => mus_access_stateFetch_now_ForElement
        get_valOfIndex => mus_stateVar_Fetch_now_fromIndex
        setup_indices => mus_accessVar_setupIndices
        nComponents = stencil%QQ
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('pressure')
        get_element => derivePressureIncomp
        get_valOfIndex => derivePressureIncomp_fromIndex
        nComponents = 1
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('bnd_force')
        get_element => deriveBndForce
        nComponents = 3
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('equilibrium')
        get_element => deriveEquilIncomp
        get_valOfIndex => deriveEquilIncomp_fromIndex
        nComponents = stencil%QQ
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('non_equilibrium')
        get_element => deriveNonEquilIncomp
        get_valOfIndex => deriveNonEquilIncomp_fromIndex
        nComponents = stencil%QQ
        allocate(input_varname(1))
        input_varname(1) = 'fetch_pdf_now'

      case ('kinetic_energy')
        get_element => deriveKeIncomp
        get_ValOfIndex => deriveKEIncomp_fromIndex
        nComponents = 1
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('temperature')
        get_element => deriveTemp
        nComponents = 1
        allocate(input_varname(0))

      case ('shear_stress')
        nComponents = 6
        if (trim(schemeHeader%kind) == 'lbm_incomp') then
          get_element => deriveShearStress
          allocate(input_varname(1))
          input_varname(1) = 'non_equilibrium'
        end if
        if (trim(schemeHeader%kind) == 'lbm_incomp_nNwtn') then
          get_element => deriveShearStressnNwtn
          allocate(input_varname(2))
          input_varname(1) = 'non_equilibrium'
          input_varname(2) = 'omega'
        end if

      case ('strain_rate')
        nComponents = 6
        if (trim(schemeHeader%kind) == 'lbm_incomp') then
          get_element => deriveStrainRateIncomp
          get_ValOfIndex => deriveStrainRateIncomp_fromIndex
          allocate(input_varname(1))
          input_varname(1) = 'fetch_pdf_now'
        end if
        if (trim(schemeHeader%kind) == 'lbm_incomp_nNwtn') then
          get_element => deriveStrainRateIncompnNwtn
          allocate(input_varname(2))
          input_varname(1) = 'pdf'
          input_varname(2) = 'omega'
        end if

      case ('shear_rate')
        get_element => deriveShearRate
        nComponents = 1
        allocate(input_varname(1))
        input_varname(1) = 'strain_rate'

      case ('wss')
        nComponents = 1
        allocate(input_varname(1))
        input_varname(1) = 'shear_stress'
        if (stencil%nDims == 2) then
          get_element => deriveWSS2D
        else if (stencil%nDims == 3) then
          get_element => deriveWSS3D
        else
          write(logUnit(1),*) 'WARNING: WSS does not support 1D'
        end if

      case ('momentum')
        get_element => deriveMomentum
        get_valOfIndex => deriveMomentum_fromIndex
        nComponents = 3 
        allocate(input_varname(1))
        input_varname(1) = 'pdf'

      case ('vel_mag')
        get_element => tem_evalMag_forElement
        get_point => tem_evalMag_forPoint
        get_valOfIndex => tem_evalMag_fromIndex
        setup_indices => tem_opVar_setupIndices
        set_params => tem_opVar_setParams 
        get_params => tem_opVar_getParams
        method_data = tem_get_new_varSys_data_ptr(method_data)
        nComponents = 1
        allocate(input_varname(1))
        input_varname(1) = 'velocity'

      case ('shear_mag')
        get_element => deriveShearMag
        nComponents = 1
        allocate(input_varname(1))
        input_varname(1) = 'shear_stress'

      case default
        write(logUnit(1),*) 'WARNING: Unknown variable: '//&
          &                 trim(derVarName_loc(iVar))
        cycle !go to next variable
      end select

      ! update variable names with field label
      varname = trim(fldLabel)//trim(adjustl(derVarName_loc(iVar)))
      do iIn = 1, size(input_varname)
        input_varname(iIn) = trim(fldLabel)//trim(input_varname(iIn))
      end do

      ! append variable to varSys
      call tem_varSys_append_derVar( me             = varSys,         &
        &                            varName        = trim(varname),  &
        &                            nComponents    = nComponents,    &
        &                            input_varname  = input_varname,  &
        &                            method_data    = method_data,    &
        &                            get_point      = get_point,      &
        &                            get_element    = get_element,    &
        &                            set_params     = set_params,     &
        &                            get_params     = get_params,     &
        &                            setup_indices  = setup_indices,  &
        &                            get_valOfIndex = get_valOfIndex, &
        &                            pos            = addedPos,       &
        &                            wasAdded       = wasAdded        )

      if (wasAdded) then
        write(logUnit(10),*) ' Appended variable: '//trim(varname)
      else if (addedpos < 1) then
        write(logUnit(1),*) 'Error: variable '//trim(varname)// &
          &                 ' is not added to variable system'
      end if

      deallocate(input_varname)
    end do

  end subroutine mus_append_derVar_lbmIncomp
  ! **************************************************************************** !


! ************************************************************************** !
!       Subroutines with common interface for the function pointers            !
! ************************************************************************** !


! ************************************************************************** !
  !> Calculate the pressure of a given set of elements (sum up all links).
  !!
  !! Pressure calculation according to the isentropic equation of state for 
  !! the LBM \( p = ( \rho - \rho_0 ) c_s^2 \)
  !! with the calculation of density as in deriveDensity
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(derivePressureIncomp)
    ! ---------------------------------------------------------------------- !
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_derivePressureIncomp

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )

  end subroutine derivePressureIncomp
! ************************************************************************** !


! ************************************************************************** !
  !> Calculate the velocity on given elements for incompressible model
  !! \( \vec{u} = \frac{\sum(f_i c_i)}{\rho_0} \)
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(deriveVelocityIncomp)
    ! ---------------------------------------------------------------------- !
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------
    fnCalcPtr => mus_deriveVelocityIncomp

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )

  end subroutine deriveVelocityIncomp
! ************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of equlibrium
  !! This routine sets the function Pointer for equlibrium calcualtion and calls 
  !! the generice get Element from PDF routine 
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(deriveEquilIncomp)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------
    fnCalcPtr => mus_deriveEquilIncomp

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )

  end subroutine deriveEquilIncomp
! ************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of NonEquil
  !! This routine sets the function Pointer for NonEquil calcualtion and calls
  !! the generice get Element from PDF routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(deriveNonEquilIncomp)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------
    fnCalcPtr => mus_deriveNonEquilIncomp

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )

  end subroutine deriveNonEquilIncomp
! ************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of StrainRate
  !! This routine sets the function Pointer for StrainRate calcualtion and calls
  !! the generice get Element from PDF routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(deriveStrainRateIncomp)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------
    fnCalcPtr => mus_deriveStrainRateIncomp

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )


  end subroutine deriveStrainRateIncomp
! ****************************************************************************** !

! **************************************************************************** !
!       Subroutines with common interface for the function pointers            !
!                              getValOfIndex                                   !
! **************************************************************************** !

! ****************************************************************************** !
  !> Initiates the calculation of Velocity.
  !! This routine sets the function Pointer for velocity calcualtion and calls
  !! the generice get Value of Index routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !!
  recursive subroutine deriveVelocityIncomp_fromIndex( fun, varSys, time, &
     &                                    iLevel, idx, idxLen, nVals, res )
      !> Description of the method to obtain the variables, here some preset
      !! values might be stored, like the space time function to use or the
      !! required variables.
      class(tem_varSys_op_type), intent(in) :: fun

      !> The variable system to obtain the variable from.
      type(tem_varSys_type), intent(in)     :: varSys

      !> Point in time at which to evaluate the variable.
      type(tem_time_type), intent(in)       :: time

      !> Level on which values are requested
      integer, intent(in)                   :: iLevel

      !> Index of points in the growing array and variable val array to
      !! return.
      !! Size: most times nVals, if contiguous arrays are used it depends
      !! on the number of first indices
      integer, intent(in)                   :: idx(:)

      !> With idx as start index in contiguous memory,
      !! idxLength defines length of each contiguous memory
      !! Size: dependes on number of first index for contiguous array,
      !! but the sum of all idxLen is equal to nVals
      integer, optional, intent(in)         :: idxLen(:)

      !> Number of values to obtain for this variable (vectorized access).
      integer, intent(in)                   :: nVals

      !> Resulting values for the requested variable.
      !!
      !! Dimension: n requested entries x nComponents of this variable
      !! Access: (iElem-1)*fun%nComponents + iComp
      real(kind=rk), intent(out)            :: res(:)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_derivevelocityIncomp

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine deriveVelocityIncomp_fromIndex
! ****************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of Pressure.
  !! This routine sets the function Pointer for pressure calcualtion and calls
  !! the generice get Value of Index routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !!
  recursive subroutine derivePressureIncomp_fromIndex( fun, varSys, time, &
     &                                    iLevel, idx, idxLen, nVals, res )
      !> Description of the method to obtain the variables, here some preset
      !! values might be stored, like the space time function to use or the
      !! required variables.
      class(tem_varSys_op_type), intent(in) :: fun

      !> The variable system to obtain the variable from.
      type(tem_varSys_type), intent(in)     :: varSys

      !> Point in time at which to evaluate the variable.
      type(tem_time_type), intent(in)       :: time

      !> Level on which values are requested
      integer, intent(in)                   :: iLevel

      !> Index of points in the growing array and variable val array to
      !! return.
      !! Size: most times nVals, if contiguous arrays are used it depends
      !! on the number of first indices
      integer, intent(in)                   :: idx(:)

      !> With idx as start index in contiguous memory,
      !! idxLength defines length of each contiguous memory
      !! Size: dependes on number of first index for contiguous array,
      !! but the sum of all idxLen is equal to nVals
      integer, optional, intent(in)         :: idxLen(:)

      !> Number of values to obtain for this variable (vectorized access).
      integer, intent(in)                   :: nVals

      !> Resulting values for the requested variable.
      !!
      !! Dimension: n requested entries x nComponents of this variable
      !! Access: (iElem-1)*fun%nComponents + iComp
      real(kind=rk), intent(out)            :: res(:)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_derivePressureIncomp

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine derivePressureIncomp_fromIndex
! ****************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of equilibrium.
  !! This routine sets the function Pointer for equilibrium calcualtion and calls
  !! the generice get Value of Index routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !!
  recursive subroutine deriveEquilIncomp_fromIndex( fun, varSys, time, iLevel, &
     &                                                 idx, idxLen, nVals, res )
      !> Description of the method to obtain the variables, here some preset
      !! values might be stored, like the space time function to use or the
      !! required variables.
      class(tem_varSys_op_type), intent(in) :: fun

      !> The variable system to obtain the variable from.
      type(tem_varSys_type), intent(in)     :: varSys

      !> Point in time at which to evaluate the variable.
      type(tem_time_type), intent(in)       :: time

      !> Level on which values are requested
      integer, intent(in)                   :: iLevel

      !> Index of points in the growing array and variable val array to
      !! return.
      !! Size: most times nVals, if contiguous arrays are used it depends
      !! on the number of first indices
      integer, intent(in)                   :: idx(:)

      !> With idx as start index in contiguous memory,
      !! idxLength defines length of each contiguous memory
      !! Size: dependes on number of first index for contiguous array,
      !! but the sum of all idxLen is equal to nVals
      integer, optional, intent(in)         :: idxLen(:)

      !> Number of values to obtain for this variable (vectorized access).
      integer, intent(in)                   :: nVals

      !> Resulting values for the requested variable.
      !!
      !! Dimension: n requested entries x nComponents of this variable
      !! Access: (iElem-1)*fun%nComponents + iComp
      real(kind=rk), intent(out)            :: res(:)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_deriveEquilIncomp

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine deriveEquilIncomp_fromIndex
! ****************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of non_equilibrium.
  !! This routine sets the function Pointer for non_equilibrium calcualtion and 
  !! calls the generice get Value of Index routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !!
  recursive subroutine deriveNonEquilIncomp_fromIndex( fun, varSys, time, iLevel, idx, &
     &                                        idxLen, nVals, res )
      !> Description of the method to obtain the variables, here some preset
      !! values might be stored, like the space time function to use or the
      !! required variables.
      class(tem_varSys_op_type), intent(in) :: fun

      !> The variable system to obtain the variable from.
      type(tem_varSys_type), intent(in)     :: varSys

      !> Point in time at which to evaluate the variable.
      type(tem_time_type), intent(in)       :: time

      !> Level on which values are requested
      integer, intent(in)                   :: iLevel

      !> Index of points in the growing array and variable val array to
      !! return.
      !! Size: most times nVals, if contiguous arrays are used it depends
      !! on the number of first indices
      integer, intent(in)                   :: idx(:)

      !> With idx as start index in contiguous memory,
      !! idxLength defines length of each contiguous memory
      !! Size: dependes on number of first index for contiguous array,
      !! but the sum of all idxLen is equal to nVals
      integer, optional, intent(in)         :: idxLen(:)

      !> Number of values to obtain for this variable (vectorized access).
      integer, intent(in)                   :: nVals

      !> Resulting values for the requested variable.
      !!
      !! Dimension: n requested entries x nComponents of this variable
      !! Access: (iElem-1)*fun%nComponents + iComp
      real(kind=rk), intent(out)            :: res(:)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_deriveNonEquilIncomp

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine deriveNonEquilIncomp_fromIndex
! ****************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of kinetic_energy.
  !! This routine sets the function Pointer for kinetic_energy calcualtion and
  !! calls the generice get Value of Index routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !!
  recursive subroutine deriveKeIncomp_fromIndex( fun, varSys, time, iLevel,    &
     &                                                 idx, idxLen, nVals, res )
      !> Description of the method to obtain the variables, here some preset
      !! values might be stored, like the space time function to use or the
      !! required variables.
      class(tem_varSys_op_type), intent(in) :: fun

      !> The variable system to obtain the variable from.
      type(tem_varSys_type), intent(in)     :: varSys

      !> Point in time at which to evaluate the variable.
      type(tem_time_type), intent(in)       :: time

      !> Level on which values are requested
      integer, intent(in)                   :: iLevel

      !> Index of points in the growing array and variable val array to
      !! return.
      !! Size: most times nVals, if contiguous arrays are used it depends
      !! on the number of first indices
      integer, intent(in)                   :: idx(:)

      !> With idx as start index in contiguous memory,
      !! idxLength defines length of each contiguous memory
      !! Size: dependes on number of first index for contiguous array,
      !! but the sum of all idxLen is equal to nVals
      integer, optional, intent(in)         :: idxLen(:)

      !> Number of values to obtain for this variable (vectorized access).
      integer, intent(in)                   :: nVals

      !> Resulting values for the requested variable.
      !!
      !! Dimension: n requested entries x nComponents of this variable
      !! Access: (iElem-1)*fun%nComponents + iComp
      real(kind=rk), intent(out)            :: res(:)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_deriveKeIncomp

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine deriveKeIncomp_fromIndex
! ****************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of StrainRate.
  !! This routine sets the function Pointer for StrainRate calcualtion and
  !! calls the generice get Value of Index routine
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !!
  recursive subroutine deriveStrainRateIncomp_fromIndex( fun, varSys, time,    &
     &                                         iLevel, idx, idxLen, nVals, res )
      !> Description of the method to obtain the variables, here some preset
      !! values might be stored, like the space time function to use or the
      !! required variables.
      class(tem_varSys_op_type), intent(in) :: fun

      !> The variable system to obtain the variable from.
      type(tem_varSys_type), intent(in)     :: varSys

      !> Point in time at which to evaluate the variable.
      type(tem_time_type), intent(in)       :: time

      !> Level on which values are requested
      integer, intent(in)                   :: iLevel

      !> Index of points in the growing array and variable val array to
      !! return.
      !! Size: most times nVals, if contiguous arrays are used it depends
      !! on the number of first indices
      integer, intent(in)                   :: idx(:)

      !> With idx as start index in contiguous memory,
      !! idxLength defines length of each contiguous memory
      !! Size: dependes on number of first index for contiguous array,
      !! but the sum of all idxLen is equal to nVals
      integer, optional, intent(in)         :: idxLen(:)

      !> Number of values to obtain for this variable (vectorized access).
      integer, intent(in)                   :: nVals

      !> Resulting values for the requested variable.
      !!
      !! Dimension: n requested entries x nComponents of this variable
      !! Access: (iElem-1)*fun%nComponents + iComp
      real(kind=rk), intent(out)            :: res(:)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_deriveStrainRateIncomp

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine deriveStrainRateIncomp_fromIndex
! ****************************************************************************** !

! **************************************************************************** !
!                           Calculation routines                               !
! **************************************************************************** !


! ****************************************************************************** !
  !> author: Jiaxing Qi
  !! Calculate the strain rate ( or rate of strain, or rate of deformation)
  !! for nonNewtonian model.
  !!
  !! The following needs to be updated!
  !! The formula is:
  !! \[
  !!  \tau_{\alpha \beta}=
  !!    -\frac{3\omega}{2{\rho}_0} \sum_{i} f^{neq}_{i} c_{i\alpha} c_{i\beta}
  !! \]
  !! where \( \tau_{\alpha \beta}\) is the stress
  !! in the \(\beta\)-direction on a face normal to the \(\alpha\)-axis,\n
  !! \( f^{neq}_i = f_i - f^{eq}_i\) is the non-equilibrium pdf.\n
  !! For more information, please refer to: equation 45 in\n
  !! Krueger T, Varnik F, Raabe D. Shear stress in lattice Boltzmann
  !! simulations. Physical Review E. 2009;79(4):1-14.\n
  !!
  !! For multi-level mesh, Omega on finer level needs to be adjusted in order to
  !! get the correct shearstress calculation.\n
  !! First, we defines c as the dx ratio between finer and coarse level.\n
  !! \( c={ \Delta dx }_{ c }/{ \Delta dx }_{ f } \)
  !! Then the viscosity on the different levels must satisfy:\n
  !! \( \frac { { \nu  }_{ f } }{ { \nu  }_{ c } } =c \)
  !! This constrain leads to a relationship of omega on different levels:\n
  !! \( {\omega}_f = \frac {1}{ {\lambda}(\frac{1}{{\omega}_c}-0.5)+0.5 } \)
  !! For more information, please refer to:\n
  !! Manuel H, Harald K, Joerg B, Sabine R. Aeroacoustic validation of the
  !! lattice boltzmann method on non-uniform grids. ECCOMAS 2012
  !!
?? copy :: get_element_headtxt(deriveStrainRateIncompnNwtn)
    ! ---------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: iLevel, iElem, iComp, iDir
    integer :: pdfPos, nCompsPDF, statePos, omegaPos, varPos, omegaVarPos
    real(kind=rk) :: dens, vel(3), omega
    real(kind=rk), allocatable :: nonEq(:)
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk), allocatable :: fEq(:)
    real(kind=rk), allocatable :: tau(:)
    integer :: nSize, nScalars
    ! ---------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    omegaPos = fun%input_varPos(2)
    omegaVarPos = varSys%method%val( omegaPos )%state_varPos(1)
    nScalars = varSys%nScalars

    allocate( tmpPDF( nCompsPDF ) )
    allocate( fEq( nCompsPDF ) )
    allocate( nonEq( nCompsPDF ) )
    allocate( tau( fun%nComponents ) )

    do iElem = 1, nElems

      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      ! transfer PDF to tmpPDF
      ! use FETCH to get pre-collision non-Eq
      do iComp = 1, nCompsPDF
        varPos = varSys%method%val(pdfPos)%state_varPos(iComp)
        tmpPDF( iComp ) = scheme%state( iLevel )%val(                       &
&?FETCH?(iComp,1,statePos,nCompsPDF,nScalars,nSize,scheme%pdf(iLevel)%neigh), &
          & scheme%pdf( iLevel )%nNow )
      end do !iComp

      ! computes density and velocity
      dens   = sum(tmpPDF)
      vel(1) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(3,:))

      ! computes equilibrium
?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)

      ! Non-Eq
      nonEq = tmpPDF - fEq

      ! get omega from state variable
      omega = scheme%state( iLevel )%val(                      &
        & ?IDX?( omegaVarPos, statePos, varSys%nScalars, nSize), &
        & scheme%pdf( iLevel )%nNow )

      ! compute shear stress
      tau(:) = secondMom( cxcx = scheme%layout%fStencil%cxcx(:,:), &
        &                 f    = nonEq(:),                         &
        &                 QQ   = scheme%layout%fStencil%QQ         )

      res( (iElem-1)*fun%nComponents+1: iElem*fun%nComponents ) = &
        &                   tau(:) * (-1.5_rk) * omega / rho0

    end do ! iElem

    deallocate( tmpPDF )
    deallocate( fEq )
    deallocate( nonEq )
    deallocate( tau )


  end subroutine deriveStrainRateIncompnNwtn
! ************************************************************************** !


! ****************************************************************************** !
  !> Initiates the calculation of kinetic energy
  !! This routine sets the function Pointer for kinetic energy calcualtion and
  !! calls the generice get Element from PDF routine 
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(deriveKeIncomp)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------
    fnCalcPtr => mus_deriveKeIncomp

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )

  end subroutine deriveKeIncomp
! ************************************************************************** !


! ****************************************************************************** !
   !> Derive sponge oemga variable defined as a source term.
?? copy :: get_element_headtxt(derive_spongeOmegaIncomp)
    ! ---------------------------------------------------------------------------
    call tem_abort('Not implemented')
    ! ---------------------------------------------------------------------------
  end subroutine derive_spongeOmegaIncomp
! ****************************************************************************** !


! ************************************************************************** !
   !> Derive external force variable defined as a source term.
   !! It evaluates spacetime function defined in lua file for force variable
   !! and convert it to state value which is to be added to the state
   !! @todo KM: Not use we need seperate force for incompressible lbm model
?? copy :: get_element_headtxt(derive_forceIncomp)
    ! ---------------------------------------------------------------------- !
    call tem_abort('Not implemented')
    ! ---------------------------------------------------------------------- !
  end subroutine derive_forceIncomp
! ************************************************************************** !

! **************************************************************************** !
   !> Update state with source variable "sponge_omega".
   !! Sponge_omega is used to increase/decrease viscosity in certain region.
   !! It just computes non-equilibrium from pre-collision pdf and equilibrium
   !! \[ f^{neq} = f^{pre} - f^{eq} \]
?? copy :: applySource_header(applySrc_spongeOmegaIncomp)
    ! --------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: spongeOmega(fun%elemLvl(iLevel)%nElems)
    real(kind=rk) :: fTmp(Q000), fEq(Q000)
    integer :: nElems, iElem, iDir, QQ, nScalars, posInTotal
    real(kind=rk) :: omega, omega_local
    real(kind=rk) :: dens, vel(3)
    ! --------------------------------------------------------------------------
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr )
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems
    ! get the correct omega value
    omega = scheme%field(1)%fieldProp%fluid%omLvl( iLevel )

    ! Get sponge_omega which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = spongeOmega                               )

    ! Subtract the spongeOmega from reference omega since non-equilibrium
    ! is already relaxred in compute kernel with reference omega
    ! spongeOmega = spongeOmega - omega

    ! constant parameter
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars

    fTmp = 0.0_rk
    fEq = 0.0_rk
    ! Get non-equil from pre- and post collision PDFs
    do iElem = 1, nElems

      ! Skip this element if spongeOmega is zero or large than omega
      if ((spongeOmega(iElem) .feq. 0.0_rk)    &
        & .or. (spongeOmega(iElem) .fge. omega)) cycle 

      ! Subtract the spongeOmega from background omega which is applied in 
      ! compute kernel 
      omega_local = spongeOmega(iElem)

      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      do iDir = 1, QQ
        fTmp(iDir) = inState(                    &
          & ?FETCH?( iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh ) ) 
      end do

      ! computes density and velocity
      dens   = sum(fTmp(1:QQ))
      vel(1) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(3,:))

      ! computes equilibrium
?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)

      do iDir = 1, QQ
        outState(?SAVE?(iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh) ) &
          !& = outState(                                                       &
          !& ?SAVE?(iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh) )      &
          !& - omega_local*( fTmp(iDir ) - fEq( iDir ))
          & = fTmp(iDir)- omega_local*( fTmp(iDir ) - fEq( iDir ))
      end do    
    end do
  end subroutine applySrc_spongeOmegaIncomp
! **************************************************************************** !

! **************************************************************************** !
   !> Update state with source variable "sponge_visc".
   !! Sponge_omega is used to increase/decrease viscosity in certain region.
   !! It just computes non-equilibrium from pre-collision pdf and equilibrium
   !! \[ f^{neq} = f^{pre} - f^{eq} \]
?? copy :: applySource_header(applySrc_spongeViscIncomp)
    ! --------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: spongeVisc(fun%elemLvl(iLevel)%nElems)
    real(kind=rk) :: fTmp(Q000), fEq(Q000)
    integer :: nElems, iElem, iDir, QQ, nScalars, posInTotal!, minLevel
    real(kind=rk) :: omega, omega_local, spongeOmega
    real(kind=rk) :: visc
    real(kind=rk) :: dens, vel(3)
    ! --------------------------------------------------------------------------
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr )
    scheme => fPtr%solverData%scheme

    !minLevel = tree%global%minLevel
    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get sponge_omega which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = spongeVisc                               )

    ! convert to lattice unit
    spongeVisc = spongeVisc / fPtr%solverData%physics%fac(iLevel)%visc

    ! actual viscosity on this level
    visc = scheme%field(1)%fieldProp%fluid%visc(iLevel)
    ! get the correct omega value
    omega = scheme%field(1)%fieldProp%fluid%omLvl( iLevel )

    ! constant parameter
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars

    fTmp = 0.0_rk
    fEq = 0.0_rk
    ! Get non-equil from pre- and post collision PDFs
    do iElem = 1, nElems

      ! Skip this element if spongeVisc is zero or spongeVisc le than actual visc
      if ((spongeVisc(iElem) .feq. 0.0_rk)    &
        & .or. (spongeVisc(iElem) .fle. visc)) cycle 

      ! convert viscosity to omega
      spongeOmega = 1.0_rk / (cs2inv*spongeVisc(iElem)+0.5_rk)  

      ! Also skip this element if spongeOmega is greater than local omega
      if (spongeOmega .fge. omega) cycle

      ! Subtract the spongeOmega from background omega which is applied in 
      ! compute kernel 
      omega_local = spongeOmega !- omega

      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      do iDir = 1, QQ
        fTmp(iDir) = inState(                    &
          & ?FETCH?( iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh ) ) 
      end do

      ! computes density and velocity
      dens   = sum(fTmp(1:QQ))
      vel(1) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(3,:))

      ! computes equilibrium
?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)

      do iDir = 1, QQ
        outState(?SAVE?(iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh) ) &
          !& = outState(                                                       &
          !& ?SAVE?(iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh) )      &
          !& - omega_local*( fTmp(iDir ) - fEq( iDir ))
          & = fTmp(iDir)- omega_local*( fTmp(iDir ) - fEq( iDir ))
      end do    
    end do

  end subroutine applySrc_spongeViscIncomp
! **************************************************************************** !

! **************************************************************************** !
   !> Update state with source variable "sponge_visc" for MRT collision. 
   !! Sponge_omega is used to increase/decrease viscosity in certain region.
   !! It just computes non-equilibrium from pre-collision pdf and equilibrium
   !! \[ f^{neq} = f^{pre} - f^{eq} \]
?? copy :: applySource_header(applySrc_spongeViscIncomp_MRT)
    ! --------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: spongeVisc(fun%elemLvl(iLevel)%nElems)
    real(kind=rk) :: fTmp(Q000), fEq(Q000), fnEq(Q000) 
    integer :: nElems, iElem, iDir, QQ, nScalars, posInTotal!, minLevel
    real(kind=rk) :: omega, omega_local, spongeOmega
    real(kind=rk) :: visc
    real(kind=rk) :: dens, vel(3)
    type(mus_mrt_type) :: mrt 
    ! --------------------------------------------------------------------------
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr )
    scheme => fPtr%solverData%scheme

    !minLevel = tree%global%minLevel
    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get sponge_omega which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = spongeVisc                               )

    ! convert to lattice unit
    spongeVisc = spongeVisc / fPtr%solverData%physics%fac(iLevel)%visc

    ! actual viscosity on this level
    visc = scheme%field(1)%fieldProp%fluid%visc(iLevel)
    ! get the correct omega value
    omega = scheme%field(1)%fieldProp%fluid%omLvl( iLevel )

    ! constant parameter
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars

    ! allocate mrt
    call mus_alloc_mrt(mrt, QQ)

    fTmp = 0.0_rk
    fEq = 0.0_rk
    ! Get non-equil from pre- and post collision PDFs
    do iElem = 1, nElems

      ! Skip this element if spongeVisc is zero or spongeVisc le than actual visc
      if ((spongeVisc(iElem) .feq. 0.0_rk)    &
        & .or. (spongeVisc(iElem) .fle. visc)) cycle 

      ! convert viscosity to omega
      spongeOmega = 1.0_rk / (cs2inv*spongeVisc(iElem)+0.5_rk)  

      ! Also skip this element if spongeOmega is greater than local omega
      if (spongeOmega .fge. omega) cycle

      ! Subtract the spongeOmega from background omega which is applied in 
      ! compute kernel 
      omega_local = spongeOmega !- omega

      call mus_set_mrtRelaxation(mrt          = mrt,                 &
        &                        omegaKine    = omega_local,         &
        &                        omegaBulk    = omega_local,         &
        &                        schemeHeader = scheme%header,       &
        &                        moment       = scheme%layout%moment )

      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      do iDir = 1, QQ
        fTmp(iDir) = inState(                    &
          & ?FETCH?( iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh ) ) 
      end do

      ! computes density and velocity
      dens   = sum(fTmp(1:QQ))
      vel(1) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(fTmp(1:QQ) * scheme%layout%fStencil%cxDirRK(3,:))

      ! computes equilibrium
?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)

      ! compute non-equilibrium
      fnEq(1:QQ) = matmul( mrt%omegaMoments, (fTmp(1:QQ)-fEq(1:QQ)) )
      do iDir = 1, QQ
        outState(?SAVE?(iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh) ) &
          & = fTmp(iDir) - fneq(iDir) 
      end do    
    end do

  end subroutine applySrc_spongeViscIncomp_MRT
! **************************************************************************** !

! ************************************************************************** !
   !> Update state with source variable "force"
   !! Force term used here is from:
   !! "Discrete lattice effects on the forcing term in the lattice Boltzmann
   !!  method", Zhaoli Guo, Chugung Zheng and Baochang Shi.
   !! In the paper, use force term is referred as Method 2 as:
   !! \[ F_i = w_i( (\vec{e}_i-\vec{u}*)/cs2 + 
   !!       (\vec{e}_i \cdot \vec{u}*)\vec{e}_i/cs4) \cdot \vec{F} \]
   !! 
   !! Force must be defined as body force per unit volume
   !! KM: If this force formula is used then velocity needs to be
   !! computed as u = \sum c_i f_i + \vec{F}/2
   !! Similar to derive routine but it updates the state whereas derive 
   !! is used for tracking
?? copy :: applySource_header(applySrc_forceIncomp)
    ! ---------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr 
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: forceField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: FF_elem(3)
    real(kind=rk) :: velocity(3), ucx, uMinusCX(3), forceTerm
    real(kind=rk), allocatable :: tmpPDF(:)
    integer :: nElems, iElem, iDir, QQ, nScalars
    integer :: posInTotal
    real(kind=rk) :: omega
    ! ---------------------------------------------------------------------- !
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme
    ! get the correct omega value
    omega = scheme%field(1)%fieldProp%fluid%omLvl( iLevel )

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = forceField                                )


!write(dbgUnit(1),*) 'ApplySrc_forceIncomp'    
!    do iElem = 1, nElems 
!      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
!      FF_elem = forceField((iElem-1)*3+1 : iElem*3)
!write(dbgUnit(1),*) 'treeID ', scheme%levelDesc(iLevel)%total(posInTotal) &
!  & , 'force ', FF_elem
!    end do

    ! convert physical to lattice
    forceField = forceField / fPtr%solverData%physics%fac(iLevel)%body_force

    ! constant parameter
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars

    allocate(tmpPDF(QQ))
    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      do iDir = 1, QQ
        tmpPDF( iDir ) =  inState(                                        &
          & ?FETCH?( iDir, 1, posInTotal, QQ, nScalars, nPdfSize, neigh ) )
      end do !iComp

      ! local velocity
      velocity(1) = dot_product( tmpPdf,               &
        &           scheme%layout%fStencil%cxDirRK(1,:)) 
      velocity(2) = dot_product( tmpPdf,               &
        &           scheme%layout%fStencil%cxDirRK(2,:)) 
      velocity(3) = dot_product( tmpPdf,               &
        &           scheme%layout%fStencil%cxDirRK(3,:)) 

      velocity = velocity / rho0

      ! force field on current element
      FF_elem = forceField((iElem-1)*3+1 : iElem*3)

      ! force term: 
      ! F_i = w_i( (\vec{e}_i-\vec{u}*)/cs2 + 
      !       (\vec{e}_i \cdot \vec{u}*)\vec{e}_i/cs4) \cdot \vec{F}
      do iDir = 1, QQ
        ucx = dot_product( scheme%layout%fStencil%cxDirRK(:, iDir), &
          &                velocity )
        uMinusCx = scheme%layout%fStencil%cxDirRK(:, iDir) - velocity

        forceTerm = dot_product( uMinusCx * cs2inv               &
          &       + ucx * scheme%layout%fStencil%cxDirRK(:,iDir) &
          &       * cs4inv, FF_elem )

        outState( ?SAVE?(iDir,1,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
          & = outState(                                                  &
          & ?SAVE?(iDir,1,posInTotal,QQ,nScalars,nPdfSize,neigh) )       &
          & + (1.0-omega*0.5_rk)*scheme%layout%weight( iDir ) * forceTerm

      end do

    end do !iElem
  end subroutine applySrc_forceIncomp
! ************************************************************************** !


! ************************************************************************** !
  !> This routine computes equilbrium from density and velocity
?? copy :: deriveFromMacro_header(deriveEquilIncomp_FromMacro)
    ! ---------------------------------------------------------------------- !
    real(kind=rk) :: fEq(layout%fStencil%QQ), vel(3), usq, ucx
    integer :: QQ, iElem, iDir
    ! ---------------------------------------------------------------------- !

    QQ = layout%fStencil%QQ
    do iElem = 1, nElems
      vel = velocity(:,iElem)

      usq = dot_product(vel, vel)*t2cs2inv

      do iDir = 1, QQ
        ucx = dot_product( layout%fStencil%cxDirRK(:, iDir), vel )

        ! calculate equilibrium density
        fEq( iDir ) = layout%weight( iDir ) * ( density(iElem) &
          & + ( cs2inv * ucx + ucx * ucx * t2cs4inv - usq ) )
      enddo

      res( (iElem-1)*QQ+1: iElem*QQ ) = fEq
    end do
  end subroutine deriveEquilIncomp_FromMacro
! ************************************************************************** !


! ************************************************************************** !
  !> This routine computes equilbrium from density and velocity
?? copy :: deriveFromMacro_header(deriveEquilIncomp_FromMacro_d3q19)
    ! ---------------------------------------------------------------------- !
    integer :: iElem
    real(kind=rk) :: fEq(19), u_x, u_y, u_z
?? copy :: fEq_d3q19_var
    ! ---------------------------------------------------------------------- !

    do iElem = 1, nElems

      u_x = velocity(1,iElem)
      u_y = velocity(2,iElem)
      u_z = velocity(3,iElem)

?? copy :: fEq_d3q19_a( fEq, u_x, u_y, u_z, density(iElem), rho0 )

      res( (iElem-1)*19 +  1) = fEq( 1)
      res( (iElem-1)*19 +  2) = fEq( 2)
      res( (iElem-1)*19 +  3) = fEq( 3)
      res( (iElem-1)*19 +  4) = fEq( 4)
      res( (iElem-1)*19 +  5) = fEq( 5)
      res( (iElem-1)*19 +  6) = fEq( 6)
      res( (iElem-1)*19 +  7) = fEq( 7)
      res( (iElem-1)*19 +  8) = fEq( 8)
      res( (iElem-1)*19 +  9) = fEq( 9)
      res( (iElem-1)*19 + 10) = fEq(10)
      res( (iElem-1)*19 + 11) = fEq(11)
      res( (iElem-1)*19 + 12) = fEq(12)
      res( (iElem-1)*19 + 13) = fEq(13)
      res( (iElem-1)*19 + 14) = fEq(14)
      res( (iElem-1)*19 + 15) = fEq(15)
      res( (iElem-1)*19 + 16) = fEq(16)
      res( (iElem-1)*19 + 17) = fEq(17)
      res( (iElem-1)*19 + 18) = fEq(18)
      res( (iElem-1)*19 + 19) = fEq(19)

    end do

  end subroutine deriveEquilIncomp_FromMacro_d3q19
! ************************************************************************** !

! ************************************************************************** !
  !> This routine computes equilbrium from auxField
?? copy :: deriveEquilFromAux_header(deriveEquilIncomp_fromAux)
    ! ---------------------------------------------------------------------- !
    real(kind=rk) :: rho, vel(3), usq, ucx
    integer :: QQ, iElem, iDir, elemOff
    integer :: dens_pos, vel_pos(3)
    ! ---------------------------------------------------------------------- !
    dens_pos = varSys%method%val(derVarPos%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos%velocity)%auxField_varPos(1:3)

    QQ = layout%fStencil%QQ
    !NEC$ ivdep
    do iElem = 1, nElems
      ! element offset
      elemoff = (iElem-1)*varSys%nAuxScalars
      ! density
      rho = auxField(elemOff + dens_pos)
      ! velocity
      vel(1) = auxField(elemOff + vel_pos(1))
      vel(2) = auxField(elemOff + vel_pos(2))
      vel(3) = auxField(elemOff + vel_pos(3))

      usq = ( vel(1)*vel(1) + vel(2)*vel(2) + vel(3)*vel(3) )*t2cs2inv

      !NEC$ shortloop
      do iDir = 1, QQ
        ucx = layout%fStencil%cxDirRK(1, iDir) * vel(1) &
          & + layout%fStencil%cxDirRK(2, iDir) * vel(2) &
          & + layout%fStencil%cxDirRK(3, iDir) * vel(3)

        ! calculate equilibrium density
        fEq( (iElem-1)*QQ + iDir ) = layout%weight( iDir ) * ( rho  &
          & + rho0*( cs2inv * ucx + ucx * ucx * t2cs4inv - usq ) )
      enddo

    end do
  end subroutine deriveEquilIncomp_fromAux
! ************************************************************************** !


! ************************************************************************** !
  !> This routine computes velocity from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveVelIncomp_FromState )
    integer :: iElem, iDir
    real(kind=rk) :: f( layout%fStencil%QQ )

    do iElem = 1, nElems
      do iDir = 1, layout%fStencil%QQ
        f(iDir) = state( iDir+(iElem-1)* varSys%nScalars )
      end do
      res( (iElem-1)*3+1 ) = sum( f * layout%fStencil%cxDirRK(1,:) )
      res( (iElem-1)*3+2 ) = sum( f * layout%fStencil%cxDirRK(2,:) )
      res( (iElem-1)*3+3 ) = sum( f * layout%fStencil%cxDirRK(3,:) )
    end do

  end subroutine deriveVelIncomp_FromState
! ************************************************************************** !


! ************************************************************************** !
  !> This routine computes velocity from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveVelIncomp_FromState_d3q19 )
    integer :: iElem
    real(kind=rk) :: f(18), u_x, u_y, u_z

    !NEC$ ivdep
    do iElem = 1, nElems

      f( 1) = state(  1+(iElem-1)*varSys%nScalars )
      f( 2) = state(  2+(iElem-1)*varSys%nScalars )
      f( 3) = state(  3+(iElem-1)*varSys%nScalars )
      f( 4) = state(  4+(iElem-1)*varSys%nScalars )
      f( 5) = state(  5+(iElem-1)*varSys%nScalars )
      f( 6) = state(  6+(iElem-1)*varSys%nScalars )
      f( 7) = state(  7+(iElem-1)*varSys%nScalars )
      f( 8) = state(  8+(iElem-1)*varSys%nScalars )
      f( 9) = state(  9+(iElem-1)*varSys%nScalars )
      f(10) = state( 10+(iElem-1)*varSys%nScalars )
      f(11) = state( 11+(iElem-1)*varSys%nScalars )
      f(12) = state( 12+(iElem-1)*varSys%nScalars )
      f(13) = state( 13+(iElem-1)*varSys%nScalars )
      f(14) = state( 14+(iElem-1)*varSys%nScalars )
      f(15) = state( 15+(iElem-1)*varSys%nScalars )
      f(16) = state( 16+(iElem-1)*varSys%nScalars )
      f(17) = state( 17+(iElem-1)*varSys%nScalars )
      f(18) = state( 18+(iElem-1)*varSys%nScalars )

?? copy :: vel_d3q19( u_x, u_y, u_z, f, 1.0_rk )

      res( (iElem-1)*3+1 ) = u_x
      res( (iElem-1)*3+2 ) = u_y
      res( (iElem-1)*3+3 ) = u_z

    end do

  end subroutine deriveVelIncomp_FromState_d3q19
! ************************************************************************** !


! ************************************************************************** !
  !> This routine computes velocity from precollision state array using FETCH
  !! macro. This must comply with interface in 
  !! [[mus_variable_module:derive_FromPreColState]]
?? copy :: deriveFromPreColState_header( deriveVelIncomp_FromPreColState )
    ! --------------------------------------------------------------------------
    integer :: iElem, iDir
    real(kind=rk) :: pdf( layout%fStencil%QQ ), vel(3)
    integer :: QQ, nScalars, nDims
    ! --------------------------------------------------------------------------
    QQ = layout%fStencil%QQ
    nScalars = varSys%nScalars
    nDims = layout%fStencil%nDims

    do iElem = 1, nElems
      do iDir = 1, QQ
        pdf(iDir) = state(                                          &
          & ?FETCH?(iDir, iField, iElem, QQ, nScalars, nSize, neigh))
      end do
      ! momentum
      vel( 1 ) = sum( pdf * layout%fStencil%cxDirRK(1,:) )
      vel( 2 ) = sum( pdf * layout%fStencil%cxDirRK(2,:) )
      vel( 3 ) = sum( pdf * layout%fStencil%cxDirRK(3,:) )
      ! return velocity field according on stencil dimensions
      res( (iElem-1)*nDims+1:iElem*nDims) = vel(1:nDims)
    end do
    ! convert to velocity
    res = res / rho0

  end subroutine deriveVelIncomp_FromPreColState
! ************************************************************************** !

! ************************************************************************** !
  !> This routine computes auxField from state array
  !! This must comply with interface in 
  !! [[mus_derVarPos_type_module:derive_AuxFromState]]
?? copy :: deriveAuxFromState_header( deriveAuxIncomp_fromState )
    ! -------------------------------------------------------------------------- !
    integer :: dens_pos, vel_pos(3), pdfPos
    integer :: iElem, iDir, elemOff
    real(kind=rk) :: pdf( stencil%QQ )
    ! -------------------------------------------------------------------------- !
    dens_pos = varSys%method%val(derVarPos%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos%velocity)%auxField_varPos(1:3)

    !NEC$ ivdep
    do iElem = 1, nElems
      !NEC$ shortloop
      do iDir = 1, stencil%QQ
        pdfPos = varSys%method%val(derVarPos%pdf)%state_varPos(iDir) 
        pdf(iDir) = state( ?IDX?(pdfPos, iElem, varSys%nScalars, nSize) )
      end do

      ! element offset
      elemoff = (iElem-1)*varSys%nAuxScalars

      ! density
      auxField(elemOff+dens_pos) = sum( pdf )
     
      ! velocity
      auxField(elemOff+vel_pos(1)) = sum( pdf * stencil%cxDirRK(1,:) ) * rho0Inv
      auxField(elemOff+vel_pos(2)) = sum( pdf * stencil%cxDirRK(2,:) ) * rho0Inv
      auxField(elemOff+vel_pos(3)) = sum( pdf * stencil%cxDirRK(3,:) ) * rho0Inv
    end do

  end subroutine deriveAuxIncomp_fromState
! ************************************************************************** !

! ************************************************************************** !
  !> This routine computes equilibirium from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
  !! Here it is assumed that rho0 = 1.0
?? copy :: deriveFromState_header( deriveEqIncomp_FromState )
    integer :: QQ, iElem, iDir
    real(kind=rk) :: f( layout%fStencil%QQ )
    real(kind=rk) :: rho, vel(3), usq, ucx

    QQ = layout%fStencil%QQ

    do iElem = 1, nElems

      do iDir = 1, QQ
        f(iDir) = state( iDir+(iElem-1)*varSys%nScalars )
      end do

      rho = sum( f )
      vel(1) = sum( f * layout%fStencil%cxDirRK(1,:) )
      vel(2) = sum( f * layout%fStencil%cxDirRK(2,:) )
      vel(3) = sum( f * layout%fStencil%cxDirRK(3,:) )

      usq = dot_product(vel, vel) * t2cs2inv

      do iDir = 1, QQ
        ucx = dot_product( layout%fStencil%cxDirRK(:, iDir), vel )

        ! calculate equilibrium density
        res( (iElem-1)*QQ+iDir ) = layout%weight( iDir ) * &
          &  ( rho + rho0 * ( cs2inv * ucx + ucx * ucx * t2cs4inv - usq ) )
      enddo

    end do

  end subroutine deriveEqIncomp_FromState
! ************************************************************************** !


! ************************************************************************** !
  !> Calculate the velocity on given elements for incompressible model
  !! \( \vec{u} = \frac{\sum(f_i c_i)}{\rho_0} \)
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
  recursive subroutine mus_deriveVelocityIncomp(fun, varsys, stencil, iLevel, &
    &                                                         pdf, res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer       :: fPtr
    type(mus_scheme_type), pointer            :: scheme
    real(kind=rk), allocatable                :: tmpPDF(:)
    integer                                   :: iComp, iVal, pdfPos, nCompsPDF
    ! ---------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    allocate( tmpPDF( nCompsPDF ) )
    res = 0.0_rk

    do iVal = 1, nVals
      tmpPDF = pdf( (iVal-1)*nCompsPDF+1 : iVal*nCompsPDF )
      do iComp = 1, fun%nComponents
        res( iComp+ (iVal-1)*fun%nComponents) =                 &
          &  sum(tmpPDF * scheme%layout%fStencil%cxDirRK(iComp,:))
      end do ! iComp
    end do ! iVal
    deallocate( tmpPDF )

  end subroutine mus_deriveVelocityIncomp
! ************************************************************************** !



! ************************************************************************** !
  !> Calculate the pressure of a given set of elements (sum up all links).
  !!
  !! Pressure calculation according to the isentropic equation of state for 
  !! the LBM \( p = ( \rho - \rho_0 ) c_s^2 \)
  !! with the calculation of density as in deriveDensity
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
  recursive subroutine mus_derivePressureIncomp(fun, varsys, stencil, iLevel, &
    &                                                         pdf, res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------------
    real(kind=rk), allocatable                :: tmpPDF(:)
    integer                                   :: pdfPos, nCompsPDF
    integer                                   :: iVal
    ! ---------------------------------------------------------------------------
    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    allocate( tmpPDF( nCompsPDF ) )
    res = 0.0_rk

    do iVal = 1, nVals
      tmpPDF = pdf( (iVal-1)*nCompsPDF+1 : iVal*nCompsPDF )
      ! convert pdf to density(sum of tmpPDF) and then density into pressure
      ! use only fluctuation part for incompressible pressure by subtracting
      ! reference density
      res(iVal) = ( sum(tmpPDF) - rho0 )  * cs2
    end do !iVal

  end subroutine mus_derivePressureIncomp
! ************************************************************************** !


! ************************************************************************** !
  !> Calculate the equlibrium of a given element number with the given input
  !! state vector for incompressible model
  !! \( \rho0 = 1.0 \)
  !! \( f_eq = w_\alpha*(\rho + \rho0( (u . c_i) + (u . c_i)^2 - u^2 ) ) \)
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
  recursive subroutine mus_deriveEquilIncomp(fun, varsys, stencil, iLevel, &
    &                                                      pdf, res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer       :: fPtr
    type(mus_scheme_type), pointer            :: scheme
    real(kind=rk), allocatable                :: tmpPDF(:)
    real(kind=rk), allocatable                :: fEq(:)
    real(kind=rk)                             :: dens, vel(3)
    integer                                   :: pdfPos, nCompsPDF, iVal, iDir
    ! ---------------------------------------------------------------------------
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    allocate( tmpPDF( nCompsPDF ) )
    allocate( fEq( fun%nComponents ) )
    res = 0.0_rk

    do iVal = 1, nVals
      tmpPDF = pdf( (iVal-1)*nCompsPDF+1 : iVal*nCompsPDF )
      dens   = sum(tmpPDF)
      vel(1) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(3,:))

?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)
      res( (iVal-1)*fun%nComponents+1: iVal*fun%nComponents ) = fEq
    end do !iVal
    deallocate( tmpPDF )
    deallocate( fEq )

  end subroutine mus_deriveEquilIncomp
! ************************************************************************** !


! ************************************************************************** !
  !> Calculate the Non-Equlibrium
  !!
  recursive subroutine mus_deriveNonEquilIncomp(fun, varsys, stencil, iLevel, &
    &                                                         pdf, res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer       :: fPtr
    type(mus_scheme_type), pointer            :: scheme
    real(kind=rk), allocatable                :: tmpPDF(:)
    real(kind=rk), allocatable                :: fEq(:)
    real(kind=rk)                             :: dens, vel(3)
    integer                                   :: iDir, iVal
    integer                                   :: pdfPos, nCompsPDF
    ! ---------------------------------------------------------------------------
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    allocate( fEq( fun%nComponents ) )
    allocate( tmpPDF( nCompsPDF ) )
    res = 0.0_rk

    do iVal = 1 , nVals
      tmpPDF = pdf( (iVal-1)*nCompsPDF+1 : iVal*nCompsPDF )
      ! computes density and velocity
      dens   = sum(tmpPDF)
      vel(1) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(3,:))

      ! computes equilibrium
?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)

      res( (iVal-1)*fun%nComponents+1: iVal*fun%nComponents ) = tmpPDF - fEq
    end do ! iVal

    deallocate( tmpPDF )
    deallocate( fEq )


  end subroutine mus_deriveNonEquilIncomp
! ************************************************************************** !


! ************************************************************************** !
  !> Calculate the kinetic energy as
  !! \[
  !!  E = \frac{1}{2} \rho ( u_x^2 + u_y^2 + u_z^2 )
  !! \]
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
  recursive subroutine mus_deriveKeIncomp(fun, varsys, stencil, iLevel, pdf,   &
    &                                                               res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer       :: fPtr
    type(mus_scheme_type), pointer            :: scheme
    real(kind=rk), allocatable                :: tmpPDF(:)
    real(kind=rk)                             :: vel(3)
    integer                                   :: pdfPos, nCompsPDF, iVal
    ! ---------------------------------------------------------------------------
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    allocate( tmpPDF( nCompsPDF ) )
    res = 0.0_rk

    do iVal = 1, nVals
      tmpPDF = pdf( (iVal-1)*nCompsPDF+1 : iVal*nCompsPDF )
      vel(1) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(3,:))
      res( iVal ) = sum( vel(:)*vel(:) ) * 0.5_rk 
    end do !iVal

    deallocate( tmpPDF )

  end subroutine mus_deriveKeIncomp
! ************************************************************************** !

! ************************************************************************** !
  !> author: Jiaxing Qi
  !! Calculate the strain rate ( or rate of strain, or rate of deformation)
  !!
  !! The formula is:
  !! \[
  !!  \tau_{\alpha \beta}=
  !!    -\frac{3\omega}{2{\rho}_0} \sum_{i} f^{neq}_{i} c_{i\alpha} c_{i\beta}
  !! \]
  !! where \( \tau_{\alpha \beta}\) is the stress
  !! in the \(\beta\)-direction on a face normal to the \(\alpha\)-axis,\n
  !! \( f^{neq}_i = f_i - f^{eq}_i\) is the non-equilibrium pdf.\n
  !! For more information, please refer to: equation 45 in\n
  !! Krueger T, Varnik F, Raabe D. Shear stress in lattice Boltzmann
  !! simulations. Physical Review E. 2009;79(4):1-14.\n
  !!
  !! For multi-level mesh, Omega on finer level needs to be adjusted in order to
  !! get the correct shearstress calculation.\n
  !! First, we defines c as the dx ratio between finer and coarse level.\n
  !! \( c={ \Delta dx }_{ c }/{ \Delta dx }_{ f } \)
  !! Then the viscosity on the different levels must satisfy:\n
  !! \( \frac { { \nu  }_{ f } }{ { \nu  }_{ c } } =c \)
  !! This constrain leads to a relationship of omega on different levels:\n
  !! \( {\omega}_f = \frac {1}{ {\lambda}(\frac{1}{{\omega}_c}-0.5)+0.5 } \)
  !! For more information, please refer to:\n
  !! Manuel H, Harald K, Joerg B, Sabine R. Aeroacoustic validation of the
  !! lattice boltzmann method on non-uniform grids. ECCOMAS 2012
  !!
  recursive subroutine mus_deriveStrainRateIncomp(fun, varsys, stencil, iLevel,&
      &                                                        pdf, res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: nCompsPDF, iVal, pdfPos, iDir
    real(kind=rk) :: dens, vel(3), omega
    real(kind=rk), allocatable :: nonEq(:)
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk), allocatable :: fEq(:)
    real(kind=rk), allocatable :: tau(:)
    integer :: nScalars
    ! ---------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    pdfPos = fun%input_varPos(1)
    nCompsPDF = varSys%method%val( pdfPos )%nComponents
    nScalars = varSys%nScalars

    allocate( tmpPDF( nCompsPDF ) )
    allocate( fEq( nCompsPDF ) )
    allocate( nonEq( nCompsPDF ) )
    allocate( tau( fun%nComponents ) )

    do iVal = 1, nVals
      tmpPDF = pdf( (iVal-1)*nCompsPDF+1 : iVal*nCompsPDF )

      ! computes density and velocity
      dens   = sum(tmpPDF)
      vel(1) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(1,:))
      vel(2) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(2,:))
      vel(3) = sum(tmpPDF * scheme%layout%fStencil%cxDirRK(3,:))

      ! computes equilibrium
?? copy :: Eq_incomp(fEq, dens, vel, scheme%layout, rho0)

      ! Non-Eq
      nonEq = tmpPDF - fEq

      ! get the correct omega value
      omega = scheme%field(1)%fieldProp%fluid%omLvl( iLevel )

      ! compute shear stress
      tau(:) = secondMom( cxcx = scheme%layout%fStencil%cxcx(:,:), &
        &                 f    = nonEq(:),                         &
        &                 QQ   = scheme%layout%fStencil%QQ         )

      res( (iVal-1)*fun%nComponents+1: iVal*fun%nComponents ) = &
        &                   tau(:) * (-1.5_rk) * omega / rho0

    end do ! iVal

    deallocate( tmpPDF )
    deallocate( fEq )
    deallocate( nonEq )
    deallocate( tau )

  end subroutine mus_deriveStrainRateIncomp
! ****************************************************************************** !

end module mus_derQuanIncomp_module
! ************************************************************************** !
