! See copyright notice in the COPYRIGHT file.
! ************************************************************************** !
!> author: Kannan Masilamani
!! Average Interpolation of flow quantities between different grid levels
!!
?? include 'header/lbm_macros.inc'
?? include 'header/lbm_deriveMacros.inc'
?? include 'header/lbm_interfaceMacros.inc'
?? include 'header/lbm_d2q9Macros.inc'
?? include 'treelm/source/logMacros.inc'
module mus_interpolate_average_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer

  ! include treelm modules
  use env_module,              only: rk
  use tem_aux_module,          only: tem_abort
  use tem_element_module,      only: eT_GhostFromCoarser, &
    &                                eT_ghostFromFiner
  use tem_param_module,        only: cs2inv, cs2, PI, div1_2, div1_9, div4_9,&
    &                                div1_36, div2_3, rho0
  use tem_comm_env_module,     only: tem_comm_env_type
  use tem_debug_module,        only: dbgUnit
  use tem_construction_module, only: tem_levelDesc_type
  use tem_matrix_module,       only: tem_matrix_type
  use tem_stencil_module,      only: tem_stencilHeader_type
  use tem_logging_module,      only: logUnit
  use tem_varSys_module,       only: tem_varSys_type
  use tem_time_module,         only: tem_time_type

?? IF (loglvl >= 1) THEN
  use env_module,            only: long_k
  use tem_topology_module,   only: tem_coordOfID
  use tem_param_module,      only: PI
?? ENDIF

  ! include musubi modules
?? IF( .not. SOA) then
  use mus_interpolate_debug_module,  only: TGV_2D
?? ENDIF
  use mus_scheme_layout_module,      only: mus_scheme_layout_type
  use mus_physics_module,            only: mus_physics_type
  use mus_interpolate_header_module, only: mus_interpolation_method_type
  use mus_interpolate_tools_module,  only: mus_intp_convertMomToPDF3D,        &
    &                                      mus_intp_convertMomToPDF3D_incomp, &
    &                                      mus_intp_convertMomToPDF2D,        &
    &                                      mus_intp_convertMomToPDF2D_incomp, &
    &                                      mus_intp_getMoments
  use mus_varSys_module,             only: mus_varSys_data_type
  use mus_scheme_type_module,        only: mus_scheme_type
  use mus_fluid_module,              only: mus_fluid_type
  use mus_mrtRelaxation_module,      only: mus_intp_getNonEqScalingFacs
  use mus_relaxationParam_module,    only: mus_calcOmegaFromVisc
  use mus_derVarPos_type_module,     only: mus_derVarPos_type  

  implicit none

  private

  public :: fillArbiMyGhostsFromFiner_avg
  public :: fillArbiFinerGhostsFromMe_weighAvg
  public :: fillMyGhostsFromFiner_avg
  public :: fillMyGhostsFromFiner_avgLES
  public :: fillMyGhostsFromFiner_avgIncomp
  public :: fillMyGhostsFromFiner_avg2D
  public :: fillMyGhostsFromFiner_avg2DIncomp
  public :: fillFinerGhostsFromMe_weighAvg
  public :: fillFinerGhostsFromMe_weighAvgLES
  public :: fillFinerGhostsFromMe_weighAvgIncomp
  public :: fillFinerGhostsFromMe_weighAvg2D
  public :: fillFinerGhostsFromMe_weighAvg2DIncomp

  contains


! **************************************************************************** !
  !> Interpolate auxiliary field from fine source to coarse target
  !!
?? copy :: interpolateRoutineArbi_header( fillArbiMyGhostsFromFiner_avg )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: inv_nSourceElems
    real(kind=rk) :: tArbi(nScalars)  ! target auxField
    real(kind=rk) :: sArbi(nScalars)  ! temp source ArbiField
    ! --------------------------------------------------------------------------
    sourceLevel = level + 1
    targetLevel = level

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromFiner)

      ! Get how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromFiner( iElem )%elem%nVals
      inv_nSourceElems = 1.0_rk / dble(nSourceElems)

      tArbi = 0.0_rk
      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element
        sourceElem = tLevelDesc%depFromFiner( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get souce auxilary variables
        sArbi(1:nScalars) = sVal((sourceElem-1)*nScalars+1:sourceElem*nScalars)

        ! target aux  
        tArbi = sArbi + tArbi

      end do  ! iSourceElem

      ! interpolate all auxiliary variables by average
      tVal((targetElem-1)*nScalars+1: targetElem*nScalars) &
        & = tArbi(:) * inv_nSourceElems

    enddo

  end subroutine fillArbiMyGhostsFromFiner_avg
! **************************************************************************** !

! **************************************************************************** !
  !> [Average interpolation](../page/features/intp_methods.html) of ghostFromFiner
  !! The interpolation procedure used in this routine is:\n
  !! 1. Calculate moments from source
  !! 2. Interpolate moments on target by simple average
  !! 3. Store target auxilary field
  !! 4. Compute viscosity on target element and compute source and target omega
  !! 5. Get nonEq scaling factor depeding on scheme layout and relaxation
  !! 6. Calculate Equilibrium and nonEquilibrium
  !! 7. calculate target: Eq + Scale * nonEquilibrium
  !!
?? copy :: interpolateRoutine_header( fillMyGhostsFromFiner_avg )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    integer :: QQ
    real(kind=rk) :: inv_nSourceElems
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    real(kind=rk) :: srcMom(layout%fStencil%QQ)  ! temp moment calculation
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, tVisc
    real(kind=rk) :: invRho
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! --------------------------------------------------------------------------
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level + 1
    targetLevel = level

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromFiner)


      ! Get how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromFiner( iElem )%elem%nVals
      inv_nSourceElems = 1.0_rk / dble(nSourceElems)

      tMom = 0.0_rk
      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element
        sourceElem = tLevelDesc%depFromFiner( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom = mus_intp_getMoments(state     = sstate,                 &
          &                          elem      = sourceElem,             &
          &                          QQ        = QQ,                     &
          &                          nScalars  = nScalars,               &
          &                          nSize     = snSize,                 &
          &                          toMoments = layout%moment%toMoments )

        tMom = srcMom + tMom

      end do  ! iSourceElem

      ! interpolate all moments by average
      tMom(:) = tMom(:) * inv_nSourceElems
    
      ! store interpolated target auxField 
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho 
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*invRho

      ! get normalized kinematic viscosity on target element 
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity: 
      ! v^s_f = 2 v^s_c 
      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(2.0_rk*tVisc) 
      tOmegaKine = mus_calcOmegaFromVisc(tVisc) 

      ! scaling factors for nonEquilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(         &
        & omegaKine_SRC = sOmegaKine,                    &
        & omegaKine_TGT = tOmegaKine,                    & 
        & omegaBulk_SRC = fluid%omLvl_bulk(sourceLevel), &
        & omegaBulk_TGT = fluid%omLvl_bulk(targetLevel), &
        & scaleFac      = 2.0_rk,                        &
        & QQ            = QQ                             )
      !write(dbgUnit(1),*) 'avg nonEqFacs', nonEqScalingFacs

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iDir = 1, QQ
        tstate(?IDX?( iDir, targetElem, nscalars, tnSize ) ) &
          & = tPDF(iDir)
      enddo ! iDir

    enddo

  end subroutine fillMyGhostsFromFiner_avg
! **************************************************************************** !

! **************************************************************************** !
  !> [Average interpolation](../page/features/intp_methods.html) of ghostFromFiner
  !! The interpolation procedure used in this routine is:\n
  !! 1. calculate moments from source
  !! 2. Interpolate moments on target by simple average
  !! 3. Calculate Equilibrium and nonEquilibrium
  !! 4. calculate target: Eq + Scale * nonEquilibrium
  !!
?? copy :: interpolateRoutine_header( fillMyGhostsFromFiner_avgLES )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    integer :: QQ
    real(kind=rk) :: inv_nSourceElems
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    real(kind=rk) :: srcMom(layout%fStencil%QQ)  ! temp moment calculation
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, sVisc, tVisc, invRho
    real(kind=rk) :: tTurbVisc
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! --------------------------------------------------------------------------
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level + 1
    targetLevel = level

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromFiner)

      ! Get how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromFiner( iElem )%elem%nVals
      inv_nSourceElems = 1.0_rk / dble(nSourceElems)

      tMom = 0.0_rk
      tTurbVisc = 0.0_rk
      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element
        sourceElem = tLevelDesc%depFromFiner( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom = mus_intp_getMoments(state     = sstate,                 &
          &                          elem      = sourceElem,             &
          &                          QQ        = QQ,                     &
          &                          nScalars  = nScalars,               &
          &                          nSize     = snSize,                 &
          &                          toMoments = layout%moment%toMoments )

        tMom = srcMom + tMom

        tTurbVisc = tTurbVisc + scheme%turbulence%dataOnLvl(sourceLevel) &
          &                                      %visc(sourceElem)
      end do  ! iSourceElem

      ! interpolate all moments by average
      tMom(:) = tMom(:) * inv_nSourceElems
    
      ! store interpolated target auxField 
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho 
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*invRho 

      ! interpolate turbulent viscosity on target element
      tTurbVisc = tTurbVisc * inv_nSourceElems
      ! scale interpolated turbulent viscosity to target element
      scheme%turbulence%dataOnLvl(targetLevel)%visc(targetElem) &
        & = scheme%turbulence%fac_f2c*tTurbVisc

      ! get normalized kinematic viscosity on target element 
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity: 
      ! v^s_f = 2 v^s_c 
      ! total viscosity on source element
      sVisc = 2.0_rk * tVisc + tTurbVisc
      ! total viscosity on target element
      tVisc = tVisc + scheme%turbulence%fac_f2c*tTurbVisc

      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(sVisc) 
      tOmegaKine = mus_calcOmegaFromVisc(tVisc) 

      ! scaling factors for nonEquilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(         &
        & omegaKine_SRC = sOmegaKine,                    &
        & omegaKine_TGT = tOmegaKine,                    & 
        & omegaBulk_SRC = fluid%omLvl_bulk(sourceLevel), &
        & omegaBulk_TGT = fluid%omLvl_bulk(targetLevel), &
        & scaleFac      = 2.0_rk,                        &
        & QQ            = QQ                             )
      !write(dbgUnit(1),*) 'avg nonEqFacs', nonEqScalingFacs

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf in the current direction to the target
      ! Element position
      do iDir = 1, QQ
        tstate(?IDX?( iDir, targetElem, nscalars, tnSize ) ) &
          & = tPDF(iDir)
      enddo ! iDir

    enddo

  end subroutine fillMyGhostsFromFiner_avgLES
! **************************************************************************** !

! **************************************************************************** !
  !> Fill coarse target ghost from fine source fluid by average interpolation.
  !! 1. For each target, calculate rho, vel and fNeq of its sources.
  !! 2. Average rho and vel, then calculate fEq.
  !! 3. Average fNeq and scale.
  !! 4. set target by f = fEq + fNeq
  !! This routine is used by 3D incomp acoustic average interpolation.
  !!
?? copy :: interpolateRoutine_header( fillMyGhostsFromFiner_avgIncomp )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    ! pdf 
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    integer :: QQ
    real(kind=rk) :: inv_nSourceElems
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    real(kind=rk) :: srcMom(layout%fStencil%QQ)  ! temp moment calculation
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: nScalars
    ! --------------------------------------------------------------------------
!write(dbgUnit(1),*) 'Entering fillCoarser average'
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ

    sourceLevel = level + 1
    targetLevel = level

    ! scaling factors for nonEquilibrium moments
    nonEqScalingFacs = mus_intp_getNonEqScalingFacs(                    &
      &                   schemeHeader = scheme%header,                 &
      &                   viscOmegaSRC = fluid%omLvl(sourceLevel),      &
      &                   viscOmegaTGT = fluid%omLvl(targetLevel),      &
      &                   bulkOmegaSRC = fluid%omLvl_bulk(sourceLevel), &
      &                   bulkOmegaTGT = fluid%omLvl_bulk(targetLevel), &
      &                   scaleFac     = 2.0_rk,                        &
      &                   QQ           = QQ                             )

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromFiner)
!write(dbgUnit(1),*) 'Target iElem ', indElem, ' ID ', tLevelDesc%total(targetElem)

      ! Find out how many fine source elements we have for interpolation.
      nSourceELems = tLevelDesc%depFromFiner( iElem )%elem%nVals
      inv_nSourceElems = 1.0_rk / dble(nSourceElems)

      tMom = 0.0_rk
      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromFiner( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom = mus_intp_getMoments(state     = sstate,                 &
          &                          elem      = sourceElem,             &
          &                          QQ        = QQ,                     &
          &                          nScalars  = nScalars,               &
          &                          nSize     = snSize,                 &
          &                          toMoments = layout%moment%toMoments )

        tMom = srcMom + tMom
      end do ! iSourceElem

      ! interpolate all moments by average
      tMom(:) = tMom(:) * inv_nSourceElems

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D_incomp(                              &
        &                               moments          = tMom,             &
        &                               nonEqScalingFacs = nonEqScalingFacs, &
        &                               layout           = layout            )

      do iDir = 1,QQ
        tstate(?IDX?( iDir, targetElem, nScalars, tnSize )) &
          & = tPDF(iDir) 
      end do
!write(dbgUnit(1),*) 'targetRho-rho0 ', rho - rho0
    end do

  end subroutine fillMyGhostsFromFiner_avgIncomp
! **************************************************************************** !

! **************************************************************************** !
  !> Fill coarse target ghost from fine source fluid by average interpolation.
  !! 1. For each target, calculate rho, vel and fNeq of its sources.
  !! 2. Average rho and vel, then calculate fEq.
  !! 3. Average fNeq and scale.
  !! 4. set target by f = fEq + fNeq
  !! This routine is used by 2D acoustic average interpolation.
  !!
?? copy :: interpolateRoutine_header( fillMyGhostsFromFiner_avg2D )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    integer :: QQ
    real(kind=rk) :: inv_nSourceElems
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    real(kind=rk) :: srcMom(layout%fStencil%QQ)  ! temp moment calculation
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: nScalars
    ! --------------------------------------------------------------------------
!write(dbgUnit(1),*) 'Entering fillCoarser average'
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ

    sourceLevel = level + 1
    targetLevel = level

    ! scaling factors for nonEquilibrium moments
    nonEqScalingFacs = mus_intp_getNonEqScalingFacs(                    &
      &                   schemeHeader = scheme%header,                 &
      &                   viscOmegaSRC = fluid%omLvl(sourceLevel),      &
      &                   viscOmegaTGT = fluid%omLvl(targetLevel),      &
      &                   bulkOmegaSRC = fluid%omLvl_bulk(sourceLevel), &
      &                   bulkOmegaTGT = fluid%omLvl_bulk(targetLevel), &
      &                   scaleFac     = 2.0_rk,                        &
      &                   QQ           = QQ                             )


    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromFiner)
!write(dbgUnit(1),*) 'Target iElem ', indElem, ' ID ', tLevelDesc%total(targetElem)

      ! Find out how many fine source elements we have for interpolation.
      nSourceELems = tLevelDesc%depFromFiner( iElem )%elem%nVals
      inv_nSourceElems = 1.0_rk / dble(nSourceElems)

      ! Now loop over all fine source elements for this target:
      tMom = 0.0_rk
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromFiner( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom = mus_intp_getMoments(state     = sstate,                 &
          &                          elem      = sourceElem,             &
          &                          QQ        = QQ,                     &
          &                          nScalars  = nScalars,               &
          &                          nSize     = snSize,                 &
          &                          toMoments = layout%moment%toMoments )

        tMom = srcMom + tMom
      end do ! iSourceElem

      ! interpolate all moments by average
      tMom(:) = tMom(:) * inv_nSourceElems

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF2D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      do iDir = 1,QQ
        tstate(?IDX?( iDir, targetElem, nScalars, tnSize )) &
          & = tPDF(iDir) 
      end do
!write(dbgUnit(1),*) 'targetRho-rho0 ', rho - rho0
    end do

  end subroutine fillMyGhostsFromFiner_avg2D
! **************************************************************************** !

! **************************************************************************** !
  !> Fill coarse target ghost from fine source fluid by average interpolation.
  !! 1. For each target, calculate rho, vel and fNeq of its sources.
  !! 2. Average rho and vel, then calculate fEq.
  !! 3. Average fNeq and scale.
  !! 4. set target by f = fEq + fNeq
  !! This routine is used by 2D incomp acoustic average interpolation.
  !!
?? copy :: interpolateRoutine_header( fillMyGhostsFromFiner_avg2DIncomp )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    ! current non-equilibrium part
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    integer :: QQ
    real(kind=rk) :: inv_nSourceElems
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    real(kind=rk) :: srcMom(layout%fStencil%QQ)  ! temp moment calculation
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: nScalars
    ! --------------------------------------------------------------------------
!write(dbgUnit(1),*) 'Entering fillCoarser average'
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ

    sourceLevel = level + 1
    targetLevel = level

    ! scaling factors for nonEquilibrium moments
    nonEqScalingFacs = mus_intp_getNonEqScalingFacs(                    &
      &                   schemeHeader = scheme%header,                 &
      &                   viscOmegaSRC = fluid%omLvl(sourceLevel),      &
      &                   viscOmegaTGT = fluid%omLvl(targetLevel),      &
      &                   bulkOmegaSRC = fluid%omLvl_bulk(sourceLevel), &
      &                   bulkOmegaTGT = fluid%omLvl_bulk(targetLevel), &
      &                   scaleFac     = 2.0_rk,                        &
      &                   QQ           = QQ                             )

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromFiner)
!write(dbgUnit(1),*) 'Target iElem ', indElem, ' ID ', tLevelDesc%total(targetElem)

      ! Find out how many fine source elements we have for interpolation.
      nSourceELems = tLevelDesc%depFromFiner( iElem )%elem%nVals
      inv_nSourceElems = 1.0_rk / dble(nSourceElems)

      ! Now loop over all fine source elements for this target:
      tMom = 0.0_rk
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromFiner( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom = mus_intp_getMoments(state     = sstate,                 &
          &                          elem      = sourceElem,             &
          &                          QQ        = QQ,                     &
          &                          nScalars  = nScalars,               &
          &                          nSize     = snSize,                 &
          &                          toMoments = layout%moment%toMoments )

        tMom = srcMom + tMom
      end do ! iSourceElem

      ! interpolate all moments by average
      tMom(:) = tMom(:) * inv_nSourceElems
!write(dbgUnit(1),*) 'before scaling tMom ', tMom

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF2D_incomp(                              &
        &                               moments          = tMom,             &
        &                               nonEqScalingFacs = nonEqScalingFacs, &
        &                               layout           = layout            )

      do iDir = 1,QQ
        tstate(?IDX?( iDir, targetElem, nScalars, tnSize )) &
          & = tPDF(iDir) 
      end do
!write(dbgUnit(1),*) 'tPDF ', tPDF
    end do

  end subroutine fillMyGhostsFromFiner_avg2DIncomp
! **************************************************************************** !

! **************************************************************************** !
  !> Interpolate auxiliary field from coarse source to fine target
  !!
?? copy :: interpolateRoutineArbi_header( fillArbiFinerGhostsFromMe_weighAvg)
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: weight( stencil%QQ )
    real(kind=rk) :: sArbi(nScalars, stencil%QQ)  ! temp source ArbiField
    integer :: iVar
    ! --------------------------------------------------------------------------
    sourceLevel = level
    targetLevel = level + 1

    ! Treat all coarse target elements
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Get how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals

      ! Read the pre-calculated weights (like in Average intp.)
      weight(1:nSourceElems) = tLevelDesc%depFromCoarser( iElem ) &
        &                                %weight(1:nSourceElems)

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get souce auxilary variables
        sArbi(:, iSourceElem) = sVal( (sourceElem-1)*nScalars+1 &
          &                           : sourceElem*nScalars       )

      end do  ! iSourceElem

      ! interpolate all auxiliary variables by average
      do iVar = 1, nScalars
        tVal((targetElem-1)*nScalars+iVar)                    &
          & = sum( weight(1:nSourceElems)*sArbi(iVar,1:nSourceElems) )
      end do  

    enddo

  end subroutine fillArbiFinerGhostsFromMe_weighAvg
! **************************************************************************** !

! **************************************************************************** !
  !> [Linear interpolation](../page/features/intp_methods.html) of ghostFromFiner
  !!
?? copy :: interpolateRoutine_header( fillFinerGhostsFromMe_weighAvg )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    real(kind=rk) :: weight( layout%fStencil%QQ )
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: iMom, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: sOmegaKine, tOmegaKine, tVisc, invRho
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! --------------------------------------------------------------------------
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Find out how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals

      ! Read the pre-calculated weights (like in Average intp.)
      weight(1:nSourceElems) = tLevelDesc%depFromCoarser( iElem ) &
        &                                %weight(1:nSourceElems)

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do ! iSourceElem

      ! interpolate all moments by weighted average
      do iMom = 1,QQ
        tMom(iMom) = sum(weight(1:nSourceElems)*srcMom(iMom,1:nSourceElems))
      end do

      ! store interpolated target auxField 
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho 
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*invRho

      ! get normalized kinematic viscosity on target element
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)
      ! relation between coarse and fine grid kinematic viscosity: 
      ! v^s_f = 2 v^s_c 
      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(0.5_rk*tVisc) 
      tOmegaKine = mus_calcOmegaFromVisc(tVisc) 

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(         &
        & omegaKine_SRC = sOmegaKine,                    &
        & omegaKine_TGT = tOmegaKine,                    &
        & omegaBulk_SRC = fluid%omLvl_bulk(sourceLevel), &
        & omegaBulk_TGT = fluid%omLvl_bulk(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf to the target element
      do iDir = 1, QQ
        tstate(?IDX?(iDir, targetElem, nScalars, tnSize ) ) &
          & = tPDF( iDir )
      end do
    enddo ! element loop

  end subroutine fillFinerGhostsFromMe_weighAvg
! **************************************************************************** !


! **************************************************************************** !
  !> [Linear interpolation](../page/features/intp_methods.html) of ghostFromFiner
  !!
?? copy :: interpolateRoutine_header( fillFinerGhostsFromMe_weighAvgLES )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    real(kind=rk) :: weight( layout%fStencil%QQ )
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: iMom, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    real(kind=rk) :: sOmegaKine, tOmegaKine, sVisc, tVisc, invRho
    real(kind=rk) :: sTurbVisc(layout%fStencil%QQ), tTurbVisc
    integer :: nScalars, tOffset
    integer :: dens_pos, vel_pos(3)
    ! --------------------------------------------------------------------------
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(:)

    sourceLevel = level
    targetLevel = level + 1

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Find out how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals

      ! Read the pre-calculated weights (like in Average intp.)
      weight(1:nSourceElems) = tLevelDesc%depFromCoarser( iElem ) &
        &                                %weight(1:nSourceElems)

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

        ! get turbulent viscosity
        sTurbVisc(iSourceElem) = scheme%turbulence%dataOnLvl(sourceLevel) &
          &                                       %visc(sourceElem)
      end do ! iSourceElem

      ! interpolate all moments by weighted average
      do iMom = 1,QQ
        tMom(iMom) = sum(weight(1:nSourceElems)*srcMom(iMom,1:nSourceElems))
      end do

      ! store interpolated target auxField 
      invRho = 1.0_rk/tMom(1)
      tOffset = (targetElem-1)*varSys%nAuxScalars
      tAuxField(tOffset+dens_pos)   = tMom(1)
      tAuxField(tOffset+vel_pos(1)) = tMom(layout%moment%first_moments(1))*invRho
      tAuxField(tOffset+vel_pos(2)) = tMom(layout%moment%first_moments(2))*invRho 
      tAuxField(tOffset+vel_pos(3)) = tMom(layout%moment%first_moments(3))*invRho

      ! interpolate turbulent viscosity
      tTurbVisc = sum(weight(1:nSourceElems)*sTurbVisc(1:nSourceElems))

      ! scale interpolated turbulent viscosity to target element
      scheme%turbulence%dataOnLvl(targetLevel)%visc(targetElem) &
        & = scheme%turbulence%fac_c2f*tTurbVisc

      ! get normalized kinematic viscosity on target element 
      tVisc = fluid%viscKine%dataOnLvl(targetLevel)%val(targetElem)

      ! relation between coarse and fine grid kinematic viscosity: 
      ! v^s_f = 2 v^s_c 
      ! total viscosity on source element
      sVisc = 0.5_rk * tVisc + tTurbVisc
      ! total viscosity on target element
      tVisc = tVisc + scheme%turbulence%fac_c2f*tTurbVisc

      ! calculate omega on source and target level
      sOmegaKine = mus_calcOmegaFromVisc(sVisc) 
      tOmegaKine = mus_calcOmegaFromVisc(tVisc) 

      ! Get scaling factors for nonequilibrium moments
      nonEqScalingFacs = fluid%nonEqScalingFacs(         &
        & omegaKine_SRC = sOmegaKine,                    &
        & omegaKine_TGT = tOmegaKine,                    &
        & omegaBulk_SRC = fluid%omLvl_bulk(sourceLevel), &
        & omegaBulk_TGT = fluid%omLvl_bulk(targetLevel), &
        & scaleFac      = 0.5_rk, QQ = QQ                )

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf to the target element
      do iDir = 1, QQ
        tstate(?IDX?(iDir, targetElem, nScalars, tnSize ) ) &
          & = tPDF( iDir )
      end do
    enddo ! element loop

  end subroutine fillFinerGhostsFromMe_weighAvgLES
! **************************************************************************** !

! **************************************************************************** !
  !> Fill finer ghost from coarser fluid.
  !! This routine is used by 2D and 3D incomp weighted average interpolation.
  !!
?? copy :: interpolateRoutine_header( fillFinerGhostsFromMe_weighAvgIncomp )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    real(kind=rk) :: weight( layout%fStencil%QQ )
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: iMom, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: nScalars
    ! --------------------------------------------------------------------------
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ

    sourceLevel = level
    targetLevel = level + 1

    ! scaling factors for nonEquilibrium moments
    nonEqScalingFacs = mus_intp_getNonEqScalingFacs(                    &
      &                   schemeHeader = scheme%header,                 &
      &                   viscOmegaSRC = fluid%omLvl(sourceLevel),      &
      &                   viscOmegaTGT = fluid%omLvl(targetLevel),      &
      &                   bulkOmegaSRC = fluid%omLvl_bulk(sourceLevel), &
      &                   bulkOmegaTGT = fluid%omLvl_bulk(targetLevel), &
      &                   scaleFac     = 0.5_rk,                        &
      &                   QQ           = QQ                             )

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Find out how many fine source elements we have for interpolation.
      ! Usually 8, but can differ at corners, obstacles, boundaries...
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals

      ! Read the pre-calculated weights (like in Average intp.)
      weight(1:nSourceElems) = &
        &            tLevelDesc%depFromCoarser( iElem )%weight(1:nSourceElems)

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do ! iSourceElem

      ! interpolate rho, momentum and shear stress by weighted average
      do iMom = 1,QQ
        tMom(iMom) = sum(weight(1:nSourceElems)*srcMom(iMom,1:nSourceElems))
      end do

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF3D_incomp(                               &
        &                                moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      do iDir = 1,QQ
        tstate(?IDX?(iDir, targetElem, nScalars, tnSize )) &
          &  = tPDF( iDir )
      end do

    end do

  end subroutine fillFinerGhostsFromMe_weighAvgIncomp
! **************************************************************************** !


! **************************************************************************** !
  !> [Linear interpolation](../page/features/intp_methods.html) of ghostFromFiner
  !!
?? copy :: interpolateRoutine_header( fillFinerGhostsFromMe_weighAvg2D )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    real(kind=rk) :: weight( layout%fStencil%QQ )
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: iMom, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: nScalars
    ! --------------------------------------------------------------------------
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ

    sourceLevel = level
    targetLevel = level + 1

    ! scaling factors for nonEquilibrium moments
    nonEqScalingFacs = mus_intp_getNonEqScalingFacs(                    &
      &                   schemeHeader = scheme%header,                 &
      &                   viscOmegaSRC = fluid%omLvl(sourceLevel),      &
      &                   viscOmegaTGT = fluid%omLvl(targetLevel),      &
      &                   bulkOmegaSRC = fluid%omLvl_bulk(sourceLevel), &
      &                   bulkOmegaTGT = fluid%omLvl_bulk(targetLevel), &
      &                   scaleFac     = 0.5_rk,                        &
      &                   QQ           = QQ                             )

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)

      ! Find out how many fine source elements we have for interpolation.
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals

      ! Read the pre-calculated weights (like in Average intp.)
      weight(1:nSourceElems) = tLevelDesc%depFromCoarser( iElem ) &
        &                                %weight(1:nSourceElems)

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems

        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do ! iSourceElem

      ! interpolate all moments by weighted average
      do iMom = 1,QQ
        tMom(iMom) = sum(weight(1:nSourceElems)*srcMom(iMom,1:nSourceElems))
      end do

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF2D( moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )

      ! Now write the resulting pdf to the target element
      do iDir = 1, QQ
        tstate(?IDX?(iDir, targetElem, nScalars, tnSize ) ) &
          & = tPDF( iDir )
      end do
    enddo ! element loop

  end subroutine fillFinerGhostsFromMe_weighAvg2D
! **************************************************************************** !

! **************************************************************************** !
  !> Fill finer ghost from coarser fluid.
  !! This routine is used by 2D and 3D incomp weighted average interpolation.
  !!
?? copy :: interpolateRoutine_header( fillFinerGhostsFromMe_weighAvg2DIncomp )
    ! --------------------------------------------------------------------------
    integer :: sourceLevel    ! level of source elements
    integer :: sourceElem     ! treeId of current source element
    integer :: targetLevel    ! level of target elements
    integer :: targetElem     ! treeId of current source element
    integer :: iDir           ! current direction (discrete velocity) for loop
    integer :: iElem          ! current target element (for outer loop)
    integer :: indElem        ! element counter for indirection list
    integer :: iSourceElem    ! current source element (for inner loop)
    integer :: nSourceElems   ! number of source elements for the current target
    ! macroscopic velocities for all source elements
    real(kind=rk) :: tPDF( layout%fStencil%QQ )
    real(kind=rk) :: weight( layout%fStencil%QQ )
    ! moments of the source elements' pdf
    real(kind=rk) :: srcMom( layout%fStencil%QQ, layout%fStencil%QQ )
    real(kind=rk) :: tMom(layout%fStencil%QQ)  ! target moment calculation
    integer :: iMom, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    type(mus_fluid_type), pointer :: fluid
    real(kind=rk) :: nonEqScalingFacs(layout%fStencil%QQ)
    integer :: nScalars
    ! --------------------------------------------------------------------------
!write(dbgUnit(1),*) 'Entering fillFiner average'
    ! use 1st variable method data access fluid information
    call c_f_pointer( varSys%method%val(1)%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme
    fluid => fPtr%solverData%scheme%field(1)%fieldProp%fluid
    nScalars = varSys%nScalars
    QQ = layout%fStencil%QQ

    sourceLevel = level
    targetLevel = level + 1

    ! scaling factors for nonEquilibrium moments
    nonEqScalingFacs = mus_intp_getNonEqScalingFacs(                    &
      &                   schemeHeader = scheme%header,                 &
      &                   viscOmegaSRC = fluid%omLvl(sourceLevel),      &
      &                   viscOmegaTGT = fluid%omLvl(targetLevel),      &
      &                   bulkOmegaSRC = fluid%omLvl_bulk(sourceLevel), &
      &                   bulkOmegaTGT = fluid%omLvl_bulk(targetLevel), &
      &                   scaleFac     = 0.5_rk,                        &
      &                   QQ           = QQ                             )
!write(dbgUnit(1),*) 'nonEqScalingFacs ', nonEqScalingFacs

    ! Treat all fine target elements:
    do indElem = 1, nTargets

      iElem = targetList( indElem )

      ! Read the target element treeId
      targetElem = iElem + tLevelDesc%offset( 1, eT_ghostFromCoarser)
!write(dbgUnit(1),*) 'Target iElem ', indElem, ' ID ', tLevelDesc%total(targetElem)

      ! Find out how many fine source elements we have for interpolation.
      ! Usually 8, but can differ at corners, obstacles, boundaries...
      nSourceElems = tLevelDesc%depFromCoarser( iElem )%elem%nVals

      ! Read the pre-calculated weights (like in Average intp.)
      weight(1:nSourceElems) = tLevelDesc%depFromCoarser( iElem ) &
        &                                %weight(1:nSourceElems)

      ! Now loop over all fine source elements for this target:
      do iSourceElem = 1, nSourceElems
        ! Get the source element position
        sourceElem = tLevelDesc%depFromCoarser( iElem ) &
          &                    %elem%val( iSourceElem )

        ! Get macroscopic moments
        srcMom(:, iSourceElem) = mus_intp_getMoments(                         &
          &                               state     = sstate,                 &
          &                               elem      = sourceElem,             &
          &                               QQ        = QQ,                     &
          &                               nScalars  = nScalars,               &
          &                               nSize     = snSize,                 &
          &                               toMoments = layout%moment%toMoments )

      end do ! iSourceElem

      ! interpolate rho, momentum and shear stress by weighted average
      do iMom = 1, QQ
        tMom(iMom) = sum(weight(1:nSourceElems)*srcMom(iMom,1:nSourceElems))
      end do
!write(dbgUnit(1),*) 'before scaling tMom ', tMom

      ! Convert moment to PDF
      tPDF = mus_intp_convertMomToPDF2D_incomp(                               &
        &                                moments          = tMom,             &
        &                                nonEqScalingFacs = nonEqScalingFacs, &
        &                                layout           = layout            )
!write(dbgUnit(1),*) 'tPDF ', tPDF
      do iDir = 1,QQ
        tstate(?IDX?(iDir, targetElem, nScalars, tnSize )) &
          &  = tPDF( iDir )
      end do

    end do

  end subroutine fillFinerGhostsFromMe_weighAvg2DIncomp
! **************************************************************************** !


end module mus_interpolate_average_module
! **************************************************************************** !
