! See copyright notice in the COPYRIGHT file.
!! ****************************************************************************** !
!> author: Kannan Masilamani
!! [[mus_hvs_construct]] "Creation of the data structures"
!! from the information in the configuration and
!! from the mesh read from disk for the musubi harvesting
!!
?? include 'header/lbm_macros.inc'
module mus_hvs_construction_module

  ! include treelm modules
  use mpi
  use env_module,              only: rk, long_k, eps, labelLen, my_status_int
  use treelmesh_module,        only: treelmesh_type
  use tem_comm_module,         only: tem_communication_type,                   &
    &                                tem_commpattern_type
  use tem_tools_module,        only: tem_horizontalSpacer
  use tem_grow_array_module,   only: grw_intArray_type, init, append, destroy, &
    &                                empty, truncate
  use tem_dyn_array_module,    only: dyn_intArray_type, init, append, expand,  &
    &                                destroy, empty, PositionOfVal, truncate,  &
    &                                dyn_labelArray_type
  use tem_construction_module, only: tem_init_elemLevels, tem_find_allElements,&
    &                                tem_build_verticalDependencies,           &
    &                                tem_build_horizontalDependencies,         &
    &                                tem_debug_horizontalDependencies,         &
    &                                tem_levelDesc_type,                       &
    &                                depSource_type,                           &
    &                                depSource_append,                         &
    &                                tem_dumpTreeIDLists
  use tem_element_module,      only: eT_fluid, eT_halo, eT_ghostFromCoarser,   &
    &                                eT_ghostFromFiner, destroy
  use tem_timer_module,        only: tem_startTimer, tem_stopTimer,            &
    &                                tem_getTimerVal
  use tem_stencil_module,      only: tem_stencilHeader_type,                   &
    &                                tem_stencil_getLabelForcxDir,             &
    &                                tem_create_d3q81, tem_create_d3q125,      &
    &                                grw_stencilHeaderArray_type, append,      &
    &                                destroy, truncate
  use tem_debug_module,        only: main_debug, dbgUnit
  use tem_aux_module,          only: tem_abort
  use tem_logging_module,      only: logUnit
  use tem_subTree_module,      only: tem_write_debugMesh
  use tem_comm_env_module,     only: tem_comm_env_type
  use tem_varSys_module,       only: tem_varSys_type
  use tem_varMap_module,       only: tem_varMap_type

  ! include musubi modules
  use mus_param_module,              only: mus_param_type
  use mus_geom_module,               only: mus_geom_type
  use mus_scheme_layout_module,      only: mus_scheme_layout_type,             &
    &                                      mus_finalize_layout
  use mus_scheme_type_module,        only: mus_scheme_type
  use mus_pdf_module,                only: mus_calc_nElems,                    &
    &                                      mus_pdf_allocate,                   &
    &                                      allocate_momBuf
  use mus_debug_module,              only: debug_normals,                      &
    &                                      debug_dependenciesFromFiner,        &
    &                                      debug_dependenciesFromCoarser,      &
    &                                      debug_connectivity
  use mus_field_module,              only: mus_field_getSymmetricBCs  
  use mus_interpolate_module,        only: mus_init_levelDescIntpArrays,   &
    &                                      mus_intp_update_depFromCoarser, &
    &                                      mus_intp_update_depFromFiner,   &
    &                                      mus_dump_levelDescIntp_nElems
  use mus_construction_module,       only: communicate_property,               &
    &                                      calculate_nElems,                   &
    &                                      set_sendHaloBits
  use mus_connectivity_module,       only: mus_construct_connectivity, &
    &                                      mus_updateConnectivity_forSymmetricBC 
  use mus_timer_module,              only: mus_timerHandles


  implicit none

  private

  public :: mus_hvs_construct

contains

! ****************************************************************************** !
  !> Initialize Musubi data strucutres based on data provided by Treelm
  !!
  !! Load the mesh and boundary conditions for this process from disk.
  !! Get the level-wise treeID lists and create the required ghost and halo
  !! elements.
  !!
  !! This is achieved by a two-folded identification of elements.
  !!
  !! -# the theoretically required elements are collected based on
  !! [[mus_scheme_layout_module:mus_scheme_layout_type]] "stencil information"
  !! The [[tem_construction_module:tem_findneighbor]] "find neighbor routine"
  !! performs this task for compute fluid elements.
  !! For boundaries which require information from neighbor elements, these
  !! required [[tem_topology_module]] "treeIDs" are collected into the
  !! [[mus_bc_header_module]]
  !! "boundary element type"
  !! -# All required elements are created in the
  !! [[tem_construction_module:tem_createleveldescriptor]]
  !! "Level Descriptor creation routine"
  !!
  !! # Additional Tasks
  !!
  !! - receive [[tem_construction_module:tem_buildhorizontaldependencies]]
  !! "horizontal"
  !! (within a level for the element updates)
  !! - and [[tem_construction:tem_buildverticaldependencies]] "vertical"
  !! dependencies (between levels for ghost-interpolations).
  !! - The main state vector and the neighbor lists on which the kernel then
  !!    acts is created
  !! - The MPI buffers are created.
  !! - For each [[mus_scheme_module]] "Scheme", the
  !! [[tem_construction_module:level_descriptor_type]] "Level Descriptor"
  !! is created
  !!
  !! # Result
  !!
  !! After this routine, all data structures for starting the main loop of the
  !! solver are allocated and ready.
  !!
  !! Only difference between this routine and mus_construct is then 
  !! creating of boundary elements and its stencil are omitted for harvesting
  !!
  subroutine mus_hvs_construct( scheme, geometry, params)
    ! ---------------------------------------------------------------------------
    !> run-time Parameters
    type( mus_param_type ), intent(inout) :: params
    !> geometric information
    type( mus_geom_type ), intent(inout) :: geometry
    !> scheme information including fluid, boundary and flow information
    type( mus_scheme_type ), intent(inout) :: scheme
    ! ---------------------------------------------------------------------------
    integer :: iLevel      ! counter for level
    integer :: minLevel, maxLevel
    integer :: hwmVal
    integer :: ii
    integer :: symmetricBCs(geometry%boundary%nBCtypes)
    integer :: nSymBCs
    ! ---------------------------------------------------------------------------
    call tem_abort('Use default mus_construct instead of mus_hvs_construct')
    minLevel = geometry%tree%global%minLevel
    maxLevel = geometry%tree%global%maxLevel

    call tem_horizontalSpacer( fUnit = dbgUnit(1), before = 1 )
    write(dbgUnit(1),*) 'Get into routine: mus_construction'

    call tem_startTimer( timerHandle =  mus_timerHandles%initLvlD )

    ! set up the data structures for the kernel

    write(logUnit(1),*) 'Starting to initialize the geometry'
    call tem_horizontalSpacer(fUnit = logUnit(1))

    ! finalize scheme layout
    ! copy growing array of stencil to allocatable array of stencil
    ! and destroy growing array
    call mus_finalize_layout( layout   = scheme%layout,            &
      &                       nElemsInTree = geometry%tree%nElems, &
      &                       minLevel = minLevel,                 &
      &                       maxLevel = maxLevel,                 &
      &                       proc     = params%general%proc       )
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After finalize layout, VmHWM: ', hwmVal

    ! 1. store treeIDs which use this stencil in the
    !    levelDesc( elemLevel )%elem
    ! 2. store the neighbors which can be loaded from the disc or accessed
    !    directly
    ! THIS MEANS THAT FOR ALL ELEMENTS (INCL. BOUNDARY ELEMENTS) NEIGHBORS
    ! ARE STORED BUT THEY DO NOT NEED TO EXIST IN THE MESH AT THIS POINT!!!
    call tem_init_elemLevels(                            &
      &                me       = scheme%levelDesc, &
      &                boundary = geometry%boundary,     &
      &                tree     = geometry%tree,         &
      &                stencils = [scheme%layout%fStencil] )
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After init elemLevels, VmHWM: ', hwmVal

    ! 1. build all dependencies for fluid, halos and ghosts
    ! 2. build the pointers for each element to its neighbors/stencil elements
    ! All this information is stored in tem_levelDesc_type.
    !
    ! THIS MEANS THAT FOR ALL ELEMENTS FOUND IN em_init_elemLevels THE CORRECT
    ! DEPENDENCIES AND NEIGHBORS ARE SET
    !
    ! This is the heart of the topologic data structure creation and
    ! takes most of the compute time, depending on the mesh you have
    !
    write(logUnit(1),*) 'Creating level descriptor ...'
    write(logUnit(6),*) 'before tem_find_allElements'
    write(dbgUnit(6),*) 'before find_allElements'
    ! The level descriptor is created from the below routine.
    ! the neigh array is created using the LD and communication buffers are filled up. 
    call tem_find_allElements(                                         &
      &                  tree            = geometry%tree,              &
      &                  levelDesc       = scheme%levelDesc,      &
      &                  levelPointer    = geometry%levelPointer,     &
      &                  computeStencil  = [scheme%layout%fStencil],   &
      &                  commPattern     = params%general%commPattern, &
      &                  cleanup         = .true.,                     &
      &                  reqNesting      = params%nNesting,            &
      &                  proc            = params%general%proc         )
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After find allElements, VmHWM: ', hwmVal
    if ( main_debug%dumpTreeIDlists ) then
      call tem_dumpTreeIDlists( minLevel, maxLevel, &
        &                       scheme%levelDesc )
    end if

    write(dbgUnit(6),*) 'before horizontal dep'
    write(logUnit(6),*) 'before tem_build_horizontalDependencies'
    call tem_build_horizontalDependencies(            &
      &       iStencil       = 1,                     &
      &       levelDesc      = scheme%levelDesc, &
      &       tree           = geometry%tree,         &
      &       computeStencil = scheme%layout%fStencil )
    if( main_debug%checkDependencies ) then
      call tem_debug_HorizontalDependencies( 1,         &
        &         scheme%levelDesc, geometry%tree, &
        &         scheme%layout%fStencil )
    end if
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After build horizontal, VmHWM: ', hwmVal

    ! Communicate property after setting missing Neighbor in
    ! tem_build_horizontalDependencies routine

    write(logUnit(6),*) 'before communicate_property'
    do iLevel = minLevel, maxLevel
      call communicate_property(                                  &
        &   send     = scheme%levelDesc( iLevel )%sendbuffer,&
        &   recv     = scheme%levelDesc( iLevel )%recvbuffer,&
        &   property = scheme%levelDesc( iLevel )%property,  &
        &   flag     = iLevel,                                    &
        &   proc     = params%general%proc,                       &
        &   pattern  = params%general%commPattern                 )
      call communicate_property(                                  &
        &   send     = scheme%levelDesc( iLevel )%           &
        &                                 sendbufferFromCoarser,  &
        &   recv     = scheme%levelDesc( iLevel )%           &
        &                                 recvbufferFromCoarser,  &
        &   property = scheme%levelDesc( iLevel )%property,  &
        &   flag     = iLevel,                                    &
        &   proc     = params%general%proc,                       &
        &   pattern  = params%general%commPattern                 )
      call communicate_property(                                  &
        &   send     = scheme%levelDesc( iLevel )%           &
        &                                 sendbufferFromFiner,    &
        &   recv     = scheme%levelDesc( iLevel )%           &
        &                                 recvbufferFromFiner,    &
        &   property = scheme%levelDesc( iLevel )%property,  &
        &   flag     = iLevel,                                    &
        &   proc     = params%general%proc,                       &
        &   pattern  = params%general%commPattern                 )
    end do
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After comm property, VmHWM: ', hwmVal

    call tem_horizontalSpacer( after = 1, fUnit = logUnit(1) )

    ! Get vertical dependencies (between levels for interpolation)
    call tem_build_verticalDependencies(          &
      &        levelDesc = scheme%levelDesc, &
      &        minlevel  = minLevel,              &
      &        maxlevel  = maxLevel               )
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After build vertical, VmHWM: ', hwmVal

    ! Only if multilevel
    if ( minLevel /= maxLevel ) then

      ! initialize the levelwise ghost and source list
      call mus_init_levelDescIntpArrays(                    &
        &  intp      = scheme%intp,                         &
        &  levelDesc = scheme%levelDesc(minLevel:maxLevel), &
        &  minLevel  = minLevel,                            &
        &  maxLevel  = maxLevel                             )

      ! Update source elements for fromCoarser ghost elements and compute
      ! weights and least square fit matric depending in interpolation method
      ! and number of source elements found
      call mus_intp_update_depFromCoarser(                  &
        &  intp      = scheme%intp,                         &
        &  levelDesc = scheme%levelDesc(minLevel:maxLevel), &
        &  stencil   = scheme%layout%fStencil,              &
        &  minLevel  = minLevel,                            &
        &  maxLevel  = maxLevel                             )

      ! Update the dependencies fromFiner (in 2d, 4 of 8 get removed)
      call mus_intp_update_depFromFiner(                    &
        &  intp      = scheme%intp,                         &
        &  levelDesc = scheme%levelDesc(minLevel:maxLevel), &
        &  minLevel  = minLevel,                            &
        &  maxLevel  = maxLevel                             )

      ! dumps global nElems in intpFromCoarser, intpFromFiner,
      ! sourcesFromCoarser ans sourcesFromFiner 
      call mus_dump_levelDescIntp_nElems(                   &
        &  intp      = scheme%intp,                         &
        &  levelDesc = scheme%levelDesc(minLevel:maxLevel), &
        &  minLevel  = minLevel,                            &
        &  maxLevel  = maxLevel,                            &
        &  root      = params%general%proc%root,            &
        &  comm      = params%general%proc%comm             )

    end if ! minLevel /= maxLevel
    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After update interpolation, VmHWM: ', hwmVal

    call calculate_nElems( levelDesc = scheme%levelDesc,    &
      &                    proc      = params%general%proc, &
      &                    minLevel  = minLevel,            &
      &                    maxLevel  = maxLevel             )

    ! allocate here since it needs to be deallocated by dynamic load
    ! balancing algorithm
    write(logUnit(6),"(A,I0,A,I0)") 'allocate PDF from Level ', minLevel, &
      &                             ' to ', maxLevel
    allocate( scheme%pdf( minLevel:maxLevel ) )
    allocate( scheme%state( minLevel:maxLevel ) )

    write(logUnit(4),*) 'Allocating PDF state and neighbor array'
    ! Allocate the PDF state array
    do iLevel = minLevel, maxLevel
      write(dbgUnit(1),*) 'calculate nElems on level: ', iLevel
      call mus_calc_nElems(                                              &
        & me                = scheme%pdf( iLevel ),                      &
        & nFluids           = scheme%levelDesc( iLevel )%elem            &
        &                                %nElems( eT_fluid ),            &
        & nGhostFromCoarser = scheme%levelDesc( iLevel )%elem            &
        &                                %nElems( eT_ghostFromCoarser ), &
        & nGhostFromFiner   = scheme%levelDesc( iLevel )%elem            &
        &                                %nElems( eT_ghostFromFiner ),   &
        & nHalos            = scheme%levelDesc( iLevel )%elem            &
        &                                %nElems( eT_halo )              )

      allocate(scheme%state(iLevel)%val( scheme%pdf(iLevel)%nSize    &
        &                                * scheme%varSys%nScalars, 2 ) )
      !For debugging purposes, set complete flow field to invalid
      do ii = 1, scheme%pdf(iLevel)%nSize * scheme%varSys%nScalars
        scheme%state(iLevel)%val(ii, 1) = -1000000.0_rk
        scheme%state(iLevel)%val(ii, 2) = -1000000.0_rk
      end do

      call mus_pdf_allocate( me              = scheme%pdf( iLevel ),      &
        &                    nScalars        = scheme%varSys%nScalars,    &
        &                    QQ              = scheme%layout%fStencil%QQ, &
        &                    nElems_bcBuffer = 0,                         &
        &                    isPDF           = scheme%readVarIsPdf        )

      call allocate_momBuf( scheme%pdf(iLevel),                        &
        &    max( scheme%levelDesc(iLevel)%sourceFromCoarser%nVals,    &
        &         scheme%levelDesc(iLevel)%sourceFromFiner%nVals    )  )

    end do

    hwmVal = my_status_int('VmHWM:')
    write(logUnit(10),"(A,I0)") 'After allocate PDF, VmHWM: ', hwmVal

    write(dbgUnit(6),*) 'clean up elem list in level desctiptor'
    write(logUnit(6),*) 'clean up elem list in level descriptor'
    ! KJ: @todo de-activate for adaptive grid refinement
    do iLevel = minLevel, maxLevel
      call destroy( me = scheme%levelDesc( iLevel )%elem )
    end do

    write(dbgUnit(6),*) 'after clean up'

    ! reconstruct connectivity vector pdf( iLevel )%neigh including
    ! bounce back rules
    if ( scheme%readVarIsPdf ) then
      do iLevel = minLevel, maxLevel
        ! construct connectivity vector pdf( iLevel )%neigh including
        ! bounce back rules
        call mus_construct_connectivity(                   &
          & neigh       = scheme%pdf(iLevel)%neigh,        &
          & nSize       = scheme%pdf(iLevel)%nSize,        &
          & nElems      = scheme%pdf(iLevel)%nElems_local, &
          & levelDesc   = scheme%levelDesc(iLevel),        &
          & stencil     = scheme%layout%fStencil,          &
          & varSys      = scheme%varSys,                   &
          & stateVarMap = scheme%stateVarMap               )
  
        ! identify boundary ids which are symmetry kind
        call mus_field_getSymmetricBCs(                &
          & symmetricBCs = symmetricBCs,               &
          & nSymBCs      = nSymBCs,                    &
          & nBCs         = geometry%boundary%nBCtypes, & 
          & nFields      = scheme%nFields,             &
          & field        = scheme%field                )
  
        ! Symmetric boundaries are defined. Update connectivity for those BC
        if ( nSymBCS > 0 ) then
          ! update connectivity to treat symmetric boundary condition implicitly
          call mus_updateConnectivity_forSymmetricBC(    &
            & neigh        = scheme%pdf(iLevel)%neigh,   &
            & nSize        = scheme%pdf(iLevel)%nSize,   &
            & iLevel       = iLevel,                     &
            & levelDesc    = scheme%levelDesc(iLevel),   &
            & layout       = scheme%layout,              &
            & varSys       = scheme%varSys,              &
            & stateVarMap  = scheme%stateVarMap,         &
            & nBCs         = geometry%boundary%nBCtypes, & 
            & globBC       = scheme%globBC,              &
            & nSymBCs      = nSymBCs,                    &
            & symmetricBCs = symmetricBCs(1:nSymBCs)     ) 
        end if   
      end do 

      ! Dump state neighbor array information to debugUnit
      if( main_debug%dumpAuxLists ) then
        call debug_connectivity( scheme   = scheme,   &
          &                      minLevel = minLevel, &
          &                      maxLevel = maxLevel  )
      end if
    end if

    write(dbgUnit(6),*)  'before init levelBuffers'

    ! Create communication buffers
    ! communicate all scalars read from restart file
    do iLevel = minLevel, maxLevel
      !    for multi-level, still all the links have to be communicated
      !    so, check for regular Buffer, if the max and min level is the same
      write(dbgUnit(1),*) "Level: ", iLevel
      write(dbgUnit(1),*) "Set send and recv buffer for normal Halo elements"
      call init_levelBuffers(                                      &
        & send       = scheme%levelDesc( iLevel )%sendbuffer,      &
        & recv       = scheme%levelDesc( iLevel )%recvbuffer,      &
        & nSize      = scheme%pdf(iLevel)%nSize,                   &
        & pattern    = params%general%commPattern,                 &
        & varSys     = scheme%varSys,                              &
        & stateVarMap = scheme%stateVarMap                         )

      write(dbgUnit(1),*) "Set send and recv buffer for fromCoarser Halo elements"
      call init_levelBuffers(                                            &
        & send       = scheme%levelDesc( iLevel )%sendbufferFromCoarser, &
        & recv       = scheme%levelDesc( iLevel )%recvbufferFromCoarser, &
        & nSize      = scheme%pdf(iLevel)%nSize,                         &
        & pattern    = params%general%commPattern,                       &
        & varSys     = scheme%varSys,                                    &
        & stateVarMap = scheme%stateVarMap                               )

      write(dbgUnit(1),*) "Set MPI buffer for fromFiner Halo elements"
      call init_levelBuffers(                                          &
        & send       = scheme%levelDesc( iLevel )%sendbufferFromFiner, &
        & recv       = scheme%levelDesc( iLevel )%recvbufferFromFiner, &
        & nSize      = scheme%pdf(iLevel)%nSize,                       &
        & pattern    = params%general%commPattern,                     &
        & varSys     = scheme%varSys,                                  &
        & stateVarMap = scheme%stateVarMap                             )
    end do

    call tem_write_debugMesh( globtree  = geometry%tree,           &
      &                       levelDesc = scheme%levelDesc,        &
      &                       myPart    = params%general%proc%rank )

    ! Dump Ghost and Source elements information to debugUnit
    if( main_debug%dumpDependencies) then
      call debug_dependenciesFromFiner(           &
        &    scheme%levelDesc(minLevel:maxLevel), &
        &    minLevel, maxLevel                   )

      call debug_dependenciesFromCoarser(         &
        &    scheme%levelDesc(minLevel:maxLevel), &
        &    minLevel, maxLevel                   )
    end if

    do iLevel = minlevel, maxLevel
      ! set the sendHalo bit for all fluid elements which are send
      ! to remote procs
      call set_sendHaloBits( property = scheme%levelDesc( iLevel )%property,    &
        &                    sendBuffer = scheme%levelDesc( iLevel )%sendBuffer )
    end do

    call tem_stopTimer( timerHandle =  mus_timerHandles%initLvlD )

    call tem_horizontalSpacer( before = 1, fUnit = logUnit(1) )

    write(logUnit(1),"(A,F10.3)") 'Done initializing geometry. Took [s]', &
      &          tem_getTimerVal( timerHandle =  mus_timerHandles%initLvlD)
    call tem_horizontalSpacer( after  = 1, fUnit = logUnit(1) )

    write(dbgUnit(1),"(A)")  'after the construction'

  end subroutine mus_hvs_construct
! ****************************************************************************** !

! ****************************************************************************** !
  !> Initialize the communication buffers for a single level
  !!
  !! Communicate all scalars read from restart file
  subroutine init_levelBuffers( send, recv, nSize, pattern, varSys, stateVarMap )
    ! ---------------------------------------------------------------------------
    !> Communication structure to initialize
    type(tem_communication_type)               :: send, recv
    !> size of state array
    integer, intent(in) :: nSize
    !> communication pattern
    type( tem_commpattern_type ), intent(in)   :: pattern
    !> Variable system
    type( tem_varSys_type ), intent(in)         :: varSys
    type( tem_varMap_type ), intent(in)         :: stateVarMap
    ! ---------------------------------------------------------------------------

    ! bitmask in mus_construct is not used here since we communicate
    ! all scalars read from restart file

    ! only initialize recvBuffers and then communicate the desired
    ! values to the neighbor processes.
    call init_commBuffers(buffer      = recv,   &
      &                   nSize       = nSize,  &
      &                   varSys      = varSys, &
      &                   stateVarMap = stateVarMap, &
      &                   pattern     = pattern )

    call init_commBuffers(buffer      = send,   &
      &                   nSize       = nSize,  &
      &                   varSys      = varSys, &
      &                   stateVarMap = stateVarMap, &
      &                   pattern     = pattern )

  end subroutine init_levelBuffers
! ****************************************************************************** !

! ****************************************************************************** !
  !> Create the communication buffers
  !!
  !! Receive buffers were before created. Now receive the buffers as
  !! send buffers from the remote processes.
  !! Assign the positions, where to get to
  !! Solver specific part to identify the actual data locations
  !! to send use within the communication buffers.
  !!
  subroutine init_commBuffers( buffer, nSize, pattern, varSys, stateVarMap )
    ! ---------------------------------------------------------------------------
    !> send communication buffer
    type(tem_communication_type), intent(inout) :: buffer
    !> size of state array
    integer,             intent(in)             :: nSize
    !> communication pattern
    type(tem_commPattern_type), intent(in)      :: pattern
    !> Variable system
    type( tem_varSys_type ), intent(in)         :: varSys
    type( tem_varMap_type ), intent(in)         :: stateVarMap
    ! ---------------------------------------------------------------------------
    ! Element positions
    integer, allocatable :: pos(:)
    integer :: iElem ! count variable for elements
    integer :: iProc, counter, elemPos
    integer :: iVar, nScalars, varPos, nComp, iComp
    integer :: state_varPos
    ! ---------------------------------------------------------------------------

    nScalars = varSys%nScalars
    allocate( pos(maxval(buffer%nElemsProc*nScalars)))

    ! Then assign the buffer buffer positions
    do iProc = 1, buffer%nProcs
      counter = 0
      do iElem = 1, buffer%nElemsProc( iProc )
        ! Now assign the exact positions in the bufferBuf
        elemPos = buffer%elemPos( iProc )%val( iElem )

        do iVar = 1, stateVarMap%varPos%nVals
          varPos = stateVarMap%varPos%val(iVar)
          nComp = varSys%method%val(varPos)%nComponents
          counter =  counter + nComp
          do iComp = 1, nComp
            state_varPos = varSys%method%val(varPos)%state_varPos(iComp)
            pos(counter) = ?IDX?(state_varPos, elemPos, nScalars, nSize )
          end do ! iComp
        end do !iVar

      end do ! iElem

      call pattern%initBuf_real( buffer%buf_real( iProc ), pos, counter )

    end do ! iProc
    deallocate(pos)

  end subroutine init_commBuffers
! ****************************************************************************** !


end module mus_hvs_construction_module
! ****************************************************************************** !
