! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> This module provides basic information on the parallel environment
module tem_comm_env_module

  !$ use omp_lib

  use mpi

  implicit none

  private

  public :: tem_comm_env_type
  public :: tem_comm_env_init, tem_comm_env_fin
  public :: tem_comm_env_init_empty

  !> Information about parallel runs
  type tem_comm_env_type
    !> size of MPI communicator
    integer :: comm_size
    !> MPI rank
    integer :: rank
    !> MPI root rank
    integer :: root
    !> MPI communicator
    integer :: comm
    !> Maximal Number of OpenMP threads
    integer :: nThreads

    !> Whether this process is the root
    logical :: isRoot

  end type tem_comm_env_type

  contains

! ****************************************************************************** !
  !> Initialize the environment. This routine is called by tem_start
  !! which should be the very first action in a program.
  !!
  subroutine tem_comm_env_init( proc, comm )
    ! --------------------------------------------------------------------------=
    !> The process communicator type
    type( tem_comm_env_type ) :: proc
    !> mpi communicator if it is predefined as in apesmate
    integer, intent(in), optional :: comm
    ! ---------------------------------------------------------------------------
    !> Error flag
    integer :: iError
    ! --------------------------------------------------------------------------

    proc%nThreads = 1
    !$ proc%nThreads = omp_get_max_threads()

    ! Init MPI rank, size and root
    ! if communicator is predefiend and passed use that one
    ! else default to mpi_comm_world
    if(present(comm)) then
      proc%comm = comm
    else
      !KM: with this call proc%comm must be freed by mpi_comm_free
      !so directly setting proc%comm = MPI_COMM_WORLD
      !call mpi_comm_dup( mpi_comm_world, proc%comm, iError )
      proc%comm = mpi_comm_world
    endif

    call mpi_comm_rank( proc%comm, proc%rank, iError )
    call mpi_comm_size( proc%comm, proc%comm_size, iError )
    proc%root = 0

    if ( proc%rank == proc%root ) then
      proc%isRoot = .true.
    else
      proc%isRoot = .false.
    end if

  end subroutine tem_comm_env_init
! ****************************************************************************** !


! ****************************************************************************** !
  !> Initialize a debug environment for just one process without envoking MPI
  !!
  subroutine tem_comm_env_init_empty( proc )
    ! ---------------------------------------------------------------------------
    !> The process communicator type
    type( tem_comm_env_type ) :: proc
    ! ---------------------------------------------------------------------------

    ! Init MPI rank, size and root
    proc%comm = 0
    proc%rank = 0
    proc%comm_size = 0
    proc%root = 0
    proc%isRoot = .false.

  end subroutine tem_comm_env_init_empty
! ****************************************************************************** !


! ****************************************************************************** !
  !> Finalize the environment. This routine is called by tem_finalize
  !! which should be the very last action in a program.
  !!
  subroutine tem_comm_env_fin( proc )
    ! ---------------------------------------------------------------------------
    !> The process communicator type
    type( tem_comm_env_type ) :: proc
    ! ---------------------------------------------------------------------------
    !> Error flag
    integer :: iError
    ! ---------------------------------------------------------------------------

    ! Free the communicator again
    call mpi_comm_free( proc%comm, iError )

  end subroutine tem_comm_env_fin
! ****************************************************************************** !

end module tem_comm_env_module
! ****************************************************************************** !
