#!ruby
# by KUMAGAI Hidetake (ggb03124@nifty.ne.jp)
# modified by YOSHIDA Kazuhiro (moriq@moriq.com)

require 'phi'
require 'rdb.so'

module RDB
  class IndexDefs   ; include Phi::Enumerable ; end
  class DBGridColumns ; include Phi::Enumerable ; end
  class Params      ; include Phi::Enumerable ; end
  class FieldDefs   ; include Phi::Enumerable ; end
  class Fields      ; include Phi::Enumerable ; end
  class SessionList ; include Phi::Enumerable ; end
end

module RDB

class Database

  alias _execute execute
  def execute(sql)
     _execute(sql)
  rescue
    puts "-- SQL is ..#{sql}--"
    raise
  end

  def new_query(sql=nil)
    ret = Query.new
    ret.session_name = self.session_name
    ret.database_name = self.database_name
    if sql
      ret.sql_text = sql
      ret.open
    end
    ret
  rescue
    puts "-- SQL is ..#{ret.sql}--"
    raise
  ensure
    GC.start unless $NO_GC_FOR_INTERBASE
  end
  alias query new_query

  def new_table(name=nil)
    ret = Table.new
    ret.session_name = self.session_name
    ret.database_name = self.database_name
    if name
      ret.table_name = name
      ret.open
    end
    ret
  ensure
    GC.start unless $NO_GC_FOR_INTERBASE
  end
  alias table new_table

  #
  #  transaction{}
  #
  #  wrapper of transaction pattern
  #
  #	2002-01-21: break/throw to commit
  #
  def transaction
    raise "no iterator" unless iterator?
    start_transaction
    begin
      yield
    rescue
      rollback if in_transaction?
      raise
    ensure
      commit if in_transaction?
    end
  end

end # of class Database

module Connection_ex
  @query = nil
  @table = nil

  #
  # This returns internal dataset.
  # attention:
  #   You may not use internal dataset as constant,
  #   and must close just after using it.
  #
  def internal_query
    @query = new_query unless @query
    return @query unless block_given?
    begin
      yield @query
    ensure
      @query.close
      GC.start unless $NO_GC_FOR_INTERBASE
    end
  end

  def internal_table
    @table = new_table unless @table
    return @table unless block_given?
    begin
      yield @table
    ensure
      @table.close
      GC.start unless $NO_GC_FOR_INTERBASE
    end
  end

  #
  # This returns the value of first field of first record.
  # ex:
  #   sum = DB.lookup "SUM(budget)", "department", "head_dept='#{val}'"
  #   sid = DB.lookup "GEN_ID(#{gen_name},1)", "dummy_for_gen", nil
  #
  def lookup( what, from, where, params=[] )
    from  = "'#{from}'" if ( not /^['"]/ =~ from ) and ( /\./ =~ from )
    sql = "SELECT #{what} FROM #{from}"
    sql << " WHERE #{where}" if where
    internal_query{|query|
      query.sql_text = sql
      query.params.each_with_index{|a,i|
        raise "no value of param [:#{a.name}] is designated" if params[i].nil?
        query.params[i] = params[i]
      }
      return nil if (rec = query.open.value[0]).nil?
      return rec[0]
    }
  rescue Exception
    if params && !params.empty?
      raise $!.to_s + "\n-- SQL is ..\n#{sql}\n-- Params is ..\n#{params.inspect}\n--"
    else
      raise $!.to_s + "\n-- SQL is ..\n#{sql}\n--"
    end
  end

  def table_drop(table_name, avoid_error=false)
    execute "DROP TABLE #{table_name}"
  rescue
    raise unless avoid_error
  end
  def table_create(table_name, col_def)
    execute "CREATE TABLE #{table_name} ( #{col_def} )"
  end

  def table_recreate(table_name, col_def)
    table_drop   table_name, true
    table_create table_name, col_def
  end

  #
  # mark:
  #   '*'               all record
  #   'ALL'             all record without NULL
  #   'DISTINCT'        all record distinct without NULL
  #
  def rec_count(from, where=nil, mark='*')
    return lookup( "COUNT(#{mark})" , from, where )
  end

  def rec_exist?(from, where=nil)
    rec_count(from, where) != 0
  end

  def table_exist?(table_name)
    internal_table{ |table|
      table.table_name = table_name
      return table.exist?
    }
  end

  #
  #  This returns Array of Hash, 
  #  pay attention NOT returns dataset.
  #
  def select( what, from, where, params=[] )
    from  = "'#{from}'" if ( not /^['"]/ =~ from ) and ( /\./ =~ from )
    sql = "SELECT #{what} FROM #{from}"
    sql << " WHERE #{where}" if where
    internal_query{|query|
      query.sql_text = sql
      query.params.each_with_index{|a,i|
        raise "no value of param [:#{a.name}] is designated" if params[i].nil?
        query.params[i] = params[i]
      }
      return query.open.values
    }
  rescue
    raise $!.to_s + "\n-- SQL is ..\n#{sql}\n--"
  end

end

#
# Database Extension for ruby
# Use Standard SQL (only), not for specific system.
# This class has contant query and table.
#
class Database
  include Connection_ex
end

class Database_ex < Database
end

class SQLConnection
  include Connection_ex

  #
  #	SQLConnection#internal_query does NOT return SQLClientDataSet, but SQLQuery object.
  #	though SQLConnection#new_query returns SQLClientDataSet object.
  #	DBGrid cannot display SQLQuery object.
  #
  # This returns internal dataset.
  # attention:
  #   You may not use internal dataset as constant,
  #   and must close just after using it.
  # 
  def internal_query
    unless @query
      @query = SQLQuery.new
      @query.sql_connection = self
    end
    return @query unless block_given?
    begin
      yield @query
    ensure
      @query.close
    end
  end

  def execute( sql,params=nil)
    if params
      _execute( sql,params )
    else
      execute_direct( sql )
    end
  end

  def new_query(sql=nil,params=nil)
    ret = SQLClientDataSet.new
    ret.db_connection = self
    ret.command_type = CT_QUERY
    if sql
      ret.sql_text = sql
      ret.open
    end
    return ret
  rescue
    puts "-- SQL is ..#{ret.sql}--"
    raise
  end
  alias query new_query

  def new_table(name=nil)
    ret = SQLClientDataSet.new
    ret.db_connection = self
    ret.command_type = RDB::CT_TABLE
    if name
      ret.command_text = name
      ret.open
    end
    return ret
  end
  alias table new_table

  def table_exist?(table_name)
    nm = table_name.upcase
    table_names(false).each{ |table|
      return true if table.upcase == nm
    }
    return false
  end

end

###########

class DataSet

end

class Fields

  def each
    (0...count).each{|i| yield self[i]}
  end

  def value
    collect{|f| f.value}
  end

  def names
    collect{|f| f.field_name}
  end

  def to_s
    names.join("\t")
  end

  def inspect
    '#<%s:0x%x>' % [self.class.to_s, self.id << 1]
  end
end

module Dataset_ex
end

module CommandTextDataSet
end

class SQLClientDataSet
  include Dataset_ex
  include CommandTextDataSet
end

module Dataset_ex

  def active=(v)
    super
    GC.start unless $NO_GC_FOR_INTERBASE
  end

  def open(sql=nil)
    if sql
      if defined? self.sql_text
        self.sql_text = sql
      else
        raise "sql_text not defined"
      end
    end
    if block_given?
      curr_active = self.active
      self.active = true
      begin
        yield
      ensure
        self.active = curr_active
      end
    else
      self.active = true
    end
    self
  end

  def close
    if block_given?
      curr_active = self.active
      self.active = false
      begin
        yield
      ensure
        self.active = curr_active
      end
    else
      self.active = false
    end
    self
  end

  def locate_ex key_field,key_value,options=[]
    options += [LO_PARTIAL_KEY]
    (1...10).each{
      ret = locate key_field,key_value,options
      return ret if ret
      if key_value.is_a? String
        if key_value.size > 1
          key_value = key_value.chop
        else
          return  ret
        end
      elsif key_value.is_a? Integer
        key_value -= 1
      else
        return ret
      end
    }
    return ret
  end

  include Enumerable

  def each
    return unless active?
    bookmark{
      self.first
      until eof?
        yield fields
        self.next
      end
    }
  end

  def to_s
    return unless active?
    collect{|a| a.collect{|f|
      if f.data_type == RDB::FT_STRING
        '"'+f.to_s+'"'
      else
        f.to_s
      end
    }.join("\t")}.join("\n")
  end

  def to_a
    return unless active?
    collect{|a| a.value}
  end

  def inspect
    '#<%s:0x%x>' % [self.class.to_s, self.id << 1]
  end

  def data_type_to_s(t)
    result = Phi::TYPE_INFO['TFieldType'].enum_name(t)
    return result[2,255] if result
    return "??(%d)" % [t]
  end

  def struct(title=true)
    result = []
    result << ["name", "type", "size"].join("\t") if title
    result+= fields.collect{|f|
      [f.field_name, data_type_to_s(f.data_type), f.size].join("\t")
    }
    result
  end

  def display(out=$defout)
    values = []
    ulines = []
    titles = []
    open{
      fields.each{|f|
        max = [f.field_name.length , f.display_width].max
        max = -max unless f.value.is_a? Numeric
        values.push "%#{max}s"
        ulines.push "-" * max.abs
        titles.push "%#{max}s" % f.field_name
      }
      format = values.join(' ') + "\n"
      titles.join(' ').display(out); "\n".display(out)
      ulines.join(' ').display(out); "\n".display(out)
      each{|r|
        (format % r.value).display(out)
        Phi::APPLICATION.process_messages
      }
    }
    self
  end
  alias list display  # dBASE like

end

class Query
  include Dataset_ex

  alias _execute execute
  def execute(*args)
     _execute(*args)
  rescue
    puts "-- SQL is ..#{sql_text}--"
    puts "-- Params is ..#{params.values.inspect}--"
    raise
  end

end

class Table
  include Dataset_ex
end

class SQLQuery
  include Dataset_ex
end

####

class Query
  def sql_text
    self.sql.text
  end
  def sql_text=(v)
    self.sql.text=v
  end
end

class SQLQuery
  def sql_text
    self.sql.text
  end
  def sql_text=(v)
    self.sql.text=v
  end
end

module CommandTextDataSet
  def sql_text
    raise "command_type != CT_QUERY" if self.command_type != CT_QUERY
    self.command_text
  end
  def sql_text=(v)
    self.command_type = CT_QUERY
    self.command_text=v
  end
  def table_name
    raise "command_type != CT_TABLE" if self.command_type != CT_TABLE
    self.command_text
  end
  def table_name=(v)
    self.command_type == CT_TABLE
    self.command_text = v
  end
end

class SQLDataSet
  include CommandTextDataSet
end

class ClientDataSet
  include Dataset_ex
  include CommandTextDataSet
end



module InterBase
  #
  # counter
  #
  def counter_exist?(counter_name)
    rec_exist? "RDB$GENERATORS", "RDB$GENERATOR_NAME='#{counter_name.upcase}'"
  end

  def counter_create(counter_name, start=0, max=nil)
    execute "CREATE GENERATOR #{counter_name}"
    counter_inc(counter_name,start) if start != 0
  end

  def counter_inc(counter_name, inc=1, avoid_error=false)
    lookup "GEN_ID(#{counter_name},#{inc})", "rdb$database", nil
  end

  def counter_drop(counter_name, avoid_error=false)
    raise "not exist counter '#{counter_name}'" unless avoid_error || counter_exist?(counter_name)
    execute "DELETE FROM RDB$GENERATORS WHERE RDB$GENERATOR_NAME='#{counter_name.upcase}'"
  end

  #
  # view
  #
  def view_exist?(view_name)
    rec_exist? "RDB$VIEW_RELATIONS", "RDB$VIEW_NAME='#{view_name.upcase}'"
  end

  def view_drop(view_name, avoid_error=false)
    ##return if avoid_error and not view_exist? view_name
    execute( "DROP VIEW #{view_name}" )
  rescue
    return if avoid_error and not view_exist? view_name
    raise
  end

  #
  # constraint
  #
  # ĂԂƕʂ̏ꏊ error ?
  # e[u͌B
  #
  def constraint_exist? table_name, constraint_name
    rec_exist? "RDB$RELATION_CONSTRAINTS", <<EOT
                RDB$CONSTRAINT_NAME = '#{constraint_name.upcase}'
            AND RDB$RELATION_NAME = '#{table_name.upcase}'
EOT
  end

  def constraint_drop(table_name, constraint_name, avoid_error=false)
    ##return if avoid_error and not constraint_exist? table_name, constraint_name
    execute "ALTER TABLE #{table_name} DROP CONSTRAINT #{constraint_name}"
  rescue
    return if avoid_error and not constraint_exist? table_name, constraint_name
    raise
  end

  #
  # "ALTER TABLE ADD FOREIGN KEY" fails
  # if DatabaseExplorer uses the database,
  # but returns no report or exception.
  # So, this routine checks fail after adding foreign key.
  #
  def named_foreign_key_add(table_name, constraint_name, key, ref)
    execute <<EOT
        ALTER TABLE #{table_name}
          ADD CONSTRAINT #{constraint_name}
              FOREIGN KEY ( #{key} ) REFERENCES #{ref}
EOT
  end
  alias foreign_key_add named_foreign_key_add

  def table_names( system_table=false )
    if system_table
      sql = "select RDB$RELATION_NAME from RDB$RELATIONS where RDB$SYSTEM_FLAG = 1"
    else
      sql = "select RDB$RELATION_NAME from RDB$RELATIONS where RDB$SYSTEM_FLAG <> 1"
    end
    internal_query{ |query| return query.open( sql ).value.flatten.collect{|i|i.strip} }
  end
  
  def view_names
    sql = "select distinct RDB$VIEW_NAME from RDB$VIEW_RELATIONS "
    internal_query{ |query| return query.open( sql ).value.flatten.collect{|i|i.strip} }
  end
  
  def procedure_names
    sql = "select RDB$PROCEDURE_NAME from RDB$PROCEDURES"
    internal_query{ |query| return query.open( sql ).value.flatten.collect{|i|i.strip} }
  end
  
  def trigger_names
    sql = "select RDB$TRIGGER_NAME from RDB$TRIGGERS where RDB$SYSTEM_FLAG is null"
    internal_query{ |query| return query.open( sql ).value.flatten.collect{|i|i.strip} }
  end
  
  def counter_names
    sql = "select RDB$GENERATOR_NAME from RDB$GENERATORS where RDB$SYSTEM_FLAG is null"
    internal_query{ |query| return query.open( sql ).value.flatten.collect{|i|i.strip} }
  end
  
  def user_names
    sql = "select distinct RDB$USER from RDB$USER_PRIVILEGES"
    internal_query{ |query| return query.open( sql ).value.flatten.collect{|i|i.strip} }
  end
  
end

class Database_ib < Database
  include InterBase
end

module Oracle

  def fix_dummy_for_counter
    unless table_exist? "dummy_for_counter"
      table_create 'dummy_for_counter' , "dmy   INTEGER"
      execute "INSERT INTO dummy_for_counter (dmy) VALUES (0) "
      execute "GRANT SELECT ON TABLE dummy_for_counter TO PUBLIC"
    end
  end
  private :fix_dummy_for_counter

  def counter_create counter_name,start=0,max=nil
    maxvalue = ''
    maxvalue = 'MAXVALUE #{max}' if max
    execute <<EOT
      CREATE SEQUENCE #{counter_name}
        INCREMENT BY 1
        START WITH #{start}
        #{maxvalue}
EOT
    fix_dummy_for_counter
  end

  #def counter_drop counter_name
  #  execute "DROP SEQUENCE #{counter_name}"
  #  raise "please teach Mr.take_tk how to realize 'drop_counter'"
  #end

  def counter_inc counter_name,inc=1
    fix_dummy_for_counter
    if inc == 0
      lookup "#{counter_name}.CURVAL", "dummy_for_counter", nil
    else
      lookup "#{counter_name}.NEXTVAL", "dummy_for_counter", nil
    end
  end

  #def counter_exist? counter_name
  #  raise "please teach Mr.take_tk how to realize 'counter_exist?'"
  #end

end

class Database_ora < Database
  include Oracle
end

######## 2001-08-30()

=begin
  Field#values
    { ږ  => e } ƂnbVԂB
  Fields#values
    { ږ1 => e1, ږ2 => e2, ...} ƂnbVԂB
  DataSet#values
    Fields#values ̔zԂB

  Field#values=(v)
    v (Hash)  { ̍ږ  => e } ΐݒ肷B
  Fields#values=(v)
    ׂĂ̍ڂɂ Field#values=(v) sB
  DataSet#values=(v)
    ()

  return Hash[*(self.collect{|field| [field.field_name, field.data_type]}.flatten)]
  Ƃ@邪..B
=end
  class Field
    def values
      return Hash[field_name, value]
    end
    def values=(hash)
      self.value = hash[field_name] if hash.has_key? field_name
      return hash
    end
    def data_types  # Read Only
      return Hash( self.field_name , self.data_type )
    end
  end

  class Fields
    def values
      hash = {}; each{|field| hash[field.field_name] = field.value     }
      return hash
    end
    def values= hash
      each{|field| field.values = hash }
      return hash
    end
    def data_types
      hash = {}; each{|field| hash[field.field_name] = field.data_type }
      return hash
    end
  end

  class DataSet
    def value
      return self.collect{|fields| fields.value  }
    end
    def values
      return self.collect{|fields| fields.values }
    end
    def data_types
      return fields.data_types
    end

  end

  class Params
=begin
: values
  Param Ƀf[^nbVŐݒ肷B

  Hash Ŏ󂯎
  {ParamName1 => Value1, ParamName2 => Value2, .. }

  nbVɂȂ̂ nil ݒ肷BvӁB
  ÕR[h̃f[^Rs[Ă܂̂h߁B
  cał NULL hȃf[^B

  L[̓p[^łătB[hłȂƂɒӁB

  炩ߌ^w肵ĂȂ
  u 'AITENAME2' ̌^킩܂v
  Əo邱ƂB̏ꍇɂ́AO
  query.params.data_types = dataset.data_types
  Ō^Rs[Ă΂njB
=end
    def values=(hash)
      each{|param| self[param.name] = hash[param.name] }
      return hash
    end
    def values
      hash = {}; each{|param| hash[param.name] = self[param.name] }
      return hash
    end

=begin
: data_types
  Param ̌^ Hash ŃZbgB

  Hash Ŏ󂯎B
  {ParamName1 => DataType1, ParamName2 => DataType2, .. }

  nbVɂȂ͉̂ȂB
  L[̓p[^łătB[hłȂƂɒӁB
=end
    def data_types=(hash)
      self.each{ |param|
        value = hash[param.name]
        self[param.name].data_type = value if value
      }
      return hash
    end
    def data_types
      hash = {} ; each{ |param| hash[param.name] = param.data_type }
      return hash
    end
  end

  class FieldDef
    def data_types
      return Hash( self.field_name , self.data_type )
    end
    def data_types= (hash)
      self.data_type = hash[self.name] if hash.has_key? self.name
      return hash
    end
  end

  class FieldDefs
    def data_types
      hash = {} ; each{ |field_def| hash[field_def.name] = field_def.data_type }
      return hash
    end
    def data_types= (hash)
      each{ |field_def| field_def.data_types = hash }
      return hash
    end
  end

class SessionList
  alias _open_session open_session
  def open_session(*args)
    session = _open_session(*args)
    yield session
  ensure
    session.active = false
  end
end

if __FILE__ == $0
  #
  #  patch for Japanese data
  #
  class String
    def inspect
      '"'+to_s+'"'
    end
  end

  db = Database.new('IBLocal','SYSDBA','masterkey')
  query = db.query('select * from employee')
  table = db.table('employee')
  #db = Database.new('Apollo','apollo','apollo')
  #query = db.query('select * from ap01den_mei')
  #table = db.table('ap01den_mei')

  puts " -- puts query.struct --"
  puts query.struct         #=> field_name,field_type & field_size

  puts " -- puts query.fields --"
  puts query.fields         #=> string of field names with \t
  puts " -- puts query --"
  puts query                #=> big string of value with \t & \n
  puts " -- puts table --"
  puts table                #=> big string of value with \t & \n

  puts " -- p query --"
  p query                   #=>     #<Query:0xcb0c74>
  puts " -- p query.fields --"
  p query.fields            #=>     #<Fields:0x25785b8>
  puts " -- p query.fields.to_a --"
  p query.fields.to_a       #=> array of field

  puts " -- p table --"
  p table                   #=>     #<Table:0x25a3514>
  puts " -- p table.to_a --"
  p table.to_a              #=> array of array of value
end

end # of module RDB
#
# one liner sample
#
# ruby -r phi -r rdb -e "puts RDB::Database.new('IBLocal','SYSDBA','masterkey').table('employee').fields"
# ruby -r phi -r rdb -e "puts RDB::Database.new('IBLocal','SYSDBA','masterkey').table('employee')"
# ruby -r phi -r rdb -e "puts RDB::Database.new('IBLocal','SYSDBA','masterkey').query('select * from employee')"
# ruby -r rdb -e "puts RDB::Database.new('IBLocal').table('employee').struct"
