unit uDatabase;

interface

uses Classes, DBTables, Rubies;

var
  cDatabase: Tvalue;

function ap_cDatabase: Tvalue;
function ap_iDatabase(real: TDatabase; owner: Tvalue): Tvalue;
procedure Init_Database;

implementation

uses SysUtils, DB, uDefUtils, Pythia, uRDB, uDataSet, uSession;

function dl_Database(This: Tvalue): TDatabase;
begin
  result := ap_data_get_struct(This);
end;

function ap_cDatabase: Tvalue;
begin
  result := cDatabase;
end;

procedure Database_setup(obj: Tvalue; real: TDatabase);
begin
  rb_iv_set(obj, '@params', ap_iStrings(real.Params, obj));
end;

procedure Database_free(real: TDatabase); cdecl;
begin
  if real.Connected then real.Close;
  CompoFree(real);
end;

// CompoAlloc modified
function Database_alloc1(This: Tvalue; real: TDatabase): Tvalue;
begin
  if real = nil then begin result := Qnil; exit; end;
  PhiObjectList.Add(real);
  result := rb_data_object_alloc(This, real, nil, @Database_free);
  Database_setup(result, real);
  rb_iv_set(result, '@events', rb_hash_new);
  real.tag := result;
end;

function Database_alloc(This: Tvalue; real: TDatabase): Tvalue;
begin
  result := TmpAlloc(This, real);
  Database_setup(result, real);
end;

function ap_iDatabase(real: TDatabase; owner: Tvalue): Tvalue;
begin
  result := Database_alloc(cDatabase, real);
  ap_owner(result, owner);
end;

function ap_iDatabase_v(var obj; owner: Tvalue): Tvalue;
begin
  result := ap_iDatabase(TDatabase(obj), owner)
end;

function Database_new(argc: integer; argv: Pointer; This: Tvalue): Tvalue; cdecl;
var
  real: TDatabase;
  args: array of Tvalue;
begin
  real := TDatabase.Create(nil);
  if argc > 0 then
  begin
    SetLength(args, argc);
    args := argv;
    try
      real.DatabaseName := dl_String(args[0]);
      if argc > 1 then
      begin
        real.Params.add('user name='+dl_String(args[1]));
        if argc > 2 then
        begin
          real.Params.add('password='+dl_String(args[2]));
          real.LoginPrompt := false;
        end;
      end;
      real.Open;
    except
      on E: Exception do
        ap_raise(ap_eDatabaseError, E.message);
    end;
  end;
  // [ap-list:0960]
  // InterBase ̗΍
  if compareText(real.params.values['user name'], 'politically') = 0 then
    ap_raise(ap_eDatabaseError, 'invalid user');

  result := Database_alloc1(This, real);
  ap_obj_call_init(result, argc, argv);
end;

function Database_execute(This, sql: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    result := INT2FIX(dl_Database(This).Execute(dl_String(sql)));
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_open(This: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    dl_Database(This).Open;
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_close(This: Tvalue): Tvalue; cdecl;
begin
  dl_Database(This).Close;
  result := This;
end;

function Database_close_datasets(This: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    dl_Database(This).CloseDatasets;
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_commit(This: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    dl_Database(This).Commit;
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_rollback(This: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    dl_Database(This).Rollback;
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_start_transaction(This: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    dl_Database(This).StartTransaction;
  except on E: Exception do
    ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_apply_updates(This, v: Tvalue): Tvalue; cdecl;
var
  real: TDatabase;
  DataSet: TDBDataSet;
  Sets: Array of TDBDataSet;
  len, i: Integer;
  ptr: Pvalue;
begin
  result := This;
  real := ap_data_get_struct(This);
  case RTYPE(v) of
  T_ARRAY:
    begin
      len := ap_ary_len(v);
      ptr := ap_ary_ptr(v);
      SetLength(Sets, len);
      for i := 0 to len-1 do
      begin
        ap_data_get_object(ptr^, TDataSet, DataSet);
        Sets[i] := DataSet;
        Inc(ptr);
      end;
    end;
  T_DATA:
    begin
      SetLength(Sets, 1);
      ap_data_get_object(v, TDataSet, DataSet);
      Sets[0] := DataSet;
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;
  try
    real.ApplyUpdates(Sets);
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_get_session(This: Tvalue): Tvalue; cdecl;
var
  real: TDatabase;
  Session: TSession;
  v: Tvalue;
begin
  real := ap_data_get_struct(This);
  Session := real.Session;
  v := Session.Tag;
  if v = 0 then v := ap_iSession(Session, This);
  result := v;
end;

function Database_get_params(This: Tvalue): Tvalue; cdecl;
var
  real: TDatabase;
  item: TStrings;
begin
  real := ap_data_get_struct(This);
//  result := ap_iStrings(real.Params, This);
  result := rb_iv_get(This, '@params');
  item := real.Params;
  if result = Qnil then begin
    result := ap_iStrings(item, This);
    rb_iv_set(This, '@params', result);
  end else begin
    Check_Type(result, T_DATA);
    PRData(result)^.data := item;
  end;
end;

function Database_get_data_sets(This: Tvalue): Tvalue; cdecl;
var
  real: TDatabase;
  i:integer;
begin
  result := rb_ary_new;
  real := ap_data_get_struct(This);
  try
    for i := 0 to real.DataSetCount-1 do
      rb_ary_push(result, DataSetTmpAlloc(real.DataSets[i], This));
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

function Database_get_in_transaction(This: Tvalue): Tvalue; cdecl;
begin
  result := ap_Bool(dl_Database(This).InTransaction);
end;

function Database_get_field_names(This,v: Tvalue): Tvalue; cdecl;
begin
  result := ap_StringList_new;
  dl_Database(This).GetFieldNames(dl_String(v), dl_Strings(result));
end;

function Database_validate_name(This, name: Tvalue): Tvalue; cdecl;
begin
  result := This;
  try
    dl_Database(This).ValidateName(dl_String(name));
  except
    on E: Exception do
      ap_raise(ap_eDatabaseError, E.message);
  end;
end;

type Tvalue_array = array of Tvalue;

function Database_get_table_names(argc: integer; argv:Tvalue_array; This: Tvalue): Tvalue; cdecl;
var
  real: TDatabase;
  SystemTable : boolean;
begin
  result := ap_StringList_new;
  real := ap_data_get_struct(This);
  SystemTable := False; // default
  if argc > 1 then ap_raise(ap_eArgError, sWrong_num_of_args);
  if argc > 0 then SystemTable := dl_Boolean(argv[0]);
  real.GetTableNames( dl_Strings(result), SystemTable );
end;

procedure Init_Database;
begin
  DefineConstSetType(mRDB, TypeInfo(TTransIsolation));

  cDatabase := DefinePersistentClass(mRDB, TDatabase, ap_cPersistent, ap_iDatabase_v);
  DefineSingletonMethod(cDatabase, 'new', Database_new);
  rb_define_method(cDatabase, 'execute', @Database_execute, 1);
  rb_define_method(cDatabase, 'open', @Database_open, 0);
  rb_define_method(cDatabase, 'close', @Database_close, 0);
  rb_define_method(cDatabase, 'close_datasets', @Database_close_datasets, 0);
  rb_define_method(cDatabase, 'commit', @Database_commit, 0);
  rb_define_method(cDatabase, 'rollback', @Database_rollback, 0);
  rb_define_method(cDatabase, 'start_transaction', @Database_start_transaction, 0);
  rb_define_method(cDatabase, 'apply_updates', @Database_apply_updates, 1);
  //rb_define_method(cDatabase, 'flush_schema_cache', @Database_flush_schema_cache, 1);
  //rb_define_method(cDatabase, 'validate_name', @Database_validate_name, 1);
  DefineAttrGet(cDatabase, 'session', Database_get_session);
  rb_define_attr(cDatabase, 'params', 1, 0);
//  DefineAttrGet(cDatabase, 'params', Database_get_params);
  DefineAttrGet(cDatabase, 'data_sets', Database_get_data_sets);
  DefineAttrGet(cDatabase, 'in_transaction?', Database_get_in_transaction);
  rb_define_method(cDatabase, 'get_field_names', @Database_get_field_names, 1);
  rb_define_method(cDatabase, 'validate_name', @Database_validate_name, 1);
  rb_define_method(cDatabase, 'table_names', @Database_get_table_names, -1);
end;

end.
