unit uParam;

interface

uses Rubies, DB;

var
  cParam, cParams: Tvalue;

function ap_iParam(real: TParam; owner: Tvalue): Tvalue;
function ap_iParams(real: TParams; owner: Tvalue): Tvalue;
procedure Init_Param;

implementation

uses SysUtils, Classes, uDefUtils, Pythia, uRDB;

function Param_alloc(This: Tvalue; real: TParam): Tvalue;
begin
  result := TmpAlloc(This, real);
end;

function ap_iParam(real: TParam; owner: Tvalue): Tvalue;
begin
  result := Param_alloc(cParam, real);
  ap_owner(result, owner);
end;

function ap_iParam_v(var obj; owner: Tvalue): Tvalue;
begin
  result := ap_iParam(TParam(obj), owner);
end;

function Param_assign(This, v: Tvalue): Tvalue; cdecl;
var
  real: TParam;
  source: TPersistent;
begin
  real := ap_data_get_struct(This);
  ap_data_get_object(v, TPersistent, source);
  real.Assign(source);
  result := This;
end;

function Param_load(This, v, blob_t: Tvalue): Tvalue; cdecl;
var
  real: TParam;
  stream: TStream;
  BlobType: TBlobType;
begin
  real := ap_data_get_struct(This);
  BlobType := TBlobType(FIX2INT(blob_t));
  case RTYPE(v) of
  T_STRING:
    try
      real.LoadFromFile(dl_String(v), BlobType);
    except
      on E: EFOpenError do
        ap_raise(ap_eIOError, E.message);
    end;
  T_DATA:
    try
      ap_data_get_object(v, TStream, stream);
      real.LoadFromStream(stream, BlobType);
    except
      on E: EReadError do;
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;
  result := This;
end;

function Param_get_type(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(Ord(real.DataType));
end;

function Param_get_to_f(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := rb_float_new(real.AsFloat);
end;

function Param_get_to_i(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := INT2FIX(real.AsInteger);
end;

function Param_get_to_s(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  case real.DataType of
  ftTime:
    result := ap_String(TimeToStr(real.AsTime));
  ftDate, ftDateTime, ftTimeStamp:
    result := ap_String(DateTimeToStr(real.AsDateTime));
  else
    result := ap_String(real.AsString);
  end;
end;

function Param_get_name(This: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  result := rb_str_new2(PChar(real.Name));
end;

function Param_set_as_memo(This,v: Tvalue): Tvalue; cdecl;
var
  real: TParam;
begin
  real := ap_data_get_struct(This);
  real.AsMemo := dl_String(v);
  result := v;
end;

procedure Params_setup(obj: Tvalue; real: TParams);
begin
  Collection_setup(obj, real);
end;

function Params_alloc(This: Tvalue; real: TParams): Tvalue;
begin
  result := TmpAlloc(This, real);
  Params_setup(result, real);
end;

function ap_iParams(real: TParams; owner: Tvalue): Tvalue;
begin
  result := Params_alloc(cParams, real);
  ap_owner(result, owner);
end;

function ap_iParams_v(var obj; owner: Tvalue): Tvalue;
begin
  result := ap_iParams(TParams(obj), owner);
end;

function Params_aref(This, p: Tvalue): Tvalue; cdecl;
var
  real: TParams;
  Param: TParam;
  S: string;
  n: Integer;
begin
  real := ap_data_get_struct(This);
  Param := nil;

  case RTYPE(p) of
  T_STRING:
    begin
      S := dl_String(p);
      Param := real.ParamByName(S);
    end;
  T_FIXNUM:
    begin
      n := FIX2INT(p);
      if (n < 0) or (real.Count <= n) then
        ap_raise(ap_eIndexError, sOut_of_range);
      Param := real[n];
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;

  if Param = nil then
    result := Qnil
  else
    result := Collection_aref(This, ap_Fixnum(Param.Index));
end;

function Params_aset(This, p, v: Tvalue): Tvalue; cdecl;
var
  n: Integer;
  real: TParams;
  Param: TParam;
  klass: Tvalue;
  klass_name: string;
  DateTime: TDateTime;
  kind_p: Tvalue;
begin
  real := ap_data_get_struct(This);
  Param := nil;

  case RTYPE(p) of
  T_STRING:
    try
      Param := real.ParamByName(dl_String(p));
    except
      on E: Exception do
        ap_raise(ap_eDatabaseError, E.message);
    end;
  T_FIXNUM:
    begin
      n := FIX2INT(p);
      if (n < 0) or (real.Count <= n) then
        ap_raise(ap_eIndexError, sOut_of_range);
      Param := real[n];
    end;
  else
    ap_raise(ap_eArgError, sWrong_arg_type);
  end;

  if Param = nil then
    result := Qnil
  else
  begin
    try
      case RTYPE(v) of
      T_NIL   : Param.Clear;
      T_STRING:
        if Param.DataType = ftMemo then
          Param.AsMemo     := dl_String(v)
        else
          Param.AsString   := dl_String(v)
        ;
      T_FIXNUM: Param.AsInteger  := FIX2INT(v);
      T_BIGNUM: Param.AsCurrency := NUM2INT(v);
      T_FLOAT : Param.AsFloat    := NUM2DBL(v);
      T_TRUE  : Param.AsBoolean  := True;
      T_FALSE : Param.AsBoolean  := False;
      T_DATA:
        begin
          klass := CLASS_OF(v);
          klass := rb_class_path(klass);
          klass_name := dl_String(klass);
          kind_p := rb_obj_is_kind_of(v, ap_cDateTime);
          if RTEST(kind_p) then
          begin
            DateTime := PDateTime(ap_data_get_struct(v))^;
            if DateTime < 1 then
              Param.AsTime := DateTime
            else
              Param.AsDateTime := DateTime;
          end
          else
            Param.AsString := 'Unknown '+klass_name+':class';
        end;
      else
        ap_raise(ap_eArgError, sWrong_arg_type);
      end;
    except
      on E: Exception do
        ap_raise(ap_eDatabaseError, E.message);
    end;
    result := ap_iParam(Param, This);
    Collection_aset(This, ap_Fixnum(Param.Index), result);
  end;
end;

procedure Init_Param;
begin
  DefineConstSetType(mRDB, TypeInfo(TParamType));

  cParam := DefinePersistentClass(mRDB, TParam, ap_cCollectionItem, ap_iParam_v);
  rb_define_method(cParam, 'assign', @Param_assign, 1);
  rb_define_method(cParam, 'load', @Param_load, 1); // vcl: load_from_file, load_from_stream
  DefineAttrGet(cParam, 'type', Param_get_type); // vcl: DataType
  DefineAttrGet(cParam, 'to_f', Param_get_to_f); // vcl: AsFloat
  DefineAttrGet(cParam, 'to_i', Param_get_to_i); // vcl: AsInteger
  DefineAttrGet(cParam, 'to_s', Param_get_to_s); // vcl: AsString
  DefineAttrGet(cParam, 'name', Param_get_name);
  DefineAttrSet(cParam, 'as_memo', Param_set_as_memo);

  cParams := DefinePersistentClass(mRDB, TParams, ap_cCollection, ap_iParams_v);
  SetCollectionItemClass(cParams, 'TParam');
  rb_define_method(cParams, '[]', @Params_aref, 1);
  rb_define_method(cParams, '[]=', @Params_aset, 2);
end;

end.
