/*
 * Copyright (c) 2000 Blue Mug, Inc.  All Rights Reserved.
 */

#include <ep7211/ep7211.h>
#include <ep7211/ioregs.h>
#include "keyboard.h"
#include "memmap.h"

/*
 * (diagram from the Cirrus Logic EDB7211 keyboard example software):
 *
 * The keyboard that comes with the CL-PS7211 evaluation board is an 83-key
 * keyboard with the following scan arrangement:
 *
 *           R0  R1  R2  R3  R4  R5  R6  R7  E0  E1  E2  E3  E4  E5  E6   E7
 *        +------------------------------------------------------------------
 *     C0 | esc   1 Tab CpL   `  Sp ArL ArU ArD ArR ShL Ctl  Fn LAlt RAlt ShR
 *     C1 |  F5   6   T   G   B   /   ;   P   - F10
 *     C2 |  F4   5   R   F   V Del   '   [   = Num
 *     C3 |  F3   4   E   D   C Ins       ]  BS Prt
 *     C4 |  F2   3   W   S   X     Ret   \     ScL
 *     C5 |  F1   2   Q   A   Z End PgD PgU  Hm Brk
 *     C6 |  F6   7   Y   H   N   .   L   O   0  F9
 *     C7 |  F7   8   U   J   M   ,   K   I   9  F8
 *
 * Blanks in table are non-existant keys.  The read_keys() function fills
 * in the given bit array left-to-right, top-to-bottom; blanks are included
 * but their value is zero.
 */

#define KBD_COLMASK	((unsigned int) 0xf)
#define IO_EXT_KBD_DATA *((volatile unsigned char*) EXT_KBD_ADDR)

void read_keys(unsigned char *bits)
{
	int col, row;
	volatile int dummy;

	/* stop driving all columns */
	IO_SYSCON1 = (IO_SYSCON1 & ~KBD_COLMASK) | KBSC_LO;

	/* loop through 8 columns */
	for (col=0; col<8; ++col) {

		/* start column drive */
		IO_SYSCON1 = (IO_SYSCON1 & ~KBD_COLMASK) | (KBSC_COL0 + col);

		/* read row */
		*bits++ = IO_PADR;
		*bits++ = IO_EXT_KBD_DATA;

		/* stop column drive */
		IO_SYSCON1 = (IO_SYSCON1 & ~KBD_COLMASK) | KBSC_X;

		/* delay */
		for (row=0; row<32*8; ++row)
			++dummy;
	}

	/* drive all 8 columns again */
	IO_SYSCON1 = (IO_SYSCON1 & ~KBD_COLMASK) | KBSC_HI;
}

int is_any_key_pressed(void)
{
	unsigned char bits [16];
	int i;

	read_keys(bits);
	for (i=0; i<sizeof bits; ++i)
		if (bits[i])
			return 1;
	return 0;
}

