# AsConverter.rb
# SWFからAS3のバイトコードを抽出しAS2に変換するコンバータクラス
# Author::    tmtysk <tmtysk@users.sourceforge.jp>
# Copyright:: Copyright (c) 2007 KLab, Inc.
# License::   Distributes under The BSD License
# Date::      2007/10/19

require 'zlib'

# AsConverter
class AsConverter

  # namespace定数配列
  NAMESPACE = {
    '8'  => 'Namespace',
    '22' => 'PackageNamespace',
    '23' => 'PackageInternalNs',
    '24' => 'ProtectedNamespace',
    '25' => 'ExplicitNamespace',
    '26' => 'StaticProtectedNs',
    '5'  => 'PrivateNs'
  }

  # multiname定数配列
  MULTINAME = {
    '7'  => 'QName',
    '13' => 'QNameA',
    '15' => 'RTQName',
    '16' => 'RTQNameA',
    '17' => 'RTQNameL',
    '18' => 'RTQNameLA',
    '9'  => 'Multiname',
    '14' => 'MultinameA',
    '27' => 'MultinameL',
    '28' => 'MultinameLA'
  }

  # vkind定数配列
  VKIND = {
    '3'  => 'signed_integer',
    '4'  => 'unsigned_integer',
    '6'  => 'double',
    '1'  => 'string',
    '8'  => 'namespace',
    '22' => 'namespace',
    '23' => 'namespace',
    '24' => 'namespace',
    '25' => 'namespace',
    '26' => 'namespace',
    '5'  => 'namespace',
  }
 
  # AbcNameSpace構造体
  AbcNameSpace = Struct.new(
    :kind,
    :name
  )

  # AbcNsSet構造体
  AbcNsSet = Struct.new(
    :nsset
  )

  # AbcMultiname構造体
  AbcMultiname = Struct.new(
    :kind,
    :ns,
    :nsset,
    :name
  )

  # AbcMethod構造体
  AbcMethod = Struct.new(
    # paramは配列扱い
    :param,
    :return_type,
    :name,
    :flags,
    :max_stack,
    :local_count,
    :init_scope_depth,
    :max_scope_depth,
    :code,
    :exception,
    # traitは配列扱い
    :trait
  )

  # AbcInstance構造体
  AbcInstance = Struct.new(
    :name,
    :super_name,
    :flags,
    :protectedNs,
    # interfaceは配列扱い
    :interface,
    :iinit,
    # traitは配列扱い
    :trait
  )

  # AbcClass構造体
  AbcClass = Struct.new(
    :cinit,
    # traitは配列扱い
    :trait
  )

  # AbcScript構造体
  AbcScript = Struct.new(
    :minit,
    # traitは配列扱い
    :trait
  )

  # AbcTrait構造体
  AbcTrait = Struct.new(
    :name,
    :kind,
    :data,
    :metadata
  )

  # コンストラクタ
  def initialize
    # ヘッド位置
    @head = 0
    # method_body中のcode配列
    @opcode = []
    # 逆アセンブル後のコード(テキスト)
    @disassembled_code = ""
    # ABC中のsigned integer定数プール
    @abcint = []
    # ABC中のunsigned integer定数プール
    @abcuint = []
    # ABC中のdouble定数プール
    @abcdouble = []
    # ABC中のstring定数プール
    @abcstring = []
    # ABC中のnamespace定数プール
    @abcnamespace = [AbcNameSpace.new]
    # ABC中のnsset定数プール
    @abcnsset = [AbcNsSet.new]
    # ABC中のmultiname定数プール
    @abcmultiname = [AbcMultiname.new]
    # ABC中のmethod構造体配列
    @abcmethod = [AbcMethod.new([], nil, nil, nil, nil, nil, nil, nil, nil, nil, [])]
    # ABC中のinstance構造体配列
    @abcinstance = [AbcInstance.new(nil, nil, nil, nil, [], nil, [])]
    # ABC中のclass構造体配列
    @abcclass = [AbcClass.new(nil, [])]
    # ABC中のscript構造体配列
    @abcscript = [AbcScript.new(nil, [])]
    # 変換後のAS2で使用する変数名のプール
    @as2_varpool = []
  end

  # readSwfFile
  # 外部SWFファイルを読み込んでインスタンス変数にバイト列を格納する
  # Param:: 読み込むファイル名
  def readSwfFile(filename)
    org = open(filename).read
    swf = org.unpack("B*").to_s
    h = 0
    # ヘッダ部
    ## CWS形式だったらswfを読み込み直す
    if swf[h,8].to_i(2).chr == "C" then
      swf = org[0,8].unpack("B*").to_s
      swf << Zlib::Inflate.inflate(org[8,org.length-8]).unpack("B*").to_s
    end
    ## 最初の64ビットを無視
    h += 64
    ## RECT構造体
    ### 構造体各要素長さ
    rl = swf[h,5].to_i(2)
    ### RECT構造体を無視
    h += 5 + (4*rl)
    ### padding
    h += (8 - h%8)
    ## ヘッダの残り32ビットを無視
    h += 32

    # タグの読み取り開始
    begin
      # タグタイプ＆長さ取得
      tag_and_length = swf[h+8,8] + swf[h,8]
      # EndTagだったら終了
      if (0 == tagid = tag_and_length[0,10].to_i(2)) then break end

      # longタグ対応
      if (63 == l = tag_and_length[10,6].to_i(2)) then
        htemp = h + 16
        lstr = ""
        4.times do
          lstr = swf[htemp,8] + lstr
          htemp += 8
        end
        l = lstr.to_i(2)
        h += 32
      end
      h += 16
      
      # タグIDが0x52(DoABC)であればABCとしてインスタンス変数へ格納
      # それ以外は無視
      # DoABCは1回しか出てこないことを期待
      if tagid == 82 then
        @abc = swf[h,8*l]
        break
      end
      h += (8*l)
    end until h > swf.length
  end

  # readAbcFile
  # 外部ABCファイルを読み込んでインスタンス変数にバイト列を格納する
  # Param:: 読み込むファイル名
  def readAbcFile(filename)
    @abc = open(filename).read.unpack("B*").to_s
  end

  # getNextBit
  # 現在の読み取りヘッドから指定分のビット列(ASCII)を取り出す
  # 読み取り後にヘッドを移動する
  # Param:: 読み取るビット数
  # Return:: 読み取ったビット列
  def getNextBit(b = 1)
    h = @head
    @head += b
    @abc[h, b]
  end

  # getNextByte
  # 現在の読み取りヘッドから指定分のバイト列(ASCII)を取り出す
  # 読み取り後にヘッドを移動する
  # Param:: 読み取るバイト数
  # Return:: 読み取ったバイト列
  def getNextByte(byte = 1)
    h = @head
    b = byte * 8
    @head += b
    @abc[h, b]
  end

  # readSwfUI32
  # 現在の読み取りヘッドからUI32データを取り出す
  # ABCフォーマットには存在しないが、DoABCタグの読み取りに使う
  # Return:: UI32データ(Dec)
  def readSwfUI32
    b = ""
    4.times do
      b = getNextByte + b
    end
    b.to_i(2)
  end

  # readSwfString
  # 現在の読み取りヘッドから有効なStringデータを取り出す
  # (NULL(0x00)に当たるまでのデータをそのまま返す
  # Return:: String
  def readSwfString
    str = ""
    until (b = getNextByte) == "00000000" do
       str += [b].pack("B8")
    end 
    str
  end
  
  # readU8
  # 現在の読み取りヘッドからU8データを取り出す
  # Return:: U8データ(Dec)
  def readU8
    getNextByte.to_i(2)
  end

  # readU16
  # 現在の読み取りヘッドからU16データを取り出す
  # Return:: U16データ(Dec)
  def readU16
    b = ""
    2.times do
      b = getNextByte + b
    end
    b.to_i(2)
  end

  # readS24
  # 現在の読み取りヘッドからS24データを取り出す
  # Return:: S24データ(Dec/負数は2の補数表現)
  def readS24
    b = ""
    i = 0
    3.times do
      b = getNextByte + b
    end
    if b[0].chr == "1" then
      i = - ((b[1,23].to_i(2) ^ ("1"*23).to_i(2)) + 1)
    else
      i = b.to_i(2)
    end
    i
  end

  # readOpcodeS24
  # @opcodeの先頭からS24データを取り出す
  # Return:: S24データ(Dec/負数は2の補数表現)
  def readOpcodeS24
    b = ""
    i = 0
    3.times do
      b = "%08d" % @opcode.shift.to_i.to_s(2) + b
    end
    if b[0].chr == "1" then
      i = - ((b[1,23].to_i(2) ^ ("1"*23).to_i(2)) + 1)
    else
      i = b.to_i(2)
    end
    i
  end

  # readU30
  # 現在の読み取りヘッドからU30データを取り出す
  # 可変長エンコード: 各バイトの先頭ビットが次のバイトの有無を表す
  # 値は各バイト下7ビットで表される
  # 最大35ビットとなるが、上位5ビットを削除して扱う(仕様明記無し)
  # Return:: U30データ(Dec)
  def readU30
    c = 1
    b1 = ""
    b2 = ""
    begin
      b1 = getNextByte
      b2 = b1[1,7] + b2
      c = c + 1
    end until (c > 5) or (b1[0].chr == "0")
    b2 = b2.slice(5,30) if b2.length > 30
    b2.to_i(2)
  end

  # readOpcodeU30
  # @opcodeの先頭からU30データを取り出す
  # 可変長エンコード: 各バイトの先頭ビットが次のバイトの有無を表す
  # 値は各バイト下7ビットで表される
  # 最大35ビットとなるが、上位5ビットを削除して扱う(仕様明記無し)
  # Return:: U30データ(Dec)
  def readOpcodeU30
    c = 1
    b1 = ""
    b2 = ""
    begin
      b1 = "%08d" % @opcode.shift.to_i.to_s(2)
      b2 = b1[1,7] + b2
      c += 1
    end until (c > 5) or (b1[0].chr == "0")
    b2 = b2.slice(5,30) if b2.length > 30
    b2.to_i(2)
  end

  # readU32
  # 現在の読み取りヘッドからU32データを取り出す
  # 可変長エンコード: 各バイトの先頭ビットが次のバイトの有無を表す
  # 値は各バイト下7ビットで表される
  # 最大35ビットとなるが、上位3ビットを削除して扱う(仕様明記無し)
  # Return:: U32データ(Dec)
  def readU32
    c = 1
    b1 = ""
    b2 = ""
    begin
      b1 = getNextByte
      b2 = b1[1,7] + b2
      c += 1
    end until (c > 5) or (b1[0].chr == "0")
    b2 = b2.slice(3,32) if b2.length > 32
    b2.to_i(2)
  end

  # readS32
  # 現在の読み取りヘッドからS32データを取り出す
  # 可変長エンコード: 各バイトの先頭ビットが次のバイトの有無を表す
  # 値は各バイト下7ビットで表される
  # 最大35ビットとなるが、上位3ビットを削除して扱う(仕様明記無し)
  # Return:: U32データ(Dec/負数は2の補数表現)
  def readS32
    c = 1
    i = 0
    b1 = ""
    b2 = ""
    begin
      b1 = getNextByte
      b2 = b1[1,7] + b2
      c += 1
    end until (c > 5) or (b1[0].chr == "0")
    if b2[0].chr == "1" then
      if b2.length > 32 then
        b2 = b2.slice(3,32)
        i = - ((b2.to_i(2) ^ ("1"*32).to_i(2)) + 1)
      else
        i = - ((b2[1,b2.length-1].to_i(2) ^ ("1"*(b2.length-1)).to_i(2)) + 1)
      end
    else
      b2 = b2.slice(3,32) if b2.length > 32
      i = b2.to_i(2)
    end
    i
  end

  # readD64
  # 現在の読み取りヘッドからD64データを取り出す
  # Return:: D64データ(丸めてしまうと処理系毎に値が変わりそうなので
  #          とりあえず現時点ではビット列で返す)
  def readD64
    b = ""
    8.times do
      b = getNextByte + b
    end
    b
  end

  # readString
  # 現在の読み取りヘッドからStringデータを取り出す
  # 1バイトめで文字列長さ(バイト)を表す
  # Return:: Stringデータ
  def readString
    s = ""
    c = getNextByte.to_i(2)
    c.times do
      s << readU8.chr
    end
    s
  end

  # abc
  # 現在読み込み済みのABCビット列を返す
  # Return:: ABCビット列
  def abc
    @abc
  end

  # abc=
  # ABCをBASE64 encoded文字列から読み込む
  # Param:: ABCをBASE64エンコードした文字列
  def abc=(str)
    require 'base64'
    @abc = Base64.decode64(str).unpack("B*").to_s
  end

  # head
  # 現在読み込み済みのABCビット列に対する読み取りヘッド位置を返す
  # Return:: 読み取りヘッド位置
  def head
    @head
  end

  # head=
  # 読み取りヘッド位置を変更する
  # Return:: 指定ヘッド位置
  def head=(h)
    @head = h
  end

  # disassembled_code
  # 逆アセンブルしたコードを返す
  # Return:: 逆アセンブル後のコード
  def disassembled_code
    @disassembled_code
  end

  # disassemble
  # 読み込み済みのABCを逆アセンブルする
  def disassemble
    s = ""
    c = 0
    @head = 0

    # Flags
    s << "DoABC_Flags: " + readSwfUI32.to_s + "\n"
    # ActionName
    s << "DoABC_ActionName: " + readSwfString.to_s + "\n"
    # abcFile
    s << "abcFile:\n"
    ## minor_version
    s << "\tminor_version: " + readU16.to_s + "\n"
    ## major_version
    s << "\tmajor_version: " + readU16.to_s + "\n"
    ## cpool_info
    s << "\tcpool_info:\n"
    ### int_count
    c = readU30
    s << "\t\tint_count: " + c.to_s + "\n"
    ### signed_integer
    (c-1).times do |i|
      s32 = readS32
      s << "\t\t\tsigned_integer[" + (i+1).to_s + "]: " + s32.to_s + "\n"
      @abcint[i+1] = s32
    end
    ### uint_count
    c = readU30
    s << "\t\tuint_count: " + c.to_s + "\n"
    ### unsigned_integer
    (c-1).times do |i|
      u32 = readU32
      s << "\t\t\tunsigned_integer[" + (i+1).to_s + "]: " + u32.to_s + "\n"
      @abcuint[i+1] = u32
    end
    ### double_count
    c = readU30
    s << "\t\tdouble_count: " + c.to_s + "\n"
    ### double_number
    (c-1).times do |i|
      d64 = readD64
      s << "\t\t\tdouble_number[" + (i+1).to_s + "]: " + d64.to_s + "\n"
      @abcdouble[i+1] = d64
    end
    ### string_count
    c = readU30
    s << "\t\tstring_count: " + c.to_s + "\n"
    ### string
    (c-1).times do |i|
      str = readString
      s << "\t\t\tstring[" + (i+1).to_s + "]: \"" + str.to_s + "\"\n"
      @abcstring[i+1] = str.to_s
    end
    ### namespace_count
    c = readU30
    s << "\t\tnamespace_count: " + c.to_s + "\n"
    ### namespace
    (c-1).times do |i|
      u8 = readU8
      u30 = readU30
      s << "\t\t\tnamespace[" + (i+1).to_s + "]: " + NAMESPACE[u8.to_s] + ": string[" + u30.to_s + "]\n"
      @abcnamespace[i+1] = AbcNameSpace.new
      @abcnamespace[i+1].kind = NAMESPACE[u8.to_s]
      @abcnamespace[i+1].name = @abcstring[u30]
    end
    ### ns_set_count
    c = readU30
    s << "\t\tns_set_count: " + c.to_s + "\n"
    ### ns_set
    (c-1).times do |i|
      s << "\t\t\tns_set[" + (i+1).to_s + "]:\n"
      @abcnsset[i+1] = AbcNsSet.new
      cnt = readU30
      cnt.times do
        u30 = readU30
        s << "\t\t\t\tnamespace[" + u30.to_s + "]\n"
        @abcnsset[i+1].nsset.push(@abcnamespace[u30])
      end
    end
    ### multiname_count
    c = readU30
    s << "\t\tmultiname_count: " + c.to_s + "\n"
    ### multiname
    (c-1).times do |i|
      k = readU8
      s << "\t\t\tmultiname[" + (i+1).to_s + "]: " + MULTINAME[k.to_s] + ":\n"
      @abcmultiname[i+1] = AbcMultiname.new
      @abcmultiname[i+1].kind = MULTINAME[k.to_s]
      case k
      when 7, 13
        ns = readU30
        name = readU30
        s << "\t\t\t\tns: namespace[" + ns.to_s + "]\n"
        s << "\t\t\t\tname: string[" + name.to_s + "]\n"
        @abcmultiname[i+1].ns = @abcnamespace[ns]
        @abcmultiname[i+1].name = @abcstring[name]
      when 15, 16
        name = readU30
        s << "\t\t\t\tname: string[" + name.to_s + "]\n"
        @abcmultiname[i+1].name = @abcstring[name]
      when 17, 18
        s << "\t\t\t\t(no data)\n"
      when 9, 14
        name = readU30
        nsset = readU30
        s << "\t\t\t\tname: string[" + name.to_s + "]\n"
        s << "\t\t\t\tns_set: ns_set[" + nsset.to_s + "]\n"
        @abcmultiname[i+1].name = @abcstring[name]
        @abcmultiname[i+1].nsset = @abcnsset[nsset]
      when 27, 28
        nsset = readU30
        s << "\t\t\t\tns_set: ns_set[" + nsset.to_s + "]\n"
        @abcmultiname[i+1].nsset = @abcnsset[nsset]
      end
    end
    ## method_count
    c = readU30
    s << "\tmethod_count: " + c.to_s + "\n"
    ## method
    c.times do |i|
      s << "\tmethod[" + (i+1).to_s + "]:\n"
      @abcmethod[i+1] = AbcMethod.new([], nil, nil, nil, nil, nil, nil, nil, nil, nil, [])
      cnt = readU30
      s << "\t\tparam_count: " + cnt.to_s + "\n"
      return_type = readU30
      s << "\t\treturn_type: multiname[" + return_type.to_s + "]\n"
      @abcmethod[i+1].return_type = @abcmultiname[return_type]
      cnt.times do |j|
        u30 = readU30
        s << "\t\tparam_type[" + (j+1).to_s + "]: multiname[" + u30.to_s + "]\n"
        @abcmethod[i+1].param.push(@abcmultiname[u30])
      end
      nm = readU30
      s << "\t\tname: string[" + nm.to_s + "]\n"
      @abcmethod[i+1].name = @abcstring[nm]
      fl = readU8
      s << "\t\tflags: " + fl.to_s + "\n"
      @abcmethod[i+1].flags = fl.to_s
      if fl & 8 == 8 then
        s << "\t\toptions:\n"
        optc = readU30
        s << "\t\t\toption_count: " + optc.to_s + "\n"
        optc.times do |k|
          s << "\t\t\toption_detail[" + (k+1).to_s + "]\n"
          s << "\t\t\t\tval: " + readU30.to_s + "\n"
          s << "\t\t\t\tkind: " + readU8.to_s + "\n"
        end
      end
      if fl & 128 == 128 then
        s << "\t\tparam_names:\n"
        cnt.times do |j|
          s << "\t\t\tparam_name[" + (j+1).to_s + "]: " + readU30.to_s + "\n"
        end
      end
    end
    ## metadata_count
    c = readU30
    s << "\tmetadata_count: " + c.to_s + "\n"
    ## metadata
    c.times do |i|
      s << "\tmetadata[" + (i+1).to_s + "]:\n"
      s << "\t\tname: string[" + readU30.to_s + "]\n"
      cnt = readU30
      s << "\t\titem_count: " + cnt.to_s + "\n"
      cnt.times do |j|
        s << "\t\titems[" + (j+1).to_s + "]:\n"
        s << "\t\t\tkey: string[" + readU30 + "]\n"
        s << "\t\t\tvalue: string[" + readU30 + "]\n"
      end
    end
    ## class_count
    c = readU30
    s << "\tclass_count: " + c.to_s + "\n"
    ## instance
    c.times do |i|
      s << "\tinstance[" + (i+1).to_s + "]:\n"
      @abcinstance[i+1] = AbcInstance.new(nil, nil, nil, nil, [], nil, [])
      nm = readU30
      snm = readU30
      s << "\t\tname: multiname[" + nm.to_s + "]\n"
      @abcinstance[i+1].name = @abcmultiname[nm]
      s << "\t\tsuper_name: multiname[" + snm.to_s + "]\n"
      @abcinstance[i+1].super_name = @abcmultiname[snm]
      flag = readU8
      s << "\t\tflags: " + flag.to_s + "\n"
      @abcinstance[i+1].flags = flag
      if flag & 8 == 8 then
        u30 = readU30
        s << "\t\tprotectedNs: namespace[" + u30.to_s + "]\n"
        @abcinstance[i+1].protectedNs = @abcnamespace[u30]
      end
      cnt = readU30
      s << "\t\tintrf_count: " + cnt.to_s + "\n"
      cnt.times do |j|
        u30 = readU30
        s << "\t\tinterface[" + (j+1).to_s + "]: multiname[" + u30.to_s + "]\n"
        @abcinstance[i+1].interface.push(@abcmultiname[u30])
      end
      u30 = readU30
      s << "\t\tiinit: method[" + u30.to_s + "]\n"
      @abcinstance[i+1].iinit = @abcmethod[u30]
      trcnt = readU30
      s << "\t\ttrait_count: " + trcnt.to_s + "\n"
      trcnt.times do |k|
        s << "\t\ttrait[" + (k+1).to_s + "]:\n"
        h = readTrait
        s << h["str"]
        @abcinstance[i+1].trait[k+1] = h["data"]
      end
    end
    ## class
    c.times do |i|
      s << "\tclass[" + (i+1).to_s + "]:\n"
      @abcclass[i+1] = AbcClass.new(nil, [])
      u30 = readU30
      s << "\t\tcinit: method[" + u30.to_s + "]\n"
      @abcclass[i+1].cinit = @abcmethod[u30]
      trcnt = readU30
      s << "\t\ttrait_count: " + trcnt.to_s + "\n"
      trcnt.times do |k|
        s << "\t\ttrait[" + (k+1).to_s + "]:\n"
        h = readTrait
        s << h["str"]
        @abcclass[i+1].trait[k+1] = h["data"]
      end
    end
    ## script_count
    c = readU30
    s << "\tscript_count: " + c.to_s + "\n"
    ## script
    c.times do |i|
      s << "\tscript[" + (i+1).to_s + "]:\n"
      @abcscript[i+1] = AbcScript.new(nil, [])
      u30 = readU30
      s << "\t\tinit: method[" + u30.to_s + "]\n"
      @abcscript[i+1].minit = @abcmethod[u30]
      trcnt = readU30
      s << "\t\ttrait_count: " + trcnt.to_s + "\n"
      trcnt.times do |k|
        s << "\t\ttrait[" + (k+1).to_s + "]:\n"
        h = readTrait
        s << h["str"]
        @abcscript[i+1].trait[k+1] = h["data"]
      end
    end
    ## method_body_count
    c = readU30
    s << "\tmethod_body_count: " + c.to_s + "\n"
    ## method_body
    c.times do |i|
      s << "\tmethod_body[" + (i+1).to_s + "]:\n"
      method_id = readU30
      s << "\t\tmethod: method[" + method_id.to_s + "]\n"
      ms = readU30
      s << "\t\tmax_stack: " + ms.to_s + "\n"
      @abcmethod[method_id].max_stack = ms
      lc = readU30
      s << "\t\tlocal_count: " + lc.to_s + "\n"
      @abcmethod[method_id].local_count = lc
      isd = readU30
      s << "\t\tinit_scope_depth: " + isd.to_s + "\n"
      @abcmethod[method_id].init_scope_depth = isd
      msd = readU30
      s << "\t\tmax_scope_depth: " + msd.to_s + "\n"
      @abcmethod[method_id].max_scope_depth = msd
      len = readU30
      s << "\t\tcode_length: " + len.to_s + "\n"
      s << "\t\tcode:\n"
      opcodearr = []
      len.times do
        opcodearr.push readU8.to_s
      end
      opcodes = readInstruct(opcodearr)
      s << opcodes
      @abcmethod[method_id].code = opcodes
      expcnt = readU30
      s << "\t\texception_count: " + expcnt.to_s + "\n"
      expcnt.times do |j|
        s << "\t\texception[" + (j+1).to_s + "]:\n"
        s << "\t\t\tfrom: " + readU30.to_s + "\n"
        s << "\t\t\tto: " + readU30.to_s + "\n"
        s << "\t\t\ttarget: " + readU30.to_s + "\n"
        s << "\t\t\texc_type: string[" + readU30.to_s + "]\n"
        s << "\t\t\tvar_name: string[" + readU30.to_s + "]\n"
      end
      trcnt = readU30
      s << "\t\ttrait_count: " + trcnt.to_s + "\n"
      trcnt.times do |k|
        s << "\t\ttrait[" + (k+1).to_s + "]:\n"
        h = readTrait
        s << h["str"]
        @abcmethod[method_id].trait[k+1] = h["data"]
      end
    end
    @disassembled_code = s
  end

  # readTrait
  # 現在のヘッド位置からTrait構造体をダンプし、
  # ダンプ文字列と構造体をセットで返す
  # Return:: Trait構造体をダンプした文字列と構造体データ
  def readTrait
    s = ""
    trait = AbcTrait.new
    
    name = readU30
    s << "\t\t\tname: multiname[" + name.to_s + "]\n"
    trait.name = @abcmultiname[name].name
    kind = readU8
    s << "\t\t\tkind: " + kind.to_s + "\n"
    trait.kind = kind

    data = []
    case kind & 15
    when 0, 6
      s << "\t\t\ttrait_slot:\n"
      slot_id = readU30
      s << "\t\t\t\tslot_id: " + slot_id.to_s + "\n"
      type_name = readU30
      s << "\t\t\t\ttype_name: multiname[" + type_name.to_s + "]\n"
      vi = readU30
      vk = 0
      if vi != 0 then
        vk = readU8
        s << "\t\t\t\tvindex: " + VKIND[vk.to_s] + "[" + vi.to_s + "]\n"
      end
      trait.data = {
        'slot_id' => slot_id,
        'type_name' => type_name,
        'vindex' => vi,
        'vkind' => vk
      }
    when 4
      s << "\t\t\ttrait_class:\n"
      slot_id = readU30
      s << "\t\t\t\tslot_id: " + slot_id.to_s + "\n"
      classi = readU30
      s << "\t\t\t\tclassi: instance[" + classi.to_s + "]\n"
      trait.data = {
        'slot_id' => slot_id,
        'classi' => @abcinstance[classi]
      }
    when 5
      s << "\t\t\ttrait_function:\n"
      slot_id = readU30
      s << "\t\t\t\tslot_id: " + slot_id.to_s + "\n"
      func = readU30
      s << "\t\t\t\tfunction: method[" + func.to_s + "]\n"
      trait.data = {
        'slot_id' => slot_id,
        'function' => @abcmethod[func]
      }
    when 1, 2, 3
      s << "\t\t\ttrait_method:\n"
      disp_id = readU30
      s << "\t\t\t\tdisp_id: " + disp_id.to_s + "\n"
      meth = readU30
      s << "\t\t\t\tmethod: method[" + meth.to_s + "]\n"
      trait.data = {
        'disp_id' => disp_id,
        'method' => @abcmethod[meth]
      }
    end
    r = {
      "str"  => s,
      "data" => trait
    }
    r
  end

  # readInstruct
  # 入力した opcode 配列を可読なコードに変換する
  # Param:: opcode 配列
  # Return:: 変換後の opcode 文字列
  def readInstruct(opcodearr)
    s = ""
    @opcode = opcodearr
    begin
      o = @opcode.shift
      case o
      when "160"
        s << "\t\t\tadd\n"
      when "197"
        s << "\t\t\tadd_i\n"
      when "134"
        s << "\t\t\tastype\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "135"
        s << "\t\t\tastypelate\n"
      when "168"
        s << "\t\t\tbitand\n"
      when "151"
        s << "\t\t\tbitnot\n"
      when "169"
        s << "\t\t\tbitor\n"
      when "170"
        s << "\t\t\tbitxor\n"
      when "65"
        s << "\t\t\tcall\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "67"
        s << "\t\t\tcallmethod\n"
        s << "\t\t\t\tmethod[" + readOpcodeU30.to_st + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "70"
        s << "\t\t\tcallproperty\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "76"
        s << "\t\t\tcollproplex\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "79"
        s << "\t\t\tcallpropvoid\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "68"
        s << "\t\t\tcallstatic\n"
        s << "\t\t\t\tmethod[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "69"
        s << "\t\t\tcallsuper\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "78"
        s << "\t\t\tcallsupervoid\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "120"
        s << "\t\t\tcheckfilter\n"
      when "128"
        s << "\t\t\tcoerce\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "130"
        s << "\t\t\tcoerce_a\n"
      when "133"
        s << "\t\t\tcoerce_s\n"
      when "66"
        s << "\t\t\tconstruct\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "74"
        s << "\t\t\tconstructprop\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "73"
        s << "\t\t\tconstructsuper\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "118"
        s << "\t\t\tconvert_b\n"
      when "115"
        s << "\t\t\tconvert_i\n"
      when "117"
        s << "\t\t\tconvert_d\n"
      when "119"
        s << "\t\t\tconvert_o\n"
      when "116"
        s << "\t\t\tconvert_u\n"
      when "112"
        s << "\t\t\tconvert_s\n"
      when "239"
        s << "\t\t\tdebug\n"
        s << "\t\t\t\tdebug_type: " + @opcode.shift + "\n"
        s << "\t\t\t\tstring[" + readOpcodeU30.to_s + "]\n"
        s << "\t\t\t\treg: " + @opcode.shift + "\n"
        s << "\t\t\t\textra: " + @opcode.shift + "\n"
      when "241"
        s << "\t\t\tdebugfile\n"
        s << "\t\t\t\tstring[" + readOpcodeU30.to_s + "]\n"
      when "240"
        s << "\t\t\tdebugline\n"
        s << "\t\t\t\tlinenum: " + readOpcodeU30.to_s + "\n"
      when "148"
        s << "\t\t\tdeclocal\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "195"
        s << "\t\t\tdeclocal_i\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "147"
        s << "\t\t\tdecrement\n"
      when "193"
        s << "\t\t\tdecrement_i\n"
      when "106"
        s << "\t\t\tdeleteproperty\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "163"
        s << "\t\t\tdivide\n"
      when "42"
        s << "\t\t\tdup\n"
      when "6"
        s << "\t\t\tdxns\n"
        s << "\t\t\t\tstring[" + readOpcodeU30.to_s + "]\n"
      when "7"
        s << "\t\t\tdxnslate\n"
      when "171"
        s << "\t\t\tequals\n"
      when "114"
        s << "\t\t\tesc_xattr\n"
      when "113"
        s << "\t\t\tesc_xelem\n"
      when "94"
        s << "\t\t\tfindproperty\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "93"
        s << "\t\t\tfindpropstrict\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "89"
        s << "\t\t\tgetdescendants\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "100"
        s << "\t\t\tgetglobalscope\n"
      when "110"
        s << "\t\t\tgetglobalslot\n"
        s << "\t\t\t\tglobalscope[" + readOpcodeU30.to_s + "]\n"
      when "96"
        s << "\t\t\tgetlex\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "98"
        s << "\t\t\tgetlocal\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "208"
        s << "\t\t\tgetlocal_0\n"
      when "209"
        s << "\t\t\tgetlocal_1\n"
      when "210"
        s << "\t\t\tgetlocal_2\n"
      when "211"
        s << "\t\t\tgetlocal_3\n"
      when "102"
        s << "\t\t\tgetproperty\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "101"
        s << "\t\t\tgetscopeobject\n"
        s << "\t\t\t\tscopeobject[" + @opcode.shift + "]\n"
      when "108"
        s << "\t\t\tgetslot\n"
        s << "\t\t\t\tobjectslot[" + readOpcodeU30.to_s + "]\n"
      when "4"
        s << "\t\t\tgetsuper\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "175"
        # graterthan と greaterequals が同じコードになってるので
        # graterthan だけ実装
        s << "\t\t\tgraterthan\n"
      when "31"
        s << "\t\t\thasnext\n"
      when "50"
        s << "\t\t\thasnext2\n"
        s << "\t\t\t\tobject: localregister[" + @opcode.shift + "]\n"
        s << "\t\t\t\tindex: localregister[" + @opcode.shift + "]\n"
      when "19"
        s << "\t\t\tifeq\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "18"
        s << "\t\t\tiffalse\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "24"
        s << "\t\t\tifge\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "23"
        s << "\t\t\tifgt\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "22"
        s << "\t\t\tifle\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "21"
        s << "\t\t\tiflt\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "15"
        s << "\t\t\tifnge\n"
      when "14"
        s << "\t\t\tifngt\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "13"
        s << "\t\t\tifnle\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "12"
        s << "\t\t\tifnlt\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "20"
        s << "\t\t\tifne\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "24"
        s << "\t\t\tifstricteq\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "25"
        s << "\t\t\tifstrictne\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "17"
        s << "\t\t\tiftrue\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "180"
        s << "\t\t\tin\n"
      when "146"
        s << "\t\t\tinclocal\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "194"
        s << "\t\t\tinclocal_i\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "145"
        s << "\t\t\tincrement\n"
      when "192"
        s << "\t\t\tincrement_i\n"
      when "104"
        s << "\t\t\tinitproperty\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "177"
        s << "\t\t\tinstanceof\n"
      when "178"
        s << "\t\t\tistype\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "179"
        s << "\t\t\tistypelate\n"
      when "16"
        s << "\t\t\tjump\n"
        s << "\t\t\t\toffset: " + readOpcodeS24.to_s + "\n"
      when "8"
        s << "\t\t\tkill\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "9"
        s << "\t\t\tlabel\n"
      when "174"
        s << "\t\t\tlessequals\n"
      when "173"
        s << "\t\t\tlessthan\n"
      when "27"
        s << "\t\t\tlookupswitch\n"
        s << "\t\t\t\tdefault_offset: " + readOpcodeS24.to_s + "\n"
        c = readOpcodeU30
        s << "\t\t\t\tcase_count: " + c.to_s + "\n"
        (c+1).times do |i|
          s << "\t\t\tcase_offsets[" + (i+1).to_s + "]: " + readOpcodeS24.to_s + "\n"
        end
      when "165"
        s << "\t\t\tlshift\n"
      when "164"
        s << "\t\t\tmodulo\n"
      when "162"
        s << "\t\t\tmultiply\n"
      when "199"
        s << "\t\t\tmultiply_i\n"
      when "144"
        s << "\t\t\tnegate\n"
      when "196"
        s << "\t\t\tnegate_i\n"
      when "87"
        s << "\t\t\tnewactivation\n"
      when "86"
        s << "\t\t\tnewobject\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "90"
        s << "\t\t\tnewcatch\n"
        s << "\t\t\t\texception[" + readOpcodeU30.to_s + "]\n"
      when "88"
        s << "\t\t\tnewclass\n"
        s << "\t\t\t\tclass[" + readOpcodeU30.to_s + "]\n"
      when "64"
        s << "\t\t\tnewfunction\n"
        s << "\t\t\t\tmethod[" + readOpcodeU30.to_s + "]\n"
      when "85"
        s << "\t\t\tnewobject\n"
        s << "\t\t\t\targ_count: " + readOpcodeU30.to_s + "\n"
      when "30"
        s << "\t\t\tnextname\n"
      when "35"
        s << "\t\t\tnextvalue\n"
      when "2"
        s << "\t\t\tnop\n"
      when "150"
        s << "\t\t\tnot\n"
      when "41"
        s << "\t\t\tpop\n"
      when "29"
        s << "\t\t\tpopscope\n"
      when "36"
        s << "\t\t\tpushbyte\n"
        s << "\t\t\t\tbyte_value: " + @opcode.shift + "\n"
      when "47"
        # 仕様上は 46 となってるけど、それだと pushuint と同じ
        # pushdouble を 47 と見る説が有力
        s << "\t\t\tpushdouble\n"
        s << "\t\t\t\tdouble[" + readOpcodeU30.to_s + "]\n"
      when "39"
        s << "\t\t\tpushfalse\n"
      when "45"
        s << "\t\t\tpushint\n"
        s << "\t\t\t\tsigned_integer[" + readOpcodeU30.to_s + "]\n"
      when "49"
        s << "\t\t\tpushnamespace\n"
        s << "\t\t\t\tnamespace[" + readOpcodeU30.to_s + "]\n"
      when "40"
        s << "\t\t\tpushnan\n"
      when "32"
        s << "\t\t\tpushnull\n"
      when "48"
        s << "\t\t\tpushscope\n"
      when "37"
        s << "\t\t\tpushshort\n"
        s << "\t\t\t\tshort_value: " + readOpcodeU30.to_s + "\n"
      when "44"
        s << "\t\t\tpushstring\n"
        s << "\t\t\t\tstring[" + readOpcodeU30.to_s + "]\n"
      when "38"
        s << "\t\t\tpushtrue\n"
      when "46"
        s << "\t\t\tpushuint\n"
        s << "\t\t\tunsined_integer[" + readOpcodeU30.to_s + "]\n"
      when "33"
        s << "\t\t\tpushundefined\n"
      when "28"
        s << "\t\t\tpushwith\n"
      when "72"
        s << "\t\t\treturnvalue\n"
      when "71"
        s << "\t\t\treturnvoid\n"
      when "166"
        s << "\t\t\trshift\n"
      when "99"
        s << "\t\t\tsetlocal\n"
        s << "\t\t\t\tlocalregister[" + readOpcodeU30.to_s + "]\n"
      when "212"
        s << "\t\t\tsetlocal_0\n"
      when "213"
        s << "\t\t\tsetlocal_1\n"
      when "214"
        s << "\t\t\tsetlocal_2\n"
      when "215"
        s << "\t\t\tsetlocal_3\n"
      when "111"
        s << "\t\t\tsetglobalslot\n"
        s << "\t\t\t\tglobalscope[" + readOpcodeU30.to_s + "]\n"
      when "97"
        s << "\t\t\tsetproperty\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "109"
        s << "\t\t\tsetslot\n"
        s << "\t\t\t\tobjectslot[" + readOpcodeU30.to_s + "]\n"
      when "5"
        s << "\t\t\tsetsuper\n"
        s << "\t\t\t\tmultiname[" + readOpcodeU30.to_s + "]\n"
      when "172"
        s << "\t\t\tstrictequals\n"
      when "161"
        s << "\t\t\tsubtract\n"
      when "198"
        s << "\t\t\tsubtract_i\n"
      when "43"
        s << "\t\t\tswap\n"
      when "3"
        s << "\t\t\tthrow\n"
      when "149"
        s << "\t\t\ttypeof\n"
      when "167"
        s << "\t\t\turshift\n"
      else
        s << "\t\t\tundefined_instruction: " + o.to_s + "\n"
      end
    end until @opcode.empty?
    s
  end

  # toAs2
  # 読み込んだ ABC を AS2 コードに変換する
  # Return:: 変換後の AS2 コード文字列
  def toAs2
    as2 = "class "
    mainclass = ""
    entry_method = []
    # script 配列の init メソッドを確認し、各行を配列へ突っ込む
    @abcscript[@abcscript.length - 1].minit.code.each_line do |l|
      entry_method.push(l.strip)
    end

    # init メソッド中のバイトコードから initproperty を発見し
    # その引数となっているクラスを参照する
    # このクラスがAS2のメインのクラスとなる
    entry_method.each_with_index do |l, i|
      case l
      when "initproperty"
        mainclass = opMultiname(entry_method[i+1]).name
        as2 << mainclass + " {\n\n"
        as2 << "  static function main() {\n"
        as2 << "    var app:" + mainclass + " = new " + mainclass + "(_root);\n"
        as2 << "  }\n\n"
      end
    end

    # initproperty されているオブジェクトの trait 構造体を確認する
    # 定義しておくべきプロパティ(メソッド)を先に評価してしまう
    # さらに、initproperty されているオブジェクトの instance 初期化
    # メソッドを確認する
    # これが mainclass のコンストラクタになる
    constructor = []
    @abcinstance.each do |instance|
      if instance.name != nil and instance.name.name == mainclass then
        # trait の method があれば、このopcodeを配列に突っ込む
        instance.trait.each_with_index do |trait,i|
          if trait != nil then
            case trait.kind & 15
            when 1,2,3
              name = trait.name
              prop = arrayOfOpcode(trait.data["method"])
              as2 << ppOpcode(prop, name)
            end
          end
        end

        # コンストラクタの opcode を配列へつっこむ
        #instance.iinit.code.each_line do |l|
        #  constructor.push(l.strip)
        #end
        constructor = arrayOfOpcode(instance.iinit)

        #as2 << "  function " + mainclass + "() {\n"

        # コンストラクタの処理を汎用メソッドにより変換する
        # コンストラクタは引数 "mc" を持つ
        as2 << ppOpcode(constructor, instance.name.name, ["mc"])
        #as2 << "  }\n"

        # mainclass 終わり
        as2 << "}\n"
      end
    end
    as2
  end 

  # arrayOdOfcode
  # 指定されたメソッドのOpcodeを配列に格納する
  # Param:: Methodの指定子
  # Return:: Opcodeを格納した配列
  def arrayOfOpcode(m)
    a = []
    m.code.each_line do |l|
      a.push(l.strip)
    end
    a
  end

  # ppOpcode
  # 読み込んだ Opcode を AS2 コードに変換する
  # 内部で別のメソッドをコールする場合はそのメソッドの Opcode を読み取り
  # 再帰的に処理する
  # Param:: 変換するOpcode
  # Param:: 変換対象のメソッド名
  # Param:: メソッドに渡される引数
  # Return:: 変換後の AS2 コード文字列
  def ppOpcode(opcode, methodName, args = [])
    # 必要なスタック操作の opcode をピックアップし、そこでのスタックの動きを
    # エミュレーションする
    # AS2 のコードに落とせるパターンをピックアップし、スタックを参照しながら
    # コード変換を行う
    # 例) ステージ上で constructprop されているクラスのコンストラクタメソッドを
    # AS2 の _root レシーバが持つメソッドに対応付けて変換する

    # 変換後のAS2コード
    opcode_as2 = ""

    # operand stack
    st = []

    # scope stack
    sst = ["this"]

    # local register
    lreg = ["this"]

    # 引数のシリアル化
    argstr = ""
    args.each_with_index do |a,i|
      argstr << a
      argstr << ", " if i < args.length-1
    end

    # エントリポイントのコンストラクタのときだけは
    # コンテキストを mc 引数にしてやる必要があるぽい
    # それ以外のときは、まだ考えてないけど
    # method 構造体から引数を取得して変数を生成することになると思う
    argstr == "mc" ? ctxt = "mc" : ctxt = "hoge"

    opcode_as2 << "  function " + methodName + "(" + argstr + ") {\n"
    # TODO: ↑の参照方法だと汎用性に欠ける

    opcode.each_with_index do |op, i|
      case op
      when "getlocal_0"
        st.push(lreg[0])
      when "setlocal_2"
        lreg[2] = st.pop
      when "pushscope"
        sst.push(st.pop)
      when "newactivation"
        # 今のところは、これが何なのかよくわかってない
        st.push("new_activation_object")
      when "dup"
        v = st.pop
        st.push(v)
        st.push(v)
      when "getscopeobject"
        
      when "constructprop"
        case opMultiname(opcode[i+1]).name
        when "TextField"
          v = genVarname
          opcode_as2 << "    var " + v + " = " + ctxt + ".createTextField(\"" + v + "\", " + ctxt + ".getNextHighestDepth(), 0, Stage.height/2, Stage.width, Stage.height/2);\n"
          st.push(ctxt + "." + v)
        end
      when "coerce"
        v = st.pop
        # v を opMultiname(opcode[i+1]).name へキャストする処理？
        st.push(v)
      when "pushstring"
        st.push("\"" + opString(opcode[i+1]) + "\"")
      when "setproperty"
        prop = st.pop
        obj = st.pop
        opcode_as2 << "    " + obj + "." + opMultiname(opcode[i+1]).name + " = " + prop + ";\n"
      when "callproperty"
        args = []
        opArgCount(opcode[i+2]).times do
          args.push(st.pop)
        end
        st.push(opMultiname(opcode[i+1]).name + "()")
      when "returnvalue"
        opcode_as2 << "    return " + st.pop + ";\n"
      end
    end
    opcode_as2 << "  }\n\n"
    opcode_as2
  end

  # genVarname
  # AS2で使用する変数名を生成し、変数名配列へストアする
  # Return:: 生成された変数名
  def genVarname
    @as2_varpool.push("as2var_aaaaaa") if @as2_varpool.length == 0
    @as2_varpool.push(@as2_varpool[@as2_varpool.length - 1].succ)
    @as2_varpool[@as2_varpool.length - 1]
  end

  # opMultiname
  # opcode中で参照されるmultiname定数を取得する
  # Return:: 参照されているAbcMultiname
  def opMultiname(mn)
    @abcmultiname[(mn.gsub(/multiname\[(\d+)\]/) { $1 }).to_i]
  end

  # opString
  # opcode中で参照されるstring定数(文字列)を取得する
  # Return:: 参照されている文字列
  def opString(str)
    @abcstring[(str.gsub(/string\[(\d+)\]/) { $1 }).to_i]
  end

  # opArgCount
  # opcodeが伴う引数の数を取得する
  # Return:: 引数の数
  def opArgCount(str)
    (str.gsub(/arg_count: (\d+)/) { $1 }).to_i
  end

end
