; -*- coding: utf-8 -*-
(ns atena.core
  (:use ;[clojure.contrib.trace]
        [clojure.contrib.def]
        [clojure.contrib.duck-streams :only (reader with-out-writer)]
        )
  (:require [atena.layouts.postcard]
            [clojure.contrib.string :as string]
            )
  (:import (java.io FileOutputStream IOException)
           (java.util ArrayList)
           (com.itextpdf.text Rectangle Document Paragraph PageSize)
           (com.itextpdf.text.pdf PdfWriter BaseFont PdfContentByte)))

;clojure.contrib.string内に定義されている関数は、
;clojure.core内に定義されている一部の関数と競合するため、
;そのままではuseできない。
;そのため、requireを使用し、かつ別名をつける。
;clojure.contrib.string内の関数(例えばsplit)を使用する場合は、
;(string/split #"," "hoge fuga moga")のように記述する。

;--------------------------------------------------
;汎用手続き
;--------------------------------------------------
(defn get-longest-string
  "文字列リスト中から最大の長さの文字列を返却"
  [strings]
  (reduce (fn [v1 v2]
            (let [s1 (str v1)
                  s2 (str v2)]
              (if (> (count s1) (count s2))
                  s1
                  s2)))
          strings))

(defn iota
  "連番生成"
  [cnt start step]
  (loop [ans '()
         i cnt
         num start]
    (if (= i 0)
        (reverse ans)
        (recur (cons num ans) (- i 1) (+ num step)))))

(defn get-font-size
  "フォントサイズの取得"
  [default-size string-height string-count]
  (if (= string-count 0)
      default-size
      (let [fsize (quot string-height string-count)]
        (if (< default-size fsize)
	    default-size
	    fsize))))


(def zen-han-map {\0 \０, \1 \１, \2 \２, \3 \３, \4 \４, \5 \５, \6 \６, \7 \７, \8 \８, \9 \９, \- \ー})

(defn to-hankaku-number-str
  "全角数字を半角数字に変換"
  [number-str]
  (apply str (replace zen-han-map number-str)))

(defn print-item!
  "PDF出力手続き"
  [cb-tmp font item]
  (let [value (to-hankaku-number-str (item :value))]
    (doto cb-tmp
      (.setFontAndSize font (item :fsize))
      (.setTextMatrix (item :left) (item :top))
      (.showText value))))

;--------------------------------------------------
;★年賀状用★PDF出力パラメータを生成
;--------------------------------------------------
(defn item-creator
  "出力データを受け取り、PDF出力用パラメータを生成"
  [page
   settings]
  ;------------------------------
  (defn field-creator
    "フィールド生成"
    [fsize top left value]
    {:fsize fsize :top top :left left :value value})

  ;------------------------------
  (defn zip-code-field-creator
    "郵便番号"
    [zip-code settings]
    (let [zcode-vec (vec (map str zip-code))]
      (map (fn [left zcode]
             (field-creator (settings :font-size)
                            (settings :top-base)
                            left
                            zcode))
           (settings :lefts)
           zcode-vec)))

  ;------------------------------
  (defn name-field-creator
    "名称エリア"
    [family-name
     first-names-string
     honer
     top-base-fn
     left-base-fn
     settings]
    (let [first-names (vec (string/split #"," first-names-string))
          first-names-cnt (count first-names)
          family-name-len (count family-name)
          first-name-max-len (count (get-longest-string first-names))
          honer-len (count honer)
          all-name-len (+ (if (= 0 honer-len) 1 2)
                          family-name-len
                          first-name-max-len
                          honer-len)
          fsize (get-font-size (settings :font-size)
                               (settings :height)
                               all-name-len)
          top-base (top-base-fn fsize all-name-len)
          first-names-top (- top-base
                             (* fsize
                                (if (= 0 family-name-len)
                                    0
                                    (+ 1 family-name-len))))
          honer-top (- first-names-top
                       (* fsize
                          (+ 1 first-name-max-len)))
          left-base (left-base-fn fsize first-names-cnt)
          lefts (iota first-names-cnt left-base (- 0 fsize))
          fsizes (repeat first-names-cnt fsize)
         ]
      (cons ;宛先姓
            (field-creator fsize top-base left-base family-name)
            (concat 
              ;宛先名（複数）
              (map field-creator
                   fsizes
                   (repeat first-names-cnt first-names-top)
                   lefts
                   first-names)
              ;敬称（複数）
              (map field-creator
                   fsizes
                   (repeat first-names-cnt honer-top)
                   lefts
                   (repeat first-names-cnt honer))))))

  ;------------------------------
  (defn get-address-vec
    "住所を半角スペースで分離し、先頭を全角スペース補正した住所をベクタ化する"
    [address]
    (let [fields (string/split #" " address)]
      (map (fn [adrs index]
             (str (apply str (repeat index "　")) adrs))
	   fields
           (range 0 (count fields)))))

  ;------------------------------
  (defn address-field-creator
    "住所エリア"
    [address
     top-base-fn
     left-base-fn
     settings
    ]
    (let [address-vec (get-address-vec address)
          address-cnt (count address-vec)
          fsize (get-font-size (settings :font-size)
                               (settings :height)
                               (count (get-longest-string address-vec)))
          top-base (top-base-fn fsize (apply max (map count address-vec)))
          left-base (left-base-fn fsize address-cnt)
          lefts (iota address-cnt left-base (- 0 fsize))
          fsizes (repeat address-cnt fsize)
         ]
      (map field-creator
           fsizes
           (repeat address-cnt top-base)
           lefts
           address-vec)))

  ;おそらく設定に持たせるべきもの。
  (let [;宛先人データ
        to-d (page :to)
        ;差出人データ
        from-d (if (= nil (page :from))
                   {:family-name "" :first-names "" :honer "" :address ""}
                   (page :from))
        ;宛先人設定
        to-s (settings :to)
        ;差出人設定
        from-s (settings :from)
        ;宛先郵便番号
        to-zcode-fields (zip-code-field-creator (to-d :zcode)
                                                (to-s :zcode))
        ;宛先人名称
        to-name-fields (name-field-creator (to-d :family-name)
                                           (to-d :first-names)
                                           (to-d :honer)
                                           (fn [fsize all-name-len]
                                             (-> to-s :name :top-base))
                                           (fn [fsize first-names-cnt]
                                             (+ (-> to-s :name :center-base)
                                                (* (quot fsize 2)
                                                   (- first-names-cnt 1))))
                                           (to-s :name))
        ;宛先人住所
        to-address-fields (address-field-creator (to-d :address)
                                                 (fn [fsize max-len]
                                                   (-> to-s :address :top-base))
                                                 (fn [fsize column]
                                                   (-> to-s :address :left-base))
                                                 (to-s :address))
        ;差出人郵便番号
        from-zcode-fields (zip-code-field-creator (from-d :zcode)
                                                  (from-s :zcode))
        ;差出人名称
        from-name-fields (name-field-creator (from-d :family-name)
                                             (from-d :first-names)
                                             ""
                                             (fn [fsize all-name-len]
                                               (+ (-> from-s :name :bottom-base)
                                                  (* fsize all-name-len)))
                                             (fn [fsize first-names-cnt]
                                               (+ (-> from-s :name :left-base)
                                                  (* fsize first-names-cnt)))
                                             (from-s :name))
        ;差出人住所
        from-address-fields (address-field-creator (from-d :address)
                                                   (fn [fsize max-len]
                                                     (+ (-> from-s :address :bottom-base)
                                                        (* fsize max-len)))
                                                   (fn [fsize column]
                                                     (let [base-field (first from-name-fields)]
                                                       (+ (base-field :left)
                                                          (* fsize (+ column 1)))))
                                                   (from-s :address))]
    ;全てのフィールドパラメータを連結
    (concat to-zcode-fields
            to-name-fields
            to-address-fields
            from-zcode-fields
            from-name-fields
            from-address-fields)))

;１エントリをPDF出力
(defn print-pages
      [from
       to-vec
       settings]
  (if (= 0 (count to-vec))
      0
      (let [
            ;PDF出力関連
            document (Document. (. PageSize POSTCARD) 0 0 0 0)
            writer (PdfWriter/getInstance document (FileOutputStream. "atena.pdf"))
            hmv (BaseFont/createFont "HeiseiMin-W3" "UniJIS-UCS2-V" BaseFont/EMBEDDED)
            hmh (BaseFont/createFont "HeiseiMin-W3" "UniJIS-UCS2-H" BaseFont/EMBEDDED)
           ]
        (.. document open)
        (doseq [to to-vec]
          (let [cb (.. writer getDirectContent)]
            (.. cb beginText)
            (doseq [item (item-creator {:to to :from from} settings)]
              (print-item! cb hmv item))
            (.. cb endText)
            (.. document newPage)))
        (.. document close)
        (count to-vec))))
