(ns atena.gui.MainFrame
  (:import (java.awt BorderLayout GridLayout GridBagLayout Insets
                     Font Color Dimension FlowLayout)
           (java.awt.event  ActionListener)
           (java.io File)
           (java.util UUID Calendar Date)
           (javax.swing JPanel JLabel JTextField JButton JCheckBox JOptionPane JScrollPane JFrame JSplitPane JFileChooser)
           (javax.swing.border TitledBorder LineBorder)
           (javax.swing.event DocumentListener)
           (javax.swing.table DefaultTableCellRenderer TableColumn
                              TableCellRenderer DefaultTableModel
                              DefaultTableColumnModel TableRowSorter)
           (javax.swing.filechooser FileNameExtensionFilter)
           (atena.gui Utility
                      FromInfoPanel
                      AddressPanel
                      DetailInfoPanel
                      Config
                      )
           )
  (:use [clojure.contrib.duck-streams :only (reader read-lines writer write-lines with-out-writer)])
  (:require [atena.utils.gridbaglayout]
            [atena.core]
            [atena.layouts.postcard]
            [clojure.contrib.string :as string])
  (:gen-class
   :extends javax.swing.JFrame
   :init init
   :post-init after-ctor
   :state state
   ))

(defn -init []
  [[]
   (ref {:address-book       (ref nil)
         :open-button        (JButton. "住所録を開く")
         :save-button        (JButton. "保存")
         :add-button         (JButton. "追加")
         :print-button       (JButton. "全印刷")
         :assign-from-button (JButton. "差出人設定")
         :from-info-panel    (atena.gui.FromInfoPanel.)
         :address-panel      (atena.gui.AddressPanel.)
         :detail-info-panel  (atena.gui.DetailInfoPanel.)
        })])

(defn getMFile
  ;汎用ファイルオープン手続き
  ;オープン用ダイアログ、セーブ用ダイアログの両方に対応。
  ;取得したファイルを返す。（指定したファイルが存在していなければ生成する。）
  [target-file
   message
   extention
   title
   open?]
  (def chooser
    (let [path (if (= nil target-file)
                   (atena.gui.Config/last-file-path)
                   target-file)]
      (doto (JFileChooser. path)
        (.setFileFilter (FileNameExtensionFilter. message (into-array [extention])))
        (.setDialogTitle title))))
  (defn get_file []
    (if (= JFileChooser/APPROVE_OPTION (if open?
                                           (.showOpenDialog chooser nil)
                                           (.showSaveDialog chooser nil)))
        (do
          (def file (.getSelectedFile chooser))
          (atena.gui.Config/last-file-path (.getAbsolutePath file))
          (if (.exists file)
              file
              (do
                (.createNewFile file)
                file)))))
  (get_file))

(defn getAddressBookFile
  ;宛先ファイル取得手続き
  [this]
  (let [{:keys [address-book
               ]} @(.state this)]
    (if (= nil @address-book)
        (do
          (def f (getMFile @address-book "宛先情報ファイル(*.adrs)" "adrs" "住所録ファイル指定" true))
          (dosync (ref-set address-book f))
          @address-book)
        @address-book)))

(defn readData
  ;ファイル読み込み手続き
  [file-path
   make-entry-from-vec]
  ; 昨年送付したかどうかの確認のため、↓このロジックは残しておく。
  ; (read (java.io.PushbackReader. (reader file-path)))
  (map (fn [line]
         (let [fields (vec (string/split #"," line))]
           (make-entry-from-vec fields)))
       (read-lines file-path)))

(defn readAddressData
  ;ファイル読み込み手続き
  [file-path]
  (readData file-path
            (fn [vec-fields]
              {:uuid        (vec-fields 0)
               :zcode       (vec-fields 1)
               :address     (vec-fields 2)
               :honer       (vec-fields 3)
               :family-name (vec-fields 4)
               :first-names (string/join "," (subvec vec-fields 5))})))

(defn readMochuHistoryData
  ;喪中履歴ファイル読み込み手続き
  [file-path]
  (readData file-path
            (fn [vec-fields]
              {:uuid  (vec-fields 0)
               :years (subvec vec-fields 1)})))

(defn saveData
  ;ファイル保存手続き
  [filePath data]
  ;(spit filePath data)
  (write-lines (writer (str filePath))
               (map (fn [item]
                      (string/join ","
                                   (list (item :uuid)
                                         (item :zcode)
                                         (item :address)
                                         (item :honer)
                                         (item :family-name)
                                         (item :first-names))))
                    data))
  ; 喪中履歴情報　　（宛先別ファイル）
  ; 送付受取履歴情報（差出人別ファイル）
  ; タグ情報　　　　（宛先別ファイル）
  )

(defn -after-ctor [this]
  (let [{:keys [open-button
                save-button
                add-button
                print-button
                assign-from-button
                from-info-panel
                address-panel
                detail-info-panel
               ]} @(.state this)
        frame this]
    ;ボタンパネル
    (def buttons-panel (doto (JPanel. (FlowLayout. FlowLayout/LEFT))
                         (.add open-button)
                         (.add save-button)
                         (.add add-button)
                         (.add print-button)
                         (.add assign-from-button)
                         ))
    ;左ペインのパネル
    (def left-panel (doto (JPanel. (GridBagLayout.))
                      (atena.utils.gridbaglayout/grid-bag-layout
                        :insets (Insets. 5 5 5 5)
                        :fill :BOTH
                        :weightx 1.0
                        :gridx 0, :gridy 0
                        buttons-panel
                        :gridx 0, :gridy 1
                        from-info-panel
                        :gridx 0, :gridy 2
                        :weighty 1.0
                        address-panel)))
    ;右ペインのパネル
    (def right-panel (doto (JPanel. (GridBagLayout.))
                       (atena.utils.gridbaglayout/grid-bag-layout
                        :insets (Insets. 5 5 5 5)
                        :fill :BOTH
                        :weightx 1.0
                        :weighty 1.0
                        :gridx 0, :gridy 0
                        detail-info-panel)))
    ;フレーム
    (doto frame
      (.setTitle "宛名印刷")
      (.setDefaultCloseOperation JFrame/EXIT_ON_CLOSE)
      (.setSize 800 600)
      (.setLocationRelativeTo nil)
      (.add (doto (JSplitPane.)
              (.setDividerLocation 500)
              (.setLeftComponent left-panel)
              (.setRightComponent right-panel))
            BorderLayout/CENTER))

    ;宛先情報ファイルオープンボタンイベント登録
    (doto open-button
      (.addActionListener
        (proxy [ActionListener] []
          (actionPerformed [evt]
            (let [file (getAddressBookFile frame)]
              (if (not (= nil file))
                  (let [path (.getAbsolutePath file)
                        data (readAddressData path)
                        mrng (str (. path substring 0 (. path lastIndexOf ".")) ".mrng")]
                    (. address-panel setData data)
                    (. detail-info-panel setMourningsFile mrng))))))))

    (doto save-button
      (.addActionListener
        (atena.gui.Utility/make-confirm-ActionListener
          "保存確認"
          "保存しますか？"
          (fn []
            (saveData (.getAbsolutePath (getAddressBookFile frame))
                      (. address-panel getAllData))))))

    (doto print-button
      (.addActionListener
        (atena.gui.Utility/make-confirm-ActionListener
          "PDF出力確認"
          "選択されている行の宛先をPDF出力しますか？"
          (fn []
            (atena.core/print-pages (. from-info-panel getData)
                                    (. address-panel getSelectedData)
                                    (atena.layout.postcard/get-page-settings))))))

    (doto add-button
      (.addActionListener
        (proxy [ActionListener] []
          (actionPerformed [evt]
            (. detail-info-panel addData)))))

    (doto assign-from-button
      (.addActionListener
        (atena.gui.Utility/make-confirm-ActionListener
          "差出人設定確認"
          "差出人設定しますか？"
          (fn []
            (let [file (getAddressBookFile frame)]
              (if (not (= nil file))
                  (let [data (. detail-info-panel getData)
                        hsty (File. (. file getParent) (str (data :uuid) ".hsty"))]
                    (. from-info-panel setData data)
                    (. detail-info-panel setHistoriesFile hsty))))))))

    (doto address-panel
      (.addSelectionChangedProc
        (fn [table]
          (let [rows (.getSelectedData table)]
            (if (= 1 (count rows))
                (let [file  (. (getAddressBookFile frame) getAbsolutePath)
                      mfile (str (. file substring 0 (. file lastIndexOf ".")) ".mrng")]
                  (doto detail-info-panel
                    (.setMourningsFile mfile)
                    (.setData (first rows))
                    )))))))

    (doto detail-info-panel
      (.addSaveButtonListener
        (atena.gui.Utility/make-confirm-ActionListener
          "保存"
          "保存しますか？"
          (fn []
            (let [uuid (let [value (. detail-info-panel uuid)]
                         (if (or (= nil value) (= "" value))
                             (. detail-info-panel createUuid)
                             value))]
              ;table（というかmodel）を更新
              (. address-panel update uuid (. detail-info-panel getData))
              (saveData (.getAbsolutePath (getAddressBookFile frame))
                        (. address-panel getAllData))))))
      (.addDeleteButtonListener
        (atena.gui.Utility/make-confirm-ActionListener
          "削除"
          "削除しますか？"
          (fn []
            (let [uuid (. detail-info-panel uuid)]
              (. address-panel remove uuid)
              (saveData (.getAbsolutePath (getAddressBookFile frame))
                        (. address-panel getAllData)))))))
))




